﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/bluetooth.h>
#include <nn/btm/debug/btm_DebugApi.h>
#include "../Common/BluetoothTools_Color.h"
#include "../Common/BluetoothTools_Utility.h"
#include "BleTestTool_GattClientScene.h"
#include "BleTestTool_Uuid.h"


namespace {
    /**
     * @brief       GATT Characteristic のProperty を表す構造体です
     */
    struct GattProperty
    {
        nn::bluetooth::GattAttributeProperty property;      //!< Property の値です
        std::string name;                                   //!< Property を文字列で表現する際の値です
    };

    static const GattProperty GATT_PROPERTY_LIST[] =
    {
        { nn::bluetooth::GattAttributeProperty::GattAttributeProperty_Broadcast,            "Broadcast" },
        { nn::bluetooth::GattAttributeProperty::GattAttributeProperty_Read,                 "Read" },
        { nn::bluetooth::GattAttributeProperty::GattAttributeProperty_WriteWithoutResponse, "Write without response" },
        { nn::bluetooth::GattAttributeProperty::GattAttributeProperty_Write,                "Write" },
        { nn::bluetooth::GattAttributeProperty::GattAttributeProperty_Notify,               "Notification" },
        { nn::bluetooth::GattAttributeProperty::GattAttributeProperty_Indicate,             "Indication" },
        { nn::bluetooth::GattAttributeProperty::GattAttributeProperty_SignedWriteCommand,   "Signed Write without response" },
        { nn::bluetooth::GattAttributeProperty::GattAttributeProperty_ExtendedProperty,     "Extended Property" }
    };

    /**
     * @brief       GATT Characteristic のProperty を文字列として取得します
     */
    const std::string ParseCharacteristicProperty(const nn::bluetooth::GattCharacteristic &characteristic)
    {
        uint8_t property = characteristic.GetProperties();

        std::string propertyString = "";

        for (int propertyIndex = 0; propertyIndex < sizeof(GATT_PROPERTY_LIST) / sizeof(GattProperty); ++propertyIndex)
        {
            if (property & GATT_PROPERTY_LIST[propertyIndex].property)
            {
                propertyString += (propertyString.length() == 0) ? GATT_PROPERTY_LIST[propertyIndex].name : " " + GATT_PROPERTY_LIST[propertyIndex].name;
            }
        }

        NN_ASSERT_NOT_EQUAL(propertyString.length(), 0);

        return propertyString;
    }

    //!< GATT Attribute へのWrite 操作時に書き込むデータです
    const uint8_t writeData[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
        0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F,
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F,
        0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F,
        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F,
        0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F,
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F,
        0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF,
        0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF,
        0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF,
        0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF,
        0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF,
        0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
        0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F,
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F,
        0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F,
        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F,
        0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F,
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F,
        0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF,
        0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF,
        0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF,
        0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF,
        0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF,
        0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF,
    };
}   // namespace

GattClientScene::GattClientScene(std::string title, uint8_t connIndex) : SceneBase(title)
{
    NN_ASSERT_RANGE(connIndex, 0, nn::bluetooth::BleConnectionCountMaxClient);

    m_ConnIndex = connIndex;

    memset(&m_ConnInfo.address, 0x00, sizeof(nn::bluetooth::Address));
    m_ConnInfo.connectionHandle = nn::bluetooth::BleInvalidConnectionHandle;

    m_ServiceNum        = 0;
    m_CharacteristicNum = 0;
    m_DescriptorNum     = 0;

    m_IsGattOperationOnGoing = false;

    m_Mtu = nn::bluetooth::BleMtuDefault;

    // GATT Attribute 操作の同期用イベントを初期化
    nn::os::InitializeEvent(&m_GattOperationInternalEvent, false, nn::os::EventClearMode_AutoClear);
}

GattClientScene::~GattClientScene()
{
    nn::os::FinalizeEvent(&m_GattOperationInternalEvent);

    if (m_ConnInfo.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
    {
        // 切断
        NN_LOG("Disconnect from GATT Server\n");
        // Result は見ない
        nn::bluetooth::DisconnectFromGattServer(m_ConnInfo.connectionHandle);
    }
}

void GattClientScene::Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    pTextWriter->SetCursor(20, 20);
    pTextWriter->SetTextColor(Color::White);
    pTextWriter->Print("%s", m_Title.c_str());

    pTextWriter->SetCursor(20, 40);
    pTextWriter->Print("%s    Handle: %d    Address: %02X:%02X:%02X:%02X:%02X:%02X    MTU: %d",
        (m_ConnInfo.connectionHandle == nn::bluetooth::BleInvalidConnectionHandle) ? "Disconnected" : "Connected",
        m_ConnInfo.connectionHandle,
        m_ConnInfo.address.address[0], m_ConnInfo.address.address[1], m_ConnInfo.address.address[2],
        m_ConnInfo.address.address[3], m_ConnInfo.address.address[4], m_ConnInfo.address.address[5],
        m_Mtu);

    float offsetY = 40;

    for (int serviceIndex = 0; serviceIndex < m_ServiceNum; ++serviceIndex)
    {
        offsetY += 30;
        pTextWriter->SetCursor(20, offsetY);

        if (GetAttributeName(m_Services[serviceIndex].GetUuid()) != nullptr)
        {
            pTextWriter->Print("Srvc Hdr: %d  E-Handle: %d  UUID: %s",
                m_Services[serviceIndex].GetHandle(), m_Services[serviceIndex].GetEndGroupHandle(), GetAttributeName(m_Services[serviceIndex].GetUuid()));
        }
        else
        {
            pTextWriter->Print("Srvc Hdr: %d  E-Handle: %d  UUID: %s",
                m_Services[serviceIndex].GetHandle(), m_Services[serviceIndex].GetEndGroupHandle(), toHexString(m_Services[serviceIndex].GetUuid()));
        }

        if (m_CharsAndDescsNum > 0)
        {
            const nn::bluetooth::GattService* pDetailedService = GetService(m_CharsAndDescs[m_Cursor.posY]->GetHandle());
            NN_ASSERT_NOT_NULL(pDetailedService);

            // カーソル位置のGATT Attribute を含むサービスのみ、GATT Characteristic とGATT Descriptor を表示
            bool showCharAndDesc = false;
            showCharAndDesc = (pDetailedService->GetHandle() == m_Services[serviceIndex].GetHandle());

            if (showCharAndDesc)
            {
                for (int charIndex = 0; charIndex < m_CharacteristicNum; ++charIndex)
                {
                    if (pDetailedService->GetHandle() <= m_Characteristics[charIndex].GetHandle() &&
                        m_Characteristics[charIndex].GetHandle() <= pDetailedService->GetEndGroupHandle())
                    {
                        if (m_CharsAndDescs[m_Cursor.posY]->GetHandle() == m_Characteristics[charIndex].GetHandle())
                        {
                            pTextWriter->SetTextColor(Color::Orange);
                        }

                        offsetY += 20;
                        pTextWriter->SetCursor(40, offsetY);

                        if (GetAttributeName(m_Characteristics[charIndex].GetUuid()) != nullptr)
                        {
                            pTextWriter->Print("Char Hdr: %d  UUID: %s  Prop: %02X (%s)",
                                m_Characteristics[charIndex].GetHandle(), GetAttributeName(m_Characteristics[charIndex].GetUuid()),
                                m_Characteristics[charIndex].GetProperties(), ParseCharacteristicProperty(m_Characteristics[charIndex]).c_str());
                        }
                        else
                        {
                            pTextWriter->Print("Char Hdr: %d  UUID: %s  Prop: %02X (%s)",
                                m_Characteristics[charIndex].GetHandle(), toHexString(m_Characteristics[charIndex].GetUuid()),
                                m_Characteristics[charIndex].GetProperties(), ParseCharacteristicProperty(m_Characteristics[charIndex]).c_str());
                        }

                        pTextWriter->SetTextColor(Color::White);

                        for (int descIndex = 0; descIndex < m_DescriptorNum; ++descIndex)
                        {
                            const nn::bluetooth::GattCharacteristic* pCharacteristic = GetCharacteristic(m_Descriptors[descIndex]);
                            NN_ASSERT_NOT_NULL(pCharacteristic);

                            if (pCharacteristic->GetHandle() == m_Characteristics[charIndex].GetHandle())
                            {
                                if (m_CharsAndDescs[m_Cursor.posY]->GetHandle() == m_Descriptors[descIndex].GetHandle())
                                {
                                    pTextWriter->SetTextColor(Color::Orange);
                                }

                                offsetY += 20;
                                pTextWriter->SetCursor(60, offsetY);

                                if (GetAttributeName(m_Descriptors[descIndex].GetUuid()) != nullptr)
                                {
                                    pTextWriter->Print("Desc  Hdr: %d  UUID: %s",
                                        m_Descriptors[descIndex].GetHandle(), GetAttributeName(m_Descriptors[descIndex].GetUuid()));
                                }
                                else
                                {
                                    pTextWriter->Print("Desc  Hdr: %d  UUID: %s",
                                        m_Descriptors[descIndex].GetHandle(), toHexString(m_Descriptors[descIndex].GetUuid()));
                                }

                                pTextWriter->SetTextColor(Color::White);
                            }
                        }
                    }
                }
            }
        }
    }

    pTextWriter->SetCursor(20, 530);
    pTextWriter->Print("GATT Operation: ");
    pTextWriter->SetCursor(200, 530);
    if (GATT_PROPERTY_LIST[m_Cursor.posX].property == nn::bluetooth::GattAttributeProperty::GattAttributeProperty_Read ||
        GATT_PROPERTY_LIST[m_Cursor.posX].property == nn::bluetooth::GattAttributeProperty::GattAttributeProperty_WriteWithoutResponse ||
        GATT_PROPERTY_LIST[m_Cursor.posX].property == nn::bluetooth::GattAttributeProperty::GattAttributeProperty_Write ||
        GATT_PROPERTY_LIST[m_Cursor.posX].property == nn::bluetooth::GattAttributeProperty::GattAttributeProperty_Notify ||
        GATT_PROPERTY_LIST[m_Cursor.posX].property == nn::bluetooth::GattAttributeProperty::GattAttributeProperty_Indicate)
    {
        pTextWriter->SetTextColor(Color::Orange);
    }
    else
    {
        // 実行できない操作はグレー表示
        pTextWriter->SetTextColor(Color::Gray);
    }
    pTextWriter->Print("%s", GATT_PROPERTY_LIST[m_Cursor.posX].name.c_str());

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(20, 550);

    const char* operationStatusString = (m_GattOperationResult.status == nn::bluetooth::BleGattOperationStatus::BleGattOperationStatus_Success) ? "Success" : "Failed";
    switch (m_GattOperationResult.operation)
    {
    case nn::bluetooth::GattOperationType_ReadCharacteristic:
        pTextWriter->Print("Result: GATT Read Characteristic %s (%d)\n", operationStatusString, m_GattOperationResult.status);
        break;
    case nn::bluetooth::GattOperationType_WriteCharacteristic:
        pTextWriter->Print("Result: GATT Write Characteristic %s (%d)\n", operationStatusString, m_GattOperationResult.status);
        break;
    case nn::bluetooth::GattOperationType_ReadDescriptor:
        pTextWriter->Print("Result: GATT Read Descriptor %s (%d)\n", operationStatusString, m_GattOperationResult.status);
        break;
    case nn::bluetooth::GattOperationType_WriteDescriptor:
        pTextWriter->Print("Result: GATT Write Descriptor %s (%d)\n", operationStatusString, m_GattOperationResult.status);
        break;
    case nn::bluetooth::GattOperationType_Notify:
        pTextWriter->Print("Result: GATT Notification %s (%d)\n", operationStatusString, m_GattOperationResult.status);
        break;
    case nn::bluetooth::GattOperationType_Indicate:
        pTextWriter->Print("Result: GATT Indication %s (%d)\n", operationStatusString, m_GattOperationResult.status);
        break;
    default:
        pTextWriter->Print("No Result");
        break;
    }

    pTextWriter->SetCursor(20, 570);
    if (GetAttributeName(m_GattOperationResult.gattService.GetUuid()) != nullptr)
    {
        pTextWriter->Print("Srvc UUID: %s", GetAttributeName(m_GattOperationResult.gattService.GetUuid()));
    }
    else
    {
        pTextWriter->Print("Srvc UUID: %s", toHexString(m_GattOperationResult.gattService.GetUuid()));
    }
    pTextWriter->SetCursor(20, 590);
    if (GetAttributeName(m_GattOperationResult.gattCharacteristic.GetUuid()) != nullptr)
    {
        pTextWriter->Print("Char UUID: %s", GetAttributeName(m_GattOperationResult.gattCharacteristic.GetUuid()));
    }
    else
    {
        pTextWriter->Print("Char UUID: %s", toHexString(m_GattOperationResult.gattCharacteristic.GetUuid()));
    }
    pTextWriter->SetCursor(20, 610);
    if (GetAttributeName(m_GattOperationResult.gattDescriptor.GetUuid()) != nullptr)
    {
        pTextWriter->Print("Desc UUID: %s", GetAttributeName(m_GattOperationResult.gattDescriptor.GetUuid()));
    }
    else
    {
        pTextWriter->Print("Desc UUID: %s", toHexString(m_GattOperationResult.gattDescriptor.GetUuid()));
    }
    pTextWriter->SetCursor(20, 630);
    pTextWriter->Print("Received Data (%d bytes. Print top 32 bytes at maximum):", m_GattOperationResult.length);
    pTextWriter->SetCursor(20, 650);
    pTextWriter->Print("%s", toHexString((m_GattOperationResult.length > 32) ? 32 : m_GattOperationResult.length, m_GattOperationResult.value));

    pTextWriter->SetCursor(20, 690);
    pTextWriter->Print("Up/Down: Select Attribute    ZL/ZR: Select Operation    A: Execute Operation    X: Disconnect    Y: Set MTU 512    +: Exit");
} // NOLINT(impl/function_size)

void GattClientScene::ProcessControllerInput(nns::hid::ButtonSet buttons)
{
    if ((buttons & nns::hid::Button::A::Mask).IsAnyOn() || (buttons & nns::hid::Button::B::Mask).IsAnyOn())
    {
        ExecuteGattOperation(!(buttons & nns::hid::Button::B::Mask).IsAnyOn());
    }
    else if ((buttons & nns::hid::Button::Down::Mask).IsAnyOn())
    {
        // 選択中のAttribute を切り替え
        m_Cursor.posY = (m_Cursor.posY < m_CharsAndDescsNum - 1) ? m_Cursor.posY + 1 : m_Cursor.posY;
    }
    else if ((buttons & nns::hid::Button::Up::Mask).IsAnyOn())
    {
        // 選択中のAttribute を切り替え
        m_Cursor.posY = (m_Cursor.posY > 0) ? m_Cursor.posY - 1 : m_Cursor.posY;
    }
    else if ((buttons & nns::hid::Button::ZL::Mask).IsAnyOn())
    {
        // 選択中のAttribute Propertyを切り替え
        m_Cursor.posX = (m_Cursor.posX > 0) ? m_Cursor.posX - 1 : m_Cursor.posX;
    }
    else if ((buttons & nns::hid::Button::ZR::Mask).IsAnyOn())
    {
        // 選択中のAttribute Propertyを切り替え
        m_Cursor.posX = (m_Cursor.posX < sizeof(GATT_PROPERTY_LIST) / sizeof(GattProperty) - 1) ? m_Cursor.posX + 1 : m_Cursor.posX;
    }
    else if ((buttons & nns::hid::Button::X::Mask).IsAnyOn())
    {
        if (m_ConnInfo.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            auto result = nn::bluetooth::DisconnectFromGattServer(m_ConnInfo.connectionHandle);

            // 接続が存在しません。接続状態を確認してください。
            if (nn::bluetooth::ResultBleNotConnected().Includes(result))
            {
                NN_LOG("Connection not found.\n");
            }
        }
    }
    else if ((buttons & nns::hid::Button::Y::Mask).IsAnyOn())
    {
        // MTU を512 Byte に設定
        auto result = nn::bluetooth::ConfigureBleMtu(m_ConnInfo.connectionHandle, 512);

        // 接続が存在しません。接続状態を確認してください。
        if (nn::bluetooth::ResultBleNotConnected().Includes(result))
        {
            NN_LOG("Connection not found.\n");
        }
        // 排他的な API の処理中です。処理の完了を待ってから再試行してください。
        // 本サンプルでは、Y ボタンを押してリトライします。
        else if (nn::bluetooth::ResultProcessingExclusiveApi().Includes(result))
        {
            NN_LOG("Failed to configure BLT MTU. Exclusive API is being proceeded. Try later.");
        }
    }
}

void GattClientScene::UpdateGattOperationStatus(const nn::bluetooth::BleClientGattOperationInfo& info)
{
    //!< 同期待ちしていたら、ブロックを解除
    if (m_IsGattOperationOnGoing && m_ResultWaitingGattOperation == info.operation)
    {
        m_IsGattOperationOnGoing = false;
        nn::os::SignalEvent(&m_GattOperationInternalEvent);
    }

    m_GattOperationResult.status = info.status;

    const nn::bluetooth::GattService* pService                  = GetService(info.serviceUuid);
    const nn::bluetooth::GattCharacteristic* pCharacteristic    = GetCharacteristic(info.charcteristicUuid);
    const nn::bluetooth::GattDescriptor* pDescriptor            = GetDescriptor(info.descriptorUuid);

    NN_ASSERT_NOT_NULL(pService);
    NN_ASSERT_NOT_NULL(pCharacteristic);

    m_GattOperationResult.gattService = *pService;
    m_GattOperationResult.gattCharacteristic = *pCharacteristic;

    if (pDescriptor)
    {
        m_GattOperationResult.gattDescriptor = *pDescriptor;
    }
    else
    {
        m_GattOperationResult.gattDescriptor = nn::bluetooth::GattDescriptor();
    }

    m_GattOperationResult.operation = info.operation;

    switch (info.operation)
    {
    case nn::bluetooth::GattOperationType_ReadCharacteristic:
    case nn::bluetooth::GattOperationType_ReadDescriptor:
    case nn::bluetooth::GattOperationType_Notify:
    case nn::bluetooth::GattOperationType_Indicate:
        memcpy(m_GattOperationResult.value, info.data, info.length);
        m_GattOperationResult.length = info.length;
        break;
    case nn::bluetooth::GattOperationType_WriteCharacteristic:
    case nn::bluetooth::GattOperationType_WriteDescriptor:
        memset(m_GattOperationResult.value, 0x00, sizeof(m_GattOperationResult.value));
        m_GattOperationResult.length = 0;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void GattClientScene::ReadCharacteristic(const nn::bluetooth::GattCharacteristic &characteristic)
{
    NN_LOG("Read Characteristic\n");

    auto result = nn::bluetooth::ReadGattCharacteristic(characteristic);
    if (result.IsFailure())
    {
        NN_LOG("Failed to read characteristic.\n");
        // 接続が存在しません。接続状態を確認してください。
        if (nn::bluetooth::ResultBleNotConnected().Includes(result))
        {
            NN_LOG("Connection not found.\n");
        }
        // GATT Characteristic が対応していない操作です。
        // 本サンプルではZR/ZL を使って、操作方法を切り替えます。
        else if (nn::bluetooth::ResultUnsupportedGattProperty().Includes(result))
        {
            NN_LOG("Unsupported operation. Property is %s\n", ParseCharacteristicProperty(characteristic).c_str());
        }
        // 同一の接続に対するGATT Attribute の操作は同時に1つまでです。
        // 本サンプルでは、操作時に処理をブロックするため、このエラーは返りませんが、アプリケーション内で、接続ごとに同期用のスレッドを推奨します
        else if (nn::bluetooth::ResultGattBusy().Includes(result))
        {
            NN_LOG("GATT Operation is on going. Wait the result.\n");
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("General error. Try again or check the parameter.\n");
        }
        return;
    }

    m_ResultWaitingGattOperation = nn::bluetooth::GattOperationType_ReadCharacteristic;
    m_IsGattOperationOnGoing = true;
    // 操作結果を待つ
    nn::os::WaitEvent(&m_GattOperationInternalEvent);
}

void GattClientScene::WriteCharacteristic(nn::bluetooth::GattCharacteristic &characteristic)
{
    NN_LOG("Write Characteristic\n");

    // 最大で書き込めるのはMTU - GATT Header(1 Byte OpCode + 2 Byte Access Address)
    uint16_t dataSize = (m_Mtu - 3 < sizeof(writeData)) ? m_Mtu - 3 : sizeof(writeData);
    uint8_t* pData = new uint8_t[dataSize];
    NN_ABORT_UNLESS_NOT_NULL(pData);

    memcpy(pData, writeData, dataSize);

    characteristic.SetValue(pData, dataSize);

    auto result = nn::bluetooth::WriteGattCharacteristic(characteristic);

    if (result.IsFailure())
    {
        NN_LOG("Failed to write characteristic.\n");
        // 接続が存在しません。接続状態を確認してください。
        if (nn::bluetooth::ResultBleNotConnected().Includes(result))
        {
            NN_LOG("Connection not found.\n");
        }
        // GATT Characteristic が対応していない操作です。
        // 本サンプルではZR/ZL を使って、操作方法を切り替えます。
        else if (nn::bluetooth::ResultUnsupportedGattProperty().Includes(result))
        {
            NN_LOG("Unsupported operation. Property is %s\n", ParseCharacteristicProperty(characteristic).c_str());
        }
        // 同一の接続に対するGATT Attribute の操作は同時に1つまでです。
        // 本サンプルでは、操作時に処理をブロックするため、このエラーは返りませんが、アプリケーション内で、接続ごとに同期用のスレッドを推奨します
        else if (nn::bluetooth::ResultGattBusy().Includes(result))
        {
            NN_LOG("GATT Operation is on going. Wait the result.\n");
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("General error. Try again or check the parameter.\n");
        }
        delete[] pData;
        return;
    }

    delete[] pData;

    m_ResultWaitingGattOperation = nn::bluetooth::GattOperationType_WriteCharacteristic;
    m_IsGattOperationOnGoing = true;
    // 操作結果を待つ
    nn::os::WaitEvent(&m_GattOperationInternalEvent);

    return;
}

void GattClientScene::EnableNotification(const nn::bluetooth::GattCharacteristic &characteristic, bool enable, bool isNotification)
{
    NN_LOG("%s Notification\n", enable ? "Enable" : "Disable");

    nn::bluetooth::GattDescriptor cccDescriptor;
    nn::Result result;
    uint8_t value[2] = { 0x00 };
    bool isCccDescriptorExisting = false;

    isCccDescriptorExisting = characteristic.GetDescriptor(&cccDescriptor, nn::bluetooth::ClientCharacteristicConfigurationDescriptorUuid);

    // Notification/Indication の受信を許可・不許可
    result = nn::bluetooth::EnableGattCharacteristicNotification(characteristic, enable);

    // Client Characteristic Descriptor が存在する場合、Descriptor にWrite
    if (result.IsSuccess() && isCccDescriptorExisting)
    {
        if (enable && isNotification)
        {
            value[0] = 0x01;
        }
        else if (enable && !isNotification)
        {
            value[0] = 0x02;
        }

        cccDescriptor.SetValue(value, NN_ARRAY_SIZE(value));
        result = nn::bluetooth::WriteGattDescriptor(cccDescriptor);
    }

    if (result.IsFailure())
    {
        NN_LOG("Failed to %s notification.\n", enable ? "enable" : "disable");
        // 接続が存在しません。接続状態を確認してください。
        if (nn::bluetooth::ResultBleNotConnected().Includes(result))
        {
            NN_LOG("Connection not found.\n");
        }
        // GATT Characteristic が対応していない操作です。
        // 本サンプルではZR/ZL を使って、操作方法を切り替えます。
        else if (nn::bluetooth::ResultUnsupportedGattProperty().Includes(result))
        {
            NN_LOG("Unsupported operation. Property is %s\n", ParseCharacteristicProperty(characteristic).c_str());
        }
        // 同一の接続に対するGATT Attribute の操作は同時に1つまでです。
        // 本サンプルでは、操作時に処理をブロックするため、このエラーは返りませんが、アプリケーション内で、接続ごとに同期用のスレッドを推奨します
        else if (nn::bluetooth::ResultGattBusy().Includes(result))
        {
            NN_LOG("GATT Operation is on going. Wait the result.\n");
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("General error. Try again or check the parameter.\n");
        }
        return;
    }

    if (isCccDescriptorExisting)
    {
        m_ResultWaitingGattOperation = nn::bluetooth::GattOperationType_WriteDescriptor;
        m_IsGattOperationOnGoing = true;
        nn::os::WaitEvent(&m_GattOperationInternalEvent);
    }
}

void GattClientScene::ReadDescriptor(const nn::bluetooth::GattDescriptor &descriptor)
{
    NN_LOG("Read Descriptor\n");

    auto result = nn::bluetooth::ReadGattDescriptor(descriptor);
    if (result.IsFailure())
    {
        NN_LOG("Failed to read descriptor.\n");
        // 接続が存在しません。接続状態を確認してください。
        if (nn::bluetooth::ResultBleNotConnected().Includes(result))
        {
            NN_LOG("Connection not found.\n");
        }
        // 同一の接続に対するGATT Attribute の操作は同時に1つまでです。
        // 本サンプルでは、操作時に処理をブロックするため、このエラーは返りませんが、アプリケーション内で、接続ごとに同期用のスレッドを推奨します
        else if (nn::bluetooth::ResultGattBusy().Includes(result))
        {
            NN_LOG("GATT Operation is on going. Wait the result.\n");
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("General error. Try again or check the parameter.\n");
        }
        return;
    }

    m_ResultWaitingGattOperation = nn::bluetooth::GattOperationType_ReadDescriptor;
    m_IsGattOperationOnGoing = true;
    nn::os::WaitEvent(&m_GattOperationInternalEvent);
}

void GattClientScene::WriteDescriptor(nn::bluetooth::GattDescriptor &descriptor)
{
    NN_LOG("Write Descriptor\n");

    uint16_t dataSize = (m_Mtu - 3 < sizeof(writeData)) ? m_Mtu - 3 : sizeof(writeData); // 最大で書き込めるのはMTU - GATT Header(1 Byte OpCode + 2 Byte Access Address)
    uint8_t* pData = new uint8_t[dataSize];
    NN_ABORT_UNLESS_NOT_NULL(pData);

    memcpy(pData, writeData, dataSize);

    descriptor.SetValue(pData, dataSize);

    auto result = nn::bluetooth::WriteGattDescriptor(descriptor);
    if (result.IsFailure())
    {
        NN_LOG("Failed to write descriptor.\n");
        // 接続が存在しません。接続状態を確認してください。
        if (nn::bluetooth::ResultBleNotConnected().Includes(result))
        {
            NN_LOG("Connection not found.\n");
        }
        // 同一の接続に対するGATT Attribute の操作は同時に1つまでです。
        // 本サンプルでは、操作時に処理をブロックするため、このエラーは返りませんが、アプリケーション内で、接続ごとに同期用のスレッドを推奨します
        else if (nn::bluetooth::ResultGattBusy().Includes(result))
        {
            NN_LOG("GATT Operation is on going. Wait the result.\n");
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("General error. Try again or check the parameter.\n");
        }
        delete[] pData;
        return;
    }

    delete[] pData;

    m_ResultWaitingGattOperation = nn::bluetooth::GattOperationType_WriteDescriptor;
    m_IsGattOperationOnGoing = true;
    nn::os::WaitEvent(&m_GattOperationInternalEvent);

    return;
}

void GattClientScene::ExecuteGattOperation(bool enable)
{
    if (m_IsGattOperationOnGoing)
    {
        NN_LOG("Previous GATT Operation is not completed yet. Ongoing operation is %d\n", m_ResultWaitingGattOperation);
        return;
    }
    // 選択中のAttribute を選択中のAttribute Property に従って操作
    switch (m_CharsAndDescs[m_Cursor.posY]->GetType())
    {
    case nn::bluetooth::GattAttributeType_Characteristic:
    {
        nn::bluetooth::GattCharacteristic *pCharacteristic = reinterpret_cast<nn::bluetooth::GattCharacteristic*>(m_CharsAndDescs[m_Cursor.posY]);

        if (!(pCharacteristic->GetProperties() & GATT_PROPERTY_LIST[m_Cursor.posX].property))
        {
            NN_LOG("Unsupported GATT Operation (%s)\n", GATT_PROPERTY_LIST[m_Cursor.posX].name.c_str());
            return;
        }

        switch (GATT_PROPERTY_LIST[m_Cursor.posX].property)
        {
        case nn::bluetooth::GattAttributeProperty_Read:
            ReadCharacteristic(*pCharacteristic);
            break;
        case nn::bluetooth::GattAttributeProperty_WriteWithoutResponse:
        case nn::bluetooth::GattAttributeProperty_Write:
            WriteCharacteristic(*pCharacteristic);
            break;
        case nn::bluetooth::GattAttributeProperty_Notify:
            EnableNotification(*pCharacteristic, enable, true);
            break;
        case nn::bluetooth::GattAttributeProperty_Indicate:
            EnableNotification(*pCharacteristic, enable, false);
            break;
        default:
            NN_LOG("Unsupported GATT Operation (%s)\n", GATT_PROPERTY_LIST[m_Cursor.posY].name.c_str());
            return;
        }
    }
    break;
    case nn::bluetooth::GattAttributeType_Descriptor:
    {
        nn::bluetooth::GattDescriptor *pDesc = reinterpret_cast<nn::bluetooth::GattDescriptor*>(m_CharsAndDescs[m_Cursor.posY]);

        switch (GATT_PROPERTY_LIST[m_Cursor.posX].property)
        {
        case nn::bluetooth::GattAttributeProperty_Read:
            ReadDescriptor(*pDesc);
            break;
        case nn::bluetooth::GattAttributeProperty_WriteWithoutResponse:
        case nn::bluetooth::GattAttributeProperty_Write:
            WriteDescriptor(*pDesc);
            break;
        default:
            NN_LOG("Unsupported GATT Operation (%s)\n", GATT_PROPERTY_LIST[m_Cursor.posX].name.c_str());
            return;
        }
    }
    break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void GattClientScene::UpdateConnectionState()
{
    nn::bluetooth::BleConnectionInfo tempInfo[nn::bluetooth::BleConnectionCountMaxClient];
    nn::bluetooth::GetBleConnectionInfoList(tempInfo, NN_ARRAY_SIZE(tempInfo));

    // 切断された or 切断され、他の接続が前詰めされた
    if ((m_ConnInfo.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle && tempInfo[m_ConnIndex].connectionHandle == nn::bluetooth::BleInvalidConnectionHandle) ||
        (m_ConnInfo.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle && m_ConnInfo.connectionHandle != tempInfo[m_ConnIndex].connectionHandle))
    {
        // 切断理由の取得
        m_ConnectionInfoDumper.UpdateDisconnectionReason();

        // 保存
        ConnectionInfoDumper::Save(m_ConnectionInfoDumper);
    }

    m_ConnInfo = tempInfo[m_ConnIndex];

    // 接続されている（新規かどうか問わない）
    if (m_ConnInfo.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
    {
        // クリア
        m_ConnectionInfoDumper.Initialize();

        m_ConnectionInfoDumper.SetConnection(m_ConnInfo.address, m_ConnInfo.connectionHandle);

        m_ConnectionInfoDumper.UpdateConnectionParameter();

        m_ConnectionInfoDumper.UpdateConnectionParameterRequest();
    }

    if (m_ConnInfo.connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
    {
        // GATT Attribute への操作中に切断が発生した場合
        if (m_IsGattOperationOnGoing)
        {
            m_IsGattOperationOnGoing = false;
            nn::os::SignalEvent(&m_GattOperationInternalEvent);
        }

        m_GattOperationResult.status                = nn::bluetooth::BleGattOperationStatus::BleGattOperationStatus_Success;
        m_GattOperationResult.gattService           = nn::bluetooth::GattService();
        m_GattOperationResult.gattCharacteristic    = nn::bluetooth::GattCharacteristic();
        m_GattOperationResult.gattDescriptor        = nn::bluetooth::GattDescriptor();
        m_GattOperationResult.length                = 0;
        memset(m_GattOperationResult.value, 0x00, sizeof(m_GattOperationResult.value));
    }

    UpdateServerProfile();
    UpdateMtu();
}

void GattClientScene::UpdateServerProfile()
{
    ClearServerProfile();

    m_ServiceNum += nn::bluetooth::GetGattServices(&m_Services[m_ServiceNum], NN_ARRAY_SIZE(m_Services) - m_ServiceNum, m_ConnInfo.connectionHandle);

    for (int i = 0; i < m_ServiceNum; ++i)
    {
        m_CharacteristicNum += m_Services[i].GetCharacteristics(&m_Characteristics[m_CharacteristicNum], NN_ARRAY_SIZE(m_Characteristics) - m_CharacteristicNum);
    }

    for (int i = 0; i < m_CharacteristicNum; ++i)
    {
        uint16_t descriptorNum = m_Characteristics[i].GetDescriptors(&m_Descriptors[m_DescriptorNum], NN_ARRAY_SIZE(m_Descriptors) - m_DescriptorNum);

        m_CharsAndDescs[m_CharsAndDescsNum++] = &m_Characteristics[i];

        for (int j = m_DescriptorNum; j < m_DescriptorNum + descriptorNum; ++j)
        {
            m_CharsAndDescs[m_CharsAndDescsNum++] = &m_Descriptors[j];
        }

        m_DescriptorNum += descriptorNum;
    }

    // アプリに通知するGATT Service を登録
    for (int i = 0; i < m_ServiceNum; ++i)
    {
        auto result = nn::bluetooth::RegisterGattOperationNotification(m_Services[i].GetUuid());

        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            NN_LOG("Registering GATT Operation Notification Failed. Trying again.\n");
            --i;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
    }
}

void GattClientScene::UpdateMtu()
{
    m_Mtu = nn::bluetooth::GetBleMtu(m_ConnInfo.connectionHandle);

    if (m_ConnInfo.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
    {
        m_ConnectionInfoDumper.UpdateMtu();
    }
}

const nn::bluetooth::GattService* GattClientScene::GetService(const nn::bluetooth::GattAttributeUuid& uuid)
{
    for (int serviceIndex = 0; serviceIndex < m_ServiceNum; ++serviceIndex)
    {
        if (m_Services[serviceIndex].GetUuid() == uuid)
        {
            return &m_Services[serviceIndex];
        }
    }

    return nullptr;
}

const nn::bluetooth::GattService* GattClientScene::GetService(uint16_t attributeHandle)
{
    for (int serviceIndex = 0; serviceIndex < m_ServiceNum; ++serviceIndex)
    {
        if (m_Services[serviceIndex].GetHandle() < attributeHandle && attributeHandle <= m_Services[serviceIndex].GetEndGroupHandle())
        {
            return &m_Services[serviceIndex];
        }
    }

    return nullptr;
}

const nn::bluetooth::GattCharacteristic* GattClientScene::GetCharacteristic(const nn::bluetooth::GattAttributeUuid& uuid)
{
    for (int charIndex = 0; charIndex < m_CharacteristicNum; ++charIndex)
    {
        if (m_Characteristics[charIndex].GetUuid() == uuid)
        {
            return &m_Characteristics[charIndex];
        }
    }

    return nullptr;
}

const nn::bluetooth::GattCharacteristic* GattClientScene::GetCharacteristic(const nn::bluetooth::GattDescriptor& descriptor)
{
    for (int charIndex = 0; charIndex < m_CharacteristicNum; ++charIndex)
    {
        if (charIndex < m_CharacteristicNum - 1)
        {
            if (m_Characteristics[charIndex].GetHandle() < descriptor.GetHandle() && descriptor.GetHandle() < m_Characteristics[charIndex + 1].GetHandle())
            {
                return &m_Characteristics[charIndex];
            }
        }
        else
        {
            if (m_Characteristics[charIndex].GetHandle() < descriptor.GetHandle())
            {
                return &m_Characteristics[charIndex];
            }
        }
    }

    return nullptr;
}

const nn::bluetooth::GattDescriptor* GattClientScene::GetDescriptor(const nn::bluetooth::GattAttributeUuid& uuid)
{
    for (int descIndex = 0; descIndex < m_DescriptorNum; ++descIndex)
    {
        if (m_Descriptors[descIndex].GetUuid() == uuid)
        {
            return &m_Descriptors[descIndex];
        }
    }

    return nullptr;
}

void GattClientScene::ClearServerProfile()
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Services); ++i)
    {
        m_Services[i] = nn::bluetooth::GattService();
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_Characteristics); ++i)
    {
        m_Characteristics[i] = nn::bluetooth::GattCharacteristic();
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_Descriptors); ++i)
    {
        m_Descriptors[i] = nn::bluetooth::GattDescriptor();
    }

    m_ServiceNum        = 0;
    m_CharacteristicNum = 0;
    m_DescriptorNum     = 0;
    m_CharsAndDescsNum  = 0;

    m_Cursor.posX = 0;
    m_Cursor.posY = 0;
}

const char* GattClientScene::GetAttributeName(const nn::bluetooth::GattAttributeUuid& uuid) const
{
    if (uuid == BleInteroperabilityTest_ReadServiceUuid)
    {
        return "Read Service";
    }
    else if (uuid == BleInteroperabilityTest_ReadCharacteristicUuid)
    {
        return "Read Characteristic";
    }
    else if (uuid == BleInteroperabilityTest_WriteServiceUuid)
    {
        return "Write Service";
    }
    else if (uuid == BleInteroperabilityTest_WriteCharacteristicUuid)
    {
        return "Write Characteristic";
    }
    else if (uuid == BleInteroperabilityTest_WriteWithoutResponseServiceUuid)
    {
        return "Write without Response Service";
    }
    else if (uuid == BleInteroperabilityTest_WriteWithoutResponseCharacteristicUuid)
    {
        return "Write without Response Characteristic";
    }
    else if (uuid == BleInteroperabilityTest_NotificationServiceUuid)
    {
        return "Notification Service";
    }
    else if (uuid == BleInteroperabilityTest_NotificationCharacteristicUuid)
    {
        return "Notification Characteristic";
    }
    else if (uuid == BleInteroperabilityTest_IndicationServiceUuid)
    {
        return "Indication Service";
    }
    else if (uuid == BleInteroperabilityTest_IndicationCharacteristicUuid)
    {
        return "Indication Characteristic";
    }
    else if (uuid == nn::bluetooth::ClientCharacteristicConfigurationDescriptorUuid)
    {
        return "Client Characteristic Configuration Descriptor";
    }
    else if (uuid == BleTestPeripheral_SmallServiceUuid)
    {
        return "Small Service";
    }
    else if (uuid == BleTestPeripheral_SmallReadCharacteristicUuid)
    {
        return "Small Read Characteristic";
    }
    else if (uuid == BleTestPeripheral_SmallWriteCharacteristicUuid)
    {
        return "Small Write Characteristic";
    }
    else if (uuid == BleTestPeripheral_SmallWriteWithoutResponseCharacteristicUuid)
    {
        return "Small Write Without Response  Characteristic";
    }
    else if (uuid == BleTestPeripheral_SmallNotificationCharacteristicUuid)
    {
        return "Small Notification Characteristic";
    }
    else if (uuid == BleTestPeripheral_LargeServiceUuid)
    {
        return "Large Service";
    }
    else if (uuid == BleTestPeripheral_LargeReadCharacteristicUuid)
    {
        return "Large Read Characteristic";
    }
    else if (uuid == BleTestPeripheral_LargeWriteCharacteristicUuid)
    {
        return "Large Write Characteristic";
    }
    else if (uuid == BleTestPeripheral_LargeWriteWithoutResponseCharacteristicUuid)
    {
        return "Large Write Without Response Characteristic";
    }
    else if (uuid == BleTestPeripheral_LargeNotificationCharacteristicUuid)
    {
        return "Large Notification Characteristic";
    }
    else if (uuid == BleTestPeripheral_DeviceConfigurationServiceUuid)
    {
        return "Device Configuration Service";
    }
    else if (uuid == BleTestPeripheral_AdvertisePacketConfigCharacteristicUuid)
    {
        return "Advertise Packet Config Characteristic";
    }
    else if (uuid == BleTestPeripheral_ConnectionParameterConfigCharacteristicUuid)
    {
        return "Connection Parameter Config Characteristic";
    }
    else if (uuid == BleTestPeripheral_GattParameterConfigCharacteristicUuid)
    {
        return "Gatt Parmeter Config Characteristic";
    }
    else if (uuid == NintendoGattPairing_ServiceUuid)
    {
        return "Nintendo Gatt Pairing Service";
    }
    else if (uuid == NintendoGattPairing_CommandCharacteristicUuid)
    {
        return "Nintendo Gatt Pairing Command Characteristic";
    }
    else if (uuid == NintendoGattPairing_AddressWriterCharacteristicUuid)
    {
        return "Nintendo Gatt Pairing Address Writer Characteristic";
    }
    else if (uuid == NintendoGattPairing_AddressReaderCharacteristicUuid)
    {
        return "Nintendo Gatt Pairing Address Reader Characteristic";
    }
    else if (uuid == NintendoGattPairing_AdvertiseDataReaderCharacteristicUuid)
    {
        return "Nintendo Gatt Pairing Advertise Data Reader Characteristic";
    }
    else if (uuid == NintendoGattPairing_ErrorHandlerCharacteristicUuid)
    {
        return "Nintendo Gatt Pairing Error Handler Characteristic";
    }

    return nullptr;
} //NOLINT(impl/function_size)
