﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/hid.h>
#include <nn/bluetooth.h>
#include <nn/btm/debug/btm_DebugApi.h>
#include <nn/fs.h>

class ConnectionInfoDumper
{
public:
    /**
     * @brief       コンストラクタです
     */
    ConnectionInfoDumper();

    /**
     * @brief       デストラクタです
     */
    virtual ~ConnectionInfoDumper() {};

    /**
     * @brief       蓄積しているすべての情報をクリアします
     *
     * @post
     *              - m_Address                     = { { 0x00 } }
     *              - m_Handle                      = nn::bluetooth::BleInvalidConnectionHandle
     *              - m_Mtu                         = nn::bluetooth::BleMtuDefault
     *              - m_Parameter.interval          = nn::btm::debug::DefaultConnectionIntervalMin
     *              - m_Parameter.latency           = nn::btm::debug::DefaultSlaveLatency
     *              - m_Parameter.timeout           = nn::btm::debug::DefaultSupervisionTimeout
     *              - m_ParameterReq.intervalMin    = nn::btm::debug::DefaultConnectionIntervalMin
     *              - m_ParameterReq.intervalMax    = nn::btm::debug::DefaultConnectionIntervalMin
     *              - m_ParameterReq.latency        = nn::btm::debug::DefaultSlaveLatency
     *              - m_ParameterReq.timeout        = nn::btm::debug::DefaultSupervisionTimeout
     */
    void Initialize();

    /**
     * @brief       現在設定されている値一覧をSD カードに保存します
     */
    static void Save(const ConnectionInfoDumper& dumper);

    /**
     * @brief       接続ハンドルをセットします
     */
    void SetConnection(const nn::bluetooth::Address& address, uint32_t connectionHandle);

    /**
     * @brief       MTU を更新します
     *
     * @pre
     *              - m_Handle  != nn::bluetooth::BleInvalidConnectionHandle
     */
    void UpdateMtu();

    /**
     * @brief       接続パラメータを更新します
     *
     * @details     下層の接続情報が更新され、値が取得できるまでの間ブロックします。
     *
     * @pre
     *              - m_Handle  != nn::bluetooth::BleInvalidConnectionHandle
     */
    void UpdateConnectionParameter();

    /**
     * @brief       接続パラメータの要求値を更新します
     *
     * @details     下層の接続情報が更新され、値が取得できるまでの間ブロックします。
     *
     * @pre
     *              - m_Handle  != nn::bluetooth::BleInvalidConnectionHandle
     */
    void UpdateConnectionParameterRequest();

    /**
     * @brief       切断理由を更新します
     *
     * @details     切断発生後に、Initialize() を呼び出さずに使用してください。
     *
     * @pre
     *              - m_Handle  != nn::bluetooth::BleInvalidConnectionHandle
     */
    void UpdateDisconnectionReason();

    /**
     * @brief       Device Name をBluetooth アドレスと紐付けてキャッシュします
     */
    static void StoreDeviceName(const char* pDeviceName, size_t length, const nn::bluetooth::Address &address);

private:
    static const char* ConvertDisconnectionReasonToString(nn::btm::debug::BleDisconnectionReason reason);

    //!< Gatt Server のアドレスです
    nn::bluetooth::Address m_Address;

    //!< 接続ハンドルです
    uint32_t m_Handle;

    //!< MTU です
    uint32_t m_Mtu;

    //!< Connection Parameter を表す構造体
    struct ConnectionParameter
    {
        uint16_t interval;
        uint16_t latency;
        uint16_t timeout;
    };

    //!< Connection Parameter です
    ConnectionParameter m_Parameter;

    //!< Connection Parameter Request を表す構造体
    struct ConnectionParameterRequset
    {
        uint16_t intervalMin;
        uint16_t intervalMax;
        uint16_t latency;
        uint16_t timeout;
    };

    //!< Connection Parameter の要求値です
    ConnectionParameterRequset m_ParameterReq;

    //!< 切断理由です
    nn::btm::debug::BleDisconnectionReason m_DisconnectionReason;

    //!< SD カードのマウント名です
    static const char* ms_SdCardMountName;
    //!< 出力先のディレクトリ名です
    static const char* ms_ResultOutputDirectoryName;
    //!< 出力先のファイル名です
    static const char* ms_ResultOutputFileName;

    //!< SD カードをマウントし、出力先のディレクトリ、ファイルを作成します
    static nn::Result InitializeSdCard();
    //!< SD カードをアンマウントします
    static nn::Result FinalizeSdCard();

    //!< 現在日時を文字列として取得します
    static void GetCalenderDateTimeString(char* pOutString, size_t* pOutLength, size_t inLength);

    //!< 接続データを文字列として取得します
    static void GetConnectionInfoString(char* pOutString, size_t* pOutLength, size_t inLength, const ConnectionInfoDumper& dumper);

    //!< コンマ区切りの .csv 形式で書き出す際のエントリを表す構造体
    struct CsvEntry
    {
        char    value[256];
        size_t  length;
    };

    //!< 指定したオープン済みのファイルハンドルに、全てのエントリを.csv 形式で書き込み、最後に改行を書き込みます
    static void WriteToCsvFile(const CsvEntry* pEntries, int entryCount, const nn::fs::FileHandle& handle);

    //!< Device Name とBluetoothAddress のキャッシュ用構造体
    struct DeviceNameDictionaryEntry
    {
        nn::bluetooth::Address  address = { { 0x00 } };
        char                    name[256] = { 0x00 };
        size_t                  length = 0;
        bool                    isRegistered = false;
    };

    //!< Device Name のキャッシュです
    static DeviceNameDictionaryEntry ms_DeviceNameDictionary[];

    //!< Device Name のキャッシュエントリの最大数です
    static const int ms_DeviceNameDisctionaryEntryCountMax;

    //!< Device Name のキャッシュから、指定したアドレスと一致するものを取得します
    static const DeviceNameDictionaryEntry* GetDeviceName(const nn::bluetooth::Address& address);
};  // class ScanScene
