﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/err.h>

#include <nn/fs/fs_Debug.h>

#include <nn/time.h>

#include <iomanip>
#include <sstream>

#include "../Common/BluetoothTools_Utility.h"
#include "BleTestTool_ConnectionInfoDumper.h"

ConnectionInfoDumper::ConnectionInfoDumper()
{
    memset(&m_Address, 0x00, sizeof(nn::bluetooth::Address));

    m_Handle = nn::bluetooth::BleInvalidConnectionHandle;

    m_Mtu = nn::bluetooth::BleMtuDefault;

    m_Parameter.interval = 0;
    m_Parameter.latency  = 0;
    m_Parameter.timeout  = 0;

    m_ParameterReq.intervalMin  = 0;
    m_ParameterReq.intervalMax  = 0;
    m_ParameterReq.latency      = 0;
    m_ParameterReq.timeout      = 0;
}

void ConnectionInfoDumper::Initialize()
{
    memset(&m_Address, 0x00, sizeof(nn::bluetooth::Address));

    m_Handle = nn::bluetooth::BleInvalidConnectionHandle;

    m_Mtu = nn::bluetooth::BleMtuDefault;

    m_Parameter.interval = 0;
    m_Parameter.latency  = 0;
    m_Parameter.timeout  = 0;

    m_ParameterReq.intervalMin  = 0;
    m_ParameterReq.intervalMax  = 0;
    m_ParameterReq.latency      = 0;
    m_ParameterReq.timeout      = 0;
}

const char* ConnectionInfoDumper::ms_SdCardMountName            = "BleTestToolSd";
const char* ConnectionInfoDumper::ms_ResultOutputDirectoryName  = "BleTestToolSd:/BleInteroperabilityTestResult";
const char* ConnectionInfoDumper::ms_ResultOutputFileName       = "BleTestToolSd:/BleInteroperabilityTestResult/BleInteroperabilityTestResult.csv";

void ConnectionInfoDumper::Save(const ConnectionInfoDumper& dumper)
{
    auto result = InitializeSdCard();

    if (result.IsFailure())
    {
        return;
    }

    CsvEntry outputs[2] = { { {0x00}, 0 }, { {0x00}, 0 } };

    GetCalenderDateTimeString(outputs[0].value, &outputs[0].length, NN_ARRAY_SIZE(outputs[0].value));

    GetConnectionInfoString(outputs[1].value, &outputs[1].length, NN_ARRAY_SIZE(outputs[1].value), dumper);

    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, ms_ResultOutputFileName, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

    WriteToCsvFile(outputs, NN_ARRAY_SIZE(outputs), fileHandle);

    nn::fs::CloseFile(fileHandle);

    NN_ABORT_UNLESS_RESULT_SUCCESS(FinalizeSdCard());
}

nn::Result ConnectionInfoDumper::InitializeSdCard()
{
    auto result = nn::fs::MountSdCardForDebug(ms_SdCardMountName);

    if (result.IsFailure())
    {
        nn::err::ShowError(result);
        return result;
    }

    result = nn::fs::CreateDirectory(ms_ResultOutputDirectoryName);

    if (nn::fs::ResultPathAlreadyExists().Includes(result))
    {
        // 何もしない
    }
    else if (nn::fs::ResultUsableSpaceNotEnough().Includes(result))
    {
        nn::err::ShowError(result);
        return result;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    nn::fs::DirectoryEntryType tempEntryType;
    result = nn::fs::GetEntryType(&tempEntryType, ms_ResultOutputFileName);

    // ファイルが存在しなければ、新規作成する
    if (nn::fs::ResultPathNotFound().Includes(result))
    {
        result = nn::fs::CreateFile(ms_ResultOutputFileName, 0);

        if (nn::fs::ResultUsableSpaceNotEnough().Includes(result))
        {
            nn::err::ShowError(result);
            return result;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }

    return nn::ResultSuccess();
}

nn::Result ConnectionInfoDumper::FinalizeSdCard()
{
    nn::fs::Unmount(ms_SdCardMountName);

    return nn::ResultSuccess();
}

void ConnectionInfoDumper::GetCalenderDateTimeString(char* pOutString, size_t* pOutLength, size_t inLength)
{
    NN_ASSERT_NOT_NULL(pOutString);
    NN_ASSERT_NOT_NULL(pOutLength);

    nn::time::Initialize();

    nn::time::PosixTime                 posixTime;
    nn::time::CalendarTime              calenderTime;
    nn::time::CalendarAdditionalInfo    calenderAdditionalInfo;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calenderTime, &calenderAdditionalInfo, posixTime));

    std::ostringstream stringStream;

    stringStream << std::setw(2) << std::setfill('0') << std::right <<
                    static_cast<uint32_t>(calenderTime.year) << "/" <<
                    static_cast<uint32_t>(calenderTime.month) << "/" <<
                    static_cast<uint32_t>(calenderTime.day) << " " <<
                    static_cast<uint32_t>(calenderTime.hour) << ":" <<
                    static_cast<uint32_t>(calenderTime.minute) << ":" <<
                    static_cast<uint32_t>(calenderTime.second) << " (" <<
                    calenderAdditionalInfo.timeZone.standardTimeName << ")";

    *pOutLength = (inLength < stringStream.str().length()) ? inLength : stringStream.str().length();

    memcpy(pOutString, stringStream.str().c_str(), *pOutLength);

    nn::time::Finalize();
}

void ConnectionInfoDumper::GetConnectionInfoString(char* pOutString, size_t* pOutLength, size_t inLength, const ConnectionInfoDumper& dumper)
{
    NN_ASSERT_NOT_NULL(pOutString);
    NN_ASSERT_NOT_NULL(pOutLength);

    const DeviceNameDictionaryEntry* pDeviceName = GetDeviceName(dumper.m_Address);

    std::ostringstream stringStream;

    if (pDeviceName != nullptr)
    {
        stringStream << pDeviceName->name << ",";
    }
    else
    {
        stringStream << "Device Name Unknown" << ",";
    }

    stringStream <<
        toHexString(sizeof(nn::bluetooth::Address), dumper.m_Address.address) << "," <<
        dumper.m_Handle << "," <<
        ConvertDisconnectionReasonToString(dumper.m_DisconnectionReason) << "," <<
        dumper.m_Mtu << "," <<
        dumper.m_Parameter.interval << "," << dumper.m_Parameter.latency << "," << dumper.m_Parameter.timeout << "," <<
        dumper.m_ParameterReq.intervalMin << "," << dumper.m_ParameterReq.intervalMax << "," << dumper.m_ParameterReq.latency << "," << dumper.m_ParameterReq.timeout;

    *pOutLength = (inLength < stringStream.str().length()) ? inLength : stringStream.str().length();

    memcpy(pOutString, stringStream.str().c_str(), *pOutLength);
}

void ConnectionInfoDumper::WriteToCsvFile(const CsvEntry* pEntries, int entryCount, const nn::fs::FileHandle& handle)
{
    NN_ASSERT_NOT_NULL(pEntries);

    int64_t fileSize = 0;
    nn::Result result = nn::ResultSuccess();

    for (int i = 0; i < entryCount; ++i)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));

        result = nn::fs::WriteFile(handle, fileSize, pEntries[i].value, pEntries[i].length, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));

        if (nn::fs::ResultUsableSpaceNotEnough().Includes(result))
        {
            nn::err::ShowError(result);
            break;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));

        // コンマ区切り
        result = nn::fs::WriteFile(handle, fileSize, ",", sizeof(","), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));

        if (nn::fs::ResultUsableSpaceNotEnough().Includes(result))
        {
            nn::err::ShowError(result);
            break;
        }
    }

    if (result.IsSuccess())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));
        // 改行を書き込むだけなので、結果は見ない
        nn::fs::WriteFile(handle, fileSize, "\n", sizeof("\n"), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    }
}

void ConnectionInfoDumper::SetConnection(const nn::bluetooth::Address& address, uint32_t connectionHandle)
{
    m_Address   = address;
    m_Handle    = connectionHandle;
}

void ConnectionInfoDumper::UpdateMtu()
{
    NN_ASSERT_NOT_EQUAL(m_Handle, nn::bluetooth::BleInvalidConnectionHandle);

    m_Mtu = nn::bluetooth::GetBleMtu(m_Handle);
}

void ConnectionInfoDumper::UpdateConnectionParameter()
{
    NN_ASSERT_NOT_EQUAL(m_Handle, nn::bluetooth::BleInvalidConnectionHandle);

    while (NN_STATIC_CONDITION(true))
    {
        if (m_Handle == nn::bluetooth::BleInvalidConnectionHandle)
        {
            break;
        }

        if (nn::btm::debug::GetBleConnectionParameter(&m_Parameter.interval, &m_Parameter.latency, &m_Parameter.timeout, m_Handle))
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    }
}

void ConnectionInfoDumper::UpdateConnectionParameterRequest()
{
    NN_ASSERT_NOT_EQUAL(m_Handle, nn::bluetooth::BleInvalidConnectionHandle);

    while (NN_STATIC_CONDITION(true))
    {
        if (m_Handle == nn::bluetooth::BleInvalidConnectionHandle)
        {
            break;
        }

        if(nn::btm::debug::GetBleConnectionParameterRequest(&m_ParameterReq.intervalMin, &m_ParameterReq.intervalMax,
                                                            &m_ParameterReq.latency, &m_ParameterReq.timeout,
                                                            m_Handle))
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    }
}

void ConnectionInfoDumper::UpdateDisconnectionReason()
{
    NN_ASSERT_NOT_EQUAL(m_Handle, nn::bluetooth::BleInvalidConnectionHandle);

    nn::btm::debug::GetGattClientDisconnectionReason(&m_DisconnectionReason, m_Handle, m_Address);
}

const int ConnectionInfoDumper::ms_DeviceNameDisctionaryEntryCountMax = 128;
ConnectionInfoDumper::DeviceNameDictionaryEntry ConnectionInfoDumper::ms_DeviceNameDictionary[ConnectionInfoDumper::ms_DeviceNameDisctionaryEntryCountMax];

void ConnectionInfoDumper::StoreDeviceName(const char* pDeviceName, size_t length, const nn::bluetooth::Address &address)
{
    DeviceNameDictionaryEntry* pTargetEntry = nullptr;

    for (auto& entry : ms_DeviceNameDictionary)
    {
        if (!entry.isRegistered)
        {
            pTargetEntry = &entry;
            break;
        }
    }

    if (pTargetEntry == nullptr)
    {
        for (int i = 0; i < NN_ARRAY_SIZE(ms_DeviceNameDictionary) - 1; ++i)
        {
            ms_DeviceNameDictionary[i] = ms_DeviceNameDictionary[i + 1];
        }

        pTargetEntry = &ms_DeviceNameDictionary[NN_ARRAY_SIZE(ms_DeviceNameDictionary) - 1];
    }

    pTargetEntry->address = address;
    pTargetEntry->length  = (NN_ARRAY_SIZE(pTargetEntry->name) < length) ? NN_ARRAY_SIZE(pTargetEntry->name) : length;
    memcpy(pTargetEntry->name, pDeviceName, pTargetEntry->length);
    pTargetEntry->isRegistered = true;
}

const ConnectionInfoDumper::DeviceNameDictionaryEntry* ConnectionInfoDumper::GetDeviceName(const nn::bluetooth::Address& address)
{
    for (const auto& entry : ms_DeviceNameDictionary)
    {
        if (entry.address == address)
        {
            return &entry;
        }
    }

    return nullptr;
}

const char* ConnectionInfoDumper::ConvertDisconnectionReasonToString(nn::btm::debug::BleDisconnectionReason reason)
{
    switch (reason)
    {
    case nn::btm::debug::BleDisconnectionReason_Unknown:
        return "Unkown";
    case nn::btm::debug::BleDisconnectionReason_L2cFailure:
        return "L2C Failure";
    case nn::btm::debug::BleDisconnectionReason_SupervisionTimeout:
        return "Supervision Timeout";
    case nn::btm::debug::BleDisconnectionReason_TerminatedByPeer:
        return "Terminated byt Peer";
    case nn::btm::debug::BleDisconnectionReason_TerminatedByHost:
        return "Terminated by Host";
    case nn::btm::debug::BleDisconnectionReason_LmpResponseTimeout:
        return "LMP response timeout";
    case nn::btm::debug::BleDisconnectionReason_EstablishFailed:
        return "Establishment Failed";
    case nn::btm::debug::BleDisconnectionReason_L2cConnectionCancled:
        return "L2C connection canceled";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

