﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/bluetooth/bluetooth_Driver.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_IBluetoothDriver.sfdl.h>
#include <nn/bluetooth/bluetooth_ServiceName.h>

class bluetoothClient
{
    private:
        static const size_t     ThreadStackSize = 8192;

        NN_ALIGNAS(4096) char   threadStack[ ThreadStackSize ];
        nn::os::ThreadType      threadForWaitEventSignals;

        nn::os::SystemEventType     systemEventForExitCore;
        nn::os::MultiWaitType       multiWaitForCore;
        nn::os::MultiWaitHolderType holderForCore;
        nn::os::MultiWaitHolderType holderForExitCore;
        static void WaitEventSignals(void* arg);

        NN_ALIGNAS(4096) char   threadStackForHid[ ThreadStackSize ];
        nn::os::ThreadType      threadForWaitEventSignalsForHid;

        nn::os::SystemEventType     systemEventForExitHid;
        nn::os::MultiWaitType       multiWaitForHid;
        nn::os::MultiWaitHolderType holderForHid;
        nn::os::MultiWaitHolderType holderForExitHid;
        nn::os::MultiWaitHolderType holderForHidReport;
        static void WaitEventSignalsForHid(void* arg);


    public:
        nn::os::SystemEventType     systemEventForInputReport;
        nn::os::SystemEventType     systemEventForHid;
        nn::os::SystemEventType     systemEventForCore;

        enum { NOT_BONDING, CANCELING_DISCOVERY, CREATING_BOND } bondingState;
        nn::bluetooth::BluetoothAddress bondingAddr;

        bool fatalErrorOccurred = false;

        int devicesPaired = 0;
        nn::settings::system::BluetoothDevicesSettings devicesSettingsArray[nn::settings::system::BluetoothDevicesSettingsCountMax];

        enum { MAX_CTRL = 10 };
        struct {
            bool                            valid;
            int                             hidCount;
            int                             tsi;
            bool                            tsiChangeAcked;
            nn::bluetooth::BluetoothAddress bdAddr;
            nn::bluetooth::BluetoothHhConnectionState      connectState;
        } controllerInfo[MAX_CTRL];

        void initControllerInfo();
        void printControllerHidCount();
        void setControllerInfoState(const nn::bluetooth::BluetoothAddress &BdAddr, nn::bluetooth::BluetoothHhConnectionState connectState);
        void setControllerHidCount(const nn::bluetooth::BluetoothAddress &BdAddr, int increment = 1);
        void setControllerTsiState(const nn::bluetooth::BluetoothAddress &BdAddr, int tsi, bool ack = 0);

        int  startBluetooth(void);
        void finishBluetooth(void);
        void restartBluetooth(void);

        void AddPairedDevices();
        void GetPairedDevice(const nn::bluetooth::BluetoothAddress &BdAddr);
        void RemovePairedDevice(const nn::bluetooth::BluetoothAddress &BdAddr);

        void setTsi(const nn::bluetooth::BluetoothAddress &bdAddr, int tsi);

        static bool        isNintendoGamepad(const nn::bluetooth::Btbdname &name);
        static const char* toHexString(uint16_t length, const uint8_t buffer[]);
        static const char* toHexString(const nn::bluetooth::BluetoothAddress &BdAddress);

        // core events
        virtual void EventFromDeviceFoundCallback(           const nn::bluetooth::InfoFromDeviceFoundCallback* pInfo);
        virtual void EventFromPinRequestCallback(            const nn::bluetooth::InfoFromPinRequestCallback* pInfo);
        virtual void EventFromDiscoveryStateChangedCallback( const nn::bluetooth::InfoFromDiscoveryStateChangedCallback* pInfo);
        virtual void EventFromBondStateChangedCallback(      const nn::bluetooth::InfoFromBondStateChangedCallback* pInfo);

        // hid events
        virtual void EventFromConnectionStateCallback(       const nn::bluetooth::InfoFromConnectionStateCallback* pInfo);
        virtual void EventFromExtensionCallbacks(            const nn::bluetooth::InfoFromExtensionCallbacks* pInfo);
        virtual void EventFromGetReportDataCallback(         const nn::bluetooth::InfoFromGetReportCallback* pInfo);
        virtual void EventFromSetReportStatusCallback(       const nn::bluetooth::InfoFromSetReportStatusCallback* pInfo);
        virtual void EventFromGetReportStatusCallback(       const nn::bluetooth::InfoFromGetReportStatusCallback* pInfo);

};


enum PeriodicDataFormat : uint8_t
{
    PeriodicDataFormat_Basic = 0,
    PeriodicDataFormat_MCU ,
    PeriodicDataFormat_Attachment ,
    PeriodicDataFormat_Audio,
};

enum ReportIdDataLength : uint16_t
{
    ReportIdDataLength_Basic      = 59,
    ReportIdDataLength_MCU        = 372,
    ReportIdDataLength_Attachment = 372,
    ReportIdDataLength_Audio      = 372
};

enum ReportId : uint8_t
{
    ReportId_Basic      = 0x30,
    ReportId_MCU        = 0x31,
    ReportId_Attachment = 0x32,
    ReportId_Audio      = 0x33,
    ReportId_Generic    = 0x3f
};
