﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include "../Common/testBle_Common.h"
#include "../Common/testBle_ScanCommon.h"

#include <nn/bluetooth/bluetooth_LeApi.h>
#include <nn/bluetooth/bluetooth_BleScanParameterIdTestPeripheral.h>

//!< General Scan
TEST(TestBleScan, GeneralScanSimple)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::scan::SetUp();

    nn::bluetooth::BleAdvertisePacketParameter param;
    nn::bluetooth::GetBleScanParameter(&param, nn::bluetooth::BleScanParameterId_TestPeripheral);

    nn::Result result = nn::ResultSuccess();
    int generalErrorRetryCount = 0;

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StartBleScanGeneral(param);

        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            generalErrorRetryCount++;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else
        {
            EXPECT_TRUE(result.IsSuccess());
            break;
        }
    }

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StopBleScanGeneral();

        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            generalErrorRetryCount++;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else
        {
            EXPECT_TRUE(result.IsSuccess());
            break;
        }
    }

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::scan::TearDown();
}

//!< Smart Device Scan
TEST(TestBleScan, SmartDeviceScanSimple)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::scan::SetUp();

    nn::Result result = nn::ResultSuccess();
    int generalErrorRetryCount = 0;

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StartBleScanSmartDevice(nnt::bluetooth::scan::testPeripheralAdvUuid);

        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            generalErrorRetryCount++;
            // 10 回までリトライを許可する
            EXPECT_LE(generalErrorRetryCount, 10);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else
        {
            EXPECT_TRUE(result.IsSuccess());
            break;
        }
    }

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StopBleScanSmartDevice();

        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            generalErrorRetryCount++;
            // 10 回までリトライを許可する
            EXPECT_LE(generalErrorRetryCount, 10);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else
        {
            EXPECT_TRUE(result.IsSuccess());
            break;
        }
    }

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::scan::TearDown();
}

//!< General Scan 境界値テスト
TEST(TestBleScan, GeneralScanBoundary)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::scan::SetUp();

    nn::bluetooth::BleAdvertisePacketParameter param;
    nn::bluetooth::GetBleScanParameter(&param, nn::bluetooth::BleScanParameterId_TestPeripheral);

    nn::Result result = nn::ResultSuccess();
    int generalErrorRetryCount = 0;
    int registeredFilterCount = 0;

    while (NN_STATIC_CONDITION(true))
    {
        param.manufacturerId[0] = static_cast<uint8_t>(registeredFilterCount);

        result = nn::bluetooth::StartBleScanGeneral(param);

        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            generalErrorRetryCount++;
            // 10 回までリトライを許可する
            EXPECT_LE(generalErrorRetryCount, 10);
        }
        else if(result.IsSuccess())
        {
            generalErrorRetryCount = 0;
            registeredFilterCount++;
        }
        else if (nn::bluetooth::ResultBleScanFull().Includes(result))
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    // 最大で5 個までフィルタ登録できる
    EXPECT_LE(registeredFilterCount, 5);

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StopBleScanGeneral();

        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            generalErrorRetryCount++;
            // 10 回までリトライを許可する
            EXPECT_LE(generalErrorRetryCount, 10);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else
        {
            break;
        }
    }

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::scan::TearDown();
}

//!< Smart Device Scan 境界値テスト
TEST(TestBleScan, SmartDeviceScanBoundary)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::scan::SetUp();

    nn::Result result = nn::ResultSuccess();
    int generalErrorRetryCount = 0;
    int registeredFilterCount = 0;

    while (NN_STATIC_CONDITION(true))
    {
        nn::bluetooth::GattAttributeUuid uuid = nnt::bluetooth::scan::testPeripheralAdvUuid;
        uuid.uu.uuid128[0] = registeredFilterCount;

        result = nn::bluetooth::StartBleScanSmartDevice(uuid);

        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            generalErrorRetryCount++;
            // 10 回までリトライを許可する
            EXPECT_LE(generalErrorRetryCount, 10);
        }
        else if (result.IsSuccess())
        {
            registeredFilterCount++;
        }
        else if (nn::bluetooth::ResultBleScanFull().Includes(result))
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
    }

    // 最大で5 個までフィルタ登録できる
    EXPECT_LE(registeredFilterCount, 5);

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StopBleScanGeneral();

        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            generalErrorRetryCount++;
            // 10 回までリトライを許可する
            EXPECT_LE(generalErrorRetryCount, 10);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
        else
        {
            break;
        }
    }

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::scan::TearDown();
}

