﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testBcat_Common.h"

#include <nn/bcat/detail/service/core/bcat_TaskManager.h>

using namespace nn::bcat::detail::service::core;

TEST(TaskManager, Initialize)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("bcat-sys", "C:\\siglo\\bcat-sys"));
    FileSystem::EnableTestMode();

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Load());

    srand(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()));
}

TEST(TaskManager, Clear)
{
    // 複数回呼出
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Clear());
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Clear());
}

TEST(TaskManager, AddOrUpdate)
{
    // 追加
    for (int i = 0; i < nn::bcat::TaskCountMax; i++)
    {
        nn::ApplicationId appId = {static_cast<nn::Bit64>(i + 1)};

        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 0));
        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 0));
    }

    // 追加（古いタスク追い出し）
    for (int i = nn::bcat::TaskCountMax; i < nn::bcat::TaskCountMax * 2; i++)
    {
        nn::ApplicationId appId = {static_cast<nn::Bit64>(i + 1)};

        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 0));
    }

    // 購読処理は行っていないため、購読解除すべきタスクはないはず。
    nn::bcat::detail::TopicId topicId;
    ASSERT_FALSE(TaskManager::GetInstance().GetNpnsTopicToBeUnsubscribed(&topicId));
}

TEST(TaskManager, RunTask)
{
    TaskManager::Record record = {};

    for (int i = 0; i < nn::bcat::TaskCountMax; i++)
    {
        ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));
        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyDone(record.appId, nn::ResultSuccess(), record.revisionHash));
    }

    ASSERT_FALSE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));
}

TEST(TaskManager, Subscribe)
{
    // 最後に登録されたもの程上位に現れる。
    for (int i = nn::bcat::TaskCountMax * 2 - 1; i >= nn::bcat::TaskCountMax; i--)
    {
        nn::bcat::detail::TopicId topicId = {};
        ASSERT_TRUE(TaskManager::GetInstance().GetNpnsTopicToBeSubscribed(&topicId));

        nn::bcat::detail::TopicId expectTopicId = {};
        nn::util::SNPrintf(expectTopicId.value, sizeof (expectTopicId.value), "nx_data_%016llx", i + 1);

        EXPECT_TRUE(nn::util::Strncmp(topicId.value, expectTopicId.value, sizeof (topicId.value)) == 0);

        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyNpnsTopicSubscriptionProcessed(topicId, nn::ResultSuccess()));
    }

    nn::bcat::detail::TopicId topicId = {};
    ASSERT_FALSE(TaskManager::GetInstance().GetNpnsTopicToBeSubscribed(&topicId));
    ASSERT_FALSE(TaskManager::GetInstance().GetNpnsTopicToBeUnsubscribed(&topicId));
}

TEST(TaskManager, Remove)
{
    for (int i = nn::bcat::TaskCountMax; i < nn::bcat::TaskCountMax * 2; i++)
    {
        nn::ApplicationId appId = {static_cast<nn::Bit64>(i + 1)};

        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Remove(appId));
        // 2 回目は無視する。
        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Remove(appId));
    }

    for (int i = 0; i < nn::bcat::TaskCountMax; i++)
    {
        nn::ApplicationId appId = {static_cast<nn::Bit64>(i + 1)};

        // 存在しない場合も成功する。
        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Remove(appId));
    }
}

TEST(TaskManager, Unsubscribe)
{
    for (int i = nn::bcat::TaskCountMax; i < nn::bcat::TaskCountMax * 2; i++)
    {
        nn::bcat::detail::TopicId topicId = {};
        ASSERT_TRUE(TaskManager::GetInstance().GetNpnsTopicToBeUnsubscribed(&topicId));

        nn::bcat::detail::TopicId expectTopicId = {};
        nn::util::SNPrintf(expectTopicId.value, sizeof (expectTopicId.value), "nx_data_%016llx", static_cast<nn::Bit64>(i + 1));

        EXPECT_TRUE(nn::util::Strncmp(topicId.value, expectTopicId.value, sizeof (topicId.value)) == 0);

        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyNpnsTopicUnsubscriptionProcessed(topicId, nn::ResultSuccess()));
    }

    nn::bcat::detail::TopicId topicId = {};
    ASSERT_FALSE(TaskManager::GetInstance().GetNpnsTopicToBeSubscribed(&topicId));
    ASSERT_FALSE(TaskManager::GetInstance().GetNpnsTopicToBeUnsubscribed(&topicId));
}

TEST(TaskManager, Wait)
{
    nn::ApplicationId appId = {static_cast<nn::Bit64>(1)};
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 0));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Wait(appId, 3));

    TaskManager::Record record = {};
    ASSERT_FALSE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
    TaskManager::GetInstance().Schedule();

    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Clear());
}

TEST(TaskManager, NotificationInProgress)
{
    nn::ApplicationId appId = {static_cast<nn::Bit64>(1)};
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 0));

    TaskManager::Record record = {};
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    nn::bcat::detail::TopicId topicId = {};
    nn::util::SNPrintf(topicId.value, sizeof (topicId.value), "nx_data_%016llx", appId.value);

    nn::bcat::detail::RevisionHash newRevisionHash1 = {{0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88}};
    nn::bcat::detail::RevisionHash newRevisionHash2 = {{0x88, 0x77, 0x66, 0x55, 0x44, 0x33, 0x22, 0x11}};

    // 処理中にリビジョンを更新。(1 回目)
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyNotificationReceived(topicId, newRevisionHash1));
    TaskManager::GetInstance().Schedule();
    // 処理中にリビジョンを更新。(2 回目)
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyNotificationReceived(topicId, newRevisionHash2));
    TaskManager::GetInstance().Schedule();

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyDone(appId, nn::ResultSuccess(), record.revisionHash));
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_TRUE(std::memcmp(&record.revisionHash, &newRevisionHash2, sizeof (record.revisionHash)) == 0);

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyDone(appId, nn::ResultSuccess(), record.revisionHash));
    ASSERT_FALSE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Clear());
}

TEST(TaskManager, ErrorNintendoAccountRequired)
{
    nn::ApplicationId appId = {static_cast<nn::Bit64>(1)};
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 0));

    TaskManager::Record record = {};
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Wait(appId));
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyDone(appId, nn::bcat::ResultNintendoAccountNotLinked(), record.revisionHash));

    ASSERT_FALSE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    TaskManager::GetInstance().NotifyNintendoAccountLinked();
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Clear());
}

TEST(TaskManager, ErrorApplicationUpdateRequired)
{
    nn::ApplicationId appId = {static_cast<nn::Bit64>(1)};
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 0));

    TaskManager::Record record = {};
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Wait(appId));
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyDone(appId, nn::bcat::ResultApplicationUpdateRequired(), record.revisionHash));

    ASSERT_FALSE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 1));
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Clear());
}

TEST(TaskManager, ErrorServiceExpired)
{
    nn::ApplicationId appId = {static_cast<nn::Bit64>(1)};
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 0));

    TaskManager::Record record = {};
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Wait(appId));
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyDone(appId, nn::bcat::ResultApplicationServiceExpired(), record.revisionHash));

    ASSERT_FALSE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    nn::bcat::detail::TopicId topicId = {};
    nn::util::SNPrintf(topicId.value, sizeof (topicId.value), "nx_data_%016llx", appId.value);

    nn::bcat::detail::RevisionHash newRevisionHash = {{0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88}};

    // サービス終了状態からの復帰。
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().NotifyNotificationReceived(topicId, newRevisionHash));
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_TRUE(std::memcmp(&record.revisionHash, &newRevisionHash, sizeof (record.revisionHash)) == 0);

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Clear());
}

TEST(TaskManager, SuspendResume)
{
    nn::ApplicationId appId = {static_cast<nn::Bit64>(1)};
    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, 0));

    TaskManager::Record record = {};
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Suspend(appId));
    ASSERT_FALSE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Resume(appId));
    ASSERT_TRUE(TaskManager::GetInstance().GetRunnableTaskRecord(&record));

    ASSERT_RESULT(TaskManager::GetInstance().Resume(appId), nn::bcat::ResultInvalidOperation);

    for (int i = 0; i < UINT8_MAX; i++)
    {
        ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Suspend(appId));
    }

    ASSERT_RESULT(TaskManager::GetInstance().Suspend(appId), nn::bcat::ResultInvalidOperation);

    ASSERT_RESULT_SUCCESS(TaskManager::GetInstance().Clear());
}
