﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testBcat_Common.h"

#include <nn/bcat/detail/service/core/bcat_DeliveryListReader.h>
#include <nn/bcat/detail/service/msgpack/bcat_FileInputStream.h>

using namespace nn::bcat::detail::service::core;

namespace
{
    struct DirectoryEntryCount
    {
        const char* dirName;
        int count;
    };

    const DirectoryEntryCount ExpectedDirectoryEntryCountsForList[] =
    {
        {"dir1", 2},
        {"dir2", 1},
        {"dir3", 7},
        {"dir4", 2},
        {"dir5", 1},
        {"empty_dir", 0}
    };

    const DirectoryEntryCount ExpectedDirectoryEntryCountsForIrregularList[] =
    {
        {"quest", 1},
        {"huge", 2},
        {"empty", 0},
        {"cdn_test", 6}
    };
}

void ReadListFile(size_t* outSize, const char* path, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    static nn::Bit8 s_MountRomCacheBuffer[4 * 1024];

    size_t mountRomCacheUseSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));

    NN_ABORT_UNLESS(mountRomCacheUseSize <= sizeof (s_MountRomCacheBuffer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("rom", s_MountRomCacheBuffer, sizeof (s_MountRomCacheBuffer)));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    nn::fs::FileHandle handle = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(outSize, handle, 0, buffer, bufferSize));
}

void DumpListHeader(const nn::bcat::detail::ListHeader& header) NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("ListHeader\n");
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("topicId:\n");
    NN_LOG("    %s\n", header.topicId.value);
    NN_LOG("isInService:\n");
    NN_LOG("    %d\n", header.isInService);
    NN_LOG("isNintendoAccountRequired:\n");
    NN_LOG("    %d\n", header.isNintendoAccountRequired);
    NN_LOG("requiredAppVersion:\n");
    NN_LOG("    %u\n", header.requiredAppVersion);
}

void DumpListDirectory(const nn::bcat::detail::ListDirectory& dir) NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("ListDirectory\n");
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("name:\n");
    NN_LOG("    %s\n", dir.name.value);
    NN_LOG("mode:\n");
    NN_LOG("    %s\n", dir.mode == nn::bcat::detail::DirectoryMode_Sync ? "sync" : "copy");
    NN_LOG("digest:\n");
    NN_LOG("    %016llx%016llx\n", dir.digest.value[0], dir.digest.value[1]);
}

void DumpListFile(const nn::bcat::detail::ListFile* entries, int count)
{
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------\n");
        NN_LOG("ListFile[%d]\n", i);
        NN_LOG("--------------------------------------------------\n");
        NN_LOG("name:\n");
        NN_LOG("    %s\n", entries[i].name.value);
        NN_LOG("dataId:\n");
        NN_LOG("    %llu\n", entries[i].dataId);
        NN_LOG("size:\n");
        NN_LOG("    %llu\n", entries[i].size);
        NN_LOG("url:\n");
        NN_LOG("    %s\n", entries[i].url.value);
        NN_LOG("digest:\n");
        NN_LOG("    %016llx%016llx\n", entries[i].digest.value[0], entries[i].digest.value[1]);
    }
}

void EnumerateCallback(const nn::bcat::detail::ListDirectory& dir, const nn::bcat::detail::ListFile* files, int count, void* param) NN_NOEXCEPT
{
    NN_UNUSED(param);

    NN_LOG("--------------------------------------------------\n");
    NN_LOG("EnumeratedDirectory\n");
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("name:\n");
    NN_LOG("    %s\n", dir.name.value);
    NN_LOG("mode:\n");
    NN_LOG("    %s\n", dir.mode == nn::bcat::detail::DirectoryMode_Sync ? "sync" : "copy");
    NN_LOG("digest:\n");
    NN_LOG("    %016llx%016llx\n", dir.digest.value[0], dir.digest.value[1]);

    DumpListFile(files, count);
}

TEST(DeliveryListReader, EnumerateList)
{
    static nn::Bit8 s_Buffer[32 * 1024];
    size_t fileSize = 0;

    ReadListFile(&fileSize, "rom:/list.msgpack", s_Buffer, sizeof (s_Buffer));

    static nn::bcat::detail::ListDirectory s_Dirs[nn::bcat::DeliveryCacheDirectoryCountMax];
    int dirCount = 0;

    static nn::bcat::detail::ListFile s_Files[nn::bcat::DeliveryCacheFileCountMaxPerDirectory];
    int fileCount = 0;

    nne::nlib::MemoryInputStream stream;

    nn::bcat::detail::ListHeader header1;

    {
        DeliveryListReader reader;

        reader.SetupForDirectoryListing(&dirCount, s_Dirs, NN_ARRAY_SIZE(s_Dirs));

        stream.Init(s_Buffer, fileSize);
        ASSERT_RESULT_SUCCESS(reader.Read(&header1, stream));
        stream.Close();

        DumpListHeader(header1);
    }

    for (int i = 0; i < dirCount; i++)
    {
        DumpListDirectory(s_Dirs[i]);

        DeliveryListReader reader;
        nn::bcat::detail::ListHeader header2;

        reader.SetupForFileListing(&fileCount, s_Files, NN_ARRAY_SIZE(s_Files), s_Dirs[i].name.value);

        stream.Init(s_Buffer, fileSize);
        ASSERT_RESULT_SUCCESS(reader.Read(&header2, stream));
        stream.Close();

        bool isFound = false;

        for (const auto& e : ExpectedDirectoryEntryCountsForList)
        {
            if (nn::util::Strncmp(e.dirName, s_Dirs[i].name.value, sizeof (s_Dirs[i].name.value)) == 0 && e.count == fileCount)
            {
                isFound = true;
                break;
            }
        }

        EXPECT_TRUE(isFound);

        DumpListFile(s_Files, fileCount);

        EXPECT_TRUE(std::memcmp(&header1, &header2, sizeof (header1)) == 0);
    }

    {
        DeliveryListReader reader;
        nn::bcat::detail::ListHeader header3;

        static DeliveryListReader::WalkWorkBuffer s_WalkBuffer;

        reader.SetupForWalking(EnumerateCallback, nullptr, s_WalkBuffer);

        stream.Init(s_Buffer, fileSize);
        ASSERT_RESULT_SUCCESS(reader.Read(&header3, stream));
        stream.Close();

        DumpListHeader(header3);

        EXPECT_TRUE(std::memcmp(&header1, &header3, sizeof (header1)) == 0);
    }
}

TEST(DeliveryListReader, EnumerateIrregularList)
{
    static nn::Bit8 s_Buffer[32 * 1024];
    size_t fileSize = 0;

    ReadListFile(&fileSize, "rom:/irregular_list.msgpack", s_Buffer, sizeof (s_Buffer));

    static nn::bcat::detail::ListDirectory s_Dirs[nn::bcat::DeliveryCacheDirectoryCountMax];
    int dirCount = 0;

    static nn::bcat::detail::ListFile s_Files[nn::bcat::DeliveryCacheFileCountMaxPerDirectory];
    int fileCount = 0;

    nne::nlib::MemoryInputStream stream;

    nn::bcat::detail::ListHeader header1;

    {
        DeliveryListReader reader;

        reader.SetupForDirectoryListing(&dirCount, s_Dirs, NN_ARRAY_SIZE(s_Dirs));

        stream.Init(s_Buffer, fileSize);
        ASSERT_RESULT_SUCCESS(reader.Read(&header1, stream));
        stream.Close();

        DumpListHeader(header1);
    }

    for (int i = 0; i < dirCount; i++)
    {
        DumpListDirectory(s_Dirs[i]);

        DeliveryListReader reader;
        nn::bcat::detail::ListHeader header2;

        reader.SetupForFileListing(&fileCount, s_Files, NN_ARRAY_SIZE(s_Files), s_Dirs[i].name.value);

        stream.Init(s_Buffer, fileSize);
        ASSERT_RESULT_SUCCESS(reader.Read(&header2, stream));
        stream.Close();

        bool isFound = false;

        for (const auto& e : ExpectedDirectoryEntryCountsForIrregularList)
        {
            if (nn::util::Strncmp(e.dirName, s_Dirs[i].name.value, sizeof (s_Dirs[i].name.value)) == 0 && e.count == fileCount)
            {
                isFound = true;
                break;
            }
        }

        EXPECT_TRUE(isFound);

        DumpListFile(s_Files, fileCount);

        EXPECT_TRUE(std::memcmp(&header1, &header2, sizeof (header1)) == 0);
    }

    {
        DeliveryListReader reader;
        nn::bcat::detail::ListHeader header3;

        static DeliveryListReader::WalkWorkBuffer s_WalkBuffer;

        reader.SetupForWalking(EnumerateCallback, nullptr, s_WalkBuffer);

        stream.Init(s_Buffer, fileSize);
        ASSERT_RESULT_SUCCESS(reader.Read(&header3, stream));
        stream.Close();

        DumpListHeader(header3);

        EXPECT_TRUE(std::memcmp(&header1, &header3, sizeof (header1)) == 0);
    }
}
