﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testBcat_Common.h"

void DumpProgress(const nn::bcat::DeliveryCacheProgress& progress)  NN_NOEXCEPT
{
    switch (progress.GetStatus())
    {
    case nn::bcat::DeliveryCacheProgressStatus_Queued:
        NN_LOG("Progress: status = queued\n");
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Connect:
        NN_LOG("Progress: status = connect\n");
        break;
    case nn::bcat::DeliveryCacheProgressStatus_ProcessList:
        NN_LOG("Progress: status = process list\n");
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Download:
        NN_LOG("Progress: status = download, current = (%s/%s, %lld/%lld), whole = (%lld/%lld)\n",
            progress.GetCurrentDirectoryName().value, progress.GetCurrentFileName().value,
            progress.GetCurrentDownloaded(), progress.GetCurrentTotal(),
            progress.GetWholeDownloaded(), progress.GetWholeTotal());
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Commit:
        NN_LOG("Progress: status = commit, current = (%s), whole = (%lld/%lld)\n",
            progress.GetCurrentDirectoryName().value,
            progress.GetWholeDownloaded(), progress.GetWholeTotal());
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Done:
        NN_LOG("Progress: status = done, whole = (%lld/%lld), result = %03d-%04d\n",
            progress.GetWholeDownloaded(), progress.GetWholeTotal(),
            progress.GetResult().GetModule(), progress.GetResult().GetDescription());
        break;
    default:
        break;
    }
}

nn::Result MountDeliveryCacheStorage() NN_NOEXCEPT
{
    nn::Result result = nn::bcat::MountDeliveryCacheStorage();

    if (result.IsSuccess())
    {
        NN_RESULT_SUCCESS;
    }
    else if (nn::bcat::ResultLocked::Includes(result))
    {
        while (nn::bcat::ResultLocked::Includes(result))
        {
            NN_LOG("The delivery cache storage is locking.\n");

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

            result = nn::bcat::MountDeliveryCacheStorage();
        }
    }

    if (nn::bcat::ResultNintendoAccountNotLinked::Includes(result))
    {
        NN_LOG("[Warning] Please link Nintendo account or register Network service account.\n");
        return result;
    }
    else
    {
        return result;
    }
}

TEST(DeliveryCacheStorageAccessor, Mount)
{
    ASSERT_RESULT_SUCCESS(MountDeliveryCacheStorage());

    nn::bcat::UnmountDeliveryCacheStorage();
}

TEST(DeliveryCacheStorageAccessor, EnumerateAll)
{
    ASSERT_RESULT_SUCCESS(MountDeliveryCacheStorage());

    NN_UTIL_SCOPE_EXIT
    {
        nn::bcat::UnmountDeliveryCacheStorage();
    };

    static nn::bcat::DirectoryName s_DirNames[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    ASSERT_RESULT_SUCCESS(nn::bcat::EnumerateDeliveryCacheDirectory(&count, s_DirNames, NN_ARRAY_SIZE(s_DirNames)));

    NN_LOG("----------------------------------------------------------------------------------------------------\n");
    NN_LOG("DirectoryCount = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("----------------------------------------------------------------------------------------------------\n");
        NN_LOG("Directory[%3d] = %s\n", i, s_DirNames[i].value);

        nn::bcat::DeliveryCacheDirectory directory;

        ASSERT_RESULT_SUCCESS(directory.Open(s_DirNames[i]));

        int entryCount1 = directory.GetCount();

        static nn::bcat::DeliveryCacheDirectoryEntry s_Entries[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
        int entryCount2 = 0;

        ASSERT_RESULT_SUCCESS(directory.Read(&entryCount2, s_Entries, NN_ARRAY_SIZE(s_Entries)));

        EXPECT_EQ(entryCount1, entryCount2);

        NN_LOG("    FileCount = %d\n", entryCount1);

        for (int e = 0; e < entryCount1; e++)
        {
            NN_LOG("    Entry[%3d] = %32s, %016llx%016llx, %lld\n", e,
                s_Entries[e].name.value, s_Entries[e].digest.value[0], s_Entries[e].digest.value[1], s_Entries[e].size);

            nn::bcat::DeliveryCacheFile file;

            ASSERT_RESULT_SUCCESS(file.Open(s_DirNames[i], s_Entries[e].name));

            int64_t size = file.GetSize();

            EXPECT_EQ(size, s_Entries[e].size);

            nn::bcat::Digest digest = file.GetDigest();

            EXPECT_TRUE(std::memcmp(digest.value, s_Entries[e].digest.value, sizeof (digest.value)) == 0);

            nn::Bit8 data[8] = {};
            size_t read;

            ASSERT_RESULT_SUCCESS(file.Read(&read, 0, data, NN_ARRAY_SIZE(data)));

            NN_LOG("               = ");

            for (int r = 0; r < static_cast<int>(read); r++)
            {
                char c = static_cast<char>(data[r]);

                NN_LOG("%c(%02X) ", (c >= 0x20 && c <= 0x7E) ? c : '?', data[r]);
            }

            NN_LOG("\n");
        }
    }

    NN_LOG("----------------------------------------------------------------------------------------------------\n");
}

// TODO: UpperCase で読み込み可能にするかどうか決めたら有効化する。
/*
TEST(DeliveryCacheStorageAccessor, EnumerateAllUpperCase)
{
    ASSERT_RESULT_SUCCESS(MountDeliveryCacheStorage());

    NN_UTIL_SCOPE_EXIT
    {
        nn::bcat::UnmountDeliveryCacheStorage();
    };

    static nn::bcat::DirectoryName s_DirNames[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    ASSERT_RESULT_SUCCESS(nn::bcat::EnumerateDeliveryCacheDirectory(&count, s_DirNames, NN_ARRAY_SIZE(s_DirNames)));

    for (int i = 0; i < count; i++)
    {
        for (int n = 0; n < sizeof (s_DirNames[i].value); n++)
        {
            if (s_DirNames[i].value[n] != '\0')
            {
                s_DirNames[i].value[n] = static_cast<char>(std::toupper(s_DirNames[i].value[n]));
            }
        }

        nn::bcat::DeliveryCacheDirectory directory;

        ASSERT_RESULT_SUCCESS(directory.Open(s_DirNames[i]));

        static nn::bcat::DeliveryCacheDirectoryEntry s_Entries[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
        int entryCount = 0;

        ASSERT_RESULT_SUCCESS(directory.Read(&entryCount, s_Entries, NN_ARRAY_SIZE(s_Entries)));

        for (int e = 0; e < entryCount; e++)
        {
            for (int n = 0; n < sizeof (s_Entries[i].name.value); n++)
            {
                if (s_Entries[i].name.value[n] != '\0')
                {
                    s_Entries[i].name.value[n] = static_cast<char>(std::toupper(s_Entries[i].name.value[n]));
                }
            }

            nn::bcat::DeliveryCacheFile file;

            ASSERT_RESULT_SUCCESS(file.Open(s_DirNames[i], s_Entries[e].name));
        }
    }
}
*/

TEST(DeliveryCacheStorageAccessor, NotExists)
{
    ASSERT_RESULT_SUCCESS(MountDeliveryCacheStorage());

    NN_UTIL_SCOPE_EXIT
    {
        nn::bcat::UnmountDeliveryCacheStorage();
    };

    nn::bcat::DeliveryCacheDirectory directory;

    ASSERT_RESULT(directory.Open("not_exists_dir"), nn::bcat::ResultNotFound);

    nn::bcat::DeliveryCacheFile file;

    ASSERT_RESULT(file.Open("not_exists_dir/not_exists_file"), nn::bcat::ResultNotFound);
}

TEST(DeliveryCacheStorageAccessor, SystemAutoDelete)
{
    ASSERT_RESULT_SUCCESS(MountDeliveryCacheStorage());

    static nn::bcat::DirectoryName s_DirNames[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    ASSERT_RESULT_SUCCESS(nn::bcat::EnumerateDeliveryCacheDirectory(&count, s_DirNames, NN_ARRAY_SIZE(s_DirNames)));

    if (count > 0)
    {
        // アプリケーションの終了時に破棄されるようにする。
        static nn::bcat::DeliveryCacheDirectory s_Directory;

        ASSERT_RESULT_SUCCESS(s_Directory.Open(s_DirNames[0]));
    }
}
