﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/applet/applet.h>

#include <mutex>

#include "../Common/applet_Configs.h"

//-----------------------------------------------------------------------------
//
void ProcessMessage(const char* pKind, nn::ae::Message message) NN_NOEXCEPT
{

    switch (message)
    {
        case nn::ae::Message_ChangeIntoForeground:
                nn::ae::AcquireForegroundRights();
                NN_LOG("%s: Done AcquireForegroundRights()\n", pKind);
                break;

        case nn::ae::Message_ChangeIntoBackground:
                nn::ae::ReleaseForegroundRights();
                NN_LOG("%s: Done ReleaseForegroundRights()\n", pKind);
                break;

        default:
                break;
    }
}

void PrintAppletMessage(const char* pKind, nn::ae::Message message) NN_NOEXCEPT
{
    NN_LOG("%s: ", pKind);

    switch (message)
    {
        case nn::ae::Message_ChangeIntoForeground:
                NN_LOG("Received Message_ChangeIntoForeground\n");
                break;

        case nn::ae::Message_ChangeIntoBackground:
                NN_LOG("Received Message_ChangeIntoBackground\n");
                break;

        case nn::ae::Message_Exit:
                NN_LOG("Received Message_Exit\n");
                break;

        case nn::ae::Message_ApplicationExited:
                NN_LOG("Received Message_ApplicationExited\n");
                break;

        case nn::ae::Message_DetectShortPressingHomeButton:
                NN_LOG("Received Message_DetectShortPressingHomeButton\n");
                break;

        case nn::ae::Message_DetectLongPressingHomeButton:
                NN_LOG("Received Message_DetectLongPressingHomeButton\n");
                break;

        case nn::ae::Message_DetectShortPressingPowerButton:
                NN_LOG("Received Message_DetectShortPressingPowerButton\n");
                break;

        case nn::ae::Message_DetectMiddlePressingPowerButton:
                NN_LOG("Received Message_DetectMiddlePressingPowerButton\n");
                break;

        case nn::ae::Message_DetectLongPressingPowerButton:
                NN_LOG("Received Message_DetectLongPressingPowerButton\n");
                break;

        case nn::ae::Message_FloatingApplicationDetected:
                NN_LOG("Received Message_FloatingApplicationDetected\n");
                break;

        case nn::ae::Message_FinishedSleepSequence:
                NN_LOG("Received Message_FinishedSleepSequence\n");
                break;

        case nn::ae::Message_RequestToPrepareSleep:
                NN_LOG("Received Message_RequestToPrepareSleep\n");
                break;

        case nn::ae::Message_LaunchApplicationRequested:
                NN_LOG("Received Message_LaunchApplicationRequested\n");
                break;

        default:
                NN_ABORT("Received unknown message= 0x%08x", message);
                break;
    }
}

bool CheckAndProcessMessageImpl(const char* pKind, nn::os::SystemEventType* pEvent, nn::ae::Message expectMessage) NN_NOEXCEPT
{
    auto message = nn::ae::WaitForNotificationMessage(pEvent);
    PrintAppletMessage(pKind, message);

    if (!(message == expectMessage))
    {
        NN_LOG("%s: Received unexpected expect=0x%08x message=0x%08x\n", pKind, expectMessage, message);
        return false;
    }

    ProcessMessage(pKind, message);
    return true;
}

bool WaitForAllMessageArrivedImpl(const char* pKind, nn::os::SystemEventType* pEvent, ...) NN_NOEXCEPT
{
    nn::Bit64 messageBits = 0;

    va_list args;
    va_start(args, pEvent);
    for (;;)
    {
        int msg = va_arg(args, int);
        if (msg < 0)
        {
            break;
        }

        NN_ASSERT(msg < sizeof(nn::Bit64) * 8);
        messageBits |= 1ul << msg;
    }
    va_end(args);

    // 全メッセージが到着するまで待機
    while (messageBits != 0)
    {
        auto message = nn::ae::WaitForNotificationMessage(pEvent);
        PrintAppletMessage(pKind, message);

        nn::Bit64  msgBits = 1ul << message;
        if (!(messageBits & msgBits))
        {
            NN_LOG("%s: Received unexpected message=0x%08x", pKind, message);
            return false;
        }
        messageBits &= ~msgBits;

        ProcessMessage(pKind, message);
    }

    return true;
}


//-----------------------------------------------------------------------------
// buffer から適当なサイズ分のメモリダンプを行なう
//
void DumpMemory(const char* pKind, const char* buffer)
{
    NN_LOG("%s: buffer=", pKind);
    for (int i=0; i<16; ++i)
    {
        NN_LOG(" %02x", buffer[i]);
    }
    NN_LOG("\n");
}

//-----------------------------------------------------------------------------
// buffer から size 分を、seed に基づくデータ列で埋める。
//
void FillMemory(char* buffer, size_t size, char seed) NN_NOEXCEPT
{
    char c = seed;
    for (int i=0; i<size; ++i)
    {
        buffer[i] = c;

        c = (c < 0x7e) ? c + 1 : ' ';
    }
}

//-----------------------------------------------------------------------------
// buffer から size 分が、seed に基づくデータ列で埋まっているかチェックする
//
void CheckMemory(const char* buffer, size_t size, char seed) NN_NOEXCEPT
{
    char c = seed;
    for (int i=0; i<size; ++i)
    {
        NN_ABORT_UNLESS(buffer[i] == c);

        c = (c < 0x7e) ? c + 1 : ' ';
    }
}

