﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/applet/applet.h>

#include "../Common/applet_Common.h"
#include "../Common/applet_Configs.h"

//-----------------------------------------------------------------------------

void WaitInterval() NN_NOEXCEPT;

namespace {

char g_StartupParamBuffer[ nn::applet::StartupParamSizeMax ];
char g_ReturnValueBuffer[ nn::applet::ReturnValueSizeMax ];

const size_t RequestCommandSize = 128 * 1024;
const size_t ReplyMessageSize   = 128 * 1024;
NN_ALIGNAS(4096) char g_RequestCommandBuffer[ RequestCommandSize ];
NN_ALIGNAS(4096) char g_ReplayMessageBuffer[ ReplyMessageSize ];

}

//-----------------------------------------------------------------------------
//  ライブラリアプレットの Foreground 起動
//
void InvokeLibraryAppletInForegroundState(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    // ライブラリアプレット起動準備
    NN_LOG("(4000)\n");
    NN_LOG("SA: Invoke nn::applet::CreateLibraryApplet( AllFG )\n");

    FillMemory(g_StartupParamBuffer, sizeof(g_StartupParamBuffer), 'a');

    nn::applet::LibraryAppletHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&handle, nn::applet::AppletId_LibraryAppletController, nn::applet::LibraryAppletMode_AllForeground));
    {
        // ストレージの作成
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(g_StartupParamBuffer)));
        // ストレージへの書き込み
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, g_StartupParamBuffer, sizeof(g_StartupParamBuffer)));
        // 入力チャンネルへの push
        nn::applet::PushToInChannel(handle, storageHandle);
        //nn::applet::ReleaseStorage(storageHandle) は不要
    }

    // ライブラリアプレットの起動待機
    NN_LOG("(4100)\n");
    NN_LOG("SA: Invoke nn::applet::StartLibraryApplet()\n");
    nn::applet::StartLibraryApplet(handle);

    // アプリケーションに対して BG 遷移要求
    NN_LOG("(4150)\n");
    NN_LOG("SA: Wait nn::applet::Message_ChangeIntoBackground()\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);
    nn::ae::ReleaseForegroundRights();


    // ライブラリアプレットの終了待機
    NN_LOG("(4200)\n");
    NN_LOG("SA: Invoke nn::applet::JoinLibraryApplet()\n");
    nn::applet::JoinLibraryApplet(handle);
    NN_ABORT_UNLESS( nn::applet::GetLibraryAppletExitReason(handle) == nn::applet::LibraryAppletExitReason_Normal );

    // 出力データの取得
    {
        // 出力チャンネルからの pop
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::applet::TryPopFromOutChannel(&storageHandle, handle));
        // ストレージデータの読み込み
        NN_ABORT_UNLESS_EQUAL(sizeof(g_ReturnValueBuffer), nn::applet::GetStorageSize(storageHandle));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(storageHandle, 0, g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer)));
        // 忘れずに解放する必要がある
        nn::applet::ReleaseStorage(storageHandle);
    }
    CheckMemory(g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer), 'B');

    // ハンドルのクローズ
    nn::applet::CloseLibraryApplet(handle);

    // アプリケーションに対して FG 遷移要求
    NN_LOG("(4250)\n");
    NN_LOG("SA: Wait nn::applet::Message_ChangeIntoForeground()\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);
    nn::ae::AcquireForegroundRights();
}

//-----------------------------------------------------------------------------
//  ライブラリアプレットの Background 起動
//
void InvokeLibraryAppletInBackgroundState(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    //-------------------------------------------------------------------------
    // ライブラリアプレット起動準備
    NN_LOG("(5000)\n");
    NN_LOG("SA: Invoke nn::applet::CreateLibraryApplet( NoUI )\n");

    FillMemory(g_StartupParamBuffer, sizeof(g_StartupParamBuffer), 'C');

    nn::applet::LibraryAppletHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&handle, nn::applet::AppletId_LibraryAppletDataErase, nn::applet::LibraryAppletMode_PartialForeground));
    {
        // ストレージの作成
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(g_StartupParamBuffer)));
        // ストレージへの書き込み
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, g_StartupParamBuffer, sizeof(g_StartupParamBuffer)));
        // 入力チャンネルへの push
        nn::applet::PushToInChannel(handle, storageHandle);
        //nn::applet::ReleaseStorage(storageHandle) は不要
    }

    // ライブラリアプレットの起動待機
    NN_LOG("(5100)\n");
    NN_LOG("SA: Invoke nn::applet::StartLibraryApplet()\n");
    nn::applet::StartLibraryApplet(handle);

    //-------------------------------------------------------------------------
    // ライブラリアプレットに IPC コマンドを送信
    NN_LOG("(5200)\n");
    NN_LOG("SA: Invoke nn::applet::PushToInteractiveInChannel()\n");

    {
        FillMemory(g_RequestCommandBuffer, sizeof(g_RequestCommandBuffer), 'D');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_RequestCommandBuffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_RequestCommandBuffer, sizeof(g_RequestCommandBuffer)));
        PushToInteractiveInChannel(handle, storageHandle);
    }


    //-------------------------------------------------------------------------
    // ライブラリアプレットから IPC 返答を受信
    NN_LOG("(5300)\n");
    NN_LOG("SA: Wait nn::applet::GetPopFromInteractiveOutChannelEvent(LA4)\n");
    nn::os::WaitSystemEvent(GetPopFromInteractiveOutChannelEvent(handle));
    NN_LOG("SA: Invoke nn::applet::TryPopFromInteractiveOutChannel()\n");
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(TryPopFromInteractiveOutChannel(&storageHandle, handle));
        NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_ReplayMessageBuffer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_ReplayMessageBuffer, sizeof(g_ReplayMessageBuffer)));
        ReleaseStorage(storageHandle);
        CheckMemory(g_ReplayMessageBuffer, sizeof(g_ReplayMessageBuffer), 'E');
    }

    //-------------------------------------------------------------------------
    // ライブラリアプレットの終了待機
    NN_LOG("(5400)\n");
    NN_LOG("SA: Invoke nn::applet::JoinLibraryApplet()\n");
    nn::applet::JoinLibraryApplet(handle);

    //-------------------------------------------------------------------------
    // ライブラリアプレットの終了要因チェック
    NN_LOG("(5500)\n");
    NN_LOG("SA: Invoke nn::applet::GetLibraryAppletExitReason()\n");
    NN_ABORT_UNLESS( nn::applet::GetLibraryAppletExitReason(handle) == nn::applet::LibraryAppletExitReason_Normal );

    // 出力データの取得
    {
        // 出力チャンネルからの pop
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::applet::TryPopFromOutChannel(&storageHandle, handle));
        // ストレージデータの読み込み
        NN_ABORT_UNLESS_EQUAL(sizeof(g_ReturnValueBuffer), nn::applet::GetStorageSize(storageHandle));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(storageHandle, 0, g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer)));
        // 忘れずに解放する必要がある
        nn::applet::ReleaseStorage(storageHandle);
    }
    CheckMemory(g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer), 'F');

    // ハンドルのクローズ
    nn::applet::CloseLibraryApplet(handle);
}

void MainLibraryAppletSequenceTestFromSA(nn::os::SystemEventType* pEvent, nn::ae::ApplicationHandle handle) NN_NOEXCEPT
{
    //--------------------------------------------
    // FG 起動のライブラリアプレット１のテスト開始
    // 4000
    WaitInterval();
    NN_LOG("SA: Wait for Message_DetectShortPressingHomeButton\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectShortPressingHomeButton);

    // SA から起動した LA3 を BG に遷移させることに成功する
    NN_LOG("SA: Invoke nn::ae::RequestToGetForeground()\n");
    nn::ae::RequestToGetForeground();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);

    WaitInterval();

    // SA から起動した LA3 を再度 FG に戻す
    NN_LOG("SA: Invoke nn::ae::RequestApplicationGetForeground()\n");
    nn::ae::RequestApplicationGetForeground(handle);
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    WaitInterval();

    //--------------------------------------------
    // BG 起動のライブラリアプレット２のテスト開始
    // 5000
    WaitInterval();
    NN_LOG("SA: Wait for Message_DetectShortPressingHomeButton\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectShortPressingHomeButton);

    // LA4 が BG 起動なので、アプリを BG に遷移させることに成功する
    NN_LOG("SA: Invoke nn::ae::RequestToGetForeground()\n");
    nn::ae::RequestToGetForeground();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);

    WaitInterval();

    // LA4 が BG 起動なので、アプリを再度 FG に戻す
    NN_LOG("SA: Invoke nn::ae::RequestApplicationGetForeground()\n");
    nn::ae::RequestApplicationGetForeground(handle);
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    WaitInterval();
}

//-----------------------------------------------------------------------------
//  ライブラリアプレット呼出しテスト
//

NN_OS_ALIGNAS_THREAD_STACK char g_Stack[0x4000];

void InvokeLibraryAppletThread(void* arg) NN_NOEXCEPT
{
    auto* pEvent = reinterpret_cast<nn::os::SystemEventType*>(arg);
    InvokeLibraryAppletInForegroundState(pEvent);

    // デフォルトのマスターボリューム期待値の取得とチェック
    float systemAppletVolume;
    float libraryAppletVolume;
    nn::ae::GetExpectedMasterVolume( &systemAppletVolume, &libraryAppletVolume );
    NN_LOG("SA: Default SystemApplet's expected volume  = %1.2f\n", systemAppletVolume);
    NN_LOG("SA: Default LibraryApplet's expected volume = %1.2f\n", libraryAppletVolume);
    NN_ABORT_UNLESS(systemAppletVolume  == 0.25f);
    NN_ABORT_UNLESS(libraryAppletVolume == 1.0f );

    // マスターボリューム期待値の変更とチェック
    nn::ae::SetExpectedMasterVolume( 0.5f, 0.2f );
    {
        float volume1;
        float volume2;
        nn::ae::GetExpectedMasterVolume( &volume1, &volume2 );
        NN_ABORT_UNLESS(volume1 == 0.5f);
        NN_ABORT_UNLESS(volume2 == 0.2f );
    }


    InvokeLibraryAppletInBackgroundState(pEvent);

    // マスターボリューム期待値をテスト前の値に戻す
    nn::ae::SetExpectedMasterVolume( systemAppletVolume, libraryAppletVolume );
}


void MainSequenceTestOfLibraryAppletInvoking(nn::os::SystemEventType* pEvent, nn::ae::ApplicationHandle handle) NN_NOEXCEPT
{
    auto id = nn::applet::GetAppletResourceUserId();
    NN_LOG("SA: ARUID = 0x%016llx\n", id.lower);

    // ここに来た時点で FG 状態
    nn::os::ThreadType thread;
    auto result = nn::os::CreateThread(&thread, InvokeLibraryAppletThread, pEvent, g_Stack, sizeof(g_Stack), nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::StartThread(&thread);

    // メインスレッド側で LA と同期
#if 0
    MainLibraryAppletSequenceTestFromSA(pEvent, handle);
#endif

    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);
}

