﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>

#include "../Common/applet_Common.h"

//-----------------------------------------------------------------------------
//  Audio 制御
//
void ControlMainAppletAudio()
{
    // アプリのボリューム操作
    auto maExpectedVolume = nn::ae::GetMainAppletExpectedMasterVolume();
    auto laExpectedVolume = nn::ae::GetLibraryAppletExpectedMasterVolume();

    NN_LOG("LA4: Current Application's expected volume   = %1.2f\n", maExpectedVolume);
    NN_LOG("LA4: Current LibraryApplet's expected volume = %1.2f\n", laExpectedVolume);
    NN_ABORT_UNLESS(maExpectedVolume == 0.5f);
    NN_ABORT_UNLESS(laExpectedVolume == 0.2f);

    nn::ae::ChangeMainAppletMasterVolume( 0.46f, nn::TimeSpan::FromMilliSeconds(14) );
    NN_LOG("LA4: Invoked nn::ae::ChangeMainAppletMasterVolume()\n");
}

//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) char g_MallocBuffer[0x200000];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x200000 );
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}


//-----------------------------------------------------------------------------
//  メイン関数
//
void LibraryAppletDataEraseMain(const nn::ae::LibraryAppletSelfInfo& info)
{
    NN_LOG("LA4: Launched an LibraryApplet4.\n");

    // 現在のシーケンスでは SA からのみ起動
    auto mainAppletIdentityInfo = nn::ae::GetMainAppletIdentityInfo();
    NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_SystemAppletMenu, mainAppletIdentityInfo.appletId);

    auto callerAppletIdentityInfo = nn::ae::GetCallerAppletIdentityInfo();
    NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_SystemAppletMenu, callerAppletIdentityInfo.appletId);


    NN_ABORT_UNLESS(info.appletId == nn::applet::AppletId_LibraryAppletDataErase);
    NN_ABORT_UNLESS(info.libraryAppletMode == nn::applet::LibraryAppletMode_PartialForeground);
    NN_ABORT_UNLESS(info.isUnwound == false);

    //-------------------------------------------------------------------------
    // ライブラリアプレットの初期化
    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);

    // 起動パラメータの検査
    {
        static char startupParameter[8192];
        nn::applet::StorageHandle handle;
        NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&handle), "入力データ数が足りない");
        NN_ABORT_UNLESS_EQUAL(sizeof(startupParameter), nn::applet::GetStorageSize(handle));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(handle, 0, startupParameter, sizeof(startupParameter)));
        nn::applet::ReleaseStorage(handle);
        CheckMemory(startupParameter, sizeof(startupParameter), 'C');
    }

    // 部分 foreground 起動なので、最初に FG 状態が通知される
    NN_LOG("LA4: Wait nn::ae::Message_ChangeIntoForeground.\n");
    CheckAndProcessMessage("LA4", &event, nn::ae::Message_ChangeIntoForeground );

    // アプリの Audio 制御
    NN_LOG("LA4: Control main applet's volume.\n");
    ControlMainAppletAudio();

    {
        const size_t RequestCommandSize = 128 * 1024;
        const size_t ReplyMessageSize   = 128 * 1024;
        static NN_ALIGNAS(4096) char g_RequestCommandBuffer[ RequestCommandSize ];
        static NN_ALIGNAS(4096) char g_ReplayMessageBuffer[ ReplyMessageSize ];

        //---------------------------------------------------------------------
        // 要求コマンドの待機と受信
        NN_LOG("LA4: Wait nn::ae::GetPopFromInteractiveInChannelEvent.\n");
        nn::os::WaitSystemEvent(nn::ae::GetPopFromInteractiveInChannelEvent());
        NN_LOG("LA4: Wait nn::ae::TryPopFromInteractiveInChannel.\n");
        {
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS(nn::ae::TryPopFromInteractiveInChannel(&storageHandle));
            NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_RequestCommandBuffer));
            NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_RequestCommandBuffer, sizeof(g_RequestCommandBuffer)));
            ReleaseStorage(storageHandle);
            CheckMemory(g_RequestCommandBuffer, sizeof(g_RequestCommandBuffer), 'D');
        }

#if 0
        //---------------------------------------------------------------------
        // 要求コマンドを契機に、SA に対して HOME ボタン短押しを発行
        NN_LOG("LA4: Press Home Button Short.\n");
        nn::ae::NotifyShortPressingHomeButtonForDebug();
#endif

        //---------------------------------------------------------------------
        // 返答メッセージの送信
        NN_LOG("LA4: Invoke nn::ae::PushToInteractiveOutChannel().\n");
        {
            FillMemory(g_ReplayMessageBuffer, sizeof(g_ReplayMessageBuffer), 'E');
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_ReplayMessageBuffer)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_ReplayMessageBuffer, sizeof(g_ReplayMessageBuffer)));
            nn::ae::PushToInteractiveOutChannel(storageHandle);
        }
    }

    // ライブラリアプレットの終了
    NN_LOG("LA4: Invoke nn::ae::ExitLibraryApplet().\n");
    {
        static char returnValue[8192];
        FillMemory(returnValue, sizeof(returnValue), 'F');

        nn::applet::StorageHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&handle, sizeof(returnValue)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(handle, 0, returnValue, sizeof(returnValue)));
        nn::ae::PushToOutChannel(handle);
    }
    nn::ae::ExitLibraryApplet();
}

extern "C" void nnMain()
{
    NN_LOG("LA4: Invoke nn::ae::InvokeLibraryAppletMain().\n");

    nn::ae::InvokeLibraryAppletMain(LibraryAppletDataEraseMain);
}

