﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/nnt_Argument.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/settings/system/settings_Language.h>
#include <nn/oe.h>
#include <nn/oe/oe_DisplayResolutionApis.h>
#include <nn/oe/oe_CrashReportPolicyApi.h>

namespace {
    class OeApiTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::settings::GetLanguageCode(&m_OriginalLanguage);
        }

        virtual void TearDown()
        {
            nn::settings::system::SetLanguageCode(m_OriginalLanguage);
        }

    private:
        nn::settings::LanguageCode m_OriginalLanguage;
    };

}

TEST_F(OeApiTest, GetDesiredLanguage)
{
    // 日本語と米国英語に対応しているので、それぞれに対して優先度が反映されているか確認する。
    auto targetLanguage = nn::settings::Language::Language_AmericanEnglish;
    nn::settings::system::SetLanguageCode(nn::settings::LanguageCode::Make(targetLanguage));
    auto language = nn::oe::GetDesiredLanguage();
    ASSERT_TRUE(language == targetLanguage);

    targetLanguage = nn::settings::Language::Language_Japanese;
    nn::settings::system::SetLanguageCode(nn::settings::LanguageCode::Make(targetLanguage));
    language = nn::oe::GetDesiredLanguage();
    ASSERT_TRUE(language == targetLanguage);

    // 対応していない場合は、本体設定の優先度に合わせて言語が選択される
    targetLanguage = nn::settings::Language::Language_CanadianFrench;
    nn::settings::system::SetLanguageCode(nn::settings::LanguageCode::Make(targetLanguage));
    language = nn::oe::GetDesiredLanguage();
    ASSERT_TRUE(language == nn::settings::Language::Language_AmericanEnglish);
}

TEST_F(OeApiTest, GetDisplayVersion)
{
    nn::oe::DisplayVersion displayVersion;
    nn::oe::GetDisplayVersion(&displayVersion);

    ASSERT_TRUE(std::strcmp(displayVersion.value, "1.0.0") == 0);
}

TEST_F(OeApiTest, GetPseudoDeviceId)
{
    nn::util::Uuid pseudoDeviceId = {};
    nn::util::Uuid compare = {};
    nn::oe::GetPseudoDeviceId(&pseudoDeviceId);

    {
        char buffer[nn::util::Uuid::StringSize];
        NN_LOG("PseudoDeviceId=%s\n", pseudoDeviceId.ToString(buffer, sizeof(buffer)));
    }

    ASSERT_TRUE(pseudoDeviceId != compare);
}

TEST_F(OeApiTest, GetDefaultDisplayResolution)
{
    int width, height;

    nn::oe::GetDefaultDisplayResolution(&width, &height);
    NN_LOG("DefaultDisplayResolution=(%d, %d)\n", width, height);

    ASSERT_TRUE(width != 0);
    ASSERT_TRUE(height != 0);

    nn::os::SystemEvent event;
    nn::oe::GetDefaultDisplayResolutionChangeEvent(&event);
    event.TryWait();
}

TEST_F(OeApiTest, EnableCrashReport)
{
    nn::oe::DisableCrashReport();
    nn::oe::EnableCrashReport();
}
