﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/vi.private.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/result/result_ErrorResult.h>
#include <nn/util/util_Color.h>

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#include <nvn/nvn_FuncPtrInline.h>
#include "../Common/Common.h"
#include "Graphics.h"

// VR モードのカーテン表示管理
extern nn::os::LightEvent g_IsVrModeCurtainRequired;

namespace Graphics {

namespace {

// クリアカラー
Rgba g_ClearColor;

// タイトル
std::string g_Title;

// 描画するオブジェクト数
int g_ObjectCount = 0;
int g_LiteObjectCount = 0;

// スリープ時間 g_SleepCount * 200us
int g_SleepCount = 0;

// 全画面クリアを行うか
bool g_ClearEnabled = false;

// バッファ数 (2 以上を指定してください)
const int g_BufferCount = 2;

// フレームワークモード
enum FrameworkMode
{
    // WindowAcquireTexture を使用するフレームワーク。
    // vsync の前にコマンドの提出を行い、提出後 vsync を待ってから GPU が処理を始める。
    // GPU を vsync 直後から使用できる。
    FrameworkMode_DeferredExecution,    // NX と Generic で挙動が異なる

    // Ocean の OA で使用されるフレームワーク。
    // 30fps で動作する。
    // 必ず 1度はこのモードで動作確認する必要がある。
    FrameworkMode_Ocean,
};

static const FrameworkMode g_FrameworkMode = FrameworkMode_DeferredExecution;

//-----------------------------------------------------------------------------

// グラフィックスフレームワークを初期化
nns::gfx::GraphicsFramework g_GraphicsFramework;
nns::gfx::GraphicsFramework::DebugFontTextWriter g_Writer;
static int g_BufferIndex = 0;
void InitializeGraphicsFramework()
{
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetColorBufferEnabled(false);
    fwInfo.SetDepthStencilBufferEnabled(false);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, 8 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 8 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, 2 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Shader, 0 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Data, 1 * 1024 * 1024);
    fwInfo.SetRootCommandBufferCommandMemorySize(2 * 1024 * 1024);
    fwInfo.SetBufferCount(g_BufferCount);
    fwInfo.SetSwapChainBufferCount(g_BufferCount);
    g_GraphicsFramework.Initialize(fwInfo);
}

// グラフィックスフレームワークを破棄
void FinalizeGraphicsFramework()
{
    g_GraphicsFramework.Finalize();
}

// プリミティブレンダラの初期化
nn::mem::StandardAllocator  g_PrimitiveRendererAllocator;
nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
void InitializePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(g_GraphicsFramework.DefaultAllocateFunction, NULL);
    info.SetAdditionalBufferSize(1024 * 4);
    info.SetDrawCallCountMax(1024 * 4);
    info.SetViewFunctionCallCountMax(1024 * 4);
    info.SetMultiBufferQuantity(g_BufferCount);

    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(g_GraphicsFramework.GetDevice(), info);
    g_pPrimitiveRenderer->SetScreenWidth(g_GraphicsFramework.GetDisplayWidth());
    g_pPrimitiveRenderer->SetScreenHeight(g_GraphicsFramework.GetDisplayHeight());
}

// プリミティブレンダラの破棄
void FinalizePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::DestroyRenderer(g_pPrimitiveRenderer, g_GraphicsFramework.GetDevice(), g_GraphicsFramework.DefaultFreeFunction, NULL);
}

// パフォーマンス計測クラスの初期化
nns::gfx::PrimitiveRenderer::MeterDrawer g_MeterDrawer;
void*  g_MeterMemory;
ptrdiff_t g_MeterMemoryPoolOffset;
void InitializeLoadMeter()
{
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(4);
    if (NN_STATIC_CONDITION(g_FrameworkMode == FrameworkMode_Ocean))
    {
        info.SetGpuBufferCount(2);
    }
    else
    {
        info.SetGpuBufferCount(3);
    }
    info.SetCpuSectionCountMax(16);
    info.SetGpuSectionCountMax(8);

    // 計測で使用するメモリの確保
    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    g_MeterMemory = g_GraphicsFramework.AllocateMemory(memorySize, memoryAlignment);

    // メモリプールの確保
    size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(g_GraphicsFramework.GetDevice(), info);
    g_MeterMemoryPoolOffset = g_GraphicsFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, NN_PERF_GET_MEMORY_POOL_ALIGNMENT(g_GraphicsFramework.GetDevice(), info));
    nn::gfx::MemoryPool* meterMemoryPool = g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Data);
    NN_PERF_INITIALIZE_METER(g_GraphicsFramework.GetDevice(), info,
        g_MeterMemory, memorySize,
        meterMemoryPool, g_MeterMemoryPoolOffset, memoryPoolSize);
    NN_ASSERT(g_MeterMemoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset, "memoryPool shortage");
}

// パフォーマンス計測クラスの破棄
void FinalizeLoadMeter()
{
    NN_PERF_FINALIZE_METER(g_GraphicsFramework.GetDevice());
    g_GraphicsFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, g_MeterMemoryPoolOffset);
    g_GraphicsFramework.FreeMemory(g_MeterMemory);
}

nns::gfx::GraphicsFramework::CommandBuffer g_CommandBufferBegin[g_BufferCount];
nns::gfx::GraphicsFramework::CommandBuffer g_CommandBufferEnd[g_BufferCount];
void InitializeQueryCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
    size_t commandMemorySize = 1024 * 1024 * 1;
    size_t controlMemorySize = 256;

    for (int bufferIndex = 0; bufferIndex < g_BufferCount; bufferIndex++)
    {
        g_GraphicsFramework.InitializeCommandBuffer(&g_CommandBufferBegin[bufferIndex], info, commandMemorySize, controlMemorySize);
        g_GraphicsFramework.InitializeCommandBuffer(&g_CommandBufferEnd[bufferIndex], info, commandMemorySize, controlMemorySize);
    }
}

void FinalizeQueryCommandBuffer()
{
    for (int bufferIndex = 0; bufferIndex < g_BufferCount; bufferIndex++)
    {
        g_GraphicsFramework.FinalizeCommandBuffer(&g_CommandBufferBegin[bufferIndex]);
        g_GraphicsFramework.FinalizeCommandBuffer(&g_CommandBufferEnd[bufferIndex]);
    }
}

void ClearColor(nn::gfx::CommandBuffer* pCommandBuffer, int x, int y, int width, int height)
{
    int displayHeight = g_GraphicsFramework.GetDisplayHeight();
    int h = displayHeight - height - y;
    nn::gfx::ViewportStateInfo vinfo;
    vinfo.SetDefault();
    vinfo.SetOriginX(static_cast<float>(x));
    vinfo.SetOriginY(static_cast<float>(h));
    vinfo.SetWidth(static_cast<float>(width));
    vinfo.SetHeight(static_cast<float>(height));
    pCommandBuffer->SetViewports(0, 1, &vinfo);

    nn::gfx::ScissorStateInfo sinfo;
    sinfo.SetDefault();
    sinfo.SetOriginX(x);
    sinfo.SetOriginY(h);
    sinfo.SetWidth(width);
    sinfo.SetHeight(height);
    pCommandBuffer->SetScissors(0, 1, &sinfo);

    float c[4];
    std::memcpy(c, &g_ClearColor, 4 * sizeof(float));
    nvnCommandBufferClearColor(
        pCommandBuffer->ToData()->pNvnCommandBuffer,
        0,
        c,
        NVN_CLEAR_COLOR_MASK_RGBA
    );
}

// VR モードのカーテンを描画する
// 全画面へのカーテン描画を行なった場合は true を、
// 行わなかった場合は false を返す。
bool DrawVrModeCurtain(nn::gfx::CommandBuffer* pCommandBuffer)
{
    bool isVrModeCurtainRequired = g_IsVrModeCurtainRequired.TryWait();

    static int vrModeCuratinBottomPos = 0;
    static const int displayHeight = g_GraphicsFramework.GetDisplayHeight();
    static const int addend = displayHeight / 10;

    if (isVrModeCurtainRequired)
    {
        vrModeCuratinBottomPos = std::min(vrModeCuratinBottomPos + addend, displayHeight);
    }
    else
    {
        vrModeCuratinBottomPos = std::max(vrModeCuratinBottomPos - addend, 0);
    }

    if (vrModeCuratinBottomPos <= 0)
    {
        return false;
    }

    // VR モード用のカーテンを描画
    auto ypos = vrModeCuratinBottomPos - displayHeight;

    const nn::util::Color4u8Type curtainColor = { { 128, 128, 64, 255 } };
    g_pPrimitiveRenderer->SetColor(curtainColor);
    g_pPrimitiveRenderer->Draw2DRect(pCommandBuffer, 0.f, ypos, g_GraphicsFramework.GetDisplayWidth(), displayHeight);

    const nn::util::Color4u8Type fontColor = { { 128, 255, 128, 255 } };
    g_Writer.object.SetTextColor(fontColor);
    g_Writer.object.SetScale(5.0f, 5.0f);
    g_Writer.object.SetCursor(300, ypos + 300);
    g_Writer.object.Print("VR-mode Curtain");

    return true;
}

void MakeCommand( unsigned int frame, bool isPrepareSuspend )
{
    static int s_LeftCountToClearAllScreen = 1;
    g_pPrimitiveRenderer->Update(g_BufferIndex);

    nn::gfx::CommandBuffer* pCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(g_BufferIndex);
    g_GraphicsFramework.BeginFrame(g_BufferIndex);
    {
        int clearLines = 140;
        if (s_LeftCountToClearAllScreen > 0)
        {
            --s_LeftCountToClearAllScreen;
            clearLines = g_GraphicsFramework.GetDisplayHeight();
        }

        if (isPrepareSuspend)
        {
            if (NN_STATIC_CONDITION(g_FrameworkMode == FrameworkMode_Ocean))
            {
                NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Green());
                NN_PERF_BEGIN_MEASURE_GPU(pCommandBuffer);
            }
            int scanBufferIndex = g_GraphicsFramework.GetNextScanBufferIndex();
            nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetScanBufferView(scanBufferIndex);

            if (g_ClearEnabled || (frame < g_BufferCount))
            {
                // 全画面クリアがオンもしくは初回描画時のときは全画面クリア
                pCommandBuffer->ClearColor(pTarget, g_ClearColor.r, g_ClearColor.g, g_ClearColor.b, g_ClearColor.a, NULL);
                pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);
            }
            else
            {
                pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);
                ClearColor(pCommandBuffer, 0, 0, g_GraphicsFramework.GetDisplayWidth(), clearLines);
            }

            pCommandBuffer->SetRenderTargets(1, &pTarget, g_GraphicsFramework.GetDepthStencilView());
            pCommandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());
            pCommandBuffer->SetRasterizerState(g_GraphicsFramework.GetRasterizerState(nns::gfx::GraphicsFramework::RasterizerStateType::RasterizerStateType_FillSolid_CullNone));

            pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
                | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);
        }
        else
        {
            if (NN_STATIC_CONDITION(g_FrameworkMode == FrameworkMode_Ocean))
            {
                NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Green());
                NN_PERF_BEGIN_MEASURE_GPU(pCommandBuffer);
            }
            int scanBufferIndex = g_GraphicsFramework.GetNextScanBufferIndex();
            nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetScanBufferView(scanBufferIndex);

            if (g_ClearEnabled || (frame < g_BufferCount))
            {
                // 全画面クリアがオンもしくは初回描画時のときは全画面クリア
                pCommandBuffer->ClearColor(pTarget, g_ClearColor.r, g_ClearColor.g, g_ClearColor.b, g_ClearColor.a, NULL);
                pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);
            }
            else
            {
                pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);
                ClearColor(pCommandBuffer, 0, 0, g_GraphicsFramework.GetDisplayWidth(), clearLines);
            }
            //pCommandBuffer->ClearDepthStencil(g_GraphicsFramework.GetDepthStencilView(), 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

            pCommandBuffer->SetRenderTargets(1, &pTarget, g_GraphicsFramework.GetDepthStencilView());
            pCommandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());
            pCommandBuffer->SetRasterizerState(g_GraphicsFramework.GetRasterizerState(nns::gfx::GraphicsFramework::RasterizerStateType::RasterizerStateType_FillSolid_CullNone));

            pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
                | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);

            // GPU 負荷変化
            //NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Yellow());
            //NN_PERF_BEGIN_MEASURE_GPU(pCommandBuffer);
            {
                nn::util::Matrix4x3fType viewMatrix;
                nn::util::Matrix4x4fType projectionMatrix;
                nn::util::Matrix4x3f modelMatrix;
                nn::util::MatrixIdentity(&modelMatrix);
                nn::util::Vector3f translate;
                nn::util::VectorSet(&translate, 0.f, 0.f, 1.f);
                nn::util::MatrixSetAxisW(&modelMatrix, translate);

                // Blend
                pCommandBuffer->SetBlendState(g_pPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal));

                // ビューとプロジェクションを設定
                float radius = 20.f;
                float x = radius * sin(0 / 500.f);
                float z = radius * cos(0 / 500.f);
                nn::util::Vector3fType camPos = { x, 10.f, z };
                nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
                nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
                nn::util::MatrixLookAtRightHanded(&viewMatrix, camPos, camTarget, camUp);

                const float fovy = nn::util::FloatPi / 3.0f;
                const float aspect = static_cast<float>(g_GraphicsFramework.GetDisplayWidth()) / static_cast<float>(g_GraphicsFramework.GetDisplayHeight());
                nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, fovy, aspect, 0.1f, 1000.f);

                g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
                g_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

                // // Depth Enable
                // g_pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);
                // Depth Disable
                g_pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

                nn::util::Vector3fType size = { 15.f, 15.f, 15.f };
                nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
                for (int i = 0; i < g_ObjectCount; ++i)
                {
                    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_IndirectBuffer);
                    pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_IndirectBuffer);
                    nn::util::VectorSet(&translate, 0, 0, 0);
                    nn::util::MatrixSetAxisW(&modelMatrix, translate);
                    g_pPrimitiveRenderer->SetColor(nn::util::Color4u8(0, 0, 0, 0));
                    g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
                    g_pPrimitiveRenderer->SetLineWidth(14.f);
                    g_pPrimitiveRenderer->DrawCone(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
                }

                for (int i = 0; i < g_LiteObjectCount / 4; ++i)
                {
                    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_IndirectBuffer);
                    pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_IndirectBuffer);
                    g_pPrimitiveRenderer->SetColor(nn::util::Color4u8(0, 0, 0, 0));
                    g_pPrimitiveRenderer->Draw2DRect(pCommandBuffer, 0, 0, 1, 1);
                }
            }
            //NN_PERF_END_MEASURE_GPU(pCommandBuffer);

            // model, view, projection行列をデフォルトに
            g_pPrimitiveRenderer->SetDefaultParameters();

            // Depth Disable
            g_pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

            nn::util::Color4u8Type green = { { 128, 255, 128, 255 } };

            // VR モードのカーテンを描画
            if (DrawVrModeCurtain(pCommandBuffer))
            {
                // 一度全画面描画を行なった場合は、
                // その後 g_BufferCount のフレーム数だけ全画面クリアを行なう。
                s_LeftCountToClearAllScreen = g_BufferCount;
            }

            // 枠を描画
            g_pPrimitiveRenderer->SetColor(green);
            g_pPrimitiveRenderer->SetLineWidth(3.f);
            g_pPrimitiveRenderer->Draw2DFrame(pCommandBuffer, 5, 10, 600, 120);

            // タイトルを描画
            g_Writer.object.SetTextColor(green);
            g_Writer.object.SetScale(1.5f, 1.5f);
            g_Writer.object.SetCursor(15, 15);
            g_Writer.object.Print(g_Title.c_str());

            // フレームカウントを描画
            g_Writer.object.SetScale(1, 1);
            g_Writer.object.SetCursor(200, 15);
            g_Writer.object.Print("Frame:%d", frame);

            // 全画面クリアモード
            g_Writer.object.SetScale(1, 1);
            g_Writer.object.SetCursor(470, 40);
            g_Writer.object.Print("Clear:%s", g_ClearEnabled ? "ON" : "OFF");

            // 負荷メーターを描画
            g_Writer.object.SetScale(1, 1);
            nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
            nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(35, 65);
            g_MeterDrawer.SetDebugFontTextWriter(&g_Writer.object);
            g_MeterDrawer.SetVisibleNoResult(false);
            g_MeterDrawer.SetPosition(pos);
            g_MeterDrawer.SetWidth(500);
            g_MeterDrawer.Draw(pCommandBuffer, g_pPrimitiveRenderer, pFrameMeter);

            float fps = 1000.f / (static_cast<float>(pFrameMeter->GetLastTotalSpan().GetNanoSeconds()) / 1000000.0f);
            g_Writer.object.SetTextColor(green);
            g_Writer.object.SetScale(1, 1);
            g_Writer.object.SetCursor(470, 15);
            g_Writer.object.Print("FPS:%4.1f", fps);

            // CPU 負荷 [AcquireTexture]
            {
                nn::TimeSpan t = NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "AcquireTexture", 0);
                g_Writer.object.SetCursor(330, 15);
                g_Writer.object.SetTextColor(nn::util::Color4u8(64, 64, 255, 255));
                g_Writer.object.Print("ACQ: %lldus", t.GetMicroSeconds());
            }

            // CPU 負荷 [PresentTexture]
            {
                nn::TimeSpan t = NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "PresentTexture", 0);
                g_Writer.object.SetCursor(330, 40);
                g_Writer.object.SetTextColor(nn::util::Color4u8(255, 255, 64, 255));
                g_Writer.object.Print("PRE: %lldus", t.GetMicroSeconds());
            }

            // GPU 負荷を描画
            {
                nn::perf::GpuMeter* pGpuMeter = NN_PERF_GET_GPU_METER();
                g_Writer.object.SetScale(1, 1);
                g_Writer.object.SetCursor(200, 40);
                g_Writer.object.SetTextColor(nn::util::Color4u8(255, 64, 64, 255));
                if (NN_STATIC_CONDITION(g_FrameworkMode == FrameworkMode_Ocean))
                {
                    nn::perf::LoadMeterBase::Section drawSection = pGpuMeter->GetLastResult(0);
                    nn::perf::LoadMeterBase::Section presentSection = pGpuMeter->GetLastResult(1);
                    int64_t drawUs = drawSection.end.ToTimeSpan().GetMicroSeconds() - drawSection.begin.ToTimeSpan().GetMicroSeconds();
                    int64_t presentUs = presentSection.end.ToTimeSpan().GetMicroSeconds() - presentSection.begin.ToTimeSpan().GetMicroSeconds();
                    g_Writer.object.Print("GPU: %lldus", drawUs + presentUs);
                }
                else
                {
                    nn::perf::LoadMeterBase::Section mainSection = pGpuMeter->GetLastResult(0);
                    g_Writer.object.Print("GPU: %lldus", mainSection.end.ToTimeSpan().GetMicroSeconds() - mainSection.begin.ToTimeSpan().GetMicroSeconds());
                }
                //nn::perf::CpuMeter* pCpuMeter = NN_PERF_GET_FRAME_METER();
                //nn::perf::LoadMeterBase::Section mainSection1 = pCpuMeter->GetLastResult(0);
                //NN_LOG("OA  %lld %lld\n", mainSection.begin.ToTimeSpan().GetMicroSeconds(), mainSection1.begin.ToTimeSpan().GetMicroSeconds());
            }

            g_Writer.object.Draw(pCommandBuffer);
            if (NN_STATIC_CONDITION(g_FrameworkMode == FrameworkMode_Ocean))
            {
                NN_PERF_END_MEASURE_GPU(pCommandBuffer);
            }
        }
    }
    pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    pCommandBuffer->End();
} // NOLINT(impl/function_size)

} // namespace

// グラフィックスの初期処理
void InitializeGraphics(Rgba clearColor, std::string title)
{
    g_ClearColor = clearColor;
    g_Title = title;

    const size_t graphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(graphicsSystemMemorySize);

    // グラフィックスフレームワークの初期化
    InitializeGraphicsFramework();

    // プリミティブレンダラの初期化
    InitializePrimitiveRenderer();

    // デバッグフォントの初期化
    g_GraphicsFramework.InitializeDebugFontTextWriter(&g_Writer, 256, g_BufferCount);

    // 負荷計測の初期化
    InitializeLoadMeter();

    InitializeQueryCommandBuffer();
}

// グラフィックスの終了処理
void FinalizeGraphics()
{
    FinalizeQueryCommandBuffer();

    // 負荷計測の破棄
    FinalizeLoadMeter();

    // デバッグフォントの破棄
    g_GraphicsFramework.FinalizeDebugFontTextWriter(&g_Writer);

    // プリミティブレンダラの破棄
    FinalizePrimitiveRenderer();

    // グラフィックスフレームワークの破棄
    FinalizeGraphicsFramework();
}

nn::vi::Layer* GetViLayer()
{
    return g_GraphicsFramework.GetLayer();
}

nn::vi::Display* GetViDisplay()
{
    return g_GraphicsFramework.GetDisplay();
}

int GetObjectCount()
{
    return g_ObjectCount;
}

void SetObjectCount(int objectCount)
{
    g_ObjectCount = std::min(objectCount, 200);
    g_ObjectCount = std::max(g_ObjectCount, 0);
}

int GetLiteObjectCount()
{
    return g_LiteObjectCount;
}

void SetLiteObjectCount(int objectCount)
{
    g_LiteObjectCount = std::min(objectCount, 400 * 4);
    g_LiteObjectCount = std::max(g_LiteObjectCount, 0);
}

int GetSleepCount()
{
    return g_SleepCount;
}

void SetSleepCount(int sleepCount)
{
    g_SleepCount = std::max(sleepCount, 0);
}

bool GetClearEnabled()
{
    return g_ClearEnabled;
}

void SetClearEnabled(bool enabled)
{
    g_ClearEnabled = enabled;
}

void WindowAcquireTexture()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());
    NN_PERF_AUTO_MEASURE_NAME("AcquireTexture");
    g_GraphicsFramework.AcquireTexture(g_BufferIndex);
}

void QueueWaitSync()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE();
    queue->SyncSemaphore(g_GraphicsFramework.GetDisplaySemaphore(g_BufferIndex));
    queue->Flush();
}

void QueueAcquireTexture()
{
    WindowAcquireTexture();
    QueueWaitSync();
}

void QueueFinish()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Black());
    NN_PERF_AUTO_MEASURE();
    g_GraphicsFramework.QueueFinish();
}

void WaitDisplaySync()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::White());
    NN_PERF_AUTO_MEASURE();
    g_GraphicsFramework.WaitDisplaySync(g_BufferIndex, nn::TimeSpan::FromSeconds(2));
}

void UpdateBufferIndex()
{
    g_BufferIndex = (g_BufferIndex + 1) % g_BufferCount;
}

void MakeAppCommands(unsigned int frame, bool isPrepareSuspend)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE_NAME("RecordCommand");
    MakeCommand(frame, isPrepareSuspend);

    // MakeBeginTimeStampQueryCommand
    g_GraphicsFramework.ResetCommandBuffer(&g_CommandBufferBegin[g_BufferIndex]);
    g_CommandBufferBegin[g_BufferIndex].object.Begin();
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
    NN_PERF_BEGIN_MEASURE_GPU(&g_CommandBufferBegin[g_BufferIndex].object);
    g_CommandBufferBegin[g_BufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferBegin[g_BufferIndex].object.End();

    // MakeEndTimeStampQueryCommand
    g_GraphicsFramework.ResetCommandBuffer(&g_CommandBufferEnd[g_BufferIndex]);
    g_CommandBufferEnd[g_BufferIndex].object.Begin();
    NN_PERF_END_MEASURE_GPU(&g_CommandBufferEnd[g_BufferIndex].object);
    g_CommandBufferEnd[g_BufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferEnd[g_BufferIndex].object.End();
}

void QueueAppCommands()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE_NAME("RecordCommand");
    queue->ExecuteCommand(g_GraphicsFramework.GetRootCommandBuffer(g_BufferIndex), NULL);
}

void QueuePresentTexture()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Yellow());
    NN_PERF_AUTO_MEASURE_NAME("PresentTexture");
    queue->Present(g_GraphicsFramework.GetSwapChain(), 1);
}

void QueueBeginTimeStampQueryCommand()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE();
    queue->ExecuteCommand(&g_CommandBufferBegin[g_BufferIndex].object, NULL);
    NN_PERF_END_MEASURE();
}

void QueueEndTimeStampQueryCommand()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE();
    queue->ExecuteCommand(&g_CommandBufferEnd[g_BufferIndex].object, NULL);
    queue->Flush();
    NN_PERF_END_MEASURE();
}

void SleepCpu()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Black());
    NN_PERF_AUTO_MEASURE();
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(g_SleepCount * 200));
}

void GraphicsRenderer(bool isPrepareSuspend)
{
    static unsigned int frame = 0;

    NN_PERF_BEGIN_FRAME();
    {
        if (NN_STATIC_CONDITION(g_FrameworkMode == FrameworkMode_DeferredExecution))
        {
            // Update bufferIndex
            UpdateBufferIndex();

            // WindowAcquireTexture
            WindowAcquireTexture();

            // Add sync command to queue
            QueueWaitSync();

            // sleep to increase CPU load
            SleepCpu();

            // Make commands
            MakeAppCommands(frame++, isPrepareSuspend);

            // AddBeginTimeStampQueryCommand
            QueueBeginTimeStampQueryCommand();

            // AddCommand
            QueueAppCommands();

            // PresentTexture
            QueuePresentTexture();

            // AddEndTimeStampQueryCommand
            QueueEndTimeStampQueryCommand();

            // Wait for vsync by waiting for the release fence from previous acquire.
            // That is, when above commands start executing on the GPU.
            WaitDisplaySync();
        }
        else if (NN_STATIC_CONDITION(g_FrameworkMode == FrameworkMode_Ocean))
        {
            nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

            // Draw Command
            {
                // Update bufferIndex
                UpdateBufferIndex();

                // WindowAcquireTexture
                WindowAcquireTexture();

                nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(2166));

                // Add sync command to queue
                QueueWaitSync();

                // Make commands
                MakeAppCommands(frame++, isPrepareSuspend);

                // AddCommand
                QueueAppCommands();

                queue->Flush();

                // Wait for vsync by waiting for the release fence from previous acquire.
                // That is, when above commands start executing on the GPU.
                WaitDisplaySync();
            }

            // Present Command
            {
                nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1000));

                // AddBeginTimeStampQueryCommand
                QueueBeginTimeStampQueryCommand();

                // PresentTexture
                QueuePresentTexture();

                // AddEndTimeStampQueryCommand
                QueueEndTimeStampQueryCommand();

                queue->Flush();

                QueueFinish();
            }
        }
    }
    NN_PERF_END_FRAME();

} // NOLINT(impl/function_size)

} // Graphics
