﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>

#include <nn/ahid/hdr/hdr.h>
#include <nn/ahid/ahid.h>
#include "../../../../../Programs/Eris/Sources/Libraries/ahid/hdr/libnn_hdrServer/ahid_Hdr.h"
#include "../../../../../Programs/Eris/Sources/Libraries/cdhid/libnn_cdhidWin32Mouse/cdhidWin32Mouse_Main.h"

//////////////////////////////////////////////////////////////////////////////
static void printString(uint8_t *p)
{
    int len = *p++;

    if (*p++ != nn::ahid::hdr::AhidDescriptorTypeString)
    {
        NN_SDK_LOG("invalid string\n");
        return;
    }

    len -= 2;

    while (len)
    {
        NN_SDK_LOG("%c", *p++);
        p++;
        len -= 2;
    }

    NN_SDK_LOG("\n");
}



//////////////////////////////////////////////////////////////////////////////
static void testInitialize(nn::ahid::Ahid *pAhid, nn::ahid::hdr::DeviceParameters *pDeviceParameters)
{
    nn::Result result;

    NN_SDK_LOG("calling Initialize\n");
    result = pAhid->Initialize(pDeviceParameters);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static nn::ahid::CodeBookHeader *pCodeBookHeader;
static nn::ahid::Item           *pItemButton[3];
static nn::ahid::Item           *pItemX;
static nn::ahid::Item           *pItemY;
static nn::ahid::Item           *pItemWheel;

static void testCodeBook(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling GetCodeBookHeader\n");
    result = pAhid->GetCodeBookHeader(&pCodeBookHeader);
    NN_SDK_LOG("result %d\n", result);

    //                  store item          report id   usage page  usage min max   index
    pAhid->GetInputItem(&pItemButton[0],    0,          0x09,       0x01, 0x01,     0);
    pAhid->GetInputItem(&pItemButton[1],    0,          0x09,       0x02, 0x02,     0);
    pAhid->GetInputItem(&pItemButton[2],    0,          0x09,       0x03, 0x03,     0);
    pAhid->GetInputItem(&pItemX,            0,          0x01,       0x30, 0x30,     0);
    pAhid->GetInputItem(&pItemY,            0,          0x01,       0x31, 0x31,     0);
    pAhid->GetInputItem(&pItemWheel,        0,          0x01,       0x38, 0x38,     0);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetString(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[64];

    NN_SDK_LOG("calling GetString\n");
    result = pAhid->GetString(data, 64, 1);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testSetIdle(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling SetIdle\n");
    result = pAhid->SetIdle(0, 0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetIdle(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t idle;

    NN_SDK_LOG("calling GetIdle\n");
    result = pAhid->GetIdle(&idle, 0);
    NN_SDK_LOG("result %d idle %d\n", result, idle);
}


//////////////////////////////////////////////////////////////////////////////
static void testSetProtocol(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling SetProtocol\n");
    result = pAhid->SetProtocol(0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetProtocol(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t protocol;

    NN_SDK_LOG("calling GetProtocol\n");
    result = pAhid->GetProtocol(&protocol);
    NN_SDK_LOG("result %d protocol %d\n", result, protocol);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetReport(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[64];

    NN_SDK_LOG("calling GetReport\n");
    result = pAhid->GetReport(data, 64, 0x01, 0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testSetReport(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[64];

    NN_SDK_LOG("calling SetReport\n");
    result = pAhid->SetReport(data, 64, 0x02, 0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testRead(nn::ahid::Ahid *pAhid)
{
    int         iteration = 0;
    uint8_t     data[64];
    uint32_t    bytesRead;
    nn::Result  result;

    NN_SDK_LOG("move the mouse, program will run for 1000 inputs\n");

    while (iteration++ < 1000)
    {
        result = pAhid->Read(&bytesRead, data, pCodeBookHeader->inputSize);

        if (result.IsSuccess())
        {
            if (bytesRead)
            {
                NN_SDK_LOG("%4d read data: ", iteration);

                for (uint32_t i = 0; i < bytesRead; i++)
                {
                    NN_SDK_LOG("%02x ", data[i]);
                }

                int32_t button[3], x, y, wheel;

                //            value         input   item
                pAhid->Decode(&button[0],   data,   pItemButton[0]);
                pAhid->Decode(&button[1],   data,   pItemButton[1]);
                pAhid->Decode(&button[2],   data,   pItemButton[2]);
                pAhid->Decode(&x,           data,   pItemX);
                pAhid->Decode(&y,           data,   pItemY);
                pAhid->Decode(&wheel,       data,   pItemWheel);

                NN_SDK_LOG(
                        "buttons: %d %d %d x: %d y: %d wheel: %d\n",
                        button[0],
                        button[1],
                        button[2],
                        x,
                        y,
                        wheel
                        );
            }
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
static void testWrite(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[4];
    uint32_t bytesWritten;

    NN_SDK_LOG("calling Write\n");
    result = pAhid->Write(&bytesWritten, data, 4);
    NN_SDK_LOG("result %d bytesWritten %d\n", result, bytesWritten);
}


//////////////////////////////////////////////////////////////////////////////
static void testFinalize(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling Finalize\n");
    result = pAhid->Finalize();
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void clientFunction()
{
    bool run = true;

    nn::ahid::hdr::Hdr  hdr;
    nn::ahid::Ahid      ahid; // use one client per device

    NN_SDK_LOG("->%s\n", __FUNCTION__);

    hdr.Initialize();

    nn::ahid::hdr::AttachFilter attachFilter;

    memset(&attachFilter, 0, sizeof(nn::ahid::hdr::AttachFilter));

    // Client queries HDR for attached devices
    while (run)
    {
        size_t outEntries, inEntries;

        nn::Result result = hdr.GetDeviceEntries(&outEntries);

        if (result.IsSuccess() && outEntries)
        {
            nn::ahid::hdr::DeviceHandle deviceHandle[nn::ahid::hdr::AhidDevicesCountMax];

            inEntries = nn::ahid::hdr::AhidDevicesCountMax;

            result = hdr.GetDeviceList(&outEntries, inEntries, deviceHandle, &attachFilter);

            attachFilter.attachedAfterTicks = nn::os::GetSystemTick().GetInt64Value();

            if (result.IsSuccess() && outEntries)
            {
                for (size_t j = 0; j < outEntries; j++)
                {
                    nn::ahid::hdr::DeviceParameters deviceParameters;

                    result = hdr.GetDeviceParameters(&deviceParameters, deviceHandle[j]);

                    if (result.IsSuccess())
                    {
                        //if (strcmp((const char*)deviceParameters.servicePath, "mouse") == 0)

                        // look for usagePage: Generic Desktop usageId: Mouse
                        if ((deviceParameters.usagePage == 0x01) && (deviceParameters.usageId == 0x02))
                        {
                            NN_SDK_LOG("DeviceHandle:       %08x\n",    deviceHandle[j]);
                            NN_SDK_LOG("    busId:          %d\n",      deviceParameters.busId);
                            NN_SDK_LOG("    servicePath:    %s\n",      deviceParameters.servicePath);
                            NN_SDK_LOG("    vendorId:       %08x\n",    deviceParameters.vendorId);
                            NN_SDK_LOG("    productId:      %08x\n",    deviceParameters.productId);
                            NN_SDK_LOG("    versionNumber:  %08x\n",    deviceParameters.versionNumber);
                            NN_SDK_LOG("    manufacturer:   ");
                            printString(deviceParameters.manufacturer);
                            NN_SDK_LOG("    product:        ");
                            printString(deviceParameters.product);
                            NN_SDK_LOG("    serialNumber:   ");
                            printString(deviceParameters.serialNumber);

                            testInitialize(&ahid, &deviceParameters);
                            testCodeBook(&ahid);
                            testGetString(&ahid);
                            testSetProtocol(&ahid);
                            testGetProtocol(&ahid);
                            testSetIdle(&ahid);
                            testGetIdle(&ahid);
                            testSetReport(&ahid);
                            testGetReport(&ahid);
                            testRead(&ahid);
                            testWrite(&ahid);
                            testFinalize(&ahid);

                            run = false;
                        }
                    }
                }
            }
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }

    hdr.Finalize();

    NN_SDK_LOG("<-%s\n", __FUNCTION__);
}


//////////////////////////////////////////////////////////////////////////////
extern "C" void nnMain()
{
    NN_SDK_LOG("HdrTest built %s %s\n", __DATE__, __TIME__);

    nn::ahid::hdr::HdrInitialize();                 // HDR server
    nn::cdhid::win32::Win32MouseInitialize();       // Win32 AHID Mouse driver

    clientFunction();

    nn::cdhid::win32::Win32MouseFinalize();
    nn::ahid::hdr::HdrFinalize();

    NN_SDK_LOG("End of HdrTest\n");
}
