﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <limits>
#include <memory>
#include <mutex>

#include "testAccount_Initializer.h"
#include "testAccount_TestExecutor.h"
#include "testAccount_TestOperator.h"

#include <nnt.h>

#include "testAccount_Thread.h"
#include "testAccount_Util.h"
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

extern "C" void nnMain()
{
    int result = 1;
    nnt::account::InitializeNetworkTest();
    NN_UTIL_SCOPE_EXIT { nnt::account::FinalizeNetworkTest(result); };

    nn::account::InitializeForAdministrator();

    result = RUN_ALL_TESTS();
}

namespace {

class JobThread
{
private:
    static void ThreadFunction(void* ptr) NN_NOEXCEPT
    {
        reinterpret_cast<JobThread*>(ptr)->ThreadFunctionImpl();
    }

private:
    nn::os::Event m_Terminator;
    nnt::account::Thread m_Thread;
    bool m_ThreadInitialized;

    nnt::account::StackMemory m_Memory;
    nn::Result (*m_Operation)(void*);
    void* m_OperationData;

    struct Statistics
    {
        char threadName[128];
        uint64_t count;
        int64_t maxMsec;
        int64_t minMsec;
        uint64_t sumMsec;

        void Clear(const char* name) NN_NOEXCEPT
        {
            std::strncpy(threadName, name, sizeof(threadName));
            count = 0;
            maxMsec = std::numeric_limits<int64_t>::min();
            minMsec = std::numeric_limits<int64_t>::max();
            sumMsec = 0;
        }
        void Update(nn::TimeSpan duration) NN_NOEXCEPT
        {
            maxMsec = std::max(maxMsec, duration.GetMilliSeconds());
            minMsec = std::min(minMsec, duration.GetMilliSeconds());
            sumMsec += static_cast<uint64_t>(duration.GetMilliSeconds());
            ++count;
        }
        void Report() const NN_NOEXCEPT
        {
            NN_LOG("-----------------------------------\n");
            NN_LOG("Thread: \"%s\"\n", threadName);
            NN_LOG(" - count: %llu\n", count);
            NN_LOG(" - avg: %lld [msec] (min: %lld, max: %lld)\n", static_cast<int64_t>(static_cast<double>(sumMsec) / count), minMsec, maxMsec);
        }
    } m_Stat;

    void ThreadFunctionImpl() NN_NOEXCEPT
    {
        m_Stat.Clear(nn::os::GetThreadNamePointer(nn::os::GetCurrentThread()));
        while (!m_Terminator.TryWait())
        {
            auto begin = nn::os::GetSystemTick();
            NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(m_Operation(m_OperationData));
            m_Stat.Update((nn::os::GetSystemTick() - begin).ToTimeSpan());
        }
    }

public:
    JobThread() NN_NOEXCEPT
        : m_Terminator(nn::os::EventClearMode_ManualClear)
        , m_ThreadInitialized(false)
        , m_Operation(nullptr)
        , m_OperationData(nullptr)
    {
    }
    ~JobThread() NN_NOEXCEPT
    {
        if (m_ThreadInitialized)
        {
            m_Thread.Finalize();
        }
    }
    template <typename T>
    void Initialize(nn::Result (*operation)(T*), T* operationData, nnt::account::StackMemory&& memory, const char* threadName) NN_NOEXCEPT
    {
        m_Operation = reinterpret_cast<decltype(m_Operation)>(operation);
        m_OperationData = operationData;

        m_Memory = std::move(memory);
        m_Thread.Initialize(
            [&]() { ThreadFunction(this); },
            m_Memory.GetAddress(), m_Memory.GetSize(), nn::os::DefaultThreadPriority, threadName);
        m_ThreadInitialized = true;
    }

    void Run() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_ThreadInitialized);
        m_Thread.Start();
    }
    void Stop() NN_NOEXCEPT
    {
        m_Terminator.Signal();
    }
    void Wait() NN_NOEXCEPT
    {
        m_Thread.Wait();
    }
    void Report() const NN_NOEXCEPT
    {
        m_Stat.Report();
    }
};

NN_ALIGNAS(4096) char g_Stack[14][4 * 4096];
NN_ALIGNAS(4096) char g_WorkMemory[4][nn::account::RequiredBufferSizeForNintendoAccountAuthorizationRequestContext];
nnt::account::StackMemoryAllocator g_StackAllocator(g_Stack);
} // ~namespace <anonymous>

TEST(AccountStress, SingleUser)
{
    // NasIdToken params
    nn::nsd::NasServiceSetting shopSetting;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::nsd::GetNasServiceSetting(&shopSetting, nn::nsd::NasServiceNameOfNxShop));
    const nn::account::NintendoAccountAuthorizationRequestParameters Params = {"openid", "a", "b"};

    // Operators
    const auto NaAuthOpCount = std::extent<decltype(g_WorkMemory)>::value;
    std::unique_ptr<nnt::account::NaIdTokenOperator> pNaIdTokenOps[NaAuthOpCount];
    for (auto i = 0; i < NaAuthOpCount; ++ i)
    {
        std::unique_ptr<nnt::account::NaIdTokenOperator> p(new nnt::account::NaIdTokenOperator(
            shopSetting.clientId, shopSetting.redirectUri.value, Params, g_WorkMemory[i], sizeof(g_WorkMemory[i])));
        pNaIdTokenOps[i] = std::move(p);
    }
    nnt::account::NaInfoCacheOperator naInfoCacheOp;
    nnt::account::NsaProfileOperator nsaProfileOp;

    // Executors
    nn::account::Uid uid;
    int count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, &uid, 1));
    ASSERT_TRUE(count > 0);
    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));

    typedef nnt::account::Executor<nn::account::NintendoAccountAuthorizationRequestContext> NaAuthExecutor;
#define NNT_ACCOUNT_NA_AUTH_TEST_INITIALIZER(_o, _a) {NaAuthExecutor::Get<>(_o, _a), "Run(" #_o ")"}
#define NNT_ACCOUNT_NA_AUTH_TEST_INITIALIZER_FOR_CANCEL(_o, _a, _t) {NaAuthExecutor::GetForCancel<>(_o, _a, _t), "RunForCancel<" #_t ">(" #_o ")"}
    struct
    {
        NaAuthExecutor executor;
        char threadName[128];
    } naAuthTests[] = {
        NNT_ACCOUNT_NA_AUTH_TEST_INITIALIZER(*pNaIdTokenOps[0], admin),
        NNT_ACCOUNT_NA_AUTH_TEST_INITIALIZER_FOR_CANCEL(*pNaIdTokenOps[1], admin, 900 * 1000),
        NNT_ACCOUNT_NA_AUTH_TEST_INITIALIZER_FOR_CANCEL(*pNaIdTokenOps[2], admin, 1000 * 1000),
        NNT_ACCOUNT_NA_AUTH_TEST_INITIALIZER_FOR_CANCEL(*pNaIdTokenOps[3], admin, 1100 * 1000),
        // NNT_ACCOUNT_NA_AUTH_TEST_INITIALIZER_FOR_CANCEL(*pNaIdTokenOps[1], admin, 1200 * 1000),
    };
    NN_STATIC_ASSERT(std::extent<decltype(naAuthTests)>::value == NaAuthOpCount);

    typedef nnt::account::Executor<nn::account::AsyncContext> AsyncExecutor;
#define NNT_ACCOUNT_ASYNC_TEST_INITIALIZER(_o, _a) {AsyncExecutor::Get<>(_o, _a), "Run(" #_o ")"}
#define NNT_ACCOUNT_ASYNC_TEST_INITIALIZER_FOR_CANCEL(_o, _a, _t) {AsyncExecutor::GetForCancel<>(_o, _a, _t), "RunForCancel<" #_t ">(" #_o ")"}
    struct
    {
        AsyncExecutor executor;
        char threadName[128];
    } asyncTexts[] = {
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER(naInfoCacheOp, admin),
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER_FOR_CANCEL(naInfoCacheOp, admin, 900 * 1000),
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER_FOR_CANCEL(naInfoCacheOp, admin, 1000 * 1000),
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER_FOR_CANCEL(naInfoCacheOp, admin, 1100 * 1000),
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER_FOR_CANCEL(naInfoCacheOp, admin, 1200 * 1000),
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER(nsaProfileOp, admin),
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER_FOR_CANCEL(nsaProfileOp, admin, 900 * 1000),
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER_FOR_CANCEL(nsaProfileOp, admin, 1000 * 1000),
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER_FOR_CANCEL(nsaProfileOp, admin, 1100 * 1000),
        NNT_ACCOUNT_ASYNC_TEST_INITIALIZER_FOR_CANCEL(nsaProfileOp, admin, 1200 * 1000),
    };
    NN_STATIC_ASSERT(std::extent<decltype(asyncTexts)>::value == std::extent<decltype(g_Stack)>::value  - NaAuthOpCount);

    // Threads
    JobThread threads[std::extent<decltype(g_Stack)>::value];
    for (int i = 0; i < NaAuthOpCount; ++ i)
    {
        auto& e = naAuthTests[i];
        threads[i].Initialize(NaAuthExecutor::Execute, &e.executor, g_StackAllocator.Allocate(), e.threadName);
    }
    for (int i = 0; i < std::extent<decltype(asyncTexts)>::value; ++ i)
    {
        auto& e = asyncTexts[i];
        threads[i + NaAuthOpCount].Initialize(AsyncExecutor::Execute, &e.executor, g_StackAllocator.Allocate(), e.threadName);
    }

    // Start
    NNT_ACCOUNT_LOG("Start signal\n");
    for (auto& t : threads)
    {
        t.Run();
    }

    NNT_ACCOUNT_LOG("Start done\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(8));
    NNT_ACCOUNT_LOG("Stop signal\n");

    // Stop
    for (auto& t : threads)
    {
        t.Stop();
    }
    for (auto& t : threads)
    {
        t.Wait();
    }
    for (auto& t : threads)
    {
        t.Report();
    }
}
