﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/** @file
    @brief 実装で使用している detail/ByteUtil と detail/UuidGenerator を評価する。
 */

#include "testAccount_NasProxy.h"
#include "testAccount_ServiceUtil.h"
#include <nn/account/detail/account_Settings.h>

#include <nn/nn_Log.h>
#include <nn/friends/friends_Api.h>
#include <nn/friends/friends_ApiAdmin.h>
#include <nn/util/util_ScopeExit.h>

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
#include <nn/oe.h>
#endif

#include "testAccount_Initializer.h"
#include <nnt.h>

extern "C" void nnMain()
{
    nn::friends::Initialize();
    nn::friends::DaemonSuspension friendsSuspension;
    nn::friends::SuspendDaemon(&friendsSuspension);

    int result = 1;
    nnt::account::InitializeNetworkTest();
    NN_UTIL_SCOPE_EXIT {
        // アカウントを削除
        nnt::account::Cleanup();
        nnt::account::FinalizeNetworkTest(result);
    };
    // アカウントを削除
    nnt::account::Cleanup();

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    nn::oe::Initialize();
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // ファームウェア設定の読み込み
    nn::account::detail::FirmwareSettings::Refresh();
#endif

    result = RUN_ALL_TESTS();
}

nn::Result LinkNintendoAccount(
    const nnt::account::NasLoginInfo& loginInfo,
    nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    bool isLinked;
    NNT_ACCOUNT_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_FALSE(isLinked);

    nn::account::NintendoAccountLinkageProcedure linkProc;
    NNT_ACCOUNT_RESULT_DO(admin.CreateProcedureToLinkWithNintendoAccount(&linkProc));

    nn::account::RequestUrl requestUrl;
    nn::account::CallbackUri callbackUri;
    NNT_ACCOUNT_RESULT_DO(linkProc.GetRequest(&requestUrl, &callbackUri));

    auto response = nnt::account::GetAuthorizationViaNasProxy(loginInfo, requestUrl.url);
    NN_LOG("NAS Helper response: %.*s\n", response.GetSize(), response.Get<char>());

    nn::account::AsyncContext linkTask;
    NNT_ACCOUNT_RESULT_DO(linkProc.ApplyResponseAsync(&linkTask, response.Get<char>(), response.GetSize()));

    nn::os::SystemEvent linkEvent;
    NNT_ACCOUNT_RESULT_DO(linkTask.GetSystemEvent(&linkEvent));
    linkEvent.Wait();

    NNT_ACCOUNT_RESULT_DO(linkTask.GetResult());

    NNT_ACCOUNT_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_TRUE(isLinked);
    NN_RESULT_SUCCESS;
}

nn::Result LinkNintendoAccountWithSuspend(
    const nnt::account::NasLoginInfo& loginInfo,
    nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    bool isLinked;
    NNT_ACCOUNT_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_FALSE(isLinked);

    nn::account::SessionId sessionId;
    nnt::account::Buffer response;

    // 中断
    {
        nn::account::NintendoAccountLinkageProcedure linkProc;
        NNT_ACCOUNT_RESULT_DO(admin.CreateProcedureToLinkWithNintendoAccount(&linkProc));

        nn::account::RequestUrl requestUrl;
        nn::account::CallbackUri callbackUri;
        NNT_ACCOUNT_RESULT_DO(linkProc.GetRequest(&requestUrl, &callbackUri));

        NNT_ACCOUNT_RESULT_DO(linkProc.Suspend(&sessionId));

        // ブラウザ呼び出し相当の処理
        response = nnt::account::GetAuthorizationViaNasProxy(loginInfo, requestUrl.url);
    }

    // 再開
    {
        NN_LOG("NAS Helper response: %.*s\n", response.GetSize(), response.Get<char>());

        nn::account::NintendoAccountLinkageProcedure linkProc;
        NNT_ACCOUNT_RESULT_DO(admin.ResumeProcedureToLinkWithNintendoAccount(&linkProc, sessionId));

        nn::account::AsyncContext linkTask;
        NNT_ACCOUNT_RESULT_DO(linkProc.ApplyResponseAsync(&linkTask, response.Get<char>(), response.GetSize()));

        nn::os::SystemEvent linkEvent;
        NNT_ACCOUNT_RESULT_DO(linkTask.GetSystemEvent(&linkEvent));
        linkEvent.Wait();

        NNT_ACCOUNT_RESULT_DO(linkTask.GetResult());

        NNT_ACCOUNT_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
        EXPECT_TRUE(isLinked);
        NN_RESULT_SUCCESS;
    }
}

nn::Result UnlinkNintendoAccount(
    nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    bool isLinked;
    NNT_ACCOUNT_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_TRUE(isLinked);

    nn::account::AsyncContext unlinkTask;
    NNT_ACCOUNT_RESULT_DO(admin.DebugUnlinkNintendoAccountAsync(&unlinkTask));

    nn::os::SystemEvent unlinkEvent;
    NNT_ACCOUNT_RESULT_DO(unlinkTask.GetSystemEvent(&unlinkEvent));
    unlinkEvent.Wait();

    NNT_ACCOUNT_RESULT_DO(unlinkTask.GetResult());

    NNT_ACCOUNT_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_FALSE(isLinked);
    NN_RESULT_SUCCESS;
}
