﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/account_RuntimeResource.h>

#include <nn/account/profile/account_ProfileStorage.h>

#include "nas\account_NasCredentialHolder.h"

#include "testAccount_File.h"
#include "testAccount_Mounter.h"
#include "testAccount_Thread.h"
#include "testAccount_Util.h"
#include "testAccount_NasProxy.h"

#include <atomic>
#include <curl/curl.h>

#include <nn/os/os_Tick.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/util/util_ScopeExit.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_NAS_OPERATOR
#define NNT_ACCOUNT_ENABLE_NAS_OPERATOR_LINKAGE
#define NNT_ACCOUNT_ENABLE_NAS_OPERATOR_IMPLICIT_RECOVERY
#define NNT_ACCOUNT_ENABLE_NAS_OPERATOR_APPLICATION
#define NNT_ACCOUNT_ENABLE_NAS_OPERATOR_GUEST
#define NNT_ACCOUNT_ENABLE_NAS_OPERATOR_OP2

#if defined(NNT_ACCOUNT_ENABLE_NAS_OPERATOR)

namespace
{

class Task
    : public nn::account::detail::Executable<nn::account::ExecutionResource>
{
private:
    typedef std::function<nn::Result(const nn::account::ExecutionResource&, nn::account::detail::Cancellable*)> FnType;
    FnType m_Fn;
public:
    explicit Task(FnType&& fn) NN_NOEXCEPT
        : m_Fn(std::move(fn))
    {
    }
    virtual nn::Result ExecuteImpl(const nn::account::ExecutionResource& resource) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Fn(resource, this);
    }
};

void ThreadFunction(
    nn::os::Event& terminator,
    a::Executor& executor, a::ExecutionResource& resource) NN_NOEXCEPT
{
    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    nn::os::MultiWaitHolderType terminatorH;
    nn::os::InitializeMultiWaitHolder(&terminatorH, terminator.GetBase());
    nn::os::LinkMultiWaitHolder(&multiWait, &terminatorH);

    nn::os::MultiWaitHolderType executorH;
    executor.InitializeMultiWaitHolder(&executorH);
    nn::os::LinkMultiWaitHolder(&multiWait, &executorH);

    nn::os::TimerEvent curlTimer(nn::os::EventClearMode_ManualClear);
    nn::os::MultiWaitHolderType curlTimerH;
    nn::os::InitializeMultiWaitHolder(&curlTimerH, curlTimer.GetBase());
    nn::os::LinkMultiWaitHolder(&multiWait, &curlTimerH);

    // CURL ハンドル周りの初期化
    bool isCurlHandleUsed = false;
    curlTimer.StartPeriodic(nn::TimeSpan::FromSeconds(10), nn::TimeSpan::FromSeconds(10));

    while (NN_STATIC_CONDITION(true))
    {
        auto pH = nn::os::WaitAny(&multiWait);
        if (pH == &terminatorH)
        {
            // Terminator は Clear しない。
            NN_LOG("[!!!] Terminate requested\n");
            break;
        }

        if (pH == &curlTimerH)
        {
            curlTimer.Clear();
            if (isCurlHandleUsed && false)
            {
                // CURL ハンドルの再取得
                curl_easy_cleanup(resource.curlHandle);
                resource.curlHandle = curl_easy_init();
                NN_ABORT_UNLESS(resource.curlHandle);
                isCurlHandleUsed = false;
                NN_LOG("[!!!] Curl refreshed\n");
            }
        }
        else if (pH == &executorH)
        {
            executor.ClearSignal();
            executor.Execute(resource);
            isCurlHandleUsed = true;
        }
    }

    // CURL ハンドル周りの解放
    curlTimer.Stop();

    nn::os::UnlinkMultiWaitHolder(&curlTimerH);
    nn::os::FinalizeMultiWaitHolder(&curlTimerH);

    nn::os::UnlinkMultiWaitHolder(&executorH);
    nn::os::FinalizeMultiWaitHolder(&executorH);

    nn::os::UnlinkMultiWaitHolder(&terminatorH);
    nn::os::FinalizeMultiWaitHolder(&terminatorH);

    nn::os::FinalizeMultiWait(&multiWait);
}

#define NNT_ACCOUNT_CREATE_READ_EVENT(name, task, clearMode) \
    nn::os::SystemEvent name((task).GetReadableHandle(), false, nn::os::InvalidNativeHandle, false, clearMode)

bool CompareLoginInfo(const t::NasLoginInfo& loginInfo, const a::nas::NasUserBase& base) NN_NOEXCEPT
{
    NN_LOG("[account] User resource comparison\n");
    if (std::strchr(loginInfo.email, '@') != nullptr)
    {
        NN_LOG("%s <-> %s\n", loginInfo.email, base.email);
        return std::strncmp(loginInfo.email, base.email, sizeof(base.email)) == 0;
    }
    NN_LOG("%s <-> %s\n", loginInfo.email, base.loginId);
    return std::strncmp(loginInfo.email, base.loginId, sizeof(base.loginId)) == 0;
}
bool CompareNasUserBaseForApplication(const a::nas::NasUserBase& base, const a::nas::NasUserBaseForApplication& base2) NN_NOEXCEPT
{
    NN_LOG("[account] User resource comparison\n");
    EXPECT_EQ(0, std::strncmp(base.birthday, base2.birthday, sizeof(base.birthday)));
    EXPECT_EQ(0, std::strncmp(base.gender, base2.gender, sizeof(base.gender)));
    EXPECT_EQ(0, std::strncmp(base.language, base2.language, sizeof(base.language)));
    EXPECT_EQ(0, std::strncmp(base.country, base2.country, sizeof(base.country)));
    EXPECT_EQ(0, std::strncmp(base.region, base2.region, sizeof(base.region)));
    EXPECT_EQ(0, std::strncmp(base.timezone, base2.timezone, sizeof(base.timezone)));
    return true;
}

nn::Result ExecuteTask(
    nn::account::detail::Executable<nn::account::ExecutionResource>& task,
    a::Executor& executor) NN_NOEXCEPT
{
    NN_RESULT_DO(task.Initialize());
    NN_RESULT_DO(executor.Register(&task));
    NN_UTIL_SCOPE_EXIT
    {
        executor.Unregister(&task);
    };
    NNT_ACCOUNT_CREATE_READ_EVENT(e, task, nn::os::EventClearMode_ManualClear);
    e.Wait();
    return task.GetResult();
}
nn::Result ExecuteTask(
    std::function<nn::Result(const nn::account::ExecutionResource& resource, nn::account::detail::Cancellable* pCancellable)> f,
    a::Executor& executor) NN_NOEXCEPT
{
    Task task(std::move(f));
    return ExecuteTask(task, executor);
}

void RunLinkage(
    const t::NasLoginInfo& LoginInfo,
    a::Executor& executor, const a::Uid& user,
    a::nas::NasOperator& nasOp, a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    a::RequestUrl requestUrl;
    a::CallbackUri callbackUri;
    t::Buffer buffer(4096);

    NN_LOG(">>> Linkage start\n");
    a::nas::NasLinkageProcedure procedure(user, nasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetRequest(&requestUrl, &callbackUri, a::NintendoAccountAuthorizationPageTheme_Intro));

    NN_LOG("[nnt::account] RequestUrl: %s\n", requestUrl.url);

    NN_LOG(">>>> Authorization via NAS proxy\n");
    auto response = t::GetAuthorizationViaNasProxy(LoginInfo, requestUrl.url);
    ASSERT_TRUE(strnlen(response.Get<char>(), response.GetSize()) < response.GetSize());
    NN_LOG("[nnt::account] Acquired authorization callback: %s\n", response.Get<char>());

    NN_LOG(">>>> Linkage execution\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.ApplyResponse(response.Get<char>(), response.GetSize()));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(
        ExecuteTask([&](const auto& r, auto* pC)-> nn::Result { return procedure.ExecutePostprocess(r, pC); },
        executor));

    // 処理結果
    NN_LOG("[Linkage result:]\n");
    auto replaced = procedure.IsNetworkServiceAccountReplaced();
    NN_LOG(" - NSA: %s\n", replaced ? "<replaced>" : "<not replaced>");

    NN_LOG(">>>> check linkage state\n");
    bool isLinked;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsLinkedWithNintendoAccount(&isLinked, user));
    ASSERT_TRUE(isLinked);

    NN_LOG(">>>> check UserResource\n");
    a::NintendoAccountId id;
    a::nas::NasUserBase base;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.LoadUserResourceCache(&id, &base, user, buffer.GetAddress(), buffer.GetSize()));
    // EXPECT_EQ(EsxpectId, id);
    EXPECT_TRUE(CompareLoginInfo(LoginInfo, base));
    a::NintendoAccountId idForApp;
    a::nas::NasUserBaseForApplication baseForApp;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.LoadUserResourceCacheForApplication(&idForApp, &baseForApp, user, buffer.GetAddress(), buffer.GetSize()));
    EXPECT_TRUE(CompareNasUserBaseForApplication(base, baseForApp));
    EXPECT_EQ(id, idForApp);

    NN_LOG(">>>> refresh UserResource\n");
    a::nas::NasUserResourceUpdateTask resourceTask(user, nasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(resourceTask, executor));

    NN_LOG(">>>> check UserResource\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.LoadUserResourceCache(&id, &base, user, buffer.GetAddress(), buffer.GetSize()));
    // EXPECT_EQ(EsxpectId, id);
    EXPECT_TRUE(CompareLoginInfo(LoginInfo, base));
}
void RunRefresh(
    const t::NasLoginInfo& LoginInfo,
    a::Executor& executor, const a::Uid& user,
    a::nas::NasOperator& nasOp, a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    a::RequestUrl requestUrl;
    a::CallbackUri callbackUri;
    t::Buffer buffer(4096);

    NN_LOG(">>> CredentialUpdate start\n");
    a::nas::NasCredentialUpdateProcedure procedure(user, nasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetRequest(&requestUrl, &callbackUri));

    NN_LOG("[nnt::account] RequestUrl: %s\n", requestUrl.url);

    NN_LOG(">>>> Authorization via NAS proxy\n");
    auto response = t::GetAuthorizationViaNasProxy(LoginInfo, requestUrl.url);
    ASSERT_TRUE(strnlen(response.Get<char>(), response.GetSize()) < response.GetSize());
    NN_LOG("[nnt::account] Acquired authorization callback: %s\n", response.Get<char>());

    NN_LOG(">>>> LinkageStateUpdate execution\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.ApplyResponse(response.Get<char>(), response.GetSize()));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(
        ExecuteTask([&](const auto& r, auto* pC)-> nn::Result { return procedure.ExecutePostprocess(r, pC); },
        executor));

    NN_LOG(">>>> check UserResource\n");
    a::NintendoAccountId id;
    a::nas::NasUserBase base;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.LoadUserResourceCache(&id, &base, user, buffer.GetAddress(), buffer.GetSize()));
    // EXPECT_EQ(EsxpectId, id);
    EXPECT_TRUE(CompareLoginInfo(LoginInfo, base));

    NN_LOG(">>>> refresh UserResource\n");
    a::nas::NasUserResourceUpdateTask resourceTask(user, nasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(resourceTask, executor));

    NN_LOG(">>>> check UserResource\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.LoadUserResourceCache(&id, &base, user, buffer.GetAddress(), buffer.GetSize()));
    // EXPECT_EQ(EsxpectId, id);
    EXPECT_TRUE(CompareLoginInfo(LoginInfo, base));
}

void RunLinkageAndRefreshTest(
    const t::NasLoginInfo& LoginInfo,
    a::Executor& executor, const a::Uid& user,
    a::nas::NasOperator& nasOp, a::baas::BaasOperator& baasOp, a::profile::ProfileStorage& profile) NN_NOEXCEPT
{
    RunLinkage(LoginInfo, executor, user, nasOp, baasOp);
    RunRefresh(LoginInfo, executor, user, nasOp, baasOp);

    NN_LOG(">>>> Unlinkage execution\n");
    {
        a::baas::NintendoAccountUnlinkageTask unlinkageTask(user, baasOp);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(unlinkageTask, executor));
    }

    NN_LOG(">>>> check linkage state\n");
    bool isLinked;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsLinkedWithNintendoAccount(&isLinked, user));
    ASSERT_FALSE(isLinked);

    NN_LOG(">>> Set invalid image \n");
    a::profile::ProfileBase base;
    a::profile::UserData userData;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(profile.GetProfile(&base, &userData, user));
    base.author = user;
    base.timeStamp += 1;
    t::Buffer image(1024);
    std::memset(image.GetAddress(), 0, image.GetSize());
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(profile.Update(user, base, userData, image.GetAddress(), image.GetSize()));

    RunLinkage(LoginInfo, executor, user, nasOp, baasOp);
    RunRefresh(LoginInfo, executor, user, nasOp, baasOp);

    NN_LOG(">>>> Unlinkage execution\n");
    {
        a::baas::NintendoAccountUnlinkageTask unlinkageTask(user, baasOp);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(unlinkageTask, executor));
    }
} // NOLINT(readability/fn_size)

NN_ALIGNAS(4096) char AuthorizationBuffer[4096];

void RunApplicationAuthorizationTest(
    const t::NasLoginInfo& LoginInfo,
    a::Executor& executor, const a::Uid& user,
    a::nas::NasOperator& nasOp, a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    const a::nas::NasClientInfo client = {0xc924fc63552653e9ull, "http://localhost:8080"};
    // const a::nas::NasClientInfo client = {0xd9a620d8bb417787ull, "http://localhost:8080"}; // dd1

    a::RequestUrl requestUrl;
    a::CallbackUri callbackUri;

    NN_LOG(">>> Revoke authorization \n");
    t::RevokeAuthorization(LoginInfo);

    NN_LOG(">>> Linkage start\n");
    a::nas::NasLinkageProcedure linkageProcedure(user, nasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(linkageProcedure.GetRequest(&requestUrl, &callbackUri, a::NintendoAccountAuthorizationPageTheme_Intro));

    NN_LOG("[nnt::account] RequestUrl: %s\n", requestUrl.url);

    NN_LOG(">>>> Authorization via NAS proxy\n");
    auto response = t::GetAuthorizationViaNasProxy(LoginInfo, requestUrl.url);
    ASSERT_TRUE(strnlen(response.Get<char>(), response.GetSize()) < response.GetSize());
    NN_LOG("[nnt::account] Acquired authorization callback: %s\n", response.Get<char>());

    NN_LOG(">>>> Linkage execution\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(linkageProcedure.ApplyResponse(response.Get<char>(), response.GetSize()));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(
        [&](const auto& r, auto* pC) -> nn::Result { return linkageProcedure.ExecutePostprocess(r, pC); },
        executor));

    NN_LOG(">>>> check linkage state\n");
    bool isLinked;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsLinkedWithNintendoAccount(&isLinked, user));
    ASSERT_TRUE(isLinked);

    NN_LOG(">>> Authorization via NAS proxy\n");
    {
        a::NintendoAccountAuthorizationRequestParameters params = {
            {"user openid"},
            {"aaaaaa"},
            {"bbbbbb"},
        };

        a::nas::NasApplicationAuthorizationProcedure procedure(user, client, nasOp);
        nn::os::TransferMemory memory(AuthorizationBuffer, sizeof(AuthorizationBuffer), nn::os::MemoryPermission_ReadWrite);
        procedure.AttachTransferMemory(memory.Detach(), sizeof(AuthorizationBuffer), true);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.Initialize(params));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(
            [&](const auto& r, auto* pC) -> nn::Result { return procedure.ExecutePreprocess(r, pC); },
            executor));

        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetRequest(&requestUrl, &callbackUri));

        NN_LOG("[nnt::account] RequestUrl: %s\n", requestUrl.url);
        auto authz = t::GetApplicationAuthorizationViaNasProxy(LoginInfo, requestUrl.url);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.ApplyResponse(authz.Get<char>(), authz.GetSize()));

        size_t sizeActual;
        t::Buffer buffer(4096);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetIdToken(&sizeActual, buffer.Get<char>(), buffer.GetSize()));
        ASSERT_TRUE(sizeActual <= a::detail::NasIdTokenSizeMax);
        ASSERT_TRUE(buffer.Get<char>()[sizeActual] == '\0');
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetAuthorizationCode(&sizeActual, buffer.Get<char>(), buffer.GetSize()));
        ASSERT_TRUE(sizeActual <= a::detail::NasAuthorizationCodeSizeMax);
        ASSERT_TRUE(buffer.Get<char>()[sizeActual] == '\0');
        a::nas::State state;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetState(&state));
        strncmp(state.data, params.state, sizeof(state.data));
    }

    NN_LOG(">>>> Acquire authorization without interaction\n");
    {
        a::NintendoAccountAuthorizationRequestParameters params = {
            {"user openid"},
            {"YYY"},
            {"ZZZ"},
        };

        a::nas::NasApplicationAuthorizationProcedure procedure(user, client, nasOp);
        nn::os::TransferMemory memory(AuthorizationBuffer, sizeof(AuthorizationBuffer), nn::os::MemoryPermission_ReadWrite);
        procedure.AttachTransferMemory(memory.Detach(), sizeof(AuthorizationBuffer), true);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.Initialize(params));

        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(
            [&](const auto& r, auto* pC) -> nn::Result { return procedure.ExecuteImplicitly(r, pC); },
            executor));

        size_t sizeActual;
        t::Buffer buffer(4096);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetIdToken(&sizeActual, buffer.Get<char>(), buffer.GetSize()));
        ASSERT_TRUE(sizeActual <= a::detail::NasIdTokenSizeMax);
        ASSERT_TRUE(buffer.Get<char>()[sizeActual] == '\0');
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetAuthorizationCode(&sizeActual, buffer.Get<char>(), buffer.GetSize()));
        ASSERT_TRUE(sizeActual <= a::detail::NasAuthorizationCodeSizeMax);
        ASSERT_TRUE(buffer.Get<char>()[sizeActual] == '\0');
        a::nas::State state;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetState(&state));
        strncmp(state.data, params.state, sizeof(state.data));
    }

    NN_LOG(">>>> Unlinkage execution\n");
    a::baas::NintendoAccountUnlinkageTask unlinkageTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(unlinkageTask, executor));

    NN_LOG(">>>> check linkage state\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsLinkedWithNintendoAccount(&isLinked, user));
    ASSERT_FALSE(isLinked);
} // NOLINT(readability/fn_size)

NN_ALIGNAS(4096) char GuestLoginBuffer[4096 * 34];

bool PrepareNintendoAccount(
    a::NetworkServiceAccountId* pOutNsaId, a::NintendoAccountId* pOutNaId,
    const t::NasLoginInfo& LoginInfo,
    const t::InFile& image, const a::profile::ProfileBase& base,
    a::Executor& executor, const a::Uid& user,
    a::nas::NasOperator& nasOp,
    a::profile::ProfileStorage& profileStorage, a::ndas::NdasOperator& ndasOp, a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{

    a::RequestUrl requestUrl;
    a::CallbackUri callbackUri;

    NN_LOG(">>> Linkage start\n");
    a::nas::NasLinkageProcedure linkageProcedure(user, nasOp);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkageProcedure.GetRequest(&requestUrl, &callbackUri, a::NintendoAccountAuthorizationPageTheme_Intro));

    auto response = t::GetAuthorizationViaNasProxy(LoginInfo, requestUrl.url);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(response.Get<char>(), response.GetSize()) < response.GetSize());
    NN_LOG("[nnt::account] Acquired authorization callback: %s\n", response.Get<char>());

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkageProcedure.ApplyResponse(response.Get<char>(), response.GetSize()));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ExecuteTask(
        [&](const auto& r, auto* pC) -> nn::Result { return linkageProcedure.ExecutePostprocess(r, pC); },
        executor));

    bool isLinked;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(baasOp.IsLinkedWithNintendoAccount(&isLinked, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(isLinked);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(baasOp.GetNetworkServiceAccountId(pOutNsaId, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(baasOp.GetLinkedNintendoAccountId(pOutNaId, user));

    NN_LOG(">>> Set profile \n");
    a::profile::UserData userData = {};
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(profileStorage.Update(user, base, userData, image.GetAddress(), image.GetSize()));
    a::baas::BaasUserProfileUploadTask ulTask(user, baasOp);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ExecuteTask(ulTask, executor));

    NN_LOG(">>>> Unregister\n");
    a::baas::BaasErasureTask unregTask(user, baasOp);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ExecuteTask(unregTask, executor));

    profileStorage.Delete(user);
    return true;
}

void RunGuestLoginTest(
    const t::NasLoginInfo& LoginInfo,
    const a::NetworkServiceAccountId& nsaId, const a::NintendoAccountId& naId,
    const t::InFile& image, const a::profile::ProfileBase& base,
    a::Executor& executor, a::nas::NasOperator& nasOp, a::ndas::NdasOperator& ndasOp, a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    a::RequestUrl requestUrl;
    a::CallbackUri callbackUri;

    NN_LOG(">>> Guest login start\n");
    const auto appInfo = a::detail::ApplicationInfo::Get(0x010000000000B123ull, 0, a::detail::ApplicationMediaType::System);
    a::nas::NasGuestLoginProcedure procedure(appInfo, ndasOp, baasOp, nasOp);

    nn::os::TransferMemory memory(GuestLoginBuffer, sizeof(GuestLoginBuffer), nn::os::MemoryPermission_None);
    procedure.AttachTransferMemory(memory.Detach(), sizeof(GuestLoginBuffer), true);
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(procedure.Initialize());

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(
        [&](const auto& r, auto* pC) -> nn::Result { return procedure.ExecutePreprocess(r, pC); },
        executor));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetRequest(&requestUrl, &callbackUri));

    NN_LOG(">>> Authorization via NAS proxy\n");
    NN_LOG("[nnt::account] RequestUrl: %s\n", requestUrl.url);
    auto authz = t::GetAuthorizationViaNasProxyForGuest(LoginInfo, requestUrl.url);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.ApplyResponse(authz.Get<char>(), authz.GetSize()));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(
        [&](const auto& r, auto* pC) -> nn::Result { return procedure.ExecutePostprocess(r, pC); },
        executor));

    NN_LOG(">>> Check result\n");
    size_t outSize;
    t::Buffer outIdToken(a::detail::BaasIdTokenSizeMax);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.LoadBaasIdTokenCache(&outSize, outIdToken.Get<char>(), outIdToken.GetSize()));
    NN_LOG("ID token: %.*s\n", outIdToken.GetSize(), outIdToken.Get<char>());

    a::NetworkServiceAccountId outNsaId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetNetworkServiceAccountId(&outNsaId));
    EXPECT_EQ(nsaId.id, outNsaId.id);
    a::NintendoAccountId outNaId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetLinkedNintendoAccountId(&outNaId));
    EXPECT_EQ(naId.id, outNaId.id);

    a::Nickname nickname;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetNickname(nickname.name, sizeof(nickname.name)));
    EXPECT_EQ(0, strncmp(nickname.name, base.nickname, sizeof(base.nickname)));

    t::Buffer outImage(a::ProfileImageBytesMax);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetProfileImage(&outSize, outImage.GetAddress(), outImage.GetSize()));
    EXPECT_EQ(image.GetSize(), outSize);
    EXPECT_EQ(0, std::memcmp(image.GetAddress(), outImage.GetAddress(), outSize));
}

void RunFloatingRegistration(
    const t::NasLoginInfo& LoginInfo,
    const a::NetworkServiceAccountId& nsaId, const a::NintendoAccountId& naId,
    const t::InFile& image, const a::profile::ProfileBase& base,
    a::Executor& executor, const a::Uid& user,
    a::nas::NasOperator& nasOp,
    a::profile::ProfileStorage& profileStorage, a::ndas::NdasOperator& ndasOp, a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    a::RequestUrl requestUrl;
    a::CallbackUri callbackUri;

    NN_LOG(">>> Guest login start\n");
    a::nas::NasFloatingRegistrationProcedure procedure(ndasOp, baasOp, nasOp);

    nn::os::TransferMemory memory(GuestLoginBuffer, sizeof(GuestLoginBuffer), nn::os::MemoryPermission_None);
    procedure.AttachTransferMemory(memory.Detach(), sizeof(GuestLoginBuffer), true);
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(procedure.Initialize());

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(
        [&](const auto& r, auto* pC) -> nn::Result { return procedure.ExecutePreprocess(r, pC); },
        executor));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetRequest(&requestUrl, &callbackUri));

    NN_LOG(">>> Authorization via NAS proxy\n");
    NN_LOG("[nnt::account] RequestUrl: %s\n", requestUrl.url);
    auto authz = t::GetAuthorizationViaNasProxy(LoginInfo, requestUrl.url);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.ApplyResponse(authz.Get<char>(), authz.GetSize()));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(
        [&](const auto& r, auto* pC) -> nn::Result { return procedure.ExecutePostprocess(r, pC); },
        executor));

    NN_LOG(">>> NsaIdToken\n");
    const auto appInfo = a::detail::ApplicationInfo::Get(0x010000000000B123ull, 0, a::detail::ApplicationMediaType::System);
    Task login([&](const a::ExecutionResource& resource, a::detail::Cancellable* pC) -> nn::Result {
        return procedure.EnsureNetworkServiceAccountIdToken(appInfo, resource, pC);
    });
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(login, executor));

    NN_LOG(">>> Check result\n");
    size_t outSize;
    t::Buffer outIdToken(a::detail::BaasIdTokenSizeMax);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.LoadBaasIdTokenCache(&outSize, outIdToken.Get<char>(), outIdToken.GetSize(), appInfo));
    NN_LOG("ID token: %.*s\n", outIdToken.GetSize(), outIdToken.Get<char>());

    a::NetworkServiceAccountId outNsaId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetNetworkServiceAccountId(&outNsaId));
    EXPECT_EQ(nsaId.id, outNsaId.id);
    a::NintendoAccountId outNaId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetLinkedNintendoAccountId(&outNaId));
    EXPECT_EQ(naId.id, outNaId.id);

    a::Nickname nickname;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetNickname(nickname.name, sizeof(nickname.name)));
    EXPECT_EQ(0, strncmp(nickname.name, base.nickname, sizeof(base.nickname)));

    t::Buffer outImage(a::ProfileImageBytesMax);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(procedure.GetProfileImage(&outSize, outImage.GetAddress(), outImage.GetSize()));
    EXPECT_EQ(image.GetSize(), outSize);
    EXPECT_EQ(0, std::memcmp(image.GetAddress(), outImage.GetAddress(), outSize));

    NN_LOG(">>> Register\n");
    profileStorage.Add(user);
    Task reg([&](const a::ExecutionResource& resource, a::detail::Cancellable* pC) -> nn::Result {
        return procedure.RegisterNetworkServiceAccount(user, resource, pC);
    });
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(reg, executor));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.CheckAvailability(user));

    a::profile::ProfileBase outProfile;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(profileStorage.GetProfile(&outProfile, nullptr, user));
    EXPECT_EQ(base.author, outProfile.author);
    EXPECT_EQ(0, strncmp(outProfile.nickname, base.nickname, sizeof(base.nickname)));
    EXPECT_EQ(base.timeStamp, outProfile.timeStamp);

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(profileStorage.LoadImage(&outSize, outImage.GetAddress(), outImage.GetSize(), user));
    EXPECT_EQ(image.GetSize(), outSize);
    EXPECT_EQ(0, std::memcmp(image.GetAddress(), outImage.GetAddress(), outSize));
}

void RunImplicitRecoveryTest(
    const t::NasLoginInfo& LoginInfo,
    a::Executor& executor, const a::Uid& user,
    a::nas::NasOperator& nasOp, a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    RunLinkage(LoginInfo, executor, user, nasOp, baasOp);

    NN_LOG(">> Set UserState Invalid\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.DegradeLinkedNintendoAccountUserState(user, a::baas::NintendoAccountUserState_TermsAgreementRequired));

    NN_LOG(">> Try Recovery UserState\n");
    a::nas::NasUserStateRecoveryTryTask recoveryTask(user, nasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(recoveryTask, executor));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.CheckAvailability(user));

    RunRefresh(LoginInfo, executor, user, nasOp, baasOp);

    NN_LOG(">>>> Unlinkage execution\n");
    {
        a::baas::NintendoAccountUnlinkageTask unlinkageTask(user, baasOp);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(unlinkageTask, executor));
    }
} // NOLINT(readability/fn_size)


#if defined(NNT_ACCOUNT_ENABLE_NAS_OPERATOR_OP2)
void RunOp2MembershipTest(
    a::NetworkServiceLicense* pOut,
    const t::NasLoginInfo& LoginInfo,
    a::Executor& executor, const a::Uid& user,
    a::nas::NasOperator& nasOp, a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    a::RequestUrl requestUrl;
    a::CallbackUri callbackUri;

    NN_LOG(">>> Linkage\n");
    a::nas::NasLinkageProcedure linkageProcedure(user, nasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(linkageProcedure.GetRequest(&requestUrl, &callbackUri, a::NintendoAccountAuthorizationPageTheme_Intro));
    auto response = t::GetAuthorizationViaNasProxy(LoginInfo, requestUrl.url);
    ASSERT_TRUE(strnlen(response.Get<char>(), response.GetSize()) < response.GetSize());
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(linkageProcedure.ApplyResponse(response.Get<char>(), response.GetSize()));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(
        [&](const auto& r, auto* pC) -> nn::Result { return linkageProcedure.ExecutePostprocess(r, pC); },
        executor));

    NN_LOG(">>> OP2 membership\n");
    a::NetworkServiceLicense license;
    nn::time::PosixTime expiration;
    nn::TimeSpan span;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.GetTimeSpanSinceLastUpdateOfNetworkServiceLicenseCache(&span, user));
    EXPECT_TRUE(span < nn::TimeSpan::FromSeconds(1));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.GetTimeSpanSinceLastUpdateOfNetworkServiceLicenseCache(&span, user));
    EXPECT_TRUE(span >= nn::TimeSpan::FromSeconds(1));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.GetNetworkServiceLicenseCache(&license, &expiration, user));
    switch (license)
    {
    case a::NetworkServiceLicense_Common:
        {
            auto calendar = nn::time::ToCalendarTimeInUtc(expiration);
            NN_LOG("[account] Acquire OP2 membership: Common (%d-%02d-%02d %02d:%02d:%02d)\n",
                calendar.year, calendar.month, calendar.day,
                calendar.hour, calendar.minute, calendar.second);
        }
        break;
    case a::NetworkServiceLicense_None:
        NN_LOG("[account] Acquire OP2 membership: None\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // Invalidate
    nasOp.InvalidateNetworkServiceLicenseCache(user);
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultResourceCacheUnavailable,
        nasOp.GetNetworkServiceLicenseCache(&license, &expiration, user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultResourceCacheUnavailable,
        nasOp.GetTimeSpanSinceLastUpdateOfNetworkServiceLicenseCache(&span, user));

    const int Count = 2;
    for (int i = 0; i < Count; ++ i)
    {
        NN_LOG(">>> Update OP2 membership\n");
        a::nas::NetworkServiceLicenseUpdateTask task(user, nasOp);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(task, executor));

        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.GetTimeSpanSinceLastUpdateOfNetworkServiceLicenseCache(&span, user));
        EXPECT_TRUE(span < nn::TimeSpan::FromSeconds(1));

        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasOp.GetNetworkServiceLicenseCache(&license, &expiration, user));
        EXPECT_TRUE(license == a::NetworkServiceLicense_Common || license == a::NetworkServiceLicense_None);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1500));
    }
    *pOut = license;
} // NOLINT(readability/fn_size)
#endif // NNT_ACCOUNT_ENABLE_NAS_OPERATOR_OP2

NN_ALIGNAS(4096) char g_Stack[4096 * 8];
} // ~namespace <anonymous>

#include <nn/os/os_MultipleWaitUtility.h>

TEST(AccountNasOperator, Basic)
{
    // ストレージ
    t::DefaultTestStorage storage;
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Mount());
    storage.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Setup());

    // ユーザーとアプリケーション
    a::Uid user = a::detail::ConvertToUid(storage.GenerateUuidWithContext());

    // NAS ユーザーリソースキャッシュ
    a::nas::NasUserResourceCache nasUserResourceCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nasUserResourceCache.Initialize(storage));

    // ユーザープロフィール
    a::profile::ProfileStorage profile;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(profile.Initialize(nullptr, 0, storage));

    // NDAS トークンキャッシュ
    nn::account::ndas::ApplicationAuthenticationCache appAuthCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(appAuthCache.Initialize(storage));

    // BaaS トークンキャッシュ
    a::baas::ClientAccessTokenCache clientAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(clientAccessTokenCache.Initialize(storage));
    a::baas::UserAccessTokenCache userAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userAccessTokenCache.Initialize(storage));
    a::baas::UserIdTokenCache userIdTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userIdTokenCache.Initialize(storage));
    nn::account::baas::BaasUserInfoHolder userInfoHolder;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userInfoHolder.Initialize(nasUserResourceCache, storage));

    // NDAS, BaaS, NAS Operator
    a::ndas::NdasOperator ndasOp(appAuthCache);
    a::baas::BaasOperator baasOp(clientAccessTokenCache, userAccessTokenCache, userIdTokenCache, userInfoHolder, profile, ndasOp, storage);
    a::nas::NasOperator nasOp(nasUserResourceCache, baasOp, ndasOp, storage);

    // グローバルなリソース
    nn::os::Event terminator(nn::os::EventClearMode_ManualClear);

    // 実行時のリソース
    t::Thread thread;
    a::Executor executor;
    t::Buffer largeBuffer(512 * 1024);
    t::Buffer workBuffer(nn::account::RequiredBufferSize);
    a::LargeBufferAllocator largeBufferAllocator(largeBuffer.GetAddress(), largeBuffer.GetSize());
    a::ExecutionResource resource = {
        curl_easy_init(),
        {workBuffer.GetAddress(), workBuffer.GetSize()},
        &largeBufferAllocator,
    };

    thread.Initialize(
        [&]{ ThreadFunction(terminator, executor, resource); },
        g_Stack, sizeof(g_Stack), nn::os::DefaultThreadPriority + 1, "AccountNasOperator:Executor");
    thread.Start();

    const auto LoginInfo = t::LoadNaList()[0];
    NN_UNUSED(LoginInfo);

    struct
    {
        const char* name;
        std::function<void()> testBody;
    } Tests[] = {
#if defined(NNT_ACCOUNT_ENABLE_NAS_OPERATOR_LINKAGE)
        // Linkage テスト
        {
            "Test linkage with Nintendo Account",
            [&]() { RunLinkageAndRefreshTest(LoginInfo, executor, user, nasOp, baasOp, profile); },
        },
#endif

#if defined(NNT_ACCOUNT_ENABLE_NAS_OPERATOR_IMPLICIT_RECOVERY)
        // Implicit recovery テスト
        {
            "Test Nintendo Account user state implicit recovery",
            [&]() { RunImplicitRecoveryTest(LoginInfo, executor, user, nasOp, baasOp); },
        },
#endif

#if defined(NNT_ACCOUNT_ENABLE_NAS_OPERATOR_APPLICATION)
        // NX App 認可
        {
            "Test acquire application authorization",
            [&]() { RunApplicationAuthorizationTest(LoginInfo, executor, user, nasOp, baasOp); },
        },
#endif

#if defined(NNT_ACCOUNT_ENABLE_NAS_OPERATOR_GUEST)
        // ゲストログイン
        {
            "Test guest login",
            [&]() {
                t::InFile image("rom:/DevUser_01.jpg");
                a::profile::ProfileBase base;
                base.author = user;
                std::strncpy(base.nickname, "foo", sizeof(base.nickname));

                a::NetworkServiceAccountId nsaId;
                a::NintendoAccountId naId;
                ASSERT_TRUE(PrepareNintendoAccount(&nsaId, &naId, LoginInfo, image, base, executor, user, nasOp, profile, ndasOp, baasOp));

                RunGuestLoginTest(LoginInfo, nsaId, naId, image, base, executor, nasOp, ndasOp, baasOp);
            },
        },
        // インポート
        {
            "Test floating registration",
            [&]() {
                t::InFile image("rom:/DevUser_02.jpg");
                a::profile::ProfileBase base;
                base.author = user;
                std::strncpy(base.nickname, "bar", sizeof(base.nickname));

                a::NetworkServiceAccountId nsaId;
                a::NintendoAccountId naId;
                ASSERT_TRUE(PrepareNintendoAccount(&nsaId, &naId, LoginInfo, image, base, executor, user, nasOp, profile, ndasOp, baasOp));

                auto user2 = a::detail::ConvertToUid(storage.GenerateUuidWithContext());
                RunFloatingRegistration(LoginInfo, nsaId, naId, image, base, executor, user2, nasOp, profile, ndasOp, baasOp);
            },
        },
#endif

#if defined(NNT_ACCOUNT_ENABLE_NAS_OPERATOR_OP2)
        // OP2
        {
            "Test acquire OP2 membership (active)",
            [&]() {
                const t::NasLoginInfo LoginInfoForOp2 = {"moriyop2_1", "N1ntend0"};
                a::NetworkServiceLicense license;
                RunOp2MembershipTest(&license, LoginInfoForOp2, executor, user, nasOp, baasOp);
                ASSERT_EQ(a::NetworkServiceLicense_None, license);
            },
        },
        {
            "Test acquire OP2 membership (inactive)",
            [&]() {
                const t::NasLoginInfo LoginInfoForOp2 = {"moriyop2_2", "N1ntend0"};
                a::NetworkServiceLicense license;
                RunOp2MembershipTest(&license, LoginInfoForOp2, executor, user, nasOp, baasOp);
                ASSERT_EQ(a::NetworkServiceLicense_Common, license);
            },
        },
#endif
    };

    for (auto& t : Tests)
    {
        // プロフィール作成
        profile.Add(user);
        NN_UTIL_SCOPE_EXIT
        {
            profile.Delete(user);
        };

        // DA 作成
        a::baas::BaasRegistrationTask regTask(user, baasOp);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ExecuteTask(regTask, executor));

        NN_UTIL_SCOPE_EXIT
        {
            a::baas::BaasErasureTask unregTask(user, baasOp);
            auto r = ExecuteTask(unregTask, executor);
            EXPECT_TRUE(r.IsSuccess() || a::ResultNetworkServiceAccountUnavailable::Includes(r));
            executor.Unregister(&unregTask);
        };

        NN_LOG("------------------------------------------------------------------\n");
        NN_LOG(">> Test start: %s\n", t.name);
        t.testBody();
        NN_LOG("<< Test end\n");
    }

    // スレッド終了
    terminator.Signal();

    thread.Finalize();
    curl_easy_cleanup(resource.curlHandle);
} // NOLINT(readability/fn_size)

#endif // NNT_ACCOUNT_ENABLE_NAS_OPERATOR
