﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/baas/account_BaasLoginDriver.h>
#include <nn/account/baas/account_BaasRegistrationDriver.h>
#include <nn/account/baas/account_BaasUserDriver.h>

#include "detail/account_UuidUtil.h"
#include "detail/account_CacheUtil.h"
#include "profile/account_ProfileAdaptor.h"

#include "testAccount_File.h"
#include "testAccount_Mounter.h"
#include "testAccount_Util.h"

#include <algorithm>
#include <curl/curl.h>

#include <nnt/nntest.h>

#include <nn/nn_Log.h>
#include <nn/account/account_RuntimeResource.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_ScopeExit.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_BAAS_LOGIN_DRIVER
#define NNT_ACCOUNT_ENABLE_BAAS_USER_DRIVER

#if defined(NNT_ACCOUNT_ENABLE_BAAS_LOGIN_DRIVER)

namespace {

void BaasDriverBasic(
    CURL* curlHandle,
    a::baas::ClientAccessTokenCache& clientAccessTokenCache,
    a::baas::UserAccessTokenCache& userAccessTokenCache,
    a::baas::UserIdTokenCache& userIdTokenCache,
    a::ndas::NdasOperator& ndasOperator,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    // BaaS Login Driver, Registration Driver
    nn::account::baas::BaasLoginDriver baasLoginDriver(clientAccessTokenCache, userAccessTokenCache, userIdTokenCache, ndasOperator, storage);
    nn::account::baas::BaasRegistrationDriver baasRegDriver(userAccessTokenCache, userIdTokenCache, clientAccessTokenCache);

    // アプリケーション
    const auto appInfo = a::detail::ApplicationInfo::Get(0x010000000000B121ull, 0, a::detail::ApplicationMediaType::System);

    // ----------------------------------------

    NN_SDK_ASSERT(curlHandle != nullptr);
    t::Buffer buffer(std::max(
        a::ndas::NdasDriver::RequiredBufferSize,
        a::baas::BaasLoginDriver::RequiredBufferSize));

    // ----------------------------------------
    a::ExecutionResource resource = {curlHandle, {buffer.GetAddress(), buffer.GetSize()}};

    // デバイス&アプリケーション認証トークン
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasOperator.AuthenticateServiceForApplications(resource, nullptr));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasOperator.AuthenticateApplication(appInfo, resource, nullptr));
    ASSERT_TRUE(ndasOperator.IsApplicationAuthenticated(appInfo));

    // ----------------------------------------

    // public_client 権限のアクセストークン
    NN_LOG("[nnt::account] Login to BaaS as PublicClient\n");
    auto begin = nn::os::GetSystemTick();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.EnsureClientAccessTokenCache(
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
    EXPECT_TRUE(baasLoginDriver.IsClientAccessTokenCacheAvailable());

    // DA の作成
    NN_LOG("[nnt::account] Register DeviceAccount\n");
    begin = nn::os::GetSystemTick();
    a::NetworkServiceAccountId id;
    a::baas::BaasCredential credential;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasRegDriver.RegisterDeviceAccount(
        &id, &credential, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());

    // public_user 権限のアクセストークン
    NN_LOG("[nnt::account] Login to BaaS as PublicUser\n");
    begin = nn::os::GetSystemTick();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.EnsureUserAccessTokenCache(
        id, credential, nullptr, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
    bool cached;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.IsUserAccessTokenCacheAvailable(&cached, id));
    ASSERT_TRUE(cached);
    NN_UTIL_SCOPE_EXIT
    {
        // DA 削除
        NN_LOG("[nnt::account] Unregister DeviceAccount\n");
        begin = nn::os::GetSystemTick();
        auto r = baasRegDriver.UnregisterDeviceAccount(id, credential, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr);
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
        NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
    };

    // public_user 権限のアクセストークン for アプリケーション
    NN_LOG("[nnt::account] Login to BaaS as PublicUser with Application calim\n");
    begin = nn::os::GetSystemTick();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.EnsureUserIdTokenCache(
        id, credential, nullptr, appInfo, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.IsUserIdTokenCacheAvailable(&cached, id, appInfo));
    ASSERT_TRUE(cached);
}
} // ~namespace <anonymous>

TEST(AccountDriver, BaasDriverBasic)
{
    // ストレージ
    t::DefaultTestStorage storage;
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Mount());
    storage.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Setup());

    // NDAS トークンキャッシュ
    nn::account::ndas::ApplicationAuthenticationCache appAuthCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(appAuthCache.Initialize(storage));

    // BaaS トークンキャッシュ
    a::baas::ClientAccessTokenCache clientAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(clientAccessTokenCache.Initialize(storage));
    a::baas::UserAccessTokenCache userAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userAccessTokenCache.Initialize(storage));
    a::baas::UserIdTokenCache userIdTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userIdTokenCache.Initialize(storage));

    // NDAS Operator
    nn::account::ndas::NdasOperator ndasOperator(appAuthCache);

    auto handle = curl_easy_init();
    ASSERT_TRUE(handle != nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(handle);
    };

    BaasDriverBasic(handle, clientAccessTokenCache, userAccessTokenCache, userIdTokenCache, ndasOperator, storage);
}

#endif // NNT_ACCOUNT_ENABLE_BAAS_LOGIN_DRIVER

#if defined(NNT_ACCOUNT_ENABLE_BAAS_USER_DRIVER)

namespace {
void TestImageApi(
    const a::NetworkServiceAccountId& id,
    const char* path, size_t expectedSize,
    CURL* curlHandle, t::Buffer& buffer,
    const a::baas::BaasUserDriver& baasUserDriver,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_UNUSED(expectedSize);

    char url[256];

    // 画像データの取得
    t::InFile inFile(path);
    // ASSERT_EQ(128 * 1024, inFile.GetSize());

    // 画像データの Base64 エンコード
    t::Buffer base64(inFile.GetSize() * 2);
    auto stat = nn::util::Base64::ToBase64String(
        base64.Get<char>(), base64.GetSize(), inFile.GetAddress(), inFile.GetSize(),
        nn::util::Base64::Mode_NormalNoLinefeed);
    ASSERT_EQ(nn::util::Base64::Status_Success, stat);
    auto length = static_cast<size_t>(strnlen(base64.Get<char>(), base64.GetSize()));

    a::detail::Uuid cacheId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId, base64.GetAddress(), length, storage));

    NN_LOG("[nnt::account] Image encoded: from=%zu bytes, to=%zu bytes\n", inFile.GetSize(), length);

    t::Buffer largeBuffer(512 * 1024);

    // 画像データのアップロード
    NN_LOG("[nnt::account] Uploading image\n");
    auto begin = nn::os::GetSystemTick();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.UploadImage(
        url, sizeof(url), id, cacheId,
        curlHandle, buffer.GetAddress(), buffer.GetSize(),
        largeBuffer.GetAddress(), largeBuffer.GetSize(),
        nullptr));
    NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
    NN_LOG("  <- %s\n", url);

    // サーバー側の処理待ちのためのウェイト (短いかも？)
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // 画像データのダウンロード
    NN_LOG("[nnt::account] Downloading image\n");
    begin = nn::os::GetSystemTick();
    size_t size;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.DownloadImage(
        &size, largeBuffer.GetAddress(), largeBuffer.GetSize(),
        url, sizeof(url), curlHandle, nullptr));
    NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
    NN_LOG("  <- %zu bytes\n", size);

    // 比較
    EXPECT_EQ(inFile.GetSize(), size);
    EXPECT_EQ(0, std::memcmp(inFile.GetAddress(), largeBuffer.GetAddress(), inFile.GetSize()));
}

void TestUserResourceApi(
    const a::NetworkServiceAccountId& id,
    CURL* curlHandle, t::Buffer& buffer,
    const a::baas::BaasUserDriver& baasUserDriver,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    // 適当なプロフィールを捏造
    static const a::baas::UserProfile Profiles[] = {
        {
            {
                "",
                ""
            },
            ""
        },
        {
            {
                "EMc9EHWDSrePxGK",
                "CPhbCtXsR6LBSdipjAYWEEngGFAs9HSL7j3jBTuBgWWfF88aN3waW9hCz7-KVjQHb_PZphFRyX2MN7KUeGhpdcALSAkG786ENNC8rLhT6pEKCDXjnCDWAWpYeyPdGQGgGEP8"
            },
            "http://6uR7n8nbFeB_-eeG2NGjMMtF6FNwsSc2-PagiyLHzkXxFi2SCQ-WSCspMZiS9CrzYSXRMRtRQ5AQt7NR9DmF3NpT52h6j85AkBCAHPFR7nhSnsVXeYmwab4a"
        },
        {
            {
                "EMc9EHWDSrePxGKeiCgLRuuPD4icym8y",
                "CPhbCtXsR6LBSdipjAYWEEngGFAs9HSL7j3jBTuBgWWfF88aN3waW9hCz7-KVjQHb_PZphFRyX2MN7KUeGhpdcALSAkG786ENNC8rLhT6pEKCDXjnCDWAWpYeyPdGQGgGEP8mKkjLZ2b7pdPbJ-9FNMQ_J3QWuEhz99nLrDBe7aDzgZ4UyaLpJy67bn4TA3wkgNETpeKuwkwJS9Huh_emXU82tcCBnBFA7YdnXyEMc9EHWDSrePxGKeiCgLRuuPD"
            },
            "http://6uR7n8nbFeB_-eeG2NGjMMtF6FNwsSc2-PagiyLHzkXxFi2SCQ-WSCspMZiS9CrzYSXRMRtRQ5AQt7NR9DmF3NpT52h6j85AkBCAHPFR7nhSnsVXeYmwab4ah7FDsF9cie-Y_Xiw_yVerh2cRPw62PGrxaWsJwtcRzxbfLPWyVnUpkrNpP48bPYH4US4b_jKKH8yx42cfkLWycYkEYZJa68hfYUgr9SsuMrgsACmQYxF4Hf2tXfpCgJH"
        },
    };

    for (auto& p : Profiles)
    {
        // upload
        NN_LOG("[nnt::account] Uploading profile\n");
        auto begin = nn::os::GetSystemTick();
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.UploadProfile(id, p, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
        NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());

        // download + verify
        a::baas::UserProfile dl;
        NN_LOG("[nnt::account] Downloading profile\n");
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.DownloadProfile(&dl, id, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
        NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
        EXPECT_EQ(strnlen(p.base.nickname, sizeof(p.base.nickname)), strnlen(dl.base.nickname, sizeof(dl.base.nickname)));
        EXPECT_EQ(0, std::strncmp(p.base.nickname, dl.base.nickname, sizeof(p.base.nickname)));
        EXPECT_EQ(strnlen(p.base.extraData, sizeof(p.base.extraData)), strnlen(dl.base.extraData, sizeof(dl.base.extraData)));
        EXPECT_EQ(0, std::strncmp(p.base.extraData, dl.base.extraData, sizeof(p.base.extraData)));
        EXPECT_EQ(strnlen(p.imageUrl, sizeof(p.imageUrl)), strnlen(dl.imageUrl, sizeof(dl.imageUrl)));
        EXPECT_EQ(0, std::strncmp(p.imageUrl, dl.imageUrl, sizeof(p.imageUrl)));
    }
}

void BaasUserDriverBasic(
    CURL* curlHandle,
    a::baas::ClientAccessTokenCache& clientAccessTokenCache,
    a::baas::UserAccessTokenCache& userAccessTokenCache,
    a::baas::UserIdTokenCache& userIdTokenCache,
    a::ndas::NdasOperator& ndasOperator,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    // BaaS Login Driver, Registration Driver
    nn::account::baas::BaasLoginDriver baasLoginDriver(clientAccessTokenCache, userAccessTokenCache, userIdTokenCache, ndasOperator, storage);
    nn::account::baas::BaasRegistrationDriver baasRegDriver(userAccessTokenCache, userIdTokenCache, clientAccessTokenCache);
    nn::account::baas::BaasUserDriver baasUserDriver(userAccessTokenCache, storage);

    // ----------------------------------------

    NN_SDK_ASSERT(curlHandle != nullptr);
    t::Buffer buffer(std::max(
        a::ndas::NdasDriver::RequiredBufferSize,
        a::baas::BaasLoginDriver::RequiredBufferSize));

    // ----------------------------------------
    a::ExecutionResource resource = {curlHandle, {buffer.GetAddress(), buffer.GetSize()}};

    // デバイス&アプリケーション認証トークン
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasOperator.AuthenticateServiceForApplications(resource, nullptr));

    // public_client 権限のアクセストークン
    NN_LOG("[nnt::account] Login to BaaS as PublicClient\n");
    auto begin = nn::os::GetSystemTick();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.EnsureClientAccessTokenCache(
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
    EXPECT_TRUE(baasLoginDriver.IsClientAccessTokenCacheAvailable());

    // DA の作成
    NN_LOG("[nnt::account] Register DeviceAccount\n");
    begin = nn::os::GetSystemTick();
    a::NetworkServiceAccountId id;
    a::baas::BaasCredential credential;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasRegDriver.RegisterDeviceAccount(
        &id, &credential, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());

    // public_user 権限のアクセストークン
    NN_LOG("[nnt::account] Login to BaaS as PublicUser\n");
    begin = nn::os::GetSystemTick();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.EnsureUserAccessTokenCache(
        id, credential, nullptr, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
    bool cached;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.IsUserAccessTokenCacheAvailable(&cached, id));
    ASSERT_TRUE(cached);
    NN_UTIL_SCOPE_EXIT
    {
        // DA 削除
        NN_LOG("[nnt::account] Unregister DeviceAccount\n");
        begin = nn::os::GetSystemTick();
        auto r = baasRegDriver.UnregisterDeviceAccount(id, credential, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr);
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
        NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
    };

    // ----------------------------------------

    TestImageApi(id, "rom:/DevUser_03_128k.jpg", 128 * 1024, curlHandle, buffer, baasUserDriver, storage);
    TestImageApi(id, "rom:/DevUser_03.jpg", 32 * 1024, curlHandle, buffer, baasUserDriver, storage);
    TestUserResourceApi(id, curlHandle, buffer, baasUserDriver, storage);
}
} // ~namespace <anonymous>

TEST(AccountDriver, BaasUserDriverBasic)
{
    // ストレージ
    t::DefaultTestStorage storage;
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Mount());
    storage.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Setup());

    // NDAS トークンキャッシュ
    nn::account::ndas::ApplicationAuthenticationCache appAuthCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(appAuthCache.Initialize(storage));

    // BaaS トークンキャッシュ
    a::baas::ClientAccessTokenCache clientAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(clientAccessTokenCache.Initialize(storage));
    a::baas::UserAccessTokenCache userAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userAccessTokenCache.Initialize(storage));
    a::baas::UserIdTokenCache userIdTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userIdTokenCache.Initialize(storage));

    // NDAS Operator
    nn::account::ndas::NdasOperator ndasOperator(appAuthCache);

    auto handle = curl_easy_init();
    ASSERT_TRUE(handle != nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(handle);
    };

    BaasUserDriverBasic(handle, clientAccessTokenCache, userAccessTokenCache, userIdTokenCache, ndasOperator, storage);
}

#endif // NNT_ACCOUNT_ENABLE_BAAS_USER_DRIVER
