﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/user/account_UserStateManager.h>

#include <nnt/nntest.h>
#include "testAccount_Mounter.h"
#include "testAccount_Util.h"

#include <nn/account/user/account_UserRegistry.h>
#include "detail/account_UuidUtil.h"

namespace {
const nn::ApplicationId TestAppIds[] = {
    {0x01000000000000001},
    {0x01000000000000002},
    {0x01000000000000003},
};

bool CheckEventsSignaled(nn::os::SystemEventType* pE[], int count)
{
    bool result = true;
    for (auto i = 0; i < count; ++ i)
    {
        result = (result && nn::os::TryWaitSystemEvent(pE[i]));
        nn::os::ClearSystemEvent(pE[i]);
    }
    return result;
};
bool CheckEventsNotSignaled(nn::os::SystemEventType* pE[], int count)
{
    bool result = false;
    for (auto i = 0; i < count; ++ i)
    {
        result = (result || nn::os::TryWaitSystemEvent(pE[i]));
        nn::os::ClearSystemEvent(pE[i]);
    }
    return !result;
};

bool TestInitialState(nn::account::user::UserStateManager& manager)
{
    nn::os::SystemEventType* pE[nn::account::detail::UserStateManagerNotifierCountMax + 1];
    for (auto i = 0; i < nn::account::detail::UserStateManagerNotifierCountMax; ++i)
    {
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.GetUserStateChangeEvent(&pE[i]));
    }
    NN_UTIL_SCOPE_EXIT
    {
        for (auto i = 0; i < nn::account::detail::UserStateManagerNotifierCountMax; ++i)
        {
            manager.ReleaseSystemEvent(pE[i]);
        }
    };

    // InitialState

    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(
        nn::account::ResultOutOfNotifier,
        manager.GetUserStateChangeEvent(&pE[std::extent<decltype(pE)>::value - 1]));

    nn::ApplicationId apps[nn::account::detail::ApplicationCountMax];
    auto appCount = manager.ListApplications(apps, std::extent<decltype(apps)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(appCount, 0);

    nn::account::Uid opens[nn::account::UserCountMax];
    auto openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 0);

    nn::account::Uid lastOpened;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.TryGetLastOpenedUser(&lastOpened));

    return true;
} // NOLINT(readability/fn_size)

bool TestSingleUser(nn::account::user::UserStateManager& manager, const nn::account::user::UserRef& ref, nn::ApplicationId appId)
{
    auto uid = static_cast<nn::account::Uid>(ref);
    nn::os::SystemEventType* pE[nn::account::detail::UserStateManagerNotifierCountMax];
    for (auto i = 0; i < nn::account::detail::UserStateManagerNotifierCountMax; ++i)
    {
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.GetUserStateChangeEvent(&pE[i]));
    }
    NN_UTIL_SCOPE_EXIT
    {
        for (auto i = 0; i < nn::account::detail::UserStateManagerNotifierCountMax; ++i)
        {
            manager.ReleaseSystemEvent(pE[i]);
        }
    };

    // InitialState

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uid));

    // Enable

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.TryEnable(appId));
    nn::ApplicationId apps[nn::account::detail::ApplicationCountMax];
    auto appCount = manager.ListApplications(apps, std::extent<decltype(apps)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(appCount >= 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::find(apps, apps + appCount, appId) < apps + appCount);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.TryEnable(appId)); // 2 回目もアリ

    // Open - Close

    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(
        nn::account::ResultUserAlreadyClosed,
        manager.SetUserStateClosed(appId, uid));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsNotSignaled(pE, std::extent<decltype(pE)>::value));

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateOpened(appId, nn::account::user::UserRef(ref)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uid));
    nn::account::Uid opens[nn::account::UserCountMax];
    auto openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(opens[0], uid);
    nn::account::Uid lastOpened;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.TryGetLastOpenedUser(&lastOpened));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(lastOpened, uid);

    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(
        nn::account::ResultUserAlreadyOpened,
        manager.SetUserStateOpened(appId, nn::account::user::UserRef(ref)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsNotSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uid));
    openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(opens[0], uid);

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateClosed(appId, uid));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uid));
    openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 0);

    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(
        nn::account::ResultUserAlreadyClosed,
        manager.SetUserStateClosed(appId, uid));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsNotSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uid));
    openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 0);

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.TryGetLastOpenedUser(&lastOpened));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(lastOpened, uid);
    manager.InvalidateLastOpenedUser();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.TryGetLastOpenedUser(&lastOpened));

    // Disable

    manager.Disable(appId);
    manager.Disable(appId);

    return true;
} // NOLINT(readability/fn_size)

bool TestMultipleUser(nn::account::user::UserStateManager& manager, const nn::account::user::UserRef refs[], int count, nn::ApplicationId appId)
{
    // Enable

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.TryEnable(appId));
    nn::ApplicationId apps[nn::account::detail::ApplicationCountMax];
    auto appCount = manager.ListApplications(apps, std::extent<decltype(apps)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(appCount >= 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::find(apps, apps + appCount, appId) < apps + appCount);

    auto OpenAll = [&manager, refs, count, appId](auto GetIndex) -> bool {
        for (int i = 0; i < count; ++ i)
        {
            auto index = GetIndex(i);
            auto uid = static_cast<nn::account::Uid>(refs[index]);

            // InitialState

            NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uid));

            NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(
                nn::account::ResultUserAlreadyClosed,
                manager.SetUserStateClosed(appId, uid));

            // Open

            NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
                manager.SetUserStateOpened(appId, nn::account::user::UserRef(refs[index])));
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uid));
            nn::account::Uid opens[nn::account::UserCountMax];
            auto openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, i + 1);
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::find(opens, opens + openCount, uid) < opens + openCount);
            nn::account::Uid lastOpened;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.TryGetLastOpenedUser(&lastOpened));
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(lastOpened, uid);

            NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(
                nn::account::ResultUserAlreadyOpened,
                manager.SetUserStateOpened(appId, nn::account::user::UserRef(refs[index])));
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uid));
        }
        return true;
    };

    auto CloseAll = [&manager, refs, count, appId](auto GetIndex) -> bool {
        for (int i = 0; i < count; ++i)
        {
            auto index = GetIndex(i);
            auto uid = static_cast<nn::account::Uid>(refs[index]);

            NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
                manager.SetUserStateClosed(appId, uid));
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uid));
            nn::account::Uid opens[nn::account::UserCountMax];
            auto openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, count - i - 1);

            NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(
                nn::account::ResultUserAlreadyClosed,
                manager.SetUserStateClosed(appId, uid));
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uid));
        }
        return true;
    };


    NNT_ACCOUNT_RETURN_FALSE_UNLESS(OpenAll([](int i) -> int { return i; }));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CloseAll([](int i) -> int { return i; }));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(OpenAll([count](int i) -> int { return count - i - 1; }));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CloseAll([](int i) -> int { return i; }));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(OpenAll([](int i) -> int { return i; }));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CloseAll([count](int i) -> int { return count - i - 1; }));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(OpenAll([count](int i) -> int { return count - i - 1; }));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CloseAll([count](int i) -> int { return count - i - 1; }));

    // Disable

    manager.Disable(appId);

    return true;
} // NOLINT(readability/fn_size)

bool TestMultipleApplication(nn::account::user::UserStateManager& manager, const nn::account::user::UserRef refs[2], const nn::ApplicationId appIds[2])
{
    nn::os::SystemEventType* pE[nn::account::detail::UserStateManagerNotifierCountMax];
    for (auto i = 0; i < nn::account::detail::UserStateManagerNotifierCountMax; ++i)
    {
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.GetUserStateChangeEvent(&pE[i]));
    }
    NN_UTIL_SCOPE_EXIT
    {
        for (auto i = 0; i < nn::account::detail::UserStateManagerNotifierCountMax; ++i)
        {
            manager.ReleaseSystemEvent(pE[i]);
        }
    };

    nn::account::Uid uids[2];

    // InitialState
    for (auto i = 0; i < 2; ++ i)
    {
        uids[i] = refs[i];
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(refs[i]));
    }

    // Enable

    for (auto i = 0; i < 2; ++i)
    {
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.TryEnable(appIds[i]));
        nn::ApplicationId apps[nn::account::detail::ApplicationCountMax];
        auto appCount = manager.ListApplications(apps, std::extent<decltype(apps)>::value);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(appCount >= 1);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::find(apps, apps + appCount, appIds[i]) < apps + appCount);
    }

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsNotSignaled(pE, std::extent<decltype(pE)>::value));

    // Open User 0

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateOpened(appIds[0], nn::account::user::UserRef(refs[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uids[0]));
    nn::account::Uid opens[nn::account::UserCountMax];
    auto openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::find(opens, opens + openCount, uids[0]) < opens + openCount);

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateOpened(appIds[1], nn::account::user::UserRef(refs[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uids[0]));

    // Close User 0

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateClosed(appIds[0], nn::account::user::UserRef(refs[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uids[0]));
    openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::find(opens, opens + openCount, uids[0]) < opens + openCount);

    // Open User 1

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateOpened(appIds[1], nn::account::user::UserRef(refs[1])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uids[0]));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uids[1]));
    openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 2);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::find(opens, opens + openCount, uids[0]) < opens + openCount);

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateOpened(appIds[0], nn::account::user::UserRef(refs[1])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uids[0]));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uids[1]));

    // Close User 0

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateClosed(appIds[1], nn::account::user::UserRef(refs[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uids[0]));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uids[1]));
    openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::find(opens, opens + openCount, uids[1]) < opens + openCount);

    // Close User 1

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateClosed(appIds[1], nn::account::user::UserRef(refs[1])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uids[0]));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(manager.IsUserOpened(uids[1]));
    openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::find(opens, opens + openCount, uids[1]) < opens + openCount);

    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        manager.SetUserStateClosed(appIds[0], nn::account::user::UserRef(refs[1])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(CheckEventsSignaled(pE, std::extent<decltype(pE)>::value));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uids[0]));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(manager.IsUserOpened(uids[0]));
    openCount = manager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(openCount, 0);

    // Disable

    for (auto i = 0; i < std::extent<decltype(appIds)>::value; ++i)
    {
        manager.Disable(appIds[i]);
    }

    return true;
} // NOLINT(readability/fn_size)

} // ~namespace <anonymous>

TEST(AccountUserState, InitialState)
{
    nn::account::user::UserStateManager manager;
    ASSERT_TRUE(TestInitialState(manager));
}

TEST(AccountUserState, OpenClose)
{
    nnt::account::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    nn::account::user::UserRegistry reg;
    NN_ABORT_UNLESS_RESULT_SUCCESS(reg.Initialize(s));
    nn::account::Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(reg.BeginUserRegistration(&uid));
    NN_ABORT_UNLESS_RESULT_SUCCESS(reg.CompleteUserRegistration(uid));
    nn::account::user::UserRef ref;
    NN_ABORT_UNLESS_RESULT_SUCCESS(reg.GetUserRef(&ref, uid));

    nn::account::user::UserStateManager manager;

    // 1 つめのアプリとして
    ASSERT_TRUE(TestSingleUser(manager, ref, TestAppIds[0]));

    // 2 つめのアプリとして
    ASSERT_TRUE(manager.TryEnable(TestAppIds[1]));
    ASSERT_TRUE(TestSingleUser(manager, ref, TestAppIds[2]));
    manager.Disable(TestAppIds[1]);
}

TEST(AccountUserState, OpenCloseMultipleUser)
{
    nnt::account::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    nn::account::user::UserRegistry reg;
    NN_ABORT_UNLESS_RESULT_SUCCESS(reg.Initialize(s));
    nn::account::user::UserRef refs[nn::account::UserCountMax];
    for (auto i = 0; i < std::extent<decltype(refs)>::value; ++i)
    {
        nn::account::Uid uid;
        NN_ABORT_UNLESS_RESULT_SUCCESS(reg.BeginUserRegistration(&uid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(reg.CompleteUserRegistration(uid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(reg.GetUserRef(&refs[i], uid));
    }

    nn::account::user::UserStateManager manager;

    // 1 つめのアプリとして
    ASSERT_TRUE(TestMultipleUser(manager, refs, std::extent<decltype(refs)>::value, TestAppIds[0]));

    // 2 つめのアプリとして
    ASSERT_TRUE(manager.TryEnable(TestAppIds[1]));
    ASSERT_TRUE(TestMultipleUser(manager, refs, std::extent<decltype(refs)>::value, TestAppIds[2]));
    manager.Disable(TestAppIds[1]);
}

TEST(AccountUserState, OpenCloseMultipleApplication)
{
    nnt::account::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    nn::account::user::UserRegistry reg;
    NN_ABORT_UNLESS_RESULT_SUCCESS(reg.Initialize(s));
    nn::account::user::UserRef refs[2];
    for (auto i = 0; i < std::extent<decltype(refs)>::value; ++i)
    {
        nn::account::Uid uid;
        NN_ABORT_UNLESS_RESULT_SUCCESS(reg.BeginUserRegistration(&uid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(reg.CompleteUserRegistration(uid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(reg.GetUserRef(&refs[i], uid));
    }

    nn::account::user::UserStateManager manager;

    ASSERT_TRUE(TestMultipleApplication(manager, refs, TestAppIds));
}

namespace {
bool TestQualification(nn::account::user::UserStateManager& manager, const nn::account::user::UserRef refs[3],  nn::ApplicationId appId)
{
    const nn::account::Uid users[3] = {refs[0], refs[1], refs[2]};

    NN_ABORT_UNLESS(manager.TryEnable(appId));
    NN_UTIL_SCOPE_EXIT
    {
        manager.Disable(appId);
    };

    nn::account::Uid qualified[nn::account::UserCountMax];
    auto qualifiedCount = manager.SelectQualifiedUsers(qualified, std::extent<decltype(qualified)>::value, appId, users, std::extent<decltype(users)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::equal(
        users, users + std::extent<decltype(users)>::value,
        qualified, qualified + qualifiedCount));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateOpened(appId, nn::account::user::UserRef(refs[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateOpened(appId, nn::account::user::UserRef(refs[1])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateOpened(appId, nn::account::user::UserRef(refs[2])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateClosed(appId, nn::account::user::UserRef(refs[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateClosed(appId, nn::account::user::UserRef(refs[1])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateClosed(appId, nn::account::user::UserRef(refs[2])));

    // 先頭 2 つを qualified にする
    manager.EnableQualificationLimitation(appId, qualified, 2);

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateOpened(appId, nn::account::user::UserRef(refs[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateOpened(appId, nn::account::user::UserRef(refs[1])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultUserUnqualified,
        manager.SetUserStateOpened(appId, nn::account::user::UserRef(refs[2])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateClosed(appId, nn::account::user::UserRef(refs[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetUserStateClosed(appId, nn::account::user::UserRef(refs[1])));

    qualifiedCount = manager.SelectQualifiedUsers(qualified, std::extent<decltype(qualified)>::value, appId, users, std::extent<decltype(users)>::value);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::equal(
        users, users + 2,
        qualified, qualified + qualifiedCount));

    return true;
}
} // ~namespace <anonymous>

TEST(AccountUserState, QualificationLimitation)
{
    nnt::account::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    nn::account::user::UserRegistry reg;
    NN_ABORT_UNLESS_RESULT_SUCCESS(reg.Initialize(s));
    nn::account::user::UserRef refs[3];
    for (auto i = 0; i < std::extent<decltype(refs)>::value; ++i)
    {
        nn::account::Uid uid;
        NN_ABORT_UNLESS_RESULT_SUCCESS(reg.BeginUserRegistration(&uid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(reg.CompleteUserRegistration(uid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(reg.GetUserRef(&refs[i], uid));
    }

    nn::account::user::UserStateManager manager;

    ASSERT_TRUE(TestQualification(manager, refs, TestAppIds[0]));
}


