﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_FileSystem.h>

#include <nn/account/account_Types.h>

#include "testAccount_Mounter.h"
#include "testAccount_RamFs.h"
#include "testAccount_Util.h"

#include <algorithm>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nnt/nntest.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_FILESYSTEM_RAM
#define NNT_ACCOUNT_ENABLE_FILESYSTEM_HOST

namespace
{
bool TestPathNotFound(
    const a::detail::AbstractFileSystem& fs,
    const char* ToPath,
    const char* InPath) NN_NOEXCEPT
{
    // Read 系の NotFound 確認
    size_t size;
    char buf[128];
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::fs::ResultPathNotFound, fs.Copy(ToPath, InPath, buf, sizeof(buf)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::fs::ResultPathNotFound, fs.GetSize(&size, InPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::fs::ResultPathNotFound, fs.Move(ToPath, InPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::fs::ResultPathNotFound, fs.Read(&size, nullptr, 0u, InPath));

    // Write 系の NotFound 確認
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::fs::ResultPathNotFound, fs.Append(InPath, 0u, nullptr, 0u));

    // Write 系の NotFound にならないもの
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(InPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Write(InPath, &buf, sizeof(buf)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&size, InPath));
    EXPECT_EQ(sizeof(buf), size);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(InPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(InPath));

    return true;
}

bool TestWriteRead(
    const a::detail::AbstractFileSystem& fs,
    const char* InPath, const void* InData, size_t InSize) NN_NOEXCEPT
{
    size_t actualSize;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, InPath));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Write(InPath, InData, InSize));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&actualSize, InPath));
    EXPECT_EQ(InSize, actualSize);
    {
        t::Buffer buffer(actualSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, buffer.GetAddress(), buffer.GetSize(), InPath));
        EXPECT_EQ(InSize, actualSize);
        EXPECT_EQ(0, std::memcmp(InData, buffer.GetAddress(), InSize));
    }

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(InPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, InPath));

    return true;
}

bool TestSetSize(
    const a::detail::AbstractFileSystem& fs,
    const char* InPath, const void* InData, size_t InSize, size_t NewSize) NN_NOEXCEPT
{
    size_t actualSize;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, InPath));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Write(InPath, InData, InSize));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&actualSize, InPath));
    EXPECT_EQ(InSize, actualSize);
    {
        t::Buffer buffer(actualSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, buffer.GetAddress(), buffer.GetSize(), InPath));
        EXPECT_EQ(InSize, actualSize);
        EXPECT_EQ(0, std::memcmp(InData, buffer.GetAddress(), InSize));
    }

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.SetSize(InPath, NewSize));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&actualSize, InPath));
    EXPECT_EQ(NewSize, actualSize);
    {
        t::Buffer buffer(actualSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, buffer.GetAddress(), buffer.GetSize(), InPath));
        EXPECT_EQ(NewSize, actualSize);
        auto SizeToCompare = std::min(InSize, NewSize);
        EXPECT_EQ(0, std::memcmp(InData, buffer.GetAddress(), SizeToCompare));
    }

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(InPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, InPath));

    return true;
}

bool TestCreateWriteReadAppendRead(
    const a::detail::AbstractFileSystem& fs,
    const char* InPath, const void* InData, size_t InSize) NN_NOEXCEPT
{
    size_t actualSize;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, InPath));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Create(InPath, InSize));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&actualSize, InPath));
    EXPECT_EQ(InSize, actualSize);

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Append(InPath, 0, InData, InSize));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&actualSize, InPath));
    EXPECT_EQ(InSize, actualSize);
    {
        t::Buffer buffer(actualSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, buffer.GetAddress(), buffer.GetSize(), InPath));
        EXPECT_EQ(InSize, actualSize);
        EXPECT_EQ(0, std::memcmp(InData, buffer.GetAddress(), InSize));
    }

    t::Buffer appendix(InSize);
    std::memset(appendix.GetAddress(), reinterpret_cast<const char*>(InData)[0], InSize);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Append(InPath, InSize, appendix.GetAddress(), appendix.GetSize()));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&actualSize, InPath));
    EXPECT_EQ(InSize * 2, actualSize);
    {
        t::Buffer buffer(actualSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, buffer.GetAddress(), buffer.GetSize(), InPath));
        EXPECT_EQ(InSize * 2, actualSize);
        auto former = buffer.GetAddress();
        auto latter = reinterpret_cast<const char*>(buffer.GetAddress()) + InSize;
        EXPECT_EQ(0, std::memcmp(InData, former, InSize));
        EXPECT_EQ(0, std::memcmp(InData, latter, InSize));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, buffer.GetAddress(), buffer.GetSize(), InPath, InSize));
        EXPECT_EQ(InSize, actualSize);
        EXPECT_EQ(0, std::memcmp(appendix.GetAddress(), buffer.GetAddress(), InSize));
    }

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(InPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, InPath));

    return true;
}

bool TestCopy(
    const a::detail::AbstractFileSystem& fs,
    const char* ToPath,
    const char* InPath, const void* InData, size_t InSize) NN_NOEXCEPT
{
    size_t actualSize;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, ToPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, InPath));

    // 初期のファイル内容
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Write(InPath, InData, InSize));

    // コピーと内容の確認
    t::Buffer buffer(1023);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Copy(ToPath, InPath, buffer.GetAddress(), buffer.GetSize()));
    t::Buffer read(InSize);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, read.GetAddress(), read.GetSize(), ToPath));
    EXPECT_EQ(InSize, actualSize);
    EXPECT_EQ(0, std::memcmp(InData, read.GetAddress(), InSize));

    // 新しいファイル内容
    t::Buffer content(InSize * 2);
    std::memset(content.GetAddress(), 0x3A, content.GetSize());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Write(InPath, content.GetAddress(), content.GetSize()));

    // コピーと内容の確認
    buffer = t::Buffer(1024 * 1024 * 4);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Copy(ToPath, InPath, buffer.GetAddress(), buffer.GetSize()));
    read = t::Buffer(content.GetSize());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, read.GetAddress(), read.GetSize(), ToPath));
    EXPECT_EQ(content.GetSize(), actualSize);
    EXPECT_EQ(0, std::memcmp(content.GetAddress(), read.GetAddress(), content.GetSize()));

    // クリーンアップ
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(ToPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(InPath));

    return true;
}

bool TestMove(
    const a::detail::AbstractFileSystem& fs,
    const char* ToPath,
    const char* InPath, const void* InData, size_t InSize) NN_NOEXCEPT
{
    size_t actualSize;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, ToPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, InPath));

    // 初期のファイル内容
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Write(InPath, InData, InSize));

    // 移動と内容の確認
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Move(ToPath, InPath));
    t::Buffer read(InSize);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, read.GetAddress(), read.GetSize(), ToPath));
    EXPECT_EQ(InSize, actualSize);
    EXPECT_EQ(0, std::memcmp(InData, read.GetAddress(), InSize));

    // 元のファイルの削除を確認
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&actualSize, InPath));

    // 元のファイルを別の内容で再度作成
    t::Buffer content(InSize * 2);
    std::memset(content.GetAddress(), 0x3A, content.GetSize());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Write(InPath, content.GetAddress(), content.GetSize()));

    // 移動と内容の確認
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Move(ToPath, InPath));
    read = t::Buffer(content.GetSize());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Read(&actualSize, read.GetAddress(), read.GetSize(), ToPath));
    EXPECT_EQ(content.GetSize(), actualSize);
    EXPECT_EQ(0, std::memcmp(content.GetAddress(), read.GetAddress(), content.GetSize()));

    // クリーンアップ
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(ToPath));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(InPath));

    return true;
}


bool TestBasic(
    const a::detail::AbstractFileSystem& fs,
    const char* VolumeName)
{
    char paths[][128] =
    {
        "0",
        "path",
        "paaaaaaaaaaaaaaaaaaaaaaaaaaaath2",
        "012345678901234567890123456789"
        "012345678901234567890123456789"
        "012345678901234567890123456789"
        "012345678901234567890123456789"
        "0123456",
    };
    size_t sizes[] =
    {
        1u, 8u, 1024u, 4096u, 1024u * 1024u
    };

    for (auto i = 0u; i < sizeof(paths) / sizeof(paths[0]); ++ i)
    {
        char InPath[256];
        nn::util::SNPrintf(InPath, sizeof(InPath), "%s:/%s", VolumeName, paths[i]);
        char ToPath[256];
        nn::util::SNPrintf(ToPath, sizeof(ToPath), "%s:/____", VolumeName);

        EXPECT_TRUE(TestPathNotFound(fs, ToPath, InPath));

        for (auto j = 0u; j < sizeof(sizes) / sizeof(sizes[0]); ++ j)
        {
            const auto InSize = sizes[j];

            t::Buffer buffer(InSize);
            std::memset(buffer.GetAddress(), (i << 4) | j, buffer.GetSize());

            EXPECT_TRUE(TestWriteRead(fs, InPath, buffer.GetAddress(), buffer.GetSize()));
            EXPECT_TRUE(TestSetSize(fs, InPath, buffer.GetAddress(), buffer.GetSize(), buffer.GetSize() / 2));
            EXPECT_TRUE(TestSetSize(fs, InPath, buffer.GetAddress(), buffer.GetSize(), buffer.GetSize() * 2));
            EXPECT_TRUE(TestCreateWriteReadAppendRead(fs, InPath, buffer.GetAddress(), buffer.GetSize()));
            EXPECT_TRUE(TestCopy(fs, ToPath, InPath, buffer.GetAddress(), buffer.GetSize()));
            EXPECT_TRUE(TestMove(fs, ToPath, InPath, buffer.GetAddress(), buffer.GetSize()));
        }
    }
    return true;
}

}

#if defined(NNT_ACCOUNT_ENABLE_FILESYSTEM_RAM)

TEST(AccountUtil, FileSystem_RamFs)
{
    const int TestCount = 100;

    t::RamFs fs;
    t::HostSaveData<> mounter;

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(mounter.Mount<t::RamFs>());
    fs.Unmount(mounter.VolumeName);

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(mounter.Mount<t::RamFs>());
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount(mounter.VolumeName);
    };

    for (auto i = 0; i < TestCount; ++ i)
    {
        auto lock = t::RamFs::AcquireWriterLock();
        EXPECT_TRUE(TestBasic(fs, mounter.VolumeName));
    }
}

#endif

#if defined(NNT_ACCOUNT_ENABLE_FILESYSTEM_HOST)

TEST(AccountUtil, FileSystem_HostFs)
{
    a::detail::DefaultFileSystem fs;
    t::HostSaveData<> mounter;

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(mounter.Mount<a::detail::DefaultFileSystem>());
    fs.Unmount(mounter.VolumeName);

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(mounter.Mount<a::detail::DefaultFileSystem>());
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount(mounter.VolumeName);
    };

    auto lock = a::detail::DefaultFileSystem::AcquireWriterLock();
    EXPECT_TRUE(TestBasic(fs, mounter.VolumeName));
}

#endif
