﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nas/account_NasLoginAdaptor.h"

#include "detail/account_CacheUtil.h"
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>

#include "testAccount_RapidJsonInputStream.h"
#include "testAccount_Mounter.h"
#include "testAccount_Util.h"

namespace a = nn::account;
namespace t = nnt::account;

/* for NAS Proxy tests

Request URI:
"https://e03a97819c9711e59510d820a52f298a-sb.accounts.nintendo.com/connect/1.0.0/authorize?client_id=ca989efaca0d0c6a&redirect_uri=&response_type=code&scope=openid+offline+napps+urn:%3Aoauth%3Ainit-sso&state=IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc&code_challenge=tqw8wQOGMxx2XwTwQcFH0PJ48q7Y6qAh4tAFf8b2_54&code_challenge_method=s256"

Authorization Code By NAS Proxy:
"{\"code\":\"eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE\",\"state\":\"IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc\",\"session_state\":\"d812fcca44ec7f428e9672c8b96a1a823a422527483e13792286cd61e55119fe\"}"

*/

namespace primaryToken {
const char* Inputs[] = {
    "{\"id_token\":\"eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiIsImtpZCI6ImI1ZWQ4ZTRjLTNkNWQtNDI3Ny1hOGRkLWQ2MmMyOWM0ZGJhZiJ9.eyJqdGkiOiJjYmQxNDJhOC1jOTJjLTQ0MTUtOWE3MS1iZWZmYmE0YzA5ZjciLCJjb3VudHJ5IjoiVVMiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsImNfaGFzaCI6IjI5aVIwUG5fLUwwRVZLbVFtSDlsNHciLCJleHAiOjE0NjUwMTYxMTUsImlhdCI6MTQ2NTAxNTIxNSwiYXRfaGFzaCI6ImduREdIdjNWWXRRM3N1Snp1VmxYNkEiLCJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwidHlwIjoiaWRfdG9rZW4iLCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.eYqFpyJRgW9FiPVYm42p12WnFDxA-ADmPfX-to-3_Oo8X_go_06zaH2Sue6wF_6acCIuXYsM_QzZpAmrv-9_iPefKv0nvlHyOwJ7xxMHgs7FbSl-6CTcOXWnV2YFstgbv6i0ORH61uGbNCCQo9rE91cL8_jqsY_pe1qAZx4WHHsA7H6TA8moiHCPRIpu5Xx_iW8gU3mA5g2gYKjtYe9KDFS5-jCcDg7PzY1BVO3yVu7nbLClRhNkflBwNCJRaqpliOnP8LB3SflS3v7ejXvcovSbfiybqgO_7IwTsnyUWaZXgvzym8JlWVD9zjdQ_xBNHtTlfAZY7GAT3XpdXSz8rA\",\"scope\":[\"openid\",\"offline\",\"napps\",\"urn:oauth:init-sso\",\"user\",\"user.birthday\",\"user.email\",\"user.mii\"],\"refresh_token\":\"eyJhbGciOiJIUzI1NiJ9.eyJqdGkiOiIzMDciLCJhYzpjYyI6InRxdzh3UU9HTXh4Mlh3VHdRY0ZIMFBKNDhxN1k2cUFoNHRBRmY4YjJfNTQiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsInR5cCI6InJlZnJlc2hfdG9rZW4iLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImF1ZCI6ImNhOTg5ZWZhY2EwZDBjNmEiLCJhYzpjY20iOiJTMjU2IiwiaWF0IjoxNDY1MDE1MjE1LCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.N-pAlNSkddZcqkMag8fC0r_XR7v11V3VhTiR4_LXDII\",\"expires_in\":900,\"access_token\":\"eyJraWQiOiI1ZDI5ZTA3OC03YTFhLTRiZDUtOTNhNi04NTFkZWZhMzg4MDIiLCJhbGciOiJSUzI1NiIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tLzEuMC4wL2NlcnRpZmljYXRlcyJ9.eyJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwiaWF0IjoxNDY1MDE1MjE1LCJ0eXAiOiJ0b2tlbiIsImFjOnNjcCI6WzAsMSw0LDUsOCw5LDEwLDE3XSwic3ViIjoiYTkwOTQ3NTU4ZTVlYzliOSIsImFjOmdydCI6MSwianRpIjoiYjJkZTIyMzEtNGJlMS00ZTg1LThhYjItY2Q5NGQ4OTZhZDQ3IiwiaXNzIjoiaHR0cHM6Ly9lMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5hY2NvdW50cy5uaW50ZW5kby5jb20iLCJleHAiOjE0NjUwMTYxMTV9.nYthkj-5YmlLxfKkdMjGDHabhnSUQOxe5KSG13cw0zAKBePL5IXdkLaRBj8bMH27C5kXhLgZ7mS15OBS9eQw-1Y7Vg5fMzBgl30RddF7tKAZBhAtBM0p34ui_o2mKF9m212fWvvhdbf7i3miFznI0iN28p72IZAGkM0MmRYwQEbPvv5yulppXhbjXWhl46MzmNwDZMrQUnER1UUyeoF6PUAgwZwl75jWECxLBSKAEbzYW6OU5BUjM5i1PQCBiKlWHpaAPm-hujLxvMTFR5LfOsFYs2R4bD2vFl-dzzki8jWqdRT-oOCsblNiqZLjLXaOyH4fBUyc8fDQyMVC5yd_Jg\",\"token_type\":\"Bearer\"}",
};

struct Expect
{
    const char* refreshToken;
    const char* idToken;
    const char* accessToken;
    int64_t expiration;
} Expects[] = {
    {
        "eyJhbGciOiJIUzI1NiJ9.eyJqdGkiOiIzMDciLCJhYzpjYyI6InRxdzh3UU9HTXh4Mlh3VHdRY0ZIMFBKNDhxN1k2cUFoNHRBRmY4YjJfNTQiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsInR5cCI6InJlZnJlc2hfdG9rZW4iLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImF1ZCI6ImNhOTg5ZWZhY2EwZDBjNmEiLCJhYzpjY20iOiJTMjU2IiwiaWF0IjoxNDY1MDE1MjE1LCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.N-pAlNSkddZcqkMag8fC0r_XR7v11V3VhTiR4_LXDII",
        "eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiIsImtpZCI6ImI1ZWQ4ZTRjLTNkNWQtNDI3Ny1hOGRkLWQ2MmMyOWM0ZGJhZiJ9.eyJqdGkiOiJjYmQxNDJhOC1jOTJjLTQ0MTUtOWE3MS1iZWZmYmE0YzA5ZjciLCJjb3VudHJ5IjoiVVMiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsImNfaGFzaCI6IjI5aVIwUG5fLUwwRVZLbVFtSDlsNHciLCJleHAiOjE0NjUwMTYxMTUsImlhdCI6MTQ2NTAxNTIxNSwiYXRfaGFzaCI6ImduREdIdjNWWXRRM3N1Snp1VmxYNkEiLCJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwidHlwIjoiaWRfdG9rZW4iLCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.eYqFpyJRgW9FiPVYm42p12WnFDxA-ADmPfX-to-3_Oo8X_go_06zaH2Sue6wF_6acCIuXYsM_QzZpAmrv-9_iPefKv0nvlHyOwJ7xxMHgs7FbSl-6CTcOXWnV2YFstgbv6i0ORH61uGbNCCQo9rE91cL8_jqsY_pe1qAZx4WHHsA7H6TA8moiHCPRIpu5Xx_iW8gU3mA5g2gYKjtYe9KDFS5-jCcDg7PzY1BVO3yVu7nbLClRhNkflBwNCJRaqpliOnP8LB3SflS3v7ejXvcovSbfiybqgO_7IwTsnyUWaZXgvzym8JlWVD9zjdQ_xBNHtTlfAZY7GAT3XpdXSz8rA",
        "eyJraWQiOiI1ZDI5ZTA3OC03YTFhLTRiZDUtOTNhNi04NTFkZWZhMzg4MDIiLCJhbGciOiJSUzI1NiIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tLzEuMC4wL2NlcnRpZmljYXRlcyJ9.eyJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwiaWF0IjoxNDY1MDE1MjE1LCJ0eXAiOiJ0b2tlbiIsImFjOnNjcCI6WzAsMSw0LDUsOCw5LDEwLDE3XSwic3ViIjoiYTkwOTQ3NTU4ZTVlYzliOSIsImFjOmdydCI6MSwianRpIjoiYjJkZTIyMzEtNGJlMS00ZTg1LThhYjItY2Q5NGQ4OTZhZDQ3IiwiaXNzIjoiaHR0cHM6Ly9lMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5hY2NvdW50cy5uaW50ZW5kby5jb20iLCJleHAiOjE0NjUwMTYxMTV9.nYthkj-5YmlLxfKkdMjGDHabhnSUQOxe5KSG13cw0zAKBePL5IXdkLaRBj8bMH27C5kXhLgZ7mS15OBS9eQw-1Y7Vg5fMzBgl30RddF7tKAZBhAtBM0p34ui_o2mKF9m212fWvvhdbf7i3miFznI0iN28p72IZAGkM0MmRYwQEbPvv5yulppXhbjXWhl46MzmNwDZMrQUnER1UUyeoF6PUAgwZwl75jWECxLBSKAEbzYW6OU5BUjM5i1PQCBiKlWHpaAPm-hujLxvMTFR5LfOsFYs2R4bD2vFl-dzzki8jWqdRT-oOCsblNiqZLjLXaOyH4fBUyc8fDQyMVC5yd_Jg",
        900,
    }
};

void RunTest(const Expect& expect, const char* json) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::nas::NasPrimaryTokenAdaptor adaptor(s);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));

    a::nas::NasCredentialCache credentialCache;
    a::nas::NasAccessTokenCache accessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.PullPrimaryTokens(&credentialCache, &accessTokenCache));

    size_t size;
    t::Buffer buffer(1024 * 1024);
    // Refresh token
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS((a::detail::CacheUtil::LoadCacheFile<a::detail::NasRefreshTokenSizeMax, a::ResultNasTokenLengthUnacceptable>(
        &size, buffer.GetAddress(), buffer.GetSize(), credentialCache.refreshTokenCacheId, s)));
    ASSERT_TRUE(size < buffer.GetSize());
    buffer.Get<char>()[size] = '\0';
    ASSERT_EQ(0, std::strncmp(expect.refreshToken, buffer.Get<char>(), a::detail::NasRefreshTokenSizeMax));

    // Id token
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS((a::detail::CacheUtil::LoadCacheFile<a::detail::NasIdTokenSizeMax, a::ResultNasTokenLengthUnacceptable>(
        &size, buffer.GetAddress(), buffer.GetSize(), credentialCache.idTokenCacheId, s)));
    ASSERT_TRUE(size < buffer.GetSize());
    buffer.Get<char>()[size] = '\0';
    ASSERT_EQ(0, std::strncmp(expect.idToken, buffer.Get<char>(), a::detail::NasIdTokenSizeMax));

    // Access token
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS((a::detail::CacheUtil::LoadCacheFile<a::detail::NasAccessTokenSizeMax, a::ResultNasTokenLengthUnacceptable>(
        &size, buffer.GetAddress(), buffer.GetSize(), accessTokenCache.cacheId, s)));
    ASSERT_TRUE(size < buffer.GetSize());
    buffer.Get<char>()[size] = '\0';
    ASSERT_EQ(0, std::strncmp(expect.accessToken, buffer.Get<char>(), a::detail::NasAccessTokenSizeMax));
}
} // ~namespace primaryToken

TEST(AccountAdaptor, NasPrimaryToken)
{
    for (int i = 0; i < sizeof(primaryToken::Expects) / sizeof(primaryToken::Expects[0]); ++ i)
    {
        primaryToken::RunTest(
            primaryToken::Expects[i],
            primaryToken::Inputs[i]);
    }
}

namespace accessToken {
const char* Inputs[] = {
    "{\"scope\":[\"openid\",\"offline\",\"napps\",\"urn:oauth:init-sso\",\"user\",\"user.birthday\",\"user.email\",\"user.mii\"],\"token_type\":\"Bearer\",\"expires_in\":900,\"access_token\":\"eyJraWQiOiI1ZDI5ZTA3OC03YTFhLTRiZDUtOTNhNi04NTFkZWZhMzg4MDIiLCJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiJ9.eyJleHAiOjE0NjUwMTYxMTcsImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwianRpIjoiZDU1OGI5MWUtMjgyNi00NGQ3LWEyYzMtNTAxMWM5Nzk1YzUzIiwiYWM6Z3J0Ijo0LCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwidHlwIjoidG9rZW4iLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImF1ZCI6ImNhOTg5ZWZhY2EwZDBjNmEiLCJpYXQiOjE0NjUwMTUyMTd9.LvWJbUTezpGDnhzke2tYXLRiz9pMV8fVdrPSNdquPlirCRTNsSFI1Ftk-ZMwrPAbDLR4o6GBpjQBU-dvSmgeTpCvOsOPWRRXvTb3vXTv95qErqWtL-m1HCIAO_YDxeergsvmY5sqN28CVD4aNjkuFQxwRM3qt8Tx1CmnHgVcZ1ozTYnVIGM2AbW2ejt8JSN4oiC4uyc7juad314WmH_aSr4-EHOWeA6_jiRbZWLiBw3OIrdNsc_n2jT1u1CH2UnpDXh-VImfWY-8vLznXWROUD62bIO3pXbfH5-DyE-f3fhZ6937Mnsqc9Ohb0IkgTQlStNsJm2-5O1K_lX14ufJBA\"}",
};

struct Expect
{
    const char* accessToken;
    int64_t expiration;
} Expects[] = {
    {
        "eyJraWQiOiI1ZDI5ZTA3OC03YTFhLTRiZDUtOTNhNi04NTFkZWZhMzg4MDIiLCJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiJ9.eyJleHAiOjE0NjUwMTYxMTcsImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwianRpIjoiZDU1OGI5MWUtMjgyNi00NGQ3LWEyYzMtNTAxMWM5Nzk1YzUzIiwiYWM6Z3J0Ijo0LCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwidHlwIjoidG9rZW4iLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImF1ZCI6ImNhOTg5ZWZhY2EwZDBjNmEiLCJpYXQiOjE0NjUwMTUyMTd9.LvWJbUTezpGDnhzke2tYXLRiz9pMV8fVdrPSNdquPlirCRTNsSFI1Ftk-ZMwrPAbDLR4o6GBpjQBU-dvSmgeTpCvOsOPWRRXvTb3vXTv95qErqWtL-m1HCIAO_YDxeergsvmY5sqN28CVD4aNjkuFQxwRM3qt8Tx1CmnHgVcZ1ozTYnVIGM2AbW2ejt8JSN4oiC4uyc7juad314WmH_aSr4-EHOWeA6_jiRbZWLiBw3OIrdNsc_n2jT1u1CH2UnpDXh-VImfWY-8vLznXWROUD62bIO3pXbfH5-DyE-f3fhZ6937Mnsqc9Ohb0IkgTQlStNsJm2-5O1K_lX14ufJBA",
        900,
    }
};

void RunTest(const Expect& expect, const char* json) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::nas::NasAccessTokenAdaptor adaptor(s);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));

    a::nas::NasAccessTokenCache accessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.PullAccessToken(&accessTokenCache));

    size_t size;
    t::Buffer buffer(1024 * 1024);

    // Access token
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS((a::detail::CacheUtil::LoadCacheFile<a::detail::NasAccessTokenSizeMax, a::ResultNasTokenLengthUnacceptable>(
        &size, buffer.GetAddress(), buffer.GetSize(), accessTokenCache.cacheId, s)));
    ASSERT_TRUE(size < buffer.GetSize());
    buffer.Get<char>()[size] = '\0';
    ASSERT_EQ(0, std::strncmp(expect.accessToken, buffer.Get<char>(), a::detail::NasAccessTokenSizeMax));
}
} // ~namespace accessToken

TEST(AccountAdaptor, NasAccessToken)
{
    for (int i = 0; i < sizeof(accessToken::Expects) / sizeof(accessToken::Expects[0]); ++ i)
    {
        accessToken::RunTest(
            accessToken::Expects[i],
            accessToken::Inputs[i]);
    }
}

namespace authroizationCode {
const char* Inputs[] = {
    "http://localhost:8080?session_state=d812fcca44ec7f428e9672c8b96a1a823a422527483e13792286cd61e55119fe&code=eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE&state=IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc",
    "nintendo://account.nx.sys?session_state=d812fcca44ec7f428e9672c8b96a1a823a422527483e13792286cd61e55119fe&code=eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE&state=IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc&id_token=eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiIsImtpZCI6ImI1ZWQ4ZTRjLTNkNWQtNDI3Ny1hOGRkLWQ2MmMyOWM0ZGJhZiJ9.eyJqdGkiOiJjYmQxNDJhOC1jOTJjLTQ0MTUtOWE3MS1iZWZmYmE0YzA5ZjciLCJjb3VudHJ5IjoiVVMiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsImNfaGFzaCI6IjI5aVIwUG5fLUwwRVZLbVFtSDlsNHciLCJleHAiOjE0NjUwMTYxMTUsImlhdCI6MTQ2NTAxNTIxNSwiYXRfaGFzaCI6ImduREdIdjNWWXRRM3N1Snp1VmxYNkEiLCJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwidHlwIjoiaWRfdG9rZW4iLCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.eYqFpyJRgW9FiPVYm42p12WnFDxA-ADmPfX-to-3_Oo8X_go_06zaH2Sue6wF_6acCIuXYsM_QzZpAmrv-9_iPefKv0nvlHyOwJ7xxMHgs7FbSl-6CTcOXWnV2YFstgbv6i0ORH61uGbNCCQo9rE91cL8_jqsY_pe1qAZx4WHHsA7H6TA8moiHCPRIpu5Xx_iW8gU3mA5g2gYKjtYe9KDFS5-jCcDg7PzY1BVO3yVu7nbLClRhNkflBwNCJRaqpliOnP8LB3SflS3v7ejXvcovSbfiybqgO_7IwTsnyUWaZXgvzym8JlWVD9zjdQ_xBNHtTlfAZY7GAT3XpdXSz8rA&expires_in=900",
    "nintendo://0123456789012345.app?session_state=d812fcca44ec7f428e9672c8b96a1a823a422527483e13792286cd61e55119fe&code=eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE&state=IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc&id_token=eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiIsImtpZCI6ImI1ZWQ4ZTRjLTNkNWQtNDI3Ny1hOGRkLWQ2MmMyOWM0ZGJhZiJ9.eyJqdGkiOiJjYmQxNDJhOC1jOTJjLTQ0MTUtOWE3MS1iZWZmYmE0YzA5ZjciLCJjb3VudHJ5IjoiVVMiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsImNfaGFzaCI6IjI5aVIwUG5fLUwwRVZLbVFtSDlsNHciLCJleHAiOjE0NjUwMTYxMTUsImlhdCI6MTQ2NTAxNTIxNSwiYXRfaGFzaCI6ImduREdIdjNWWXRRM3N1Snp1VmxYNkEiLCJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwidHlwIjoiaWRfdG9rZW4iLCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.eYqFpyJRgW9FiPVYm42p12WnFDxA-ADmPfX-to-3_Oo8X_go_06zaH2Sue6wF_6acCIuXYsM_QzZpAmrv-9_iPefKv0nvlHyOwJ7xxMHgs7FbSl-6CTcOXWnV2YFstgbv6i0ORH61uGbNCCQo9rE91cL8_jqsY_pe1qAZx4WHHsA7H6TA8moiHCPRIpu5Xx_iW8gU3mA5g2gYKjtYe9KDFS5-jCcDg7PzY1BVO3yVu7nbLClRhNkflBwNCJRaqpliOnP8LB3SflS3v7ejXvcovSbfiybqgO_7IwTsnyUWaZXgvzym8JlWVD9zjdQ_xBNHtTlfAZY7GAT3XpdXSz8rA&expires_in=900",
};

struct Expect
{
    char redirectUri[256];
    a::nas::State state;
    const char* code;
} Expects[] = {
    {
        "http://localhost:8080",
        {"IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc"},
        "eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE",
    },
    {
        "nintendo://account.nx.sys",
        {"IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc"},
        "eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE",
    },
    {
        "nintendo://0123456789012345.app",
        {"IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc"},
        "eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE",
    },
};

void RunTest(const Expect& expect, const char* uri, size_t uriSize) NN_NOEXCEPT
{
    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::nas::NasServiceAuthorizationAdaptor adaptor(expect.redirectUri, expect.state, s);
    ASSERT_TRUE(adaptor.Parse(uri, uriSize));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt());

    a::detail::Uuid codeCacheId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.PullAuthorizationCode(&codeCacheId));

    size_t size;
    t::Buffer buffer(1024 * 1024);

    // code
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS((a::detail::CacheUtil::LoadCacheFile<a::detail::NasAuthorizationCodeSizeMax, a::ResultNasTokenLengthUnacceptable>(
        &size, buffer.GetAddress(), buffer.GetSize(), codeCacheId, s)));
    ASSERT_TRUE(size < buffer.GetSize());
    buffer.Get<char>()[size] = '\0';
    ASSERT_EQ(0, std::strncmp(expect.code, buffer.Get<char>(), a::detail::NasAuthorizationCodeSizeMax));
}
} // ~namespace authorizationCode

TEST(AccountAdaptor, NasAuthorizationCode)
{
    for (int i = 0; i < sizeof(authroizationCode::Expects) / sizeof(authroizationCode::Expects[0]); ++ i)
    {
        authroizationCode::RunTest(
            authroizationCode::Expects[i],
            authroizationCode::Inputs[i], strlen(authroizationCode::Inputs[i]) + 1);
    }
}

namespace appAuthorization {
const char* Inputs[] = {
    "http://localhost:8080#id_token=a&session_state=d812fcca44ec7f428e9672c8b96a1a823a422527483e13792286cd61e55119fe&code=eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE&state=IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc",
    "nintendo://account.nx.sys#session_state=d812fcca44ec7f428e9672c8b96a1a823a422527483e13792286cd61e55119fe&code=eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE&state=IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc&id_token=eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiIsImtpZCI6ImI1ZWQ4ZTRjLTNkNWQtNDI3Ny1hOGRkLWQ2MmMyOWM0ZGJhZiJ9.eyJqdGkiOiJjYmQxNDJhOC1jOTJjLTQ0MTUtOWE3MS1iZWZmYmE0YzA5ZjciLCJjb3VudHJ5IjoiVVMiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsImNfaGFzaCI6IjI5aVIwUG5fLUwwRVZLbVFtSDlsNHciLCJleHAiOjE0NjUwMTYxMTUsImlhdCI6MTQ2NTAxNTIxNSwiYXRfaGFzaCI6ImduREdIdjNWWXRRM3N1Snp1VmxYNkEiLCJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwidHlwIjoiaWRfdG9rZW4iLCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.eYqFpyJRgW9FiPVYm42p12WnFDxA-ADmPfX-to-3_Oo8X_go_06zaH2Sue6wF_6acCIuXYsM_QzZpAmrv-9_iPefKv0nvlHyOwJ7xxMHgs7FbSl-6CTcOXWnV2YFstgbv6i0ORH61uGbNCCQo9rE91cL8_jqsY_pe1qAZx4WHHsA7H6TA8moiHCPRIpu5Xx_iW8gU3mA5g2gYKjtYe9KDFS5-jCcDg7PzY1BVO3yVu7nbLClRhNkflBwNCJRaqpliOnP8LB3SflS3v7ejXvcovSbfiybqgO_7IwTsnyUWaZXgvzym8JlWVD9zjdQ_xBNHtTlfAZY7GAT3XpdXSz8rA&expires_in=900",
    "nintendo://0123456789012345.app#session_state=d812fcca44ec7f428e9672c8b96a1a823a422527483e13792286cd61e55119fe&code=eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE&state=IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc&id_token=eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiIsImtpZCI6ImI1ZWQ4ZTRjLTNkNWQtNDI3Ny1hOGRkLWQ2MmMyOWM0ZGJhZiJ9.eyJqdGkiOiJjYmQxNDJhOC1jOTJjLTQ0MTUtOWE3MS1iZWZmYmE0YzA5ZjciLCJjb3VudHJ5IjoiVVMiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsImNfaGFzaCI6IjI5aVIwUG5fLUwwRVZLbVFtSDlsNHciLCJleHAiOjE0NjUwMTYxMTUsImlhdCI6MTQ2NTAxNTIxNSwiYXRfaGFzaCI6ImduREdIdjNWWXRRM3N1Snp1VmxYNkEiLCJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwidHlwIjoiaWRfdG9rZW4iLCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.eYqFpyJRgW9FiPVYm42p12WnFDxA-ADmPfX-to-3_Oo8X_go_06zaH2Sue6wF_6acCIuXYsM_QzZpAmrv-9_iPefKv0nvlHyOwJ7xxMHgs7FbSl-6CTcOXWnV2YFstgbv6i0ORH61uGbNCCQo9rE91cL8_jqsY_pe1qAZx4WHHsA7H6TA8moiHCPRIpu5Xx_iW8gU3mA5g2gYKjtYe9KDFS5-jCcDg7PzY1BVO3yVu7nbLClRhNkflBwNCJRaqpliOnP8LB3SflS3v7ejXvcovSbfiybqgO_7IwTsnyUWaZXgvzym8JlWVD9zjdQ_xBNHtTlfAZY7GAT3XpdXSz8rA&expires_in=900",
};

struct Expect
{
    char redirectUri[256];
    a::nas::State state;
    const char* code;
    const char* idToken;
} Expects[] = {
    {
        "http://localhost:8080",
        {"IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc"},
        "eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE",
        "a",
    },
    {
        "nintendo://account.nx.sys",
        {"IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc"},
        "eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE",
        "eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiIsImtpZCI6ImI1ZWQ4ZTRjLTNkNWQtNDI3Ny1hOGRkLWQ2MmMyOWM0ZGJhZiJ9.eyJqdGkiOiJjYmQxNDJhOC1jOTJjLTQ0MTUtOWE3MS1iZWZmYmE0YzA5ZjciLCJjb3VudHJ5IjoiVVMiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsImNfaGFzaCI6IjI5aVIwUG5fLUwwRVZLbVFtSDlsNHciLCJleHAiOjE0NjUwMTYxMTUsImlhdCI6MTQ2NTAxNTIxNSwiYXRfaGFzaCI6ImduREdIdjNWWXRRM3N1Snp1VmxYNkEiLCJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwidHlwIjoiaWRfdG9rZW4iLCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.eYqFpyJRgW9FiPVYm42p12WnFDxA-ADmPfX-to-3_Oo8X_go_06zaH2Sue6wF_6acCIuXYsM_QzZpAmrv-9_iPefKv0nvlHyOwJ7xxMHgs7FbSl-6CTcOXWnV2YFstgbv6i0ORH61uGbNCCQo9rE91cL8_jqsY_pe1qAZx4WHHsA7H6TA8moiHCPRIpu5Xx_iW8gU3mA5g2gYKjtYe9KDFS5-jCcDg7PzY1BVO3yVu7nbLClRhNkflBwNCJRaqpliOnP8LB3SflS3v7ejXvcovSbfiybqgO_7IwTsnyUWaZXgvzym8JlWVD9zjdQ_xBNHtTlfAZY7GAT3XpdXSz8rA",
    },
    {
        "nintendo://0123456789012345.app",
        {"IbvCtJEvkLljP71beXe-Zr7d13UM1tcWb2HHDfuagnc"},
        "eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJhOTA5NDc1NThlNWVjOWI5IiwiYXVkIjoiY2E5ODllZmFjYTBkMGM2YSIsImlhdCI6MTQ2NTAxNTIxNCwiYWM6Y2NtIjoiUzI1NiIsInR5cCI6ImNvZGUiLCJhYzpzY3AiOlswLDEsNCw1LDgsOSwxMCwxN10sImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiYWM6Y2MiOiJ0cXc4d1FPR014eDJYd1R3UWNGSDBQSjQ4cTdZNnFBaDR0QUZmOGIyXzU0IiwiZXhwIjoxNDY1MDE4ODE0LCJqdGkiOiIxODE0In0.59Tio_liRjdSbMfOOtbN2vwLn-tmThjIkKRaOHfayRE",
        "eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbS8xLjAuMC9jZXJ0aWZpY2F0ZXMiLCJhbGciOiJSUzI1NiIsImtpZCI6ImI1ZWQ4ZTRjLTNkNWQtNDI3Ny1hOGRkLWQ2MmMyOWM0ZGJhZiJ9.eyJqdGkiOiJjYmQxNDJhOC1jOTJjLTQ0MTUtOWE3MS1iZWZmYmE0YzA5ZjciLCJjb3VudHJ5IjoiVVMiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmFjY291bnRzLm5pbnRlbmRvLmNvbSIsImNfaGFzaCI6IjI5aVIwUG5fLUwwRVZLbVFtSDlsNHciLCJleHAiOjE0NjUwMTYxMTUsImlhdCI6MTQ2NTAxNTIxNSwiYXRfaGFzaCI6ImduREdIdjNWWXRRM3N1Snp1VmxYNkEiLCJhdWQiOiJjYTk4OWVmYWNhMGQwYzZhIiwidHlwIjoiaWRfdG9rZW4iLCJzdWIiOiJhOTA5NDc1NThlNWVjOWI5In0.eYqFpyJRgW9FiPVYm42p12WnFDxA-ADmPfX-to-3_Oo8X_go_06zaH2Sue6wF_6acCIuXYsM_QzZpAmrv-9_iPefKv0nvlHyOwJ7xxMHgs7FbSl-6CTcOXWnV2YFstgbv6i0ORH61uGbNCCQo9rE91cL8_jqsY_pe1qAZx4WHHsA7H6TA8moiHCPRIpu5Xx_iW8gU3mA5g2gYKjtYe9KDFS5-jCcDg7PzY1BVO3yVu7nbLClRhNkflBwNCJRaqpliOnP8LB3SflS3v7ejXvcovSbfiybqgO_7IwTsnyUWaZXgvzym8JlWVD9zjdQ_xBNHtTlfAZY7GAT3XpdXSz8rA",
    },
};

void RunTest(const Expect& expect, const char* uri, size_t uriSize) NN_NOEXCEPT
{
    a::nas::NasApplicationAuthorization auth;
    a::nas::NasApplicationAuthorizationAdaptor adaptor(&auth, expect.redirectUri, expect.state);
    ASSERT_TRUE(adaptor.Parse(uri, uriSize));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt());

    // code
    ASSERT_EQ(strlen(expect.code), strnlen(auth.code, sizeof(auth.code)));
    ASSERT_EQ(0, std::strncmp(expect.code, auth.code, sizeof(auth.code)));

    // id token
    ASSERT_EQ(strlen(expect.idToken), strnlen(auth.idToken, sizeof(auth.idToken)));
    ASSERT_EQ(0, std::strncmp(expect.idToken, auth.idToken, sizeof(auth.idToken)));
}
} // ~namespace appAuthorization

TEST(AccountAdaptor, NasApplicationAuthorizationCode)
{
    for (int i = 0; i < sizeof(appAuthorization::Expects) / sizeof(appAuthorization::Expects[0]); ++ i)
    {
        appAuthorization::RunTest(
            appAuthorization::Expects[i],
            appAuthorization::Inputs[i], strlen(appAuthorization::Inputs[i]) + 1);
    }
}
