﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAccount_RamFs.h"

#include "detail/account_PathUtil.h"
#include <nn/account/detail/account_FileSystem.h>
#include <nn/account/account_ResultPrivate.h>

#include <cstdlib>
#include <cstring>
#include <algorithm>
#include <map>
#include <mutex>
#include <vector>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>

namespace nnt { namespace account {

#ifdef NNT_ACCOUNT_RAMFS_LOG_ENABLE
    #define NNT_ACCOUNT_RAMFS_LOG(...) NN_LOG(__VA_ARGS__)
#else
    #define NNT_ACCOUNT_RAMFS_LOG(...)
#endif

namespace {
struct Path
{
    char _data[128];

    Path() NN_NOEXCEPT
    {
    }
    explicit Path(const char* str)
    {
        strncpy(_data, str, sizeof(_data));
        _data[sizeof(_data) - 1] = '\0';
    }
    NN_EXPLICIT_OPERATOR const char*() const NN_NOEXCEPT
    {
        return _data;
    }
};

class Less
{
public:
    bool operator() (Path lhs, Path rhs) const NN_NOEXCEPT
    {
        return std::strncmp(lhs, rhs, sizeof(lhs)) < 0;
    }
};
typedef std::pair<void*, size_t> FileType;
std::map<Path, FileType, Less> g_Storage;

inline void DeleteFilesOnImpl(const char* path) NN_NOEXCEPT
{
    size_t len = strnlen(path, 128);
    std::vector<Path> v;

    NNT_ACCOUNT_RAMFS_LOG("DeleteFilesOnImpl - Query: \"%s\"\n", path);

    for (auto& f: g_Storage)
    {
        if (strncmp(f.first, path, len) == 0)
        {
            NNT_ACCOUNT_RAMFS_LOG("%s is cache\n", f.first);
            v.push_back(f.first);
        }
        else
        {
            NNT_ACCOUNT_RAMFS_LOG("%s is not cache\n", f.first);
        }
    }
    for (auto& p: v)
    {
        g_Storage.erase(p);
    }
}

inline size_t GetEntryCountOnImpl(const char* path) NN_NOEXCEPT
{
    size_t len = strnlen(path, 128);
    std::vector<Path> v;

    NNT_ACCOUNT_RAMFS_LOG("GetEntryCountOnImpl - Query: \"%s\"\n", path);

    for (auto& f: g_Storage)
    {
        if (strncmp(f.first, path, len) == 0)
        {
            NNT_ACCOUNT_RAMFS_LOG("%s is cache\n", f.first);
            v.push_back(f.first);
        }
        else
        {
            NNT_ACCOUNT_RAMFS_LOG("%s is not cache\n", f.first);
        }
    }
    return v.size();
}
} //

nn::os::SdkRecursiveMutex RamFs::s_WriterLock;
nn::os::SdkMutex RamFs::s_TableLock;

nn::Result RamFs::DeleteDirectoryImpl(const char* path) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    auto len = strlen(path);
    if (path[len - 1] == '/')
    {
        // 既に '/' で終端されていればそのまま
        DeleteFilesOnImpl(path);
        NN_RESULT_SUCCESS;
    }
    char newPath[128];
    nn::util::SNPrintf(newPath, sizeof(newPath), "%s/", path);
    DeleteFilesOnImpl(newPath);
    NN_RESULT_SUCCESS;
}

nn::Result RamFs::MountHost(const char* volumeName, const char* hostLocation) NN_NOEXCEPT
{
    NN_UNUSED(volumeName);
    NN_UNUSED(hostLocation);
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::MountSave(const char* volumeName, nn::fs::SystemSaveDataId saveDataId, int64_t dataSize, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    NN_UNUSED(volumeName);
    NN_UNUSED(saveDataId);
    NN_UNUSED(dataSize);
    NN_UNUSED(journalSize);
    NN_UNUSED(flags);
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::Unmount(const char* volumeName) NN_NOEXCEPT
{
    NN_UNUSED(volumeName);
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::Commit(const char* volumeName) NN_NOEXCEPT
{
    NN_UNUSED(volumeName);
    NN_RESULT_SUCCESS;
}

nn::Result RamFs::Setup(const char* rootPath) NN_NOEXCEPT
{
    ClearCache(rootPath);
    NN_RESULT_SUCCESS;
}
void RamFs::ClearProfile(const char* rootPath) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    char path[64];
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteDirectoryImpl(
        nn::account::detail::PathUtil::GetProfileDirectoryPath(path, sizeof(path), rootPath)));
}
void RamFs::ClearCache(const char* rootPath) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    char path[64];
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteDirectoryImpl(
        nn::account::detail::PathUtil::GetCacheDirectoryPath(path, sizeof(path), rootPath)));
}
void RamFs::Clear(const char* rootPath) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteDirectoryImpl(rootPath));
}
nn::account::detail::FsLockGuard RamFs::AcquireWriterLock() NN_NOEXCEPT
{
    return nn::account::detail::FsLockGuard(s_WriterLock);
}

size_t RamFs::GetEntryCountOn(const char* path) NN_NOEXCEPT
{
    return GetEntryCountOnImpl(path);
}

// ファイルの操作
nn::Result RamFs::Create(const char* path, size_t sizeToReserve) const NN_NOEXCEPT
{
    NN_UNUSED(sizeToReserve);

    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    std::lock_guard<decltype(s_TableLock)> lock(s_TableLock);

    Path p(path);
    NNT_ACCOUNT_RAMFS_LOG("Create: %s\n", p);

    auto& s = g_Storage;
    if (s.count(p) != 0)
    {
        NN_RESULT_THROW(nn::fs::ResultPathAlreadyExists());
    }
    s[p] = FileType(std::malloc(sizeToReserve), sizeToReserve);
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::Delete(const char* path) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    std::lock_guard<decltype(s_TableLock)> lock(s_TableLock);

    Path p(path);
    NNT_ACCOUNT_RAMFS_LOG("Delete: %s\n", p);

    auto& s = g_Storage;
    if (s.count(p) != 0)
    {
        free(s[p].first);
        s.erase(p);
    }
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::Write(const char* path, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    std::lock_guard<decltype(s_TableLock)> lock(s_TableLock);

    Path p(path);
    NNT_ACCOUNT_RAMFS_LOG("Write: %s\n", p);

    auto& s = g_Storage;
    if (s.count(p) != 0)
    {
        free(s[p].first);
    }
    auto f = FileType(std::malloc(bufferSize), bufferSize);;
    std::memcpy(f.first, buffer, bufferSize);
    s[p] = std::move(f);
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::Append(const char* path, size_t offset, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    std::lock_guard<decltype(s_TableLock)> lock(s_TableLock);

    Path p(path);
    NNT_ACCOUNT_RAMFS_LOG("Append: %s\n", p);

    auto& s = g_Storage;
    if (s.count(p) == 0)
    {
        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
    }
    auto f = s[p];
    if (f.second < offset + bufferSize)
    {
        auto m = std::malloc(offset + bufferSize);
        std::memcpy(m, f.first, f.second);
        free(f.first);
        f.first = m;
        f.second = offset + bufferSize;
    }
    std::memcpy(reinterpret_cast<char*>(f.first) + offset, buffer, bufferSize);
    s[p] = std::move(f);
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::GetSize(size_t* pOutSize, const char* path) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_TableLock)> lock(s_TableLock);

    Path p(path);
    NNT_ACCOUNT_RAMFS_LOG("GetSize: %s\n", p);

    auto& s = g_Storage;
    if (s.count(p) == 0)
    {
        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
    }
    *pOutSize = s[p].second;
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::SetSize(const char* path, size_t size) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_TableLock)> lock(s_TableLock);

    Path p(path);
    NNT_ACCOUNT_RAMFS_LOG("GetSize: %s\n", p);

    auto& s = g_Storage;
    if (s.count(p) == 0)
    {
        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
    }
    auto f = s[p];
    auto m = std::malloc(size);
    std::memcpy(m, f.first, std::min(f.second, size));
    std::free(f.first);
    f.first = m;
    f.second = size;
    s[p] = std::move(f);
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_TableLock)> lock(s_TableLock);

    Path p(path);
    NNT_ACCOUNT_RAMFS_LOG("Read: %s\n", p);

    auto& s = g_Storage;
    if (s.count(p) == 0)
    {
        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
    }
    auto f = s[p];
    *pOutActualSize = std::min(f.second - offset, bufferSize);
    std::memcpy(buffer, reinterpret_cast<const char*>(s[p].first) + offset, *pOutActualSize);
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path) const NN_NOEXCEPT
{
    return Read(pOutActualSize, buffer, bufferSize, path, 0u);
}
nn::Result RamFs::Move(const char* to, const char* from) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    std::lock_guard<decltype(s_TableLock)> lock(s_TableLock);

    Path pf(from);
    Path pt(to);
    NNT_ACCOUNT_RAMFS_LOG("Move: %s <- %s\n", pt, pf);

    auto& s = g_Storage;
    if (s.count(pf) == 0)
    {
        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
    }
    auto f = s[pf];
    s.erase(pf);
    if (s.count(pt) != 0)
    {
        free(s[pt].first);
    }
    s[pt] = std::move(f);
    NN_RESULT_SUCCESS;
}
nn::Result RamFs::Copy(const char* to, const char* from, void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    NN_UNUSED(buffer);
    NN_UNUSED(bufferSize);

    std::lock_guard<decltype(s_TableLock)> lock(s_TableLock);

    Path pf(from);
    Path pt(to);
    NNT_ACCOUNT_RAMFS_LOG("Copy: %s <- %s\n", pt, pf);

    auto& s = g_Storage;
    if (s.count(pf) == 0)
    {
        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
    }

    auto ff = s[pf];
    FileType ft = FileType(std::malloc(ff.second), ff.second);
    std::memcpy(ft.first, ff.first, ff.second);

    if (s.count(pt) != 0)
    {
        free(s[pt].first);
    }
    s[pt] = std::move(ft);
    NN_RESULT_SUCCESS;
}

}} // ~namespace nnt::account
