﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @brief
 * HidHandler class for processing controller input data. Registered clients will be signaled when a controller
 * registered key press is detected.
 *
 * @details
 *
 * Purpose:
 *     The main purpose of HidHandler class is to read controller input data process it and signal any registered
 * clients for the input. A limited set of controller inputs are tracked for use with movie players.
 *
 * Setup:
 *     Client creates an instance of HidHandler. When initialized, a thread is created to sample controller
 * inputs. Client needs to call RegisterEvent() API to register an event and a type of controller key type.
 *
 * Main Processing:
 *     The HidHandler class thread function will call ProcessHidEvent() to process events. If it finds any key
 * is press and if there is an event registered, the event will be signaled. The thread will sample for controller inputs
 *  every 16ms.
 *
 * Teardown:
 *    Client needs to call Finalize() to stop HidHandler to stop sampling controller inputs. It will stop and destroy the
 * thread. All the resources will be freed. Finally client needs to delete the instance.
 *
 */

#include <nns/mm/mm_HidHandler.h>
HidHandler::HidHandler()
    : m_ThreadCreated(false),
      m_ThreadStarted(false),
      m_ThreadDone(false),
      m_Threadstacksize(1024 * 128),
      m_Threadstack(nullptr),
      m_ThreadSleepTimeMs(16),
      m_NpadIdCount(4),
      m_PrevPadSamplingNumber(0)
{
}

bool HidHandler::Initialize()
{
    nn::Result nnResult = nn::ResultSuccess();
    bool retStatus = false;
    m_NpadIds = new nn::hid::NpadIdType[ m_NpadIdCount ];
    m_OldNpadJoyDualState = new nn::hid::NpadJoyDualState[ m_NpadIdCount * 2];
    m_CurrentNpadJoyDualState = new nn::hid::NpadJoyDualState[ m_NpadIdCount * 2 ];

    if( ( m_NpadIds == nullptr ) || ( m_OldNpadJoyDualState == nullptr ) || ( m_CurrentNpadJoyDualState == nullptr ) )
    {
        return retStatus;
    }

    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadIdType(m_NpadIds, m_NpadIdCount);
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleJoyDual::Mask);

    // Allocate memory for HidHandler thread stack
    m_Threadstack = aligned_alloc(nn::os::ThreadStackAlignment, m_Threadstacksize);
    if( m_Threadstack == nullptr )
    {
        return retStatus;
    }
    // Create HidHandler thread
    nnResult = nn::os::CreateThread(&m_ThreadType,
        &HidHandlerThreadFunction,
        ( void* )this,
        m_Threadstack,
        m_Threadstacksize,
        nn::os::DefaultThreadPriority);
        m_ThreadCreated = true;
    if( nnResult.IsSuccess() )
    {
        nn::os::SetThreadName(&m_ThreadType, "HidHandlerThread");
        nn::os::StartThread(&m_ThreadType);
        m_ThreadStarted = true;
        retStatus = true;
    }
    return retStatus;
}

void HidHandler::Finalize()
{
    // Release all resources.
    m_ThreadDone = true;
    if( m_ThreadCreated == true )
    {
        if( m_ThreadStarted == true )
        {
            nn::os::WaitThread(&m_ThreadType);
            m_ThreadStarted = false;
        }
        nn::os::DestroyThread(&m_ThreadType);
        m_ThreadCreated = false;
    }

    if( m_NpadIds != nullptr )
    {
        delete[] m_NpadIds;
        m_NpadIds = nullptr;
    }
    if( m_OldNpadJoyDualState != nullptr )
    {
        delete[] m_OldNpadJoyDualState;
        m_OldNpadJoyDualState = nullptr;
    }
    if( m_CurrentNpadJoyDualState != nullptr )
    {
        delete[] m_CurrentNpadJoyDualState;
        m_CurrentNpadJoyDualState = nullptr;
    }

    m_EventMap.clear();
    if( m_Threadstack != nullptr )
    {
        free(m_Threadstack);
        m_Threadstack = nullptr;
    }
}

int64_t HidHandler::GetThreadSleepTimeMs()
{
    return m_ThreadSleepTimeMs;
}

bool HidHandler::ExitThread()
{
    return m_ThreadDone;
}

void HidHandler::RegisterEvent(nn::os::EventType *nnOsEvent, HidHandlerKeyType keyType)
{
    m_EventMap.insert(std::pair<HidHandlerKeyType, nn::os::EventType*>(keyType, nnOsEvent));
}

void HidHandler::ProcessHidEvent()
{
    nn::hid::GetDebugPadStates(m_DebugPadState, nn::hid::DebugPadStateCountMax);
    m_AllCurrentNpadButtonsPressed.Reset();
    m_AllCurrentNpadButtonsHeld.Reset();
    for( int i = 0; i < m_NpadIdCount; i++ )
    {
        m_OldNpadJoyDualState[ i ] = m_CurrentNpadJoyDualState[ i ];

        nn::hid::GetNpadState(&( m_CurrentNpadJoyDualState[ i ] ), m_NpadIds[ i ]);

        if( ( m_CurrentNpadJoyDualState[ i ].buttons ^ m_OldNpadJoyDualState[ i ].buttons & m_CurrentNpadJoyDualState[ i ].buttons ).IsAnyOn() )
        {
            m_AllCurrentNpadButtonsPressed |= m_CurrentNpadJoyDualState[ i ].buttons ^ m_OldNpadJoyDualState[ i ].buttons;
        }
        if( ( m_CurrentNpadJoyDualState[ i ].buttons & m_OldNpadJoyDualState[ i ].buttons ).IsAnyOn() )
        {
            m_AllCurrentNpadButtonsHeld |= m_CurrentNpadJoyDualState[ i ].buttons & m_OldNpadJoyDualState[ i ].buttons;
        }
    }

    int64_t padSamplingCount = m_DebugPadState[ 0 ].samplingNumber - m_PrevPadSamplingNumber;
    if( padSamplingCount >= nn::hid::DebugPadStateCountMax )
    {
        padSamplingCount = nn::hid::DebugPadStateCountMax - 1;
    }
    m_PrevPadSamplingNumber = m_DebugPadState[ 0 ].samplingNumber;
    nn::hid::DebugPadButtonSet debugPadButtonDown(m_DebugPadState[ 0 ].buttons & ~m_DebugPadState[ padSamplingCount ].buttons);

    if( debugPadButtonDown.Test< ::nn::hid::DebugPadButton::A >() || m_AllCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::A>() )
    {
        nn::os::EventType* eventButtonAPress = nullptr;
        if( m_EventMap.find(HidHandlerKeyType_ButtonA) != m_EventMap.end() )
        {
           eventButtonAPress = m_EventMap.find(HidHandlerKeyType_ButtonA)->second;
        }
        if( eventButtonAPress != nullptr )
        {
            nn::os::SignalEvent(eventButtonAPress);
        }
    }

    if( debugPadButtonDown.Test< ::nn::hid::DebugPadButton::B >() || m_AllCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::B>() )
    {
        nn::os::EventType* eventButtonBPress = nullptr;
        if( m_EventMap.find(HidHandlerKeyType_ButtonB) != m_EventMap.end() )
        {
            eventButtonBPress = m_EventMap.find(HidHandlerKeyType_ButtonB)->second;
        }
        if( eventButtonBPress != nullptr )
        {
            nn::os::SignalEvent(eventButtonBPress);
        }
    }

    if( debugPadButtonDown.Test< ::nn::hid::DebugPadButton::X >() || m_AllCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::X>() )
    {
        nn::os::EventType* eventButtonXPress = nullptr;
        if( m_EventMap.find(HidHandlerKeyType_ButtonX) != m_EventMap.end() )
        {
            eventButtonXPress = m_EventMap.find(HidHandlerKeyType_ButtonX)->second;
        }
        if( eventButtonXPress != nullptr )
        {
            nn::os::SignalEvent(eventButtonXPress);
        }
    }

    if( debugPadButtonDown.Test< ::nn::hid::DebugPadButton::ZR >() ||
        m_AllCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::ZR>() )
    {
        nn::os::EventType* eventButtonZr = nullptr;
        if( m_EventMap.find(HidHandlerKeyType_ButtonZr) != m_EventMap.end() )
        {
            eventButtonZr = m_EventMap.find(HidHandlerKeyType_ButtonZr)->second;
        }
        if( eventButtonZr != nullptr )
        {
            nn::os::SignalEvent(eventButtonZr);
        }
    }

    if( debugPadButtonDown.Test< ::nn::hid::DebugPadButton::L >() ||
        m_AllCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::L>() )
    {
        nn::os::EventType* eventButtonL = nullptr;
        if( m_EventMap.find(HidHandlerKeyType_ButtonL) != m_EventMap.end() )
        {
            eventButtonL = m_EventMap.find(HidHandlerKeyType_ButtonL)->second;
        }
        if( eventButtonL != nullptr )
        {
            nn::os::SignalEvent(eventButtonL);
        }
    }

    if( debugPadButtonDown.Test< ::nn::hid::DebugPadButton::R >() ||
        m_AllCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::R>() )
    {
        nn::os::EventType* eventButtonR = nullptr;
        if( m_EventMap.find(HidHandlerKeyType_ButtonR) != m_EventMap.end() )
        {
            eventButtonR = m_EventMap.find(HidHandlerKeyType_ButtonR)->second;
        }
        if( eventButtonR != nullptr )
        {
            nn::os::SignalEvent(eventButtonR);
        }
    }
}

void HidHandlerThreadFunction(void *arg)
{
    HidHandler *hidHandler = ( HidHandler* ) arg;
    if( hidHandler != nullptr )
    {
        for( ;; )
        {
            if( hidHandler->ExitThread() )
            {
                break;
            }
            hidHandler->ProcessHidEvent();
            int64_t threadSleepTimeMs = hidHandler->GetThreadSleepTimeMs();
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(threadSleepTimeMs));
        }
    }
}
