﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <vector>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid/hid_Button.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerManager.h>
#include <nns/hid/hid_DeviceAssetId.h>
#include <nns/hid/hid_KeyboardAndMouse.h>

namespace nns { namespace hid {

namespace {

/**
 * @brief       NumLock が作用するか否かを表す値を返します。
 *
 * @param[in]   modifiers                   キー修飾です。
 *
 * @return      NumLock が作用するか否かを表す値です。
 */
bool IsNumLockEffective(nn::hid::KeyboardModifierSet modifiers) NN_NOEXCEPT
{
    return (modifiers.Test<nn::hid::KeyboardModifier::NumLock>() &&
           !modifiers.Test<nn::hid::KeyboardModifier::Shift>());
}

/**
 * @brief       印字可能な文字を生成するキーが作用するか否かを表す値を返します。
 *
 * @param[in]   usageId                     Usage ID です。
 * @param[in]   isNumLockEffective          NumLock が作用するか否かを表す値です。
 *
 * @return      印字可能な文字を生成するキーが作用するか否かを表す値です。
 */
bool IsPrintingKeyEffective(int usageId, bool isNumLockEffective) NN_NOEXCEPT
{
    if (usageId >= nn::hid::KeyboardKey::A::Index &&
        usageId <= nn::hid::KeyboardKey::Z::Index)
    {
        return true;
    }

    if (usageId >= nn::hid::KeyboardKey::D1::Index &&
        usageId <= nn::hid::KeyboardKey::D0::Index)
    {
        return true;
    }

    if (usageId == nn::hid::KeyboardKey::Space::Index)
    {
        return true;
    }

    if (usageId >= nn::hid::KeyboardKey::Minus::Index &&
        usageId <= nn::hid::KeyboardKey::Slash::Index)
    {
        return true;
    }

    if (usageId >= nn::hid::KeyboardKey::NumPadDivide::Index &&
        usageId <= nn::hid::KeyboardKey::NumPadAdd::Index)
    {
        return true;
    }

    if (isNumLockEffective &&
        usageId >= nn::hid::KeyboardKey::NumPad1::Index &&
        usageId <= nn::hid::KeyboardKey::NumPadDot::Index)
    {
        return true;
    }

    if (usageId == nn::hid::KeyboardKey::Backslash::Index)
    {
        return true;
    }

    if (usageId == nn::hid::KeyboardKey::NumPadEquals::Index)
    {
        return true;
    }

    if (usageId == nn::hid::KeyboardKey::NumPadComma::Index)
    {
        return true;
    }

    if (usageId == nn::hid::KeyboardKey::Ro::Index)
    {
        return true;
    }

    if (usageId == nn::hid::KeyboardKey::Yen::Index)
    {
        return true;
    }

    if (usageId == nn::hid::KeyboardKey::NumPadCommaPc98::Index)
    {
        return true;
    }

    return false;
}

/**
 * @brief       ポインティングデバイスの座標を有効範囲に収めます。
 *
 * @param[in]   pointer                     座標値です。
 * @param[in]   pointerMin                  有効範囲の下限座標です。
 * @param[in]   pointerMax                  有効範囲の上限座標です。
 *
 * @return      有効範囲に収められたポインティングデバイスの座標です。
 */
nn::util::Float2 ClampPointer(
    const nn::util::Float2& pointer,
    const nn::util::Float2& pointerMin,
    const nn::util::Float2& pointerMax) NN_NOEXCEPT
{
    nn::util::Float2 value = {};
    value.x = std::min(std::max(pointerMin.x, pointer.x), pointerMax.x);
    value.y = std::min(std::max(pointerMin.y, pointer.y), pointerMax.y);
    return value;
}

} // namespace

bool KeyEvent::IsPrintingKey() const NN_NOEXCEPT
{
    return IsPrintingKeyEffective(
        this->usageId, IsNumLockEffective(this->modifiers));
}

KeyboardAndMouse::KeyboardAndMouse(ControllerManager* pManager) NN_NOEXCEPT
    : Controller(pManager)
    , m_IsKeyboardConnected(false)
    , m_IsMouseConnected(false)
    , m_IsMouseCursorAvailable(false)
    , m_IsMouseCursorAbsoluteCoordinateUsed(false)
    , m_MouseWheel(0)
{
    NN_ASSERT_NOT_NULL(pManager);

    // 対応するキーボードのデバイスアセットを取得します。
    m_pKeyboardAsset = static_cast<KeyboardAsset*>(
        pManager->GetDeviceAsset(DeviceAssetId_Keyboard, 0));

    m_Keys.Reset();
    m_KeysDown.Reset();
    m_KeysUp.Reset();
    m_KeysRepeat.Reset();
    m_KeysToggle.Reset();

    for (size_t i = 0; i < KeyCountMax; ++i)
    {
        m_KeysDurationCounts[i] = 0;
        m_RepeatDelayCounts[i] = DefaultRepeatDelayCount;
        m_RepeatIntervalCounts[i] = DefaultRepeatIntervalCount;
    }

    m_KeyModifiers.Reset();

    // 対応するマウスのデバイスアセットを取得します。
    m_pMouseAsset = static_cast<MouseAsset*>(
        pManager->GetDeviceAsset(DeviceAssetId_Mouse, 0));

    m_MouseButtons.Reset();
    m_MouseButtonsDown.Reset();
    m_MouseButtonsUp.Reset();
    m_MouseButtonsDoubleClick.Reset();

    for (size_t i = 0; i < MouseButtonCountMax; ++i)
    {
        m_MouseButtonsUpDurationCounts[i] = 0;
        m_DoubleClickWaitCounts[i] = DefaultDoubleClickWaitCount;
    }

    m_MouseCursor = Controller::GetInvalidPointer();
}

KeyboardAndMouse::~KeyboardAndMouse() NN_NOEXCEPT {}

void KeyboardAndMouse::Update() NN_NOEXCEPT
{
    ButtonSet buttons = {};

    this->UpdateKeyboard(&buttons);

    this->UpdateMouse();

    const nn::util::Float2 stick = {};

    const bool isTouched = m_MouseButtons.Test<nn::hid::MouseButton::Left>();

    // コントローラの基底の状態を更新します。
    this->UpdateBase(buttons, stick, stick, isTouched, m_MouseCursor);
}

bool KeyboardAndMouse::HasAnyPrintingKey(
    const nn::hid::KeyboardKeySet& keys) const NN_NOEXCEPT
{
    if (keys.CountPopulation() == 0)
    {
        return false;
    }

    const bool isNumLockEffective = IsNumLockEffective(m_KeyModifiers);

    for (int i = 0; i < static_cast<int>(KeyCountMax); ++i)
    {
        if (keys.Test(i) && IsPrintingKeyEffective(i, isNumLockEffective))
        {
            return true;
        }
    }

    return false;
}

void KeyboardAndMouse::SetKeysRepeatOption(
    const nn::hid::KeyboardKeySet& keys,
    uint8_t delayCount, uint8_t intervalCount) NN_NOEXCEPT
{
    for (size_t i = 0; i < KeyCountMax; ++i)
    {
        if (keys.Test(static_cast<int>(i)))
        {
            // 指定されたキーにリピートを開始するまでのフレーム数を設定します。
            m_RepeatDelayCounts[i] = delayCount;

            // 指定されたキーにリピートの間隔となるフレーム数を設定します。
            m_RepeatIntervalCounts[i] = intervalCount;
        }
    }
}

void KeyboardAndMouse::SetKeysToggle(
    const nn::hid::KeyboardKeySet& flags) NN_NOEXCEPT
{
    m_KeysToggle = flags;

    // ロックキーについてはロック状態を優先します。
    m_KeysToggle.Set<nn::hid::KeyboardKey::CapsLock>(
        m_KeyModifiers.Test<nn::hid::KeyboardModifier::CapsLock>());
    m_KeysToggle.Set<nn::hid::KeyboardKey::ScrollLock>(
        m_KeyModifiers.Test<nn::hid::KeyboardModifier::ScrollLock>());
    m_KeysToggle.Set<nn::hid::KeyboardKey::NumLock>(
        m_KeyModifiers.Test<nn::hid::KeyboardModifier::NumLock>());
}

void KeyboardAndMouse::SetMouseButtonsDoubleClickOption(
    const nn::hid::MouseButtonSet& buttons, uint8_t waitCount) NN_NOEXCEPT
{
    for (size_t i = 0; i < MouseButtonCountMax; ++i)
    {
        if (buttons.Test(static_cast<int>(i)))
        {
            // 指定されたマウスボタンにダブルクリックの待ち受けフレーム数を設定します。
            m_DoubleClickWaitCounts[i] = waitCount;
        }
    }
}

bool KeyboardAndMouse::NeedsKeyRepeat(int usageId) const NN_NOEXCEPT
{
    NN_ASSERT_RANGE(usageId, 0, static_cast<int>(KeyCountMax));

    const uint32_t duration = m_KeysDurationCounts[usageId];
    const uint8_t delay = m_RepeatDelayCounts[usageId];
    const uint8_t interval = m_RepeatIntervalCounts[usageId];

    // 指定されたキーにリピートが設定されているか確認します。
    if (interval > 0)
    {
        // リピート条件を満たしているか判定します。
        if (duration >= delay && (duration - delay) % interval == 0)
        {
            return true;
        }
    }

    return false;
}

void KeyboardAndMouse::UpdateKeysToggle() NN_NOEXCEPT
{
    for (size_t i = 0; i < KeyCountMax; ++i)
    {
        if (m_KeysDown.Test(static_cast<int>(i)))
        {
            m_KeysToggle.Flip(static_cast<int>(i));
        }
    }

    // トグル状態変更時の共通処理を実行します。
    this->SetKeysToggle(m_KeysToggle);
}

void KeyboardAndMouse::UpdateKeyEvents() NN_NOEXCEPT
{
    // 前回のキーイベントを削除します。
    m_KeyEvents.clear();

    for (size_t i = 0; i < KeyCountMax; ++i)
    {
        KeyActionSet action = {};

        if (m_KeysDown.Test(static_cast<int>(i)))
        {
            action.Set<KeyAction::Down>();
        }
        else if (m_KeysUp.Test(static_cast<int>(i)))
        {
            action.Set<KeyAction::Up>();
        }
        else if (m_KeysRepeat.Test(static_cast<int>(i)))
        {
            action.Set<KeyAction::Repeat>();
        }

        if (action.IsAnyOn())
        {
            const KeyEvent event = {
                action,
                m_KeyModifiers,
                m_KeysToggle.Test(static_cast<int>(i)),
                static_cast<int32_t>(i)
            };

            m_KeyEvents.push_back(event);
        }
    }
}

void KeyboardAndMouse::UpdateKeyboard(ButtonSet* pOutButtons) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutButtons);

    // 対応するキーボードのデバイスアセットが存在しない場合は何も行いません。
    if (!m_pKeyboardAsset)
    {
        return;
    }

    const nn::hid::KeyboardState state = m_pKeyboardAsset->GetKeyboardState();

    // 接続状態を設定します。
    m_IsKeyboardConnected =
        state.attributes.Test<nn::hid::KeyboardAttribute::IsConnected>();

    // 新たに押下されたキーを最新の値に更新します。
    m_KeysDown = ~m_Keys & state.keys;

    // 新たに開放されたキーを最新の値に更新します。
    m_KeysUp = m_Keys & ~state.keys;

    // 押下中のキーを最新の値に更新します。
    m_Keys = state.keys;

    // キーのリピート状態をリセットします。
    m_KeysRepeat.Reset();

    for (int i = 0; i < static_cast<int>(KeyCountMax); ++i)
    {
        if (state.keys.Test(i))
        {
            // リピートが必要ならば割り当てます。
            if (this->NeedsKeyRepeat(i))
            {
                m_KeysRepeat.Set(i, true);
            }

            // キーが押下状態であれば経過フレーム数を更新します。
            ++m_KeysDurationCounts[i];
        }
        else
        {
            // キーが押下状態でなければ経過フレーム数をリセットします。
            m_KeysDurationCounts[i] = 0;
        }
    }

    // トグルされたキーを更新します。
    this->UpdateKeysToggle();

    // キー修飾を更新します。
    m_KeyModifiers = state.modifiers;

    // キーイベントを更新します。
    this->UpdateKeyEvents();

    // デジタルボタンの押下状態を対応付けます。
    ButtonSet& buttons = *pOutButtons;
    buttons.Set<Button::A>(state.keys.Test<nn::hid::KeyboardKey::A>());
    buttons.Set<Button::B>(state.keys.Test<nn::hid::KeyboardKey::B>());
    buttons.Set<Button::X>(state.keys.Test<nn::hid::KeyboardKey::X>());
    buttons.Set<Button::Y>(state.keys.Test<nn::hid::KeyboardKey::Y>());
    buttons.Set<Button::Minus>(state.keys.Test<nn::hid::KeyboardKey::M>());
    buttons.Set<Button::Plus>(state.keys.Test<nn::hid::KeyboardKey::P>());
    buttons.Set<Button::Start>(state.keys.Test<nn::hid::KeyboardKey::P>());
    buttons.Set<Button::Select>(state.keys.Test<nn::hid::KeyboardKey::M>());
    buttons.Set<Button::L>(state.keys.Test<nn::hid::KeyboardKey::L>());
    buttons.Set<Button::R>(state.keys.Test<nn::hid::KeyboardKey::R>());

    // キーパッドのものも含めて方向キーを対応付けます。
    const bool isNumLockOff = !IsNumLockEffective(state.modifiers);
    buttons.Set<Button::Up>(
        state.keys.Test<nn::hid::KeyboardKey::UpArrow>() ||
        (isNumLockOff && state.keys.Test<nn::hid::KeyboardKey::NumPad8>()));
    buttons.Set<Button::Down>(
        state.keys.Test<nn::hid::KeyboardKey::DownArrow>() ||
        (isNumLockOff && state.keys.Test<nn::hid::KeyboardKey::NumPad2>()));
    buttons.Set<Button::Left>(
        state.keys.Test<nn::hid::KeyboardKey::LeftArrow>() ||
        (isNumLockOff && state.keys.Test<nn::hid::KeyboardKey::NumPad4>()));
    buttons.Set<Button::Right>(
        state.keys.Test<nn::hid::KeyboardKey::RightArrow>() ||
        (isNumLockOff && state.keys.Test<nn::hid::KeyboardKey::NumPad6>()));
}

void KeyboardAndMouse::UpdateMouseButtonsDoubleClick() NN_NOEXCEPT
{
    m_MouseButtonsDoubleClick.Reset();

    for (size_t i = 0; i < MouseButtonCountMax; ++i)
    {
        uint8_t& duration =  m_MouseButtonsUpDurationCounts[i];

        if (!m_MouseButtonsUp.Test(static_cast<int>(i)))
        {
            if (duration != 0)
            {
                duration -= 1;
            }
        }
        else
        {
            if (duration == 0)
            {
                duration = m_DoubleClickWaitCounts[i];
            }
            else
            {
                duration = 0;

                m_MouseButtonsDoubleClick.Set(static_cast<int>(i));
            }
        }
    }
}

void KeyboardAndMouse::UpdateMouse() NN_NOEXCEPT
{
    // 対応するマウスのデバイスアセットが存在しない場合は何も行いません。
    if (!m_pMouseAsset)
    {
        return;
    }

    // 差分値を破棄します。
    m_MouseWheel = 0;

    typedef std::vector<nn::hid::MouseState> MouseStates;

    const MouseStates& states = m_pMouseAsset->GetMouseStates();

    // 新規の入力状態が得られなかった場合は何も行いません。
    if (states.empty())
    {
        return;
    }

    // マウスの入力状態を古いものから順にカーソル座標に反映します。
    for (MouseStates::const_reverse_iterator rit =
             states.rbegin(); rit != states.rend(); ++rit)
    {
        const nn::hid::MouseState& state = *rit;

        if (state.attributes.Test<nn::hid::MouseAttribute::Transferable>())
        {
            if (m_IsMouseCursorAvailable)
            {
                // マウスのフォーカスアウトを処理します。
                m_IsMouseCursorAvailable = false;
                m_MouseCursor = Controller::GetInvalidPointer();
                m_MouseWheel = 0;
            }
        }
        else
        {
            if (!m_IsMouseCursorAvailable)
            {
                // マウスフォーカスインを処理します。
                m_IsMouseCursorAvailable = true;
                m_MouseCursor.x = static_cast<float>(state.x);
                m_MouseCursor.y = static_cast<float>(state.y);
            }
            else
            {
                // マウスの差分値を処理します。
                m_MouseWheel += state.wheelDelta;

                if (m_IsMouseCursorAbsoluteCoordinateUsed)
                {
                    m_MouseCursor.x = static_cast<float>(state.x);
                    m_MouseCursor.y = static_cast<float>(state.y);
                }
                else
                {
                    m_MouseCursor.x += static_cast<float>(state.deltaX);
                    m_MouseCursor.y += static_cast<float>(state.deltaY);
                }
            }
        }
    }

    if (m_IsMouseCursorAvailable)
    {
        m_MouseCursor = ClampPointer(
            m_MouseCursor,
            this->GetPointerBoundaryMin(), this->GetPointerBoundaryMax());
    }

    {
        // マウスの入力状態を取得します。
        const nn::hid::MouseState& state = states[0];

        // 接続状態を設定します。
        m_IsMouseConnected =
            state.attributes.Test<nn::hid::MouseAttribute::IsConnected>();

        // 新たに押下されたボタンを最新の値に更新します。
        m_MouseButtonsDown = ~m_MouseButtons & state.buttons;

        // 新たに開放されたボタンを最新の値に更新します。
        m_MouseButtonsUp = m_MouseButtons & ~state.buttons;

        // 押下中のボタンを最新の値に更新します。
        m_MouseButtons = state.buttons;

        // ダブルクリックされたマウスボタンを更新します。
        this->UpdateMouseButtonsDoubleClick();
    }
}

}} // namespace nns::hid
