﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid/hid_Button.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerManager.h>
#include <nns/hid/hid_DebugPad.h>
#include <nns/hid/hid_DeviceAssetId.h>

namespace nns { namespace hid {

namespace {

/**
 * @brief       コントローラ用に DebugPad のスティックの座標値を正規化します。
 *
 * @param[in]   x                           DebugPad のスティックの座標値です。
 *
 * @return      正規化された DebugPad のスティックの座標値です。
 */
float GetNormalizedPosition(int x)
{
    NN_ASSERT_MINMAX(x, -::nn::hid::AnalogStickMax, ::nn::hid::AnalogStickMax);

    // 境界値については丸め込み誤差が発生しないよう即値を返します。
    if (x == 0)
    {
        return 0.0f;
    }

    if (x == ::nn::hid::AnalogStickMax)
    {
        return 1.0f;
    }

    if (x == -::nn::hid::AnalogStickMax)
    {
        return -1.0f;
    }

    // 正規化した位置を返します。
    return static_cast<float>(x) / ::nn::hid::AnalogStickMax;
}

} // namespace

DebugPad::DebugPad(ControllerManager* pManager) NN_NOEXCEPT
    : Controller(pManager)
    , m_IsConnected(false)
{
    NN_ASSERT_NOT_NULL(pManager);

    // 対応する DebugPad のデバイスアセットを取得します。
    m_pDebugPadAsset = static_cast<DebugPadAsset*>(
        pManager->GetDeviceAsset(DeviceAssetId_DebugPad, 0));
}

DebugPad::~DebugPad() NN_NOEXCEPT {}

void DebugPad::Update() NN_NOEXCEPT
{
    // 対応する DebugPad のデバイスアセットが存在しない場合は何も行いません。
    if (!m_pDebugPadAsset)
    {
        return;
    }

    // DebugPad の入力状態を取得します。
    const ::nn::hid::DebugPadState state = m_pDebugPadAsset->GetDebugPadState();

    // 接続状態を設定します。
    m_IsConnected =
        state.attributes.Test<nn::hid::DebugPadAttribute::IsConnected>();

    // デジタルボタンの押下状態を対応付けます。
    const ::nn::hid::DebugPadButtonSet& debugPadButtons = state.buttons;
    ButtonSet buttons = {};
    buttons.Set<Button::A>(
        debugPadButtons.Test<nn::hid::DebugPadButton::A>());
    buttons.Set<Button::B>(
        debugPadButtons.Test<nn::hid::DebugPadButton::B>());
    buttons.Set<Button::ZL>(
        debugPadButtons.Test<nn::hid::DebugPadButton::ZL>());
    buttons.Set<Button::X>(
        debugPadButtons.Test<nn::hid::DebugPadButton::X>());
    buttons.Set<Button::Y>(
        debugPadButtons.Test<nn::hid::DebugPadButton::Y>());
    buttons.Set<Button::ZR>(
        debugPadButtons.Test<nn::hid::DebugPadButton::ZR>());
    buttons.Set<Button::Minus>(
        debugPadButtons.Test<nn::hid::DebugPadButton::Select>());
    buttons.Set<Button::Plus>(
        debugPadButtons.Test<nn::hid::DebugPadButton::Start>());
    buttons.Set<Button::Start>(
        debugPadButtons.Test<nn::hid::DebugPadButton::Start>());
    buttons.Set<Button::Select>(
        debugPadButtons.Test<nn::hid::DebugPadButton::Select>());
    buttons.Set<Button::L>(
        debugPadButtons.Test<nn::hid::DebugPadButton::L>());
    buttons.Set<Button::R>(
        debugPadButtons.Test<nn::hid::DebugPadButton::R>());
    buttons.Set<Button::Up>(
        debugPadButtons.Test<nn::hid::DebugPadButton::Up>());
    buttons.Set<Button::Down>(
        debugPadButtons.Test<nn::hid::DebugPadButton::Down>());
    buttons.Set<Button::Left>(
        debugPadButtons.Test<nn::hid::DebugPadButton::Left>());
    buttons.Set<Button::Right>(
        debugPadButtons.Test<nn::hid::DebugPadButton::Right>());

    // 左スティックの入力座標を対応付けます。
    nn::util::Float2 leftStick = {};
    leftStick.x = GetNormalizedPosition(state.analogStickL.x);
    leftStick.y = GetNormalizedPosition(state.analogStickL.y);

    // 右スティックの入力座標を対応付けます。
    nn::util::Float2 rightStick = {};
    rightStick.x = GetNormalizedPosition(state.analogStickR.x);
    rightStick.y = GetNormalizedPosition(state.analogStickR.y);

    // コントローラの基底の状態を更新します。
    this->UpdateBase(buttons,
                     leftStick,
                     rightStick,
                     false,
                     Controller::GetInvalidPointer());
}

}} // namespace nns::hid
