﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfxLogGpuBuffer.h"

namespace nns
{
namespace gfxLog
{

//----------------------------------------
bool GpuBuffer::Initialize(
    nn::gfx::Device* pDevice,
    const InitializeArg&  arg,
    nn::gfx::MemoryPool* pMemoryPool,
    size_t memoryPoolOffset
    ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(arg.m_BufferSize > 0);
    NN_ASSERT(arg.m_BufferCount >= 1);

    // バッファのアライメント値を保存
    m_Alignment = GetGpuBufferAlignement(pDevice, arg);
    // 渡されたメモリプールオフセットのアライメントチェック
    NN_ASSERT( (memoryPoolOffset & ( m_Alignment - 1 ))  == 0 );

    // アライメントしてバッファのサイズを決定
    m_Size = nn::util::align_up(arg.m_BufferSize, m_Alignment);

    // バッファの個数
    m_Count = arg.m_BufferCount;

    // nn::gfx::Bufferを確保する
    m_AllBufferSize = CalculateGpuBufferPoolSize(pDevice, arg);
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(arg.m_GpuAccessFlag);
    info.SetSize(m_AllBufferSize);
    m_Buffer.Initialize(pDevice, info, pMemoryPool, memoryPoolOffset, m_AllBufferSize);

    m_IsInitialized = true;

    return true;
}

//----------------------------------------
void GpuBuffer::Finalize(
    nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    m_Buffer.Finalize(pDevice);

    // 再利用に備えてすべてのパラメータをクリアする。
    m_Size = 0;
    m_Alignment = 0;
    m_Count = 0;
    m_AllBufferSize = 0;
    m_MappedBufferIndex = -1;
    m_pMappedPointer = NULL;
    m_IsInitialized = false;
}

//----------------------------------------
void GpuBuffer::Map(int bufferIndex) NN_NOEXCEPT
{
    if(m_MappedBufferIndex >= 0)
    {
        return;
    }

    NN_ASSERT(m_IsInitialized);

    m_MappedBufferIndex = bufferIndex;

    m_pMappedPointer = m_Buffer.Map();

    m_Raw = 0;
}

//----------------------------------------
void GpuBuffer::Unmap() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized);

    if(m_MappedBufferIndex < 0)
    {
        return;
    }

    m_Buffer.Unmap();

    m_MappedBufferIndex = -1;

    m_Raw = 0;
}

//----------------------------------------
size_t GpuBuffer::GetGpuBufferAlignement(
    nn::gfx::Device* pDevice,
    const InitializeArg& arg) NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(arg.m_GpuAccessFlag);
    return nn::gfx::Buffer::GetBufferAlignment(pDevice, info);
}

//----------------------------------------
size_t GpuBuffer::CalculateGpuBufferPoolSize(
    nn::gfx::Device* pDevice,
    const InitializeArg& arg) NN_NOEXCEPT
{
    size_t alignmentSize = GetGpuBufferAlignement(pDevice, arg);
    return nn::util::align_up(arg.m_BufferSize, alignmentSize) * arg.m_BufferCount;
}

} // namespace gfxLog
} // namespace nns
