﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif

#if NN_GFX_IS_TARGET_GL
#include <GL/glew.h>
#include <nn/gfx/gfx_Variation-api.gl.h>

#ifdef NN_BUILD_TARGET_PLATFORM_OS_NN
#include <EGL/egl.h>
#endif

#endif

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
    #if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
        #include <nvnTool/nvnTool_GlslcInterface.h>
    #endif
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
#include <nv/nv_MemoryManagement.h>
#include <cstdlib>
#endif

namespace {
#if NN_GFX_IS_TARGET_GL
    nn::gfx::ImageFormat swapChainFormat;
#endif // NN_GFX_IS_TARGET_GL
}

namespace nns {
namespace gfx {


GraphicsFramework::GraphicsFramework() NN_NOEXCEPT :
m_pDisplay(nullptr),
m_pLayer(nullptr),
m_pScanBufferViews(nullptr),
m_pScanBuffers(nullptr),
m_pDisplayFence(nullptr),
m_pGpuFence(nullptr),
m_pDisplaySemaphore(nullptr),
m_pCommandBuffer(nullptr),
m_pShaderScratchMemory(nullptr),
m_pColorBuffer(nullptr),
m_pColorTargetView(nullptr),
m_pDepthStencilBuffer(nullptr),
m_pDepthStencilView(nullptr),
m_AllocateFunction(nullptr),
m_FreeFunction(nullptr),
m_pAllocateFunctionUserData(nullptr),
m_IsInitialized(false)
{}

//------------------------------------------------------------------------------
// 初期化・終了
//------------------------------------------------------------------------------
void GraphicsFramework::InitializeGraphicsSystem(
    size_t graphicsSystemMemorySize,
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
    nn::FreeFunctionWithUserData pFreeFunction,
    ReallocateFunctionWithUserData pReallocateFunction,
    void* pAllocateFunctionUserData
) NN_NOEXCEPT
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        if ((pAllocateFunction != nullptr) && (pFreeFunction != nullptr) && (pReallocateFunction != nullptr)) {
            void* memory = pAllocateFunction(graphicsSystemMemorySize, 1, pAllocateFunctionUserData);
            nv::SetGraphicsAllocator(pAllocateFunction, pFreeFunction, pReallocateFunction, pAllocateFunctionUserData);
            nv::SetGraphicsDevtoolsAllocator(pAllocateFunction, pFreeFunction, pReallocateFunction, pAllocateFunctionUserData);
            nv::InitializeGraphics(memory, graphicsSystemMemorySize);
        }
        else {
            void* memory = DefaultAllocateFunction(graphicsSystemMemorySize, 1, nullptr);
            nv::SetGraphicsAllocator(DefaultAllocateFunction, DefaultFreeFunction, DefaultReallocateFunction, nullptr);
            nv::SetGraphicsDevtoolsAllocator(DefaultAllocateFunction, DefaultFreeFunction, DefaultReallocateFunction, nullptr);
            nv::InitializeGraphics(memory, graphicsSystemMemorySize);
        }
    }
#else
    NN_UNUSED(graphicsSystemMemorySize);
    NN_UNUSED(pAllocateFunction);
    NN_UNUSED(pFreeFunction);
    NN_UNUSED(pReallocateFunction);
    NN_UNUSED(pAllocateFunctionUserData);
#endif

#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    // GLSLC のためのメモリアロケータを設定します。
    if ((pAllocateFunction != nullptr) && (pFreeFunction != nullptr) && (pReallocateFunction != nullptr)) {
        glslcSetAllocator(pAllocateFunction, pFreeFunction, pReallocateFunction, pAllocateFunctionUserData);
    }
    else {
        glslcSetAllocator(DefaultAllocateFunction, DefaultFreeFunction, DefaultReallocateFunction, nullptr);
    }
#endif
}

void GraphicsFramework::InitializeGraphicsSystem(
    size_t graphicsSystemMemorySize
) NN_NOEXCEPT
{
    return InitializeGraphicsSystem(
        graphicsSystemMemorySize,
        DefaultAllocateFunction,
        DefaultFreeFunction,
        DefaultReallocateFunction,
        nullptr
    );
}

void GraphicsFramework::Initialize(
    const FrameworkInfo& info,
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
    nn::FreeFunctionWithUserData pFreeFunction,
    void* pAllocateFunctionUserData
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());

    if ((pAllocateFunction != nullptr) && (pFreeFunction != nullptr)) {
        m_AllocateFunction = pAllocateFunction;
        m_FreeFunction = pFreeFunction;
        m_pAllocateFunctionUserData = pAllocateFunctionUserData;
    }
    else {
        m_AllocateFunction = DefaultAllocateFunction;
        m_FreeFunction = DefaultFreeFunction;
        m_pAllocateFunctionUserData = nullptr;
    }

    m_DisplayWidth = info.GetDisplayWidth();
    m_DisplayHeight = info.GetDisplayHeight();
    m_BufferCount = info.GetBufferCount();
    m_SwapChainBufferCount = info.GetSwapChainBufferCount();
    m_NextScanBufferIndex = 0;

    if (m_DisplayWidth <= 0)
    {
        m_DisplayWidth = DefaultDisplayWidth;
    }
    if (m_DisplayHeight <= 0)
    {
        m_DisplayHeight = DefaultDisplayHeight;
    }

    // VI
    {
        // VI 初期化
        nn::vi::Initialize();

        // ディスプレイ, レイヤー
        nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
        NN_ASSERT(result.IsSuccess());
        NN_UNUSED(result);

        result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
        NN_ASSERT(result.IsSuccess());

        result = nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);
        NN_ASSERT(result.IsSuccess());
    }

    // GFX
    {
        // GFX 初期化
        nn::gfx::Initialize();

        // Device
        {
            nn::gfx::Device::InfoType deviceInfo;
            deviceInfo.SetDefault();
            deviceInfo.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
            deviceInfo.SetDebugMode(info.GetDebugMode());
            m_Device.Initialize(deviceInfo);
        }

        // デプスモード、ウインドウ座標 ( 0,0 )位置モード の設定
        nn::gfx::util::ScalableViewport::OriginMode originMode = info.GetPhysicalWindowCoordinate().GetOriginMode();

#if NN_GFX_IS_TARGET_NVN
        // 深度モード
        if (info.GetDepthMode() == DepthMode_Near_Is_Zero)
        {
            nvnDeviceSetDepthMode(m_Device.ToData()->pNvnDevice, NVN_DEPTH_MODE_NEAR_IS_ZERO);
        }
        else if (info.GetDepthMode() == DepthMode_Near_Is_Minus_W)
        {
            nvnDeviceSetDepthMode(m_Device.ToData()->pNvnDevice, NVN_DEPTH_MODE_NEAR_IS_MINUS_W);
        }

        // ウィンドウ原点モード
        if (originMode == nn::gfx::util::ScalableViewport::OriginMode_LowerLeft)
        {
            nvnDeviceSetWindowOriginMode(m_Device.ToData()->pNvnDevice, NVN_WINDOW_ORIGIN_MODE_LOWER_LEFT);
        }
        else if (originMode == nn::gfx::util::ScalableViewport::OriginMode_UpperLeft)
        {
            nvnDeviceSetWindowOriginMode(m_Device.ToData()->pNvnDevice, NVN_WINDOW_ORIGIN_MODE_UPPER_LEFT);
        }
#endif

        // Queue
        {
            nn::gfx::Queue::InfoType queueInfo;
            queueInfo.SetDefault();
            queueInfo.SetCapability(nn::gfx::QueueCapability_Graphics
                | nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Copy);
            m_Queue.Initialize(&m_Device, queueInfo);
        }

#if NN_GFX_IS_TARGET_GL
        GLenum windowOriginMode = GL_LOWER_LEFT;
        GLenum depthMode = GL_NEGATIVE_ONE_TO_ONE;

        if (info.GetDepthMode() == DepthMode_Near_Is_Zero)
        {
            depthMode = GL_ZERO_TO_ONE;
        }

        if (originMode == nn::gfx::util::ScalableViewport::OriginMode_UpperLeft)
        {
            windowOriginMode = GL_UPPER_LEFT;
        }

#ifdef NN_BUILD_TARGET_PLATFORM_OS_WIN
        BOOL result = TRUE;
        HDC hDc = m_Queue.ToData()->hDc;
        HGLRC hGlRc = m_Queue.ToData()->renderingContext.hGlRc;
        ::wglMakeCurrent(hDc, hGlRc);
        glClipControl(windowOriginMode, depthMode);
        auto error = glGetError();
        NN_ASSERT(error == GL_NO_ERROR);
        result = ::wglMakeCurrent(NULL, NULL);
        NN_ASSERT(result == TRUE);
#else // NN_BUILD_TARGET_PLATFORM_OS_NN
        nn::gfx::Queue::DataType& data = m_Queue.ToData();
        EGLBoolean result = eglMakeCurrent(data.hDisplay, data.hSurface, data.hSurface, data.renderingContext.hGlRc);
        NN_ASSERT(result);
        glClipControl(windowOriginMode, depthMode);
        GLenum error = glGetError();
        NN_ASSERT(error == GL_NO_ERROR);
#endif // NN_BUILD_TARGET_PLATFORM_OS_WIN
#endif

        // DisplayFence
        {
            nn::gfx::Fence::InfoType fenceInfo;
            fenceInfo.SetDefault();

            int count = m_BufferCount;
            m_pDisplayFence = reinterpret_cast<nn::gfx::Fence*>(m_AllocateFunction(sizeof(nn::gfx::Fence) * count, NN_ALIGNOF(nn::gfx::Fence), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pDisplayFence);
            for (int i = 0; i < count; i++) {
                new(&m_pDisplayFence[i])nn::gfx::Fence;

                m_pDisplayFence[i].Initialize(&m_Device, fenceInfo);
            }
        }
        // GpuFence
        {
            nn::gfx::Fence::InfoType fenceInfo;
            fenceInfo.SetDefault();

            int count = m_BufferCount;
            m_pGpuFence = reinterpret_cast<nn::gfx::Fence*>(m_AllocateFunction(sizeof(nn::gfx::Fence) * count, NN_ALIGNOF(nn::gfx::Fence), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pGpuFence);
            for (int i = 0; i < count; i++) {
                new(&m_pGpuFence[i])nn::gfx::Fence;

                m_pGpuFence[i].Initialize(&m_Device, fenceInfo);
            }
        }

        // DisplaySemaphore
        {
            nn::gfx::Semaphore::InfoType semaphoreInfo;
            semaphoreInfo.SetDefault();

            int count = m_BufferCount;
            m_pDisplaySemaphore = reinterpret_cast<nn::gfx::Semaphore*>(m_AllocateFunction(sizeof(nn::gfx::Semaphore) * count, NN_ALIGNOF(nn::gfx::Semaphore), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pDisplaySemaphore);
            for (int i = 0; i < count; i++) {
                new(&m_pDisplaySemaphore[i])nn::gfx::Semaphore;

                m_pDisplaySemaphore[i].Initialize(&m_Device, semaphoreInfo);
            }
        }

        // MemoryPool
        {
            nn::gfx::MemoryPool::InfoType memorypoolInfo;
            int memoryPoolProperty[MemoryPoolType_End] = {
                /* CommandBuffer */     nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached,
                /* RenderTarget */      nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible,
                /* ConstantBuffer */    nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
                /* Shader */            nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_ShaderCode,
                /* Data */              nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached,
                /* Others */            nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible | nn::gfx::MemoryPoolProperty_ShaderCode,
            };

            for (int i = 0; i < MemoryPoolType_End; i++)
            {
                memorypoolInfo.SetDefault();
                memorypoolInfo.SetMemoryPoolProperty(memoryPoolProperty[i]);

                size_t size = info.GetMemoryPoolSize(static_cast<MemoryPoolType>(i));
                if (size == 0) {
                    m_MemoryPool[i].size = 0;
                    continue;
                }
                size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, memorypoolInfo);
                void* pAlignedMemory = m_AllocateFunction(size, alignment, m_pAllocateFunctionUserData);
                NN_ASSERT_NOT_NULL(pAlignedMemory);

                memorypoolInfo.SetPoolMemory(pAlignedMemory, nn::util::align_down(size,
                    nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, memorypoolInfo)));
                m_MemoryPool[i].object.Initialize(&m_Device, memorypoolInfo);

                m_MemoryPool[i].allocator.Initialize(
                    MemoryPoolAllocatorMalloc,
                    reinterpret_cast<void*>(this),
                    MemoryPoolAllocatorFree,
                    reinterpret_cast<void*>(this),
                    &m_MemoryPool[i].object,
                    0,
                    size,
                    nn::gfx::util::MemoryPoolAllocator::AlignmentMax,
                    true
                );
                NN_ASSERT(m_MemoryPool[i].allocator.IsInitialized());

                m_MemoryPool[i].pMemory = pAlignedMemory;
                m_MemoryPool[i].size = size;
            }
        }

        // DescriptorPool
        {
            int descriptorBaseIndex[nn::gfx::DescriptorPoolType_End];
            descriptorBaseIndex[nn::gfx::DescriptorPoolType_BufferView] = 0;
            descriptorBaseIndex[nn::gfx::DescriptorPoolType_TextureView] = 0;
            descriptorBaseIndex[nn::gfx::DescriptorPoolType_Sampler] = 0;
#if NN_GFX_IS_TARGET_NVN
            nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(m_Device);
            nvnDeviceGetInteger(deviceData.pNvnDevice,
                NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &descriptorBaseIndex[nn::gfx::DescriptorPoolType_TextureView]);
            nvnDeviceGetInteger(deviceData.pNvnDevice,
                NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &descriptorBaseIndex[nn::gfx::DescriptorPoolType_Sampler]);
#endif
            nn::gfx::DescriptorPool::InfoType descriptorpoolInfo;

            for (int i = 0; i < nn::gfx::DescriptorPoolType_End; i++)
            {
                nn::gfx::DescriptorPoolType descriptorpoolType = static_cast<nn::gfx::DescriptorPoolType>(i);
                m_DescriptorPoolSlotCount[i] = info.GetDescriptorPoolSlotCount(descriptorpoolType);
                if (m_DescriptorPoolSlotCount[i] > 0) {
                    NN_ASSERT_GREATER(m_DescriptorPoolSlotCount[i], descriptorBaseIndex[i]);

                    descriptorpoolInfo.SetDefault();
                    descriptorpoolInfo.SetDescriptorPoolType(descriptorpoolType);
                    descriptorpoolInfo.SetSlotCount(m_DescriptorPoolSlotCount[i]);

                    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, descriptorpoolInfo);
                    MemoryPoolType memorypoolType = MemoryPoolType_ConstantBuffer;
                    size_t alignment = nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, descriptorpoolInfo);
                    ptrdiff_t offset = m_MemoryPool[memorypoolType].allocator.Allocate(size, alignment);
                    NN_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);

                    m_DescriptorPool[i].Initialize(&m_Device, descriptorpoolInfo, &m_MemoryPool[memorypoolType].object, offset, size);

                    m_DescriptorPoolAllocator[i].Initialize(
                        DescriptorPoolAllocatorMalloc,
                        reinterpret_cast<void*>(this),
                        DescriptorPoolAllocatorFree,
                        reinterpret_cast<void*>(this),
                        &m_DescriptorPool[i],
                        descriptorBaseIndex[i],
                        m_DescriptorPoolSlotCount[i] - descriptorBaseIndex[i],
                        true
                    );
                    NN_ASSERT(m_DescriptorPoolAllocator[i].IsInitialized());
                }
            }
        }

        // RootCommandBuffer
        {
            int count = m_BufferCount;
            m_pCommandBuffer = reinterpret_cast<GraphicsFramework::CommandBuffer*>(m_AllocateFunction(sizeof(GraphicsFramework::CommandBuffer) * count, NN_ALIGNOF(GraphicsFramework::CommandBuffer), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pCommandBuffer);
            for (int i = 0; i < count; i++) {
                new(&m_pCommandBuffer[i])GraphicsFramework::CommandBuffer;
                m_pCommandBuffer[i].commandMemorySize = info.GetRootCommandBufferCommandMemorySize();
                m_pCommandBuffer[i].controlMemorySize = info.GetRootCommandBufferControlMemorySize();
            }
            m_ShaderScratchMemorySize = 0;

            // CommandBuffer
            for (int i = 0; i < count; i++) {
                {
                    nn::gfx::CommandBuffer::InfoType commandbufferInfo;
                    commandbufferInfo.SetDefault();
                    commandbufferInfo.SetQueueCapability(nn::gfx::QueueCapability_Graphics
                        | nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Copy);
                    commandbufferInfo.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);

                    m_pCommandBuffer[i].object.Initialize(&m_Device, commandbufferInfo);
                }

                // CommandMemory
                {
                    MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
                    size_t alignment = nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device);
                    size_t size = m_pCommandBuffer[i].commandMemorySize;
                    m_pCommandBuffer[i].commandMemoryOffset = m_MemoryPool[memorypoolType].allocator.Allocate(size, alignment);
                    NN_ASSERT(m_pCommandBuffer[i].commandMemoryOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                }

                // ControlMemory
                {
                    size_t alignment = nn::gfx::CommandBuffer::GetControlMemoryAlignment(&m_Device);
                    size_t size = m_pCommandBuffer[i].controlMemorySize;
                    m_pCommandBuffer[i].pControlMemory = m_AllocateFunction(size, alignment, m_pAllocateFunctionUserData);
                    NN_ASSERT_NOT_NULL(m_pCommandBuffer[i].pControlMemory);
                }
            }
        }

        // ShaderScratchMemory
        {
            int count = m_BufferCount;
            m_pShaderScratchMemory = reinterpret_cast<ShaderScratchMemoryInfo*>(m_AllocateFunction(sizeof(ShaderScratchMemoryInfo) * count, NN_ALIGNOF(ShaderScratchMemoryInfo), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pShaderScratchMemory);
            for (int i = 0; i < count; i++) {
                new(&m_pShaderScratchMemory[i])ShaderScratchMemoryInfo;
                m_pShaderScratchMemory[i].offset = nn::gfx::util::MemoryPoolAllocator::InvalidOffset;
                m_pShaderScratchMemory[i].size = 0;
            }
        }

        // SwapChain
        {
            nn::gfx::SwapChain::InfoType swapchainInfo;
            swapchainInfo.SetDefault();
            swapchainInfo.SetLayer(m_pLayer);
            swapchainInfo.SetWidth(m_DisplayWidth);
            swapchainInfo.SetHeight(m_DisplayHeight);
            swapchainInfo.SetFormat(info.GetSwapChainFormat());
            swapchainInfo.SetBufferCount(m_SwapChainBufferCount);
            {
                MemoryPoolType type = MemoryPoolType_RenderTarget;
                size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, swapchainInfo);
                size_t alignment = nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, swapchainInfo);
                ptrdiff_t offset = m_MemoryPool[type].allocator.Allocate(size, alignment);
                NN_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                m_SwapChain.Initialize(&m_Device, swapchainInfo, &m_MemoryPool[type].object, offset, size);
            }

#if NN_GFX_IS_TARGET_GL
            swapChainFormat = info.GetSwapChainFormat();
#endif
        }

        // ScanBuffer
        {
            int count = m_SwapChainBufferCount;
            m_pScanBufferViews = reinterpret_cast<nn::gfx::ColorTargetView**>(m_AllocateFunction(sizeof(nn::gfx::ColorTargetView*) * count, NN_ALIGNOF(nn::gfx::ColorTargetView*), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pScanBufferViews);
            m_pScanBuffers = reinterpret_cast<nn::gfx::Texture**>(m_AllocateFunction(sizeof(nn::gfx::Texture*) * count, NN_ALIGNOF(nn::gfx::Texture*), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pScanBuffers);

            m_SwapChain.GetScanBufferViews(m_pScanBufferViews, count);
            m_SwapChain.GetScanBuffers(m_pScanBuffers, count);
        }

        if (info.IsColorBufferEnabled())
        {
            // ColorBuffer
            nn::gfx::Texture::InfoType textureInfo;
            textureInfo.SetDefault();
            textureInfo.SetWidth(m_DisplayWidth);
            textureInfo.SetHeight(m_DisplayHeight);
            textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
            textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
            textureInfo.SetImageFormat(info.GetColorBufferFormat());
            textureInfo.SetMipCount(1);
            textureInfo.SetDepth(1);

            MemoryPoolType type = MemoryPoolType_RenderTarget;
            size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, textureInfo);
            size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, textureInfo);

            int count = 1; //m_BufferCount;
            m_pColorBuffer = reinterpret_cast<nn::gfx::Texture*>(m_AllocateFunction(sizeof(nn::gfx::Texture) * count, NN_ALIGNOF(nn::gfx::Texture), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pColorBuffer);
            for (int i = 0; i < count; i++) {
                new(&m_pColorBuffer[i])nn::gfx::Texture;
                {
                    ptrdiff_t offset = m_MemoryPool[type].allocator.Allocate(size, alignment);
                    NN_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                    m_pColorBuffer[i].Initialize(&m_Device, textureInfo, &m_MemoryPool[type].object, offset, size);
                }
            }

            // ColorTargetView
            nn::gfx::ColorTargetView::InfoType colorTargetViewInfo;
            colorTargetViewInfo.SetDefault();
            colorTargetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
            colorTargetViewInfo.SetImageFormat(info.GetColorBufferFormat());

            m_pColorTargetView = reinterpret_cast<nn::gfx::ColorTargetView*>(m_AllocateFunction(sizeof(nn::gfx::ColorTargetView) * count, NN_ALIGNOF(nn::gfx::ColorTargetView), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pColorTargetView);
            for (int i = 0; i < count; i++) {
                new(&m_pColorTargetView[i])nn::gfx::ColorTargetView;
                colorTargetViewInfo.SetTexturePtr(&m_pColorBuffer[i]);
                m_pColorTargetView[i].Initialize(&m_Device, colorTargetViewInfo);
            }
        }
        else
        {
            m_pColorBuffer = nullptr;
            m_pColorTargetView = nullptr;
        }

        if (info.IsDepthStencilBufferEnabled())
        {
            // DepthStencilBuffer
            nn::gfx::Texture::InfoType textureInfo;
            textureInfo.SetDefault();
            textureInfo.SetWidth(m_DisplayWidth);
            textureInfo.SetHeight(m_DisplayHeight);
            textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
            textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
            textureInfo.SetImageFormat(info.GetDepthStencilBufferFormat());
            textureInfo.SetMipCount(1);
            textureInfo.SetDepth(1);

            MemoryPoolType type = MemoryPoolType_RenderTarget;
            size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, textureInfo);
            size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, textureInfo);

            int count = 1; //m_BufferCount;
            m_pDepthStencilBuffer = reinterpret_cast<nn::gfx::Texture*>(m_AllocateFunction(sizeof(nn::gfx::Texture) * count, NN_ALIGNOF(nn::gfx::Texture), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pDepthStencilBuffer);
            for (int i = 0; i < count; i++) {
                new(&m_pDepthStencilBuffer[i])nn::gfx::Texture;
                {
                    ptrdiff_t offset = m_MemoryPool[type].allocator.Allocate(size, alignment);
                    NN_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                    m_pDepthStencilBuffer[i].Initialize(&m_Device, textureInfo, &m_MemoryPool[type].object, offset, size);
                }
            }

            // DepthStencilView
            nn::gfx::DepthStencilView::InfoType depthStencilViewInfo;
            depthStencilViewInfo.SetDefault();
            depthStencilViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);

            m_pDepthStencilView = reinterpret_cast<nn::gfx::DepthStencilView*>(m_AllocateFunction(sizeof(nn::gfx::DepthStencilView) * count, NN_ALIGNOF(nn::gfx::DepthStencilView), m_pAllocateFunctionUserData));
            NN_ASSERT_NOT_NULL(m_pDepthStencilView);
            for (int i = 0; i < count; i++) {
                new(&m_pDepthStencilView[i])nn::gfx::DepthStencilView;
                depthStencilViewInfo.SetTexturePtr(&m_pDepthStencilBuffer[i]);
                m_pDepthStencilView[i].Initialize(&m_Device, depthStencilViewInfo);
            }
        }
        else
        {
            m_pDepthStencilBuffer = nullptr;
            m_pDepthStencilView = nullptr;
        }
    }

    m_IsInitialized = true;

    // その他組み込み GFX オブジェクト
    {
        // ScalableViewport
        m_ScalableViewport.Initialize(info.GetVirtualWindowCoordinate(), info.GetPhysicalWindowCoordinate());

        // 描画する領域を設定
        ResizeWindow();

        // ViewportScissorState
        IntializeViewportScissorState();

        // VertexState
        {
            nn::gfx::VertexState::InfoType vertexInfo;

            // VertexStateType_Float3_Float2
            vertexInfo.SetDefault();
            ptrdiff_t stride = sizeof(float) * 5;
            nn::gfx::VertexAttributeStateInfo attribs[2];
            {
                attribs[0].SetDefault();
                //attribs[0].SetNamePtr("i_Position");
                attribs[0].SetBufferIndex(0);
                attribs[0].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
                attribs[0].SetOffset(0);
                attribs[0].SetShaderSlot(0);
            }
            {
                attribs[1].SetDefault();
                //attribs[1].SetNamePtr("i_TexCoord");
                attribs[1].SetBufferIndex(0);
                attribs[1].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
                attribs[1].SetOffset(sizeof(float) * 3);
                attribs[1].SetShaderSlot(1);
            }
            nn::gfx::VertexBufferStateInfo buffer;
            {
                buffer.SetDefault();
                buffer.SetStride(stride);
            }
            vertexInfo.SetVertexAttributeStateInfoArray(attribs, 2);
            vertexInfo.SetVertexBufferStateInfoArray(&buffer, 1);

            InitializeVertexState(&m_VertexState[VertexStateType_Float3_Float2], vertexInfo);
        }

        // RasterizerState
        {
            nn::gfx::RasterizerState::InfoType rasterizerInfo;

            // RasterizerStateType_FillSolid_CullNone
            rasterizerInfo.SetDefault();
            rasterizerInfo.SetCullMode(nn::gfx::CullMode_None);
            rasterizerInfo.SetScissorEnabled(true);
            rasterizerInfo.SetDepthClipEnabled(false);
            m_RasterizerState[RasterizerStateType_FillSolid_CullNone].Initialize(GetDevice(), rasterizerInfo);
        }

        // BlendState
        {
            nn::gfx::BlendState::InfoType blendInfo;
            nn::gfx::BlendTargetStateInfo targetInfo;

            // BlendStateType_Disabled
            blendInfo.SetDefault();
            targetInfo.SetDefault();
            targetInfo.SetBlendEnabled(false);
            blendInfo.SetBlendTargetStateInfoArray(&targetInfo, 1);
            InitializeBlendState(&m_BlendState[BlendStateType_Disabled], blendInfo);

            // BlendStateType_Alpha
            blendInfo.SetDefault();
            targetInfo.SetDefault();
            targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue);
            targetInfo.SetBlendEnabled(true);
            targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
            targetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
            targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
            blendInfo.SetBlendTargetStateInfoArray(&targetInfo, 1);
            InitializeBlendState(&m_BlendState[BlendStateType_Alpha], blendInfo);
        }

        // DepthStencilState
        {
            nn::gfx::DepthStencilState::InfoType depthStencilInfo;

            // DepthStencilStateType_Disabled
            depthStencilInfo.SetDefault();
            depthStencilInfo.SetDepthTestEnabled(false);
            depthStencilInfo.SetDepthWriteEnabled(false);
            m_DepthStencilState[DepthStencilStateType_Disabled].Initialize(GetDevice(), depthStencilInfo);
        }

        // Sampler
        {
            nn::gfx::Sampler::InfoType samplerInfo;

            // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Repeat
            samplerInfo.SetDefault();
            samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
            samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Repeat);
            samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Repeat);
            samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Repeat);
            m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Repeat].Initialize(GetDevice(), samplerInfo);

            // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Mirror
            samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
            samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
            samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
            m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Mirror].Initialize(GetDevice(), samplerInfo);

            // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp
            samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToEdge);
            samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToEdge);
            samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
            m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp].Initialize(GetDevice(), samplerInfo);

            // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Border
            samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToBorder);
            samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToBorder);
            samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToBorder);
            m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Border].Initialize(GetDevice(), samplerInfo);
        }
    }

    // フレーム処理用
    m_FrameworkMode = FrameworkMode_DeferredSubmission;
    m_pCalculateCallback = nullptr;
    m_pCalculateCallbackUserData = nullptr;
    m_pMakeCommandCallback = nullptr;
    m_pMakeCommandCallbackUserData = nullptr;
    m_PresentInterval = 1;
    m_WaitSyncTimeout = nn::TimeSpan::FromSeconds(1);
    m_CurrentFrameIndex = 0;
    m_PrevFrameworkMode = m_FrameworkMode;
}// NOLINT(impl/function_size)

void GraphicsFramework::Initialize(
    const FrameworkInfo& info
) NN_NOEXCEPT
{
    Initialize(
        info,
        DefaultAllocateFunction,
        DefaultFreeFunction,
        nullptr
    );
}

void GraphicsFramework::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // GFX 終了
    {
        // m_MemoryPoolAllocator, m_DescriptorPoolAllocator は Finalize() するので
        // 確保していた offset, index の個々の Free() は不要

        for (int i = 0; i < SamplerType_End; i++) {
            m_Sampler[i].Finalize(GetDevice());
        }
        for (int i = 0; i < DepthStencilStateType_End; i++) {
            m_DepthStencilState[i].Finalize(GetDevice());
        }
        for (int i = 0; i < BlendStateType_End; i++) {
            FinalizeBlendState(&m_BlendState[i]);
        }
        for (int i = 0; i < RasterizerStateType_End; i++) {
            m_RasterizerState[i].Finalize(GetDevice());
        }
        for (int i = 0; i < VertexStateType_End; i++) {
            FinalizeVertexState(&m_VertexState[i]);
        }
        FinalizeViewportScissorState();
        m_ScalableViewport.Finalize();
        {
            m_FreeFunction(m_pScanBufferViews, m_pAllocateFunctionUserData);
            m_FreeFunction(m_pScanBuffers, m_pAllocateFunctionUserData);
        }
        m_SwapChain.Finalize(&m_Device);

        if (m_pDepthStencilBuffer != nullptr) {
            for (int i = 0; i < 1/*m_BufferCount*/; i++) {
                m_pDepthStencilView[i].Finalize(&m_Device);
                m_pDepthStencilBuffer[i].Finalize(&m_Device);
            }
            m_FreeFunction(m_pDepthStencilView, m_pAllocateFunctionUserData);
            m_FreeFunction(m_pDepthStencilBuffer, m_pAllocateFunctionUserData);
        }
        if (m_pColorBuffer != nullptr) {
            for (int i = 0; i < 1/*m_BufferCount*/; i++) {
                m_pColorTargetView[i].Finalize(&m_Device);
                m_pColorBuffer[i].Finalize(&m_Device);
            }
            m_FreeFunction(m_pColorTargetView, m_pAllocateFunctionUserData);
            m_FreeFunction(m_pColorBuffer, m_pAllocateFunctionUserData);
        }
        for (int i = 0; i < m_BufferCount; i++) {
            m_pCommandBuffer[i].object.Finalize(&m_Device);
            m_FreeFunction(m_pCommandBuffer[i].pControlMemory, m_pAllocateFunctionUserData);
        }
        m_FreeFunction(m_pCommandBuffer, m_pAllocateFunctionUserData);
        m_FreeFunction(m_pShaderScratchMemory, m_pAllocateFunctionUserData);

        for (int i = 0; i < nn::gfx::DescriptorPoolType_End; i++)
        {
            if (m_DescriptorPoolSlotCount[i] > 0) {
                m_DescriptorPool[i].Finalize(&m_Device);
                m_DescriptorPoolAllocator[i].Finalize();
            }
            m_DescriptorPoolSlotCount[i] = 0;
        }

        for (int i = 0; i < MemoryPoolType_End; i++)
        {
            if (m_MemoryPool[i].size == 0) {
                continue;
            }
            m_MemoryPool[i].allocator.Finalize();
            m_MemoryPool[i].object.Finalize(&m_Device);
            m_MemoryPool[i].size = 0;
            m_FreeFunction(m_MemoryPool[i].pMemory, m_pAllocateFunctionUserData);
        }

        for (int i = 0; i < m_BufferCount; i++) {
            m_pDisplayFence[i].Finalize(&m_Device);
        }
        m_FreeFunction(m_pDisplayFence, m_pAllocateFunctionUserData);

        for (int i = 0; i < m_BufferCount; i++) {
            m_pGpuFence[i].Finalize(&m_Device);
        }
        m_FreeFunction(m_pGpuFence, m_pAllocateFunctionUserData);

        for (int i = 0; i < m_BufferCount; i++) {
            m_pDisplaySemaphore[i].Finalize(&m_Device);
        }
        m_FreeFunction(m_pDisplaySemaphore, m_pAllocateFunctionUserData);

        m_Queue.Finalize(&m_Device);
        m_Device.Finalize();
        nn::gfx::Finalize();
    }

    // VI 終了
    {
        nn::vi::DestroyLayer(m_pLayer);
        nn::vi::CloseDisplay(m_pDisplay);
        nn::vi::Finalize();
    }

    m_AllocateFunction = nullptr;
    m_FreeFunction = nullptr;
    m_pAllocateFunctionUserData = nullptr;

    m_IsInitialized = false;
}

bool GraphicsFramework::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

//------------------------------------------------------------------------------
// 初期化パラメータ取得
//------------------------------------------------------------------------------
int GraphicsFramework::GetDisplayWidth() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_DisplayWidth;
}

int GraphicsFramework::GetDisplayHeight() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_DisplayHeight;
}

int GraphicsFramework::GetBufferCount() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_BufferCount;
}

nn::AlignedAllocateFunctionWithUserData GraphicsFramework::GetAllocateFunction() const NN_NOEXCEPT
{
    //NN_SDK_REQUIRES(IsInitialized());
    return m_AllocateFunction;
}

nn::FreeFunctionWithUserData GraphicsFramework::GetFreeFunction() const NN_NOEXCEPT
{
    //NN_SDK_REQUIRES(IsInitialized());
    return m_FreeFunction;
}

void* GraphicsFramework::GetAllocateFunctionUserData() const NN_NOEXCEPT
{
    //NN_SDK_REQUIRES(IsInitialized());
    return m_pAllocateFunctionUserData;
}

//------------------------------------------------------------------------------
// メモリアロケータ(初期化時に渡したアロケータ関数を使用)
//------------------------------------------------------------------------------
void* GraphicsFramework::AllocateMemory(size_t size, size_t alignment) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_AllocateFunction(size, alignment, m_pAllocateFunctionUserData);
}

void GraphicsFramework::FreeMemory(void* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_FreeFunction(ptr, m_pAllocateFunctionUserData);
}

//------------------------------------------------------------------------------
// 内部 GFX オブジェクト取得
//------------------------------------------------------------------------------
nn::vi::Display* GraphicsFramework::GetDisplay() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pDisplay;
}

nn::vi::Layer* GraphicsFramework::GetLayer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pLayer;
}

nn::gfx::Device* GraphicsFramework::GetDevice() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_Device;
}

nn::gfx::Queue* GraphicsFramework::GetQueue() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_Queue;
}

nn::gfx::SwapChain* GraphicsFramework::GetSwapChain() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_SwapChain;
}

nn::gfx::Fence* GraphicsFramework::GetDisplayFence(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_pDisplayFence[bufferIndex];
}

nn::gfx::Fence* GraphicsFramework::GetGpuFence(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_pGpuFence[bufferIndex];
}

nn::gfx::Semaphore* GraphicsFramework::GetDisplaySemaphore(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_pDisplaySemaphore[bufferIndex];
}

nn::gfx::ColorTargetView* GraphicsFramework::GetScanBufferView(int scanBufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES((0 <= scanBufferIndex) && (scanBufferIndex < m_SwapChainBufferCount));
    return m_pScanBufferViews[scanBufferIndex];
}

nn::gfx::Texture* GraphicsFramework::GetScanBuffer(int scanBufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES((0 <= scanBufferIndex) && (scanBufferIndex < m_SwapChainBufferCount));
    return m_pScanBuffers[scanBufferIndex];
}

nn::gfx::ViewportScissorState* GraphicsFramework::GetViewportScissorState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_ViewportScissorState;
}

nn::gfx::VertexState* GraphicsFramework::GetVertexState(VertexStateType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_VertexState[type].object;
}

nn::gfx::RasterizerState* GraphicsFramework::GetRasterizerState(RasterizerStateType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_RasterizerState[type];
}

nn::gfx::BlendState* GraphicsFramework::GetBlendState(BlendStateType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_BlendState[type].object;
}

nn::gfx::DepthStencilState* GraphicsFramework::GetDepthStencilState(DepthStencilStateType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_DepthStencilState[type];
}

nn::gfx::Sampler* GraphicsFramework::GetSampler(SamplerType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_Sampler[type];
}

const nn::gfx::util::ScalableViewport* GraphicsFramework::GetScalableViewport() const NN_NOEXCEPT
{
    return &m_ScalableViewport;
}

//------------------------------------------------------------------------------
// カラーバッファ, 深度ステンシルバッファ
//------------------------------------------------------------------------------
nn::gfx::Texture* GraphicsFramework::GetColorBuffer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    if (m_pColorBuffer == nullptr) {
        return nullptr;
    }
    return &m_pColorBuffer[0];
}

nn::gfx::ColorTargetView* GraphicsFramework::GetColorTargetView() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    if (m_pColorTargetView == nullptr) {
        return nullptr;
    }
    return &m_pColorTargetView[0];
}

nn::gfx::Texture* GraphicsFramework::GetDepthStencilBuffer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    if (m_pDepthStencilBuffer == nullptr) {
        return nullptr;
    }
    return &m_pDepthStencilBuffer[0];
}

nn::gfx::DepthStencilView* GraphicsFramework::GetDepthStencilView() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    if (m_pDepthStencilView == nullptr) {
        return nullptr;
    }
    return &m_pDepthStencilView[0];
}

//------------------------------------------------------------------------------
// メモリプール
//------------------------------------------------------------------------------
nn::gfx::MemoryPool* GraphicsFramework::GetMemoryPool(MemoryPoolType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_MemoryPool[type].size > 0);
    return &m_MemoryPool[type].object;
}

nn::gfx::util::MemoryPoolAllocator* GraphicsFramework::GetMemoryPoolAllocator(MemoryPoolType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_MemoryPool[type].size > 0);
    return &m_MemoryPool[type].allocator;
}

ptrdiff_t GraphicsFramework::AllocatePoolMemory(MemoryPoolType type, size_t size, size_t alignment) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_MemoryPool[type].size > 0);
    return m_MemoryPool[type].allocator.Allocate(size, alignment);
}

void GraphicsFramework::FreePoolMemory(MemoryPoolType type, ptrdiff_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_MemoryPool[type].size > 0);
    m_MemoryPool[type].allocator.Free(offset);
}

//------------------------------------------------------------------------------
// デスクリプタプール
//------------------------------------------------------------------------------
nn::gfx::DescriptorPool* GraphicsFramework::GetDescriptorPool(nn::gfx::DescriptorPoolType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[type] > 0);
    return &m_DescriptorPool[type];
}

nn::gfx::util::DescriptorPoolAllocator* GraphicsFramework::GetDescriptorPoolAllocator(nn::gfx::DescriptorPoolType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[type] > 0);
    return &m_DescriptorPoolAllocator[type];
}

int GraphicsFramework::AllocateDescriptorSlot(nn::gfx::DescriptorPoolType type, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[type] > 0);
    return m_DescriptorPoolAllocator[type].Allocate(count);
}

void GraphicsFramework::FreeDescriptorSlot(nn::gfx::DescriptorPoolType type, int indexSlot) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[type] > 0);
    m_DescriptorPoolAllocator[type].Free(indexSlot);
}

void GraphicsFramework::SetBufferViewToDescriptorPool(int indexSlot, const nn::gfx::GpuAddress& gpuAddress, size_t size)
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_BufferView] > 0);

    nn::gfx::DescriptorPool& bufferDescriptorPool = m_DescriptorPool[nn::gfx::DescriptorPoolType_BufferView];
    bufferDescriptorPool.BeginUpdate();
    {
        bufferDescriptorPool.SetBufferView(indexSlot, gpuAddress, size);
    }
    bufferDescriptorPool.EndUpdate();
}

void GraphicsFramework::SetTextureViewToDescriptorPool(int indexSlot, const nn::gfx::TextureView* pTextureView)
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_TextureView] > 0);

    nn::gfx::DescriptorPool& textureViewDescriptorPool = m_DescriptorPool[nn::gfx::DescriptorPoolType_TextureView];
    textureViewDescriptorPool.BeginUpdate();
    {
        textureViewDescriptorPool.SetTextureView(indexSlot, pTextureView);
    }
    textureViewDescriptorPool.EndUpdate();
}

void GraphicsFramework::SetSamplerToDescriptorPool(int indexSlot, const nn::gfx::Sampler* pSampler)
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_Sampler] > 0);

    nn::gfx::DescriptorPool& samplerDescriptorPool = m_DescriptorPool[nn::gfx::DescriptorPoolType_Sampler];
    samplerDescriptorPool.BeginUpdate();
    {
        samplerDescriptorPool.SetSampler(indexSlot, pSampler);
    }
    samplerDescriptorPool.EndUpdate();
}

//------------------------------------------------------------------------------
// ルートコマンドバッファ
//------------------------------------------------------------------------------
nn::gfx::CommandBuffer* GraphicsFramework::GetRootCommandBuffer(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_pCommandBuffer[bufferIndex].object;
}

void GraphicsFramework::ResetRootCommandBuffer(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    GraphicsFramework::CommandBuffer& commandBuffer = m_pCommandBuffer[bufferIndex];

    commandBuffer.object.Reset();

    MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
    commandBuffer.object.AddCommandMemory(
        GetMemoryPool(memorypoolType),
        commandBuffer.commandMemoryOffset,
        commandBuffer.commandMemorySize
    );

    commandBuffer.object.AddControlMemory(
        commandBuffer.pControlMemory,
        commandBuffer.controlMemorySize
    );
}

void GraphicsFramework::BeginRootCommandBuffer(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::CommandBuffer& commandBuffer = m_pCommandBuffer[bufferIndex].object;

    commandBuffer.Begin();
    commandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor);

#if NN_GFX_IS_TARGET_NVN
    int scratchMemoryIndex = bufferIndex;
    ShaderScratchMemoryInfo& shaderScratchMemory = m_pShaderScratchMemory[scratchMemoryIndex];

    if (shaderScratchMemory.size != m_ShaderScratchMemorySize && m_ShaderScratchMemorySize > 0) {
        CreateShaderScratchMemory(scratchMemoryIndex);
    }

    if (shaderScratchMemory.size > 0) {
        nvnCommandBufferSetShaderScratchMemory(commandBuffer.ToData()->pNvnCommandBuffer,
            m_MemoryPool[MemoryPoolType_RenderTarget].object.ToData()->pNvnMemoryPool,
            shaderScratchMemory.offset, shaderScratchMemory.size);
    }
#endif

    if (m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_BufferView] > 0) {
        commandBuffer.SetDescriptorPool(&m_DescriptorPool[nn::gfx::DescriptorPoolType_BufferView]);
    }
    if (m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_TextureView] > 0) {
        commandBuffer.SetDescriptorPool(&m_DescriptorPool[nn::gfx::DescriptorPoolType_TextureView]);
    }
    if (m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_Sampler] > 0) {
        commandBuffer.SetDescriptorPool(&m_DescriptorPool[nn::gfx::DescriptorPoolType_Sampler]);
    }
}

void GraphicsFramework::EndRootCommandBuffer(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_pCommandBuffer[bufferIndex].object.End();
}

//------------------------------------------------------------------------------
// シェーダスクラッチメモリ
//------------------------------------------------------------------------------
void GraphicsFramework::SetShaderScratchMemorySize(size_t size) NN_NOEXCEPT
{
#if NN_GFX_IS_TARGET_NVN
    NN_SDK_REQUIRES(IsInitialized());
    int granularity;
    nvnDeviceGetInteger(m_Device.ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_GRANULARITY, &granularity);

    m_ShaderScratchMemorySize = nn::util::align_up(size, granularity);
#else
    NN_UNUSED(size);
#endif
}

void GraphicsFramework::CreateShaderScratchMemory(int bufferIndex) NN_NOEXCEPT
{
#if NN_GFX_IS_TARGET_NVN
    NN_SDK_REQUIRES(IsInitialized());
    ShaderScratchMemoryInfo& shaderScratchMemory = m_pShaderScratchMemory[bufferIndex];

    MemoryPoolType memorypoolType = MemoryPoolType_RenderTarget;
    shaderScratchMemory.size = m_ShaderScratchMemorySize;

    int alignment;
    nvnDeviceGetInteger(m_Device.ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_ALIGNMENT, &alignment);

    m_MemoryPool[memorypoolType].allocator.Free(shaderScratchMemory.offset);
    shaderScratchMemory.offset = m_MemoryPool[memorypoolType].allocator.Allocate(
        shaderScratchMemory.size, static_cast<size_t>(alignment));
    NN_ASSERT(shaderScratchMemory.offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
#else
    NN_UNUSED(bufferIndex);
#endif
}

void GraphicsFramework::BeginFrame(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    ResetRootCommandBuffer(bufferIndex);
    BeginRootCommandBuffer(bufferIndex);
}

void GraphicsFramework::EndFrame(int bufferIndex) NN_NOEXCEPT
{
    EndFrame(bufferIndex, true);
}

void GraphicsFramework::EndFrame(int bufferIndex, bool enabledCopyImage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // カラーバッファからスキャンバッファへコピー
    if (enabledCopyImage) {
        nn::gfx::CommandBuffer* commandBuffer = GetRootCommandBuffer(bufferIndex);
        nn::gfx::Texture* pScanBuffer = GetScanBuffer(GetNextScanBufferIndex());
        nn::gfx::Texture* pColorBuffer = GetColorBuffer();
        NN_ASSERT_NOT_NULL(pColorBuffer);

        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(GetDisplayWidth());
        region.SetHeight(GetDisplayHeight());
        commandBuffer->BlitImage(pScanBuffer, region, pColorBuffer, region, 0);
    }

    EndRootCommandBuffer(bufferIndex);
}

void GraphicsFramework::ExecuteCommand(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::Queue* queue = GetQueue();
    queue->ExecuteCommand(GetRootCommandBuffer(bufferIndex), GetGpuFence(bufferIndex));
}

void GraphicsFramework::ExecuteCommand(int bufferIndex, nn::gfx::Fence* pFence) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::Queue* queue = GetQueue();
    queue->ExecuteCommand(GetRootCommandBuffer(bufferIndex), pFence);
}

void GraphicsFramework::AcquireTexture(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_NextScanBufferIndex = -1;
    nn::gfx::AcquireScanBufferResult acquireResult = GetSwapChain()->AcquireNextScanBufferIndex(
        &m_NextScanBufferIndex, &m_pDisplaySemaphore[bufferIndex], &m_pDisplayFence[bufferIndex]);
    NN_ASSERT(acquireResult == nn::gfx::AcquireScanBufferResult_Success);
    NN_UNUSED(acquireResult);
}

int GraphicsFramework::GetNextScanBufferIndex() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_NextScanBufferIndex;
}

void GraphicsFramework::QueueFinish() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::Queue* queue = GetQueue();
    queue->Sync();
}

void GraphicsFramework::QueuePresentTexture(int presentInterval) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::Queue* queue = GetQueue();
    queue->Present(GetSwapChain(), presentInterval);
}

void GraphicsFramework::QueueWaitSync(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::Queue* queue = GetQueue();
    queue->SyncSemaphore(&m_pDisplaySemaphore[bufferIndex]);
    queue->Flush();
}

nn::gfx::SyncResult GraphicsFramework::WaitDisplaySync(int bufferIndex, nn::TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pDisplayFence[bufferIndex].Sync(timeout);
}

nn::gfx::SyncResult GraphicsFramework::WaitGpuSync(int bufferIndex, nn::TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pGpuFence[bufferIndex].Sync(timeout);
}

void GraphicsFramework::Calculate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    if (m_pCalculateCallback != nullptr)
    {
        m_pCalculateCallback(this, m_pCalculateCallbackUserData);
    }
}

void GraphicsFramework::MakeCommand(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    if (m_pMakeCommandCallback != nullptr)
    {
        m_pMakeCommandCallback(this, bufferIndex, m_pMakeCommandCallbackUserData);
    }
}

//------------------------------------------------------------------------------
// フレーム処理
//------------------------------------------------------------------------------
void GraphicsFramework::SetFrameworkMode(FrameworkMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_PrevFrameworkMode = m_FrameworkMode;
    m_FrameworkMode = mode;
}

nns::gfx::GraphicsFramework::FrameworkMode GraphicsFramework::GetFrameworkMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_FrameworkMode;
}

void GraphicsFramework::SetCalculateCallback(GraphicsFrameworkCalculateCallback pFunction, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_pCalculateCallback = pFunction;
    m_pCalculateCallbackUserData = pUserData;
}

GraphicsFrameworkCalculateCallback GraphicsFramework::GetCalculateCallback() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pCalculateCallback;
}

void* GraphicsFramework::GetCalculateCallbackUserData() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pCalculateCallbackUserData;
}

void GraphicsFramework::SetMakeCommandCallback(GraphicsFrameworkMakeCommandCallback pFunction, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_pMakeCommandCallback = pFunction;
    m_pMakeCommandCallbackUserData = pUserData;
}

GraphicsFrameworkMakeCommandCallback GraphicsFramework::GetMakeCommandCallback() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pMakeCommandCallback;
}

void* GraphicsFramework::GetMakeCommandCallbackUserData() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pMakeCommandCallbackUserData;
}

void GraphicsFramework::SetPresentInterval(int presentInterval) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_PresentInterval = presentInterval;
}

int GraphicsFramework::GetPresentInterval() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_PresentInterval;
}

void GraphicsFramework::SetWaitSyncTimeout(nn::TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_WaitSyncTimeout = timeout;
}

nn::TimeSpan GraphicsFramework::GetWaitSyncTimeout() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_WaitSyncTimeout;
}

void GraphicsFramework::ProcessFrame() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    if (m_FrameworkMode != m_PrevFrameworkMode)
    {
        if (m_CurrentFrameIndex > 0)
        {
            if (m_PrevFrameworkMode == FrameworkMode_DeferredSubmission || m_PrevFrameworkMode == FrameworkMode_Immediate)
            {
                QueuePresentTexture(m_PresentInterval);
            }
            QueueFinish();
        }

        m_CurrentFrameIndex = 0;
        m_PrevFrameworkMode = m_FrameworkMode;
    }

    int currentBufferIndex = m_CurrentFrameIndex % m_BufferCount;
    int previousBufferIndex = (currentBufferIndex + m_BufferCount - 1) % m_BufferCount;
    int nextBufferIndex = (currentBufferIndex + 1) % m_BufferCount;

    switch (m_FrameworkMode)
    {
    case FrameworkMode_DeferredExecution:
        {
            Calculate();

            AcquireTexture(currentBufferIndex);

            QueueWaitSync(currentBufferIndex);

            MakeCommand(currentBufferIndex);

            ExecuteCommand(currentBufferIndex);

            QueuePresentTexture(m_PresentInterval);

            WaitDisplaySync(currentBufferIndex, m_WaitSyncTimeout);

            if (m_CurrentFrameIndex > 0)
            {
                WaitGpuSync(previousBufferIndex, m_WaitSyncTimeout);
            }
        }
        break;
    case FrameworkMode_DeferredSubmission:
        {
            if (m_CurrentFrameIndex > 0)
            {
                ExecuteCommand(previousBufferIndex);

                QueuePresentTexture(m_PresentInterval);
            }

            Calculate();

            AcquireTexture(currentBufferIndex);

            //QueueWaitSync(currentBufferIndex); // 不要

            MakeCommand(currentBufferIndex);

            WaitDisplaySync(currentBufferIndex, m_WaitSyncTimeout);

            if (m_CurrentFrameIndex > 0)
            {
                WaitGpuSync(previousBufferIndex, m_WaitSyncTimeout);
            }
        }
        break;
    case FrameworkMode_Immediate:
        {
            if (m_CurrentFrameIndex == 0)
            {
                AcquireTexture(currentBufferIndex);

                //QueueWaitSync(currentBufferIndex); // 不要

                WaitDisplaySync(currentBufferIndex, m_WaitSyncTimeout);
            }

            Calculate();

            MakeCommand(currentBufferIndex);

            ExecuteCommand(currentBufferIndex);

            QueuePresentTexture(m_PresentInterval);

            AcquireTexture(nextBufferIndex);

            //QueueWaitSync(nextBufferIndex); // 不要

            WaitDisplaySync(nextBufferIndex, m_WaitSyncTimeout);

            WaitGpuSync(currentBufferIndex, m_WaitSyncTimeout);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_CurrentFrameIndex++;
    if (m_CurrentFrameIndex > 10000000)
    {
        m_CurrentFrameIndex = (m_CurrentFrameIndex % m_BufferCount) + m_BufferCount;
    }
}

void GraphicsFramework::ResetFrame() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    if (m_CurrentFrameIndex > 0)
    {
        if (m_PrevFrameworkMode == FrameworkMode_DeferredSubmission || m_PrevFrameworkMode == FrameworkMode_Immediate)
        {
            QueuePresentTexture(m_PresentInterval);
        }
        QueueFinish();
    }

    m_CurrentFrameIndex = 0;
    m_PrevFrameworkMode = m_FrameworkMode;
}

//------------------------------------------------------------------------------
// メモリプールアロケータ, デスクリプタプールアロケータ用アロケータ関数
//------------------------------------------------------------------------------
void* GraphicsFramework::MemoryPoolAllocatorMalloc(size_t size, void* pUserData)
{
    GraphicsFramework* pFramework = reinterpret_cast<GraphicsFramework*> (pUserData);
    return pFramework->GetAllocateFunction()(size, 1, pFramework->GetAllocateFunctionUserData());
    //return pFramework->AllocateMemory(size, 1);
}

void GraphicsFramework::MemoryPoolAllocatorFree(void* ptr, void* pUserData)
{
    GraphicsFramework* pFramework = reinterpret_cast<GraphicsFramework*> (pUserData);
    pFramework->GetFreeFunction()(ptr, pFramework->GetAllocateFunctionUserData());
    //pFramework->FreeMemory(ptr);
}

void* GraphicsFramework::DescriptorPoolAllocatorMalloc(size_t size, void* pUserData)
{
    GraphicsFramework* pFramework = reinterpret_cast<GraphicsFramework*> (pUserData);
    return pFramework->GetAllocateFunction()(size, 1, pFramework->GetAllocateFunctionUserData());
    //return pFramework->AllocateMemory(size, 1);
}

void GraphicsFramework::DescriptorPoolAllocatorFree(void* ptr, void* pUserData)
{
    GraphicsFramework* pFramework = reinterpret_cast<GraphicsFramework*> (pUserData);
    pFramework->GetFreeFunction()(ptr, pFramework->GetAllocateFunctionUserData());
    //pFramework->FreeMemory(ptr);
}

//------------------------------------------------------------------------------
// 外部 gfx オブジェクト コマンドバッファ
//------------------------------------------------------------------------------
void GraphicsFramework::InitializeCommandBuffer(GraphicsFramework::CommandBuffer* pOutCommandBuffer, nn::gfx::CommandBuffer::InfoType& info, size_t commandMemorySize, size_t controlMemorySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // CommandBuffer
    pOutCommandBuffer->object.Initialize(GetDevice(), info);

    // CommandMemory
    MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
    size_t alignment = nn::gfx::CommandBuffer::GetCommandMemoryAlignment(GetDevice());
    size_t size = commandMemorySize;
    pOutCommandBuffer->commandMemoryOffset = AllocatePoolMemory(memorypoolType, size, alignment);
    NN_ASSERT(pOutCommandBuffer->commandMemoryOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
    pOutCommandBuffer->commandMemorySize = size;

    // ControlMemory
    alignment = nn::gfx::CommandBuffer::GetControlMemoryAlignment(GetDevice());
    size = controlMemorySize;
    pOutCommandBuffer->pControlMemory = AllocateMemory(size, alignment);
    pOutCommandBuffer->controlMemorySize = size;
}

void GraphicsFramework::FinalizeCommandBuffer(GraphicsFramework::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // CommandBuffer
    pCommandBuffer->object.Finalize(GetDevice());

    // CommandMemory
    nns::gfx::GraphicsFramework::MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
    FreePoolMemory(memorypoolType, pCommandBuffer->commandMemoryOffset);

    // ControlMemory
    FreeMemory(pCommandBuffer->pControlMemory);
}

void GraphicsFramework::ResetCommandBuffer(GraphicsFramework::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    pCommandBuffer->object.Reset();

    nns::gfx::GraphicsFramework::MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
    pCommandBuffer->object.AddCommandMemory(
        GetMemoryPool(memorypoolType),
        pCommandBuffer->commandMemoryOffset,
        pCommandBuffer->commandMemorySize
    );

    pCommandBuffer->object.AddControlMemory(
        pCommandBuffer->pControlMemory,
        pCommandBuffer->controlMemorySize
    );
}

void GraphicsFramework::SetPhysicalWindowCoordinate(const nn::gfx::util::ScalableViewport::WindowCoordinate& physicalWindowCoordinate) NN_NOEXCEPT
{
    NN_UNUSED(physicalWindowCoordinate);
    const nn::gfx::util::ScalableViewport::WindowCoordinate* pPhysicalWindowCoordinate = m_ScalableViewport.GetPhysicalWindowCoordinate();
    NN_ASSERT_EQUAL(pPhysicalWindowCoordinate->GetOriginMode(), physicalWindowCoordinate.GetOriginMode());

#ifdef NN_BUILD_TARGET_PLATFORM_OS_NN
    int width = static_cast<int>(physicalWindowCoordinate.GetWidth());
    int height = static_cast<int>(physicalWindowCoordinate.GetHeight());

    NN_UNUSED(width);
    NN_UNUSED(height);
    NN_ASSERT_GREATER_EQUAL(GetDisplayWidth(), width);
    NN_ASSERT_GREATER_EQUAL(GetDisplayHeight(), height);
    NN_ASSERT(physicalWindowCoordinate.IsValid());

    m_ScalableViewport.SetPhysicalWindowCoordinate(physicalWindowCoordinate);

    // デフォルトビューポートを再構築
    FinalizeViewportScissorState();
    IntializeViewportScissorState();

    ResizeWindow();
#endif // NN_BUILD_TARGET_PLATFORM_OS_NN
}

nn::gfx::util::ScalableViewport* GraphicsFramework::GetScalableViewport() NN_NOEXCEPT
{
    return &m_ScalableViewport;
}

//------------------------------------------------------------------------------
// gfx オブジェクト 頂点ステート
//------------------------------------------------------------------------------
void GraphicsFramework::InitializeVertexState(GraphicsFramework::VertexState* pOutVertexState, nn::gfx::VertexState::InfoType& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize(info);
    if (size) {
        pOutVertexState->pMemory = AllocateMemory(size, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
        pOutVertexState->object.SetMemory(pOutVertexState->pMemory, size);
    }
    else {
        pOutVertexState->pMemory = nullptr;
    }

    pOutVertexState->object.Initialize(GetDevice(), info);
}

void GraphicsFramework::FinalizeVertexState(GraphicsFramework::VertexState* pVertexState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    pVertexState->object.Finalize(GetDevice());
    if (pVertexState->pMemory) {
        FreeMemory(pVertexState->pMemory);
    }
}

//------------------------------------------------------------------------------
// gfx オブジェクト ブレンドステート
//------------------------------------------------------------------------------
void GraphicsFramework::InitializeBlendState(GraphicsFramework::BlendState* pOutBlendState, nn::gfx::BlendState::InfoType& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize(info);
    if (size) {
        pOutBlendState->pMemory = AllocateMemory(size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        pOutBlendState->object.SetMemory(pOutBlendState->pMemory, size);
    }
    else {
        pOutBlendState->pMemory = nullptr;
    }

    pOutBlendState->object.Initialize(GetDevice(), info);
}

void GraphicsFramework::FinalizeBlendState(GraphicsFramework::BlendState* pBlendState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    pBlendState->object.Finalize(GetDevice());
    if (pBlendState->pMemory) {
        FreeMemory(pBlendState->pMemory);
    }
}

//------------------------------------------------------------------------------
// gfx オブジェクト ビューポートシザーステート
//------------------------------------------------------------------------------
void GraphicsFramework::IntializeViewportScissorState() NN_NOEXCEPT
{
    const nn::gfx::util::ScalableViewport::WindowCoordinate* pVirtualWindowCoord = m_ScalableViewport.GetVirtualWindowCoordinate();
    nn::gfx::util::ScalableViewport::Rect rect;
    rect.originX = rect.originY = .0f;
    rect.width = pVirtualWindowCoord->GetWidth();
    rect.height = pVirtualWindowCoord->GetHeight();
    m_ScalableViewport.ConvertRectVirtualToPhysical(&rect, rect);

    nn::gfx::ViewportStateInfo viewportInfo;
    viewportInfo.SetDefault();
    m_ScalableViewport.SetupViewportStateInfo(&viewportInfo, rect);

    nn::gfx::ScissorStateInfo scissorInfo;
    scissorInfo.SetDefault();
    m_ScalableViewport.SetupScissorStateInfo(&scissorInfo, rect);

    nn::gfx::ViewportScissorState::InfoType viewportScissorInfo;
    viewportScissorInfo.SetDefault();
    viewportScissorInfo.SetScissorEnabled(true);
    viewportScissorInfo.SetViewportStateInfoArray(&viewportInfo, 1);
    viewportScissorInfo.SetScissorStateInfoArray(&scissorInfo, 1);

    NN_SDK_ASSERT(nn::gfx::ViewportScissorState::GetRequiredMemorySize(viewportScissorInfo) == 0);
    m_ViewportScissorState.Initialize(GetDevice(), viewportScissorInfo);
}

void GraphicsFramework::FinalizeViewportScissorState() NN_NOEXCEPT
{
    m_ViewportScissorState.Finalize(&m_Device);
}

void GraphicsFramework::ResizeWindow() NN_NOEXCEPT
{
#ifdef NN_BUILD_TARGET_PLATFORM_OS_NN
    const nn::gfx::util::ScalableViewport::WindowCoordinate* pPhysicalWindowCoordinate = m_ScalableViewport.GetPhysicalWindowCoordinate();
    int width = static_cast<int>(pPhysicalWindowCoordinate->GetWidth() + 0.5f);
    int height = static_cast<int>(pPhysicalWindowCoordinate->GetHeight() + 0.5f);

    NN_UNUSED(pPhysicalWindowCoordinate);
    NN_UNUSED(width);
    NN_UNUSED(height);

#if NN_GFX_IS_TARGET_NVN
    // NVN を利用していた場合には、描画する画面サイズを変更します。
    nvnWindowSetCrop(m_SwapChain.ToData()->pNvnWindow, 0, 0, width, height);
#elif NN_GFX_IS_TARGET_GL
    m_SwapChain.Finalize(&m_Device);
    nn::vi::DestroyLayer(m_pLayer);

    nn::vi::CreateLayer(&m_pLayer, m_pDisplay, width, height);
    nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_ScaleAndCrop);

    // スワップチェーンを再構築
    {
        nn::gfx::SwapChain::InfoType swapchainInfo;
        swapchainInfo.SetDefault();
        swapchainInfo.SetLayer(m_pLayer);
        swapchainInfo.SetWidth(width);
        swapchainInfo.SetHeight(height);
        swapchainInfo.SetFormat(swapChainFormat);
        swapchainInfo.SetBufferCount(m_SwapChainBufferCount);
        {
            NN_ASSERT(!nn::gfx::SwapChain::IsMemoryPoolRequired);
            m_SwapChain.Initialize(&m_Device, swapchainInfo, NULL, 0, 0);
        }
    }

    // ScanBuffer
    {
        int count = m_SwapChainBufferCount;
        NN_ASSERT_NOT_NULL(m_pScanBufferViews);
        NN_ASSERT_NOT_NULL(m_pScanBuffers);

        m_SwapChain.GetScanBufferViews(m_pScanBufferViews, count);
        m_SwapChain.GetScanBuffers(m_pScanBuffers, count);
    }
#else
    // unsupported low level graphics API
#endif // NN_GFX_IS_TARGET
#endif // NN_BUILD_TARGET_PLATFORM_OS_NN
}

//------------------------------------------------------------------------------
// 外部 gfx オブジェクト デバッグフォント
//------------------------------------------------------------------------------
void GraphicsFramework::InitializeDebugFontTextWriter(GraphicsFramework::DebugFontTextWriter* pOutWriter, int charCountMax) NN_NOEXCEPT
{
    InitializeDebugFontTextWriter(pOutWriter, charCountMax, 2);
}

void GraphicsFramework::InitializeDebugFontTextWriter(GraphicsFramework::DebugFontTextWriter* pOutWriter, int charCountMax, int bufferCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetBufferCount(bufferCount);
    info.SetCharCountMax(charCountMax);
    info.SetUserMemoryPoolEnabled(true);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(GetDevice(), info);
    void* pDebugFontMemory = AllocateMemory(debugFontHeapSize, 1);

    size_t debugFontMemoryPoolSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(GetDevice(), info);

    nns::gfx::GraphicsFramework::MemoryPoolType type = MemoryPoolType_ConstantBuffer;
    ptrdiff_t offset = AllocatePoolMemory(type, debugFontMemoryPoolSize, 1);
    NN_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);

    nn::gfx::util::DebugFontTextWriter* pWriter = &pOutWriter->object;
    pWriter->Initialize(
        GetDevice(),
        info,
        pDebugFontMemory,
        debugFontHeapSize,
        GetMemoryPool(type),
        offset,
        debugFontMemoryPoolSize
    );

    int textureDescriptorIndex = AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    NN_ASSERT(textureDescriptorIndex != nn::gfx::util::DescriptorPoolAllocator::InvalidIndex);
    int samplerDescriptorIndex = AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
    NN_ASSERT(samplerDescriptorIndex != nn::gfx::util::DescriptorPoolAllocator::InvalidIndex);

    pWriter->SetDisplayWidth(GetDisplayWidth());
    pWriter->SetDisplayHeight(GetDisplayHeight());
    pWriter->SetTextureDescriptor(GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView), textureDescriptorIndex);
    pWriter->SetSamplerDescriptor(GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler), samplerDescriptorIndex);
}

void GraphicsFramework::FinalizeDebugFontTextWriter(GraphicsFramework::DebugFontTextWriter* pWriter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    void* pDebugFontMemory = pWriter->object.GetMemory();
    ptrdiff_t memoryPoolOffset = pWriter->object.GetMemoryPoolOffset();
    int textureDescriptorIndex = pWriter->object.GetTextureDescriptorIndexSlot();
    int samplerDescriptorIndex = pWriter->object.GetSamplerDescriptorIndexSlot();
    pWriter->object.Finalize();

    FreeMemory(pDebugFontMemory);
    nns::gfx::GraphicsFramework::MemoryPoolType type = MemoryPoolType_ConstantBuffer;
    FreePoolMemory(type, memoryPoolOffset);
    FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, textureDescriptorIndex);
    FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, samplerDescriptorIndex);
}

//------------------------------------------------------------------------------
// デフォルトのメモリアロケータ
//------------------------------------------------------------------------------
void* GraphicsFramework::DefaultAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
        NN_UNUSED(pUserData);
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
#else
    return _aligned_malloc(size, alignment);
#endif
}

void GraphicsFramework::DefaultFreeFunction(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    free(ptr);
#else
    _aligned_free(ptr);
#endif
}

void* GraphicsFramework::DefaultReallocateFunction(void* ptr, size_t newSize, void* pUserData)
{
    NN_UNUSED(pUserData);
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    return realloc(ptr, newSize);
#else
    return _aligned_realloc(ptr, newSize, 1);
#endif
}


} // namespace gfx
} // namespace nns
