﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/g3d/g3d_Utility.h>

namespace nns { namespace g3d {

nn::g3d::ModelObj* CreateModelObj(nn::gfx::Device* pDevice, nn::g3d::ModelObj::Builder& builder) NN_NOEXCEPT
{
    if (builder.IsMemoryCalculated() == false)
    {
        builder.CalculateMemorySize();
    }

    // ModelObjのサイズ + ModelObj作成に必要なサイズ分のメモリーを確保する
    size_t alignment = nn::g3d::ModelObj::Alignment_Buffer;
    size_t bufferSize = builder.GetWorkMemorySize();
    size_t objectSize = nn::util::align_up(sizeof(nn::g3d::ModelObj), alignment);
    void* ptr = Allocate(objectSize + bufferSize, alignment);

    nn::util::BytePtr bytePtr(ptr);
    nn::g3d::ModelObj* pModelObj = new(bytePtr.Get()) nn::g3d::ModelObj();
    bytePtr.Advance(objectSize);

    bool success = builder.Build(pModelObj, bytePtr.Get(), bufferSize);
    (void)success;
    NN_ASSERT(success == true);

    size_t size = pModelObj->CalculateBlockBufferSize(pDevice);
    if (size > 0)
    {
        ptrdiff_t offset = AllocateMemoryPool(size, pModelObj->GetBlockBufferAlignment(pDevice));

        success = pModelObj->SetupBlockBuffer(pDevice, GetMemoryPool(), offset, size);
        NN_ASSERT(success == true);
    }
    return pModelObj;
}

void DestroyModelObj(nn::gfx::Device* pDevice, nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
{
    if (pModelObj->IsBlockBufferValid())
    {
        ptrdiff_t offset = pModelObj->GetMemoryPoolOffset();
        pModelObj->CleanupBlockBuffer(pDevice);
        FreeMemoryPool(offset);
    }
    pModelObj->~ModelObj();
    Free(pModelObj);
}

nns::g3d::ModelAnimObj* CreateModelAnimObj(nn::gfx::Device* pDevice, nn::g3d::ModelObj::Builder& builder) NN_NOEXCEPT
{
    return CreateModelAnimObj(pDevice, builder, 8, 8, 8, 8);
}

nns::g3d::ModelAnimObj* CreateModelAnimObj(
    nn::gfx::Device* pDevice,
    nn::g3d::ModelObj::Builder& builder,
    int handleSkeletalAnimCount,
    int handleMaterialAnimCount,
    int handleShapeAnimCount,
    int handleBoneVisibilityAnimCount
) NN_NOEXCEPT
{
    nns::g3d::ModelAnimObj* pModelAnimObj = Allocate<nns::g3d::ModelAnimObj>(sizeof(nns::g3d::ModelAnimObj));
    new(pModelAnimObj) nns::g3d::ModelAnimObj();
    nn::g3d::ModelObj* pModelObj = CreateModelObj(pDevice, builder);
    pModelAnimObj->Initialize(pModelObj, handleSkeletalAnimCount, handleMaterialAnimCount, handleShapeAnimCount, handleBoneVisibilityAnimCount);

    return pModelAnimObj;
}

void DestroyModelAnimObj(nn::gfx::Device* pDevice, nns::g3d::ModelAnimObj* pModelAnimObj) NN_NOEXCEPT
{
    pModelAnimObj->Finalize();
    DestroyModelObj(pDevice, pModelAnimObj->GetModelObj());
    pModelAnimObj->~ModelAnimObj();
    Free(pModelAnimObj);
}

RenderModel* CreateRenderModel(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel, int passCount) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));
    NN_ASSERT_NOT_NULL(pResModel);

    RenderModel* pRenderModel = Allocate<RenderModel>(sizeof(RenderModel));
    pRenderModel = new(pRenderModel) RenderModel();
    pRenderModel->Initialize(pDevice, pResModel, passCount);

    return pRenderModel;
}

RenderModel* CreateRenderModel(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel) NN_NOEXCEPT
{
    return CreateRenderModel(pDevice, pResModel, 1);
}

void DestroyRenderModel(nn::gfx::Device* pDevice, RenderModel* pRenderModel) NN_NOEXCEPT
{
    pRenderModel->Finalize(pDevice);
    pRenderModel->~RenderModel();
    Free(pRenderModel);
}

RenderModelObj* CreateRenderModelObj(nn::g3d::ModelObj* pModelObj, RenderModel* pRenderModel) NN_NOEXCEPT
{
    RenderModelObj* pRenderModelObj = Allocate<RenderModelObj>(sizeof(RenderModelObj));
    pRenderModelObj = new(pRenderModelObj) RenderModelObj();
    pRenderModelObj->Initialize(pModelObj, pRenderModel);

    return pRenderModelObj;
}

void DestroyRenderModelObj(RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    pRenderModelObj->Finalize();
    pRenderModelObj->~RenderModelObj();
    Free(pRenderModelObj);
}

nn::g3d::ShadingModelObj* CreateShadingModelObj(nn::gfx::Device* pDevice, nn::g3d::ShadingModelObj::Builder& builder) NN_NOEXCEPT
{
    nn::g3d::ShadingModelObj* pShadingModelObj = NULL;
    {
        if (builder.IsMemoryCalculated() == false)
        {
            builder.CalculateMemorySize();
        }
        size_t alignment = nn::g3d::ShadingModelObj::Alignment_Buffer;
        size_t bufferSize = builder.GetWorkMemorySize();
        size_t objectSize = nn::util::align_up(sizeof(nn::g3d::ShadingModelObj), alignment);
        void* ptr = Allocate(objectSize + bufferSize, alignment);

        pShadingModelObj = new(ptr) nn::g3d::ShadingModelObj();
        ptr = nn::g3d::AddOffset(ptr, objectSize);

        bool success = builder.Build(pShadingModelObj, ptr, bufferSize);
        (void)success;
        NN_ASSERT(success == true);
    }

    {
        // Uniform Block のバッファーは所定のアライメントが必要です。
        size_t size = pShadingModelObj->CalculateBlockBufferSize(pDevice);
        if (size > 0)
        {
            ptrdiff_t offset = AllocateMemoryPool(size, pShadingModelObj->GetBlockBufferAlignment(pDevice));

            bool success = pShadingModelObj->SetupBlockBuffer(pDevice, GetMemoryPool(), offset, size);
            (void)success;
            NN_ASSERT(success == true);
        }
    }

    return pShadingModelObj;
}

void DestroyShadingModelObj(nn::gfx::Device* pDevice, nn::g3d::ShadingModelObj* pShadingModelObj) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pShadingModelObj);

    if (pShadingModelObj->IsBlockBufferValid())
    {
        pShadingModelObj->CleanupBlockBuffer(pDevice);
    }

    // メモリープールから確保したオフセットを解放
    if (pShadingModelObj->GetMemoryPoolPtr())
    {
        ptrdiff_t offset = pShadingModelObj->GetMemoryPoolOffset();
        FreeMemoryPool(offset);
    }

    // BufferPtr は pShadingModelObj のバッファーの一部として確保したので解放不要。
    pShadingModelObj->~ShadingModelObj();
    Free(pShadingModelObj);
}

nn::g3d::ShaderSelector* CreateShaderSelector(nn::g3d::ShaderSelector::Builder& builder) NN_NOEXCEPT
{
    nn::g3d::ShaderSelector* pShaderSelector = NULL;
    {
        if (builder.IsMemoryCalculated() == false)
        {
            builder.CalculateMemorySize();
        }
        size_t alignment = nn::g3d::ShaderSelector::Alignment_Buffer;
        size_t bufferSize = builder.GetWorkMemorySize();
        size_t objectSize = nn::util::align_up(sizeof(*pShaderSelector), alignment);
        void* ptr = Allocate(objectSize + bufferSize, alignment);

        pShaderSelector = new(ptr) nn::g3d::ShaderSelector();
        ptr = nn::g3d::AddOffset(ptr, objectSize);

        bool success = builder.Build(pShaderSelector, ptr, bufferSize);
        (void)success;
        NN_ASSERT(success == true);
    }

    return pShaderSelector;
}

void DestroyShaderSelector(nn::g3d::ShaderSelector* pShaderSelector) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pShaderSelector);

    // BufferPtr は pShaderSelector のバッファーの一部として確保したので解放不要。
    pShaderSelector->~ShaderSelector();
    Free(pShaderSelector);
}

//------------------------------------------------------------------------------------------------------------------

void RenderView::Initialize(nn::gfx::Device* pDevice, int viewCount, int bufferCount) NN_NOEXCEPT
{
    // 必要なメモリー量を計算し、確保します
    nn::util::MemorySplitter::MemoryBlock memoryBlock[MemoryBlock_Max];
    for (int index = MemoryBlock_View; index < MemoryBlock_Max; ++index)
    {
        memoryBlock[index].Initialize();
    }
    memoryBlock[MemoryBlock_View].SetSizeBy<View>(1, viewCount);
    memoryBlock[MemoryBlock_View].SetAlignment(NN_ALIGNOF(View));
    memoryBlock[MemoryBlock_ViewUniformBlock].SetSizeBy<nn::gfx::Buffer>(1, viewCount * bufferCount);
    memoryBlock[MemoryBlock_ViewUniformBlock].SetAlignment(NN_ALIGNOF(nn::gfx::Buffer));
    memoryBlock[MemoryBlock_ViewVolume].SetSizeBy<nn::g3d::ViewVolume>(1, viewCount);
    memoryBlock[MemoryBlock_ViewVolume].SetAlignment(NN_ALIGNOF(nn::g3d::ViewVolume));
    memoryBlock[MemoryBlock_ViewUniformBlockOffset].SetSizeBy<ptrdiff_t>(1, viewCount * bufferCount);
    memoryBlock[MemoryBlock_ViewUniformBlockOffset].SetAlignment(NN_ALIGNOF(ptrdiff_t));

    nn::util::MemorySplitter memorySplitter;
    memorySplitter.Initialize();
    for (int index = MemoryBlock_View; index < MemoryBlock_Max; ++index)
    {
        memorySplitter.Append(&memoryBlock[index]);
    }

    void* ptr = Allocate(memorySplitter.GetSize(), memorySplitter.GetAlignment());
    m_pView = memoryBlock[MemoryBlock_View].Get<View>(ptr);
    m_pViewUniformBlock = memoryBlock[MemoryBlock_ViewUniformBlock].Get<nn::gfx::Buffer>(ptr);
    m_pViewVolume = memoryBlock[MemoryBlock_ViewVolume].Get<nn::g3d::ViewVolume>(ptr);
    m_pViewUniformBlockOffset = memoryBlock[MemoryBlock_ViewUniformBlockOffset].Get<ptrdiff_t>(ptr);

    m_ViewCount = viewCount;
    m_BufferCount = bufferCount;

    // マトリクスを初期化
    for (int index = 0; index < viewCount; ++index)
    {
        nn::util::MatrixIdentity(&m_pView[index].projectionMtx);
        nn::util::MatrixIdentity(&m_pView[index].inverseViewMtx);
        nn::util::MatrixIdentity(&m_pView[index].viewMtx);
    }
    // ユニフォームブロックを初期化
    for (int index = 0; index < viewCount * bufferCount; ++index)
    {
        nn::gfx::BufferInfo bufferInfo;
        bufferInfo.SetDefault();
        bufferInfo.SetSize(sizeof(ViewUniformBlock));
        bufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        {
            m_pViewUniformBlockOffset[index] = AllocateMemoryPool(bufferInfo.GetSize(), nn::gfx::Buffer::GetBufferAlignment(pDevice, bufferInfo));
            new(&m_pViewUniformBlock[index]) nn::gfx::Buffer();
            m_pViewUniformBlock[index].Initialize(pDevice, bufferInfo, GetMemoryPool(), m_pViewUniformBlockOffset[index], bufferInfo.GetSize());
        }
    }
    // コンストラクタ呼び出し
    for (int index = 0; index < viewCount; ++index)
    {
        new(&m_pView[index]) View();
        new(&m_pViewVolume[index]) nn::g3d::ViewVolume();
    }

    // 計算結果をユニフォームブロックに反映させておく
    for (int index = 0; index < bufferCount; ++index)
    {
        Calculate(index);
    }
}

void RenderView::Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    for (int index = 0; index < m_ViewCount * m_BufferCount ; ++index)
    {
        m_pViewUniformBlock[index].Finalize(pDevice);
        m_pViewUniformBlock[index].nn::gfx::Buffer::~TBuffer();
        FreeMemoryPool(m_pViewUniformBlockOffset[index]);
    }

    Free(m_pView);
    m_pView = NULL;
}

void RenderView::Calculate(int viewIndex, int bufferIndex) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);

    nn::gfx::Buffer& viewUniformBlockBuffer = m_pViewUniformBlock[m_BufferCount * viewIndex + bufferIndex];
    ViewUniformBlock* pViewUniformBlock = viewUniformBlockBuffer.Map<ViewUniformBlock>();
    nn::util::Matrix4x4fType& projectionMtx = m_pView[viewIndex].projectionMtx;
    nn::util::Matrix4x3fType& viewMtx = m_pView[viewIndex].viewMtx;
    nn::util::Matrix4x3fType& inverseViewMtx = m_pView[viewIndex].inverseViewMtx;
    MatrixInverse(&inverseViewMtx, viewMtx);
    MatrixStore(&pViewUniformBlock->projectionMtx, projectionMtx);
    MatrixStore(&pViewUniformBlock->viewMtx, viewMtx);
    MatrixStore(&pViewUniformBlock->inverseViewMtx, inverseViewMtx);
    viewUniformBlockBuffer.FlushMappedRange(0, sizeof(ViewUniformBlock));
    viewUniformBlockBuffer.Unmap();
}

//------------------------------------------------------------------------------------------------------------------

void ModelAnimObj::Initialize(
    nn::g3d::ModelObj* pModelObj,
    int handleSkeletalAnimCount,
    int handleMaterialAnimCount,
    int handleShapeAnimCount,
    int handleBoneVisibilityAnimCount
) NN_NOEXCEPT
{
    NN_ASSERT(!IsInitialized());
    NN_ASSERT_NOT_NULL(pModelObj);

    m_pModelObj = pModelObj;

    if(handleSkeletalAnimCount > 0)
    {
        m_pSkeletalAnimList = Allocate<SkeletalAnim>(sizeof(SkeletalAnim) * handleSkeletalAnimCount);
        m_HandleSkeletalAnimCount = handleSkeletalAnimCount;
        for (int index = 0; index < m_HandleSkeletalAnimCount; ++index)
        {
            new(&m_pSkeletalAnimList[index]) SkeletalAnim();
        }
    }
    if (handleMaterialAnimCount > 0)
    {
        m_pMaterialAnimList = Allocate<MaterialAnim>(sizeof(MaterialAnim) * handleMaterialAnimCount);
        m_HandleMaterialAnimCount = handleMaterialAnimCount;
        for (int index = 0; index < m_HandleMaterialAnimCount; ++index)
        {
            new(&m_pMaterialAnimList[index]) MaterialAnim();
        }
    }
    if (handleShapeAnimCount > 0)
    {
        m_pShapeAnimList = Allocate<ShapeAnim>(sizeof(ShapeAnim) * handleShapeAnimCount);
        m_HandleShapeAnimCount = handleShapeAnimCount;
        for (int index = 0; index < m_HandleShapeAnimCount; ++index)
        {
            new(&m_pShapeAnimList[index]) ShapeAnim();
        }
    }
    if (handleBoneVisibilityAnimCount > 0)
    {
        m_pBoneVisibilityAnimList = Allocate<BoneVisibilityAnim>(sizeof(BoneVisibilityAnim) * handleBoneVisibilityAnimCount);
        m_HandleBoneVisibilityAnimCount = handleBoneVisibilityAnimCount;
        for (int index = 0; index < m_HandleBoneVisibilityAnimCount; ++index)
        {
            new(&m_pBoneVisibilityAnimList[index]) BoneVisibilityAnim();
        }
    }
    {
        nn::g3d::SkeletalAnimBlender::Builder skeletalAnimBlenderBuilder;
        skeletalAnimBlenderBuilder.Reserve(m_pModelObj->GetResource());
        skeletalAnimBlenderBuilder.CalculateMemorySize();
        size_t bufferSize = skeletalAnimBlenderBuilder.GetWorkMemorySize();
        void* buffer = Allocate(bufferSize, nn::g3d::SkeletalAnimBlender::Alignment_Buffer);
        skeletalAnimBlenderBuilder.Build(&m_SkeletalAnimBlender, buffer, bufferSize);
        m_SkeletalAnimBlender.SetBoneCount(m_pModelObj->GetSkeleton()->GetBoneCount());
    }

    m_IsInitialized = true;
}

void ModelAnimObj::Finalize() NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    if (m_SkeletalAnimBlender.GetBufferPtr())
    {
        Free(m_SkeletalAnimBlender.GetBufferPtr());
    }
    for (int index = 0; index < m_HandleSkeletalAnimCount; ++index)
    {
        DestroySkeletalAnim(index);
    }
    Free(m_pSkeletalAnimList);
    m_pSkeletalAnimList = NULL;
    for (int index = 0; index < m_HandleMaterialAnimCount; ++index)
    {
        DestroyMaterialAnim(index);
    }
    Free(m_pMaterialAnimList);
    m_pMaterialAnimList = NULL;
    for (int index = 0; index < m_HandleShapeAnimCount; ++index)
    {
        DestroyShapeAnim(index);
    }
    Free(m_pShapeAnimList);
    m_pShapeAnimList = NULL;
    for (int index = 0; index < m_HandleBoneVisibilityAnimCount; ++index)
    {
        DestroyBoneVisibilityAnim(index);
    }
    Free(m_pBoneVisibilityAnimList);
    m_pBoneVisibilityAnimList = NULL;

    m_IsInitialized = false;
}

ModelAnimObj::AnimType ModelAnimObj::GetAnimType(void* pResAnim) NN_NOEXCEPT
{
    nn::util::BinaryBlockHeader* pHeader = reinterpret_cast<nn::util::BinaryBlockHeader*>(pResAnim);
    if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResSkeletalAnim::Signature)))
    {
        return AnimType::Skeletal;
    }
    else if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResBoneVisibilityAnim::Signature)))
    {
        return AnimType::BoneVisibility;
    }
    else if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResMaterialAnim::Signature)))
    {
        return AnimType::Material;
    }
    else if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResShapeAnim::Signature)))
    {
        return AnimType::Shape;
    }
    return AnimType::Invalid;
}

int ModelAnimObj::CreateAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim, const nn::g3d::SkeletalAnimObj::BindArgument& bindArgument, AnimPlayState state) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pResSkeletalAnim);

    int index = CreateSkeletalAnim(pResSkeletalAnim, bindArgument, 1.0f);
    if (index == InvalidAnimIndex)
    {
        return InvalidAnimId;
    }
    else
    {
        SetSkeletalAnimState(index, state);
        return CreateAnimId(AnimType::Skeletal, index);
    }
}

int ModelAnimObj::CreateAnim(const nn::g3d::ResMaterialAnim* pResMaterialAnim, AnimPlayState state) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pResMaterialAnim);

    int index = CreateMaterialAnim(pResMaterialAnim);
    if (index == InvalidAnimIndex)
    {
        return InvalidAnimId;
    }
    else
    {
        SetMaterialAnimState(index, state);
        return CreateAnimId(AnimType::Material, index);
    }
}

int ModelAnimObj::CreateAnim(const nn::g3d::ResShapeAnim* pResShapeAnim, AnimPlayState state) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pResShapeAnim);

    int index = CreateShapeAnim(pResShapeAnim);
    if (index == InvalidAnimIndex)
    {
        return InvalidAnimId;
    }
    else
    {
        SetShapeAnimState(index, state);
        return CreateAnimId(AnimType::Shape, index);
    }
}

int ModelAnimObj::CreateAnim(const nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim, AnimPlayState state) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pResBoneVisibilityAnim);

    int index = CreateBoneVisibilityAnim(pResBoneVisibilityAnim);
    if (index == InvalidAnimIndex)
    {
        return InvalidAnimId;
    }
    else
    {
        SetBoneVisibilityAnimState(index, state);
        return CreateAnimId(AnimType::BoneVisibility, index);
    }
}

int ModelAnimObj::FindAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim) const NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pResSkeletalAnim);

    int index = FindSkeletalAnim(pResSkeletalAnim);
    return index != -1 ? CreateAnimId(AnimType::Skeletal, index) : InvalidAnimId;
}

int ModelAnimObj::FindAnim(const nn::g3d::ResMaterialAnim* pResMaterialAnim) const NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pResMaterialAnim);

    int index = FindMaterialAnim(pResMaterialAnim);
    return index != -1 ? CreateAnimId(AnimType::Material, index) : InvalidAnimId;
}

int ModelAnimObj::FindAnim(const nn::g3d::ResShapeAnim* pResShapeAnim) const NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pResShapeAnim);

    int index = FindShapeAnim(pResShapeAnim);
    return index != -1 ? CreateAnimId(AnimType::Shape, index) : InvalidAnimId;
}

int ModelAnimObj::FindAnim(const nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim) const NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pResBoneVisibilityAnim);

    int index = FindBoneVisibilityAnim(pResBoneVisibilityAnim);
    return index != -1 ? CreateAnimId(AnimType::BoneVisibility, index) : InvalidAnimId;
}

void ModelAnimObj::DestroyAnim(int id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    AnimType type = GetAnimType(id);
    int index = GetAnimIndex(id);
    switch (type)
    {
        case AnimType::Skeletal:
            DestroySkeletalAnim(index);
            break;
        case AnimType::Material:
            DestroyMaterialAnim(index);
            break;
        case AnimType::Shape:
            DestroyShapeAnim(index);
            break;
        case AnimType::BoneVisibility:
            DestroyBoneVisibilityAnim(index);
            break;
        default:
            break;
    }
}

ModelAnimObj::AnimPlayState ModelAnimObj::GetAnimState(int id) const NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    AnimType type = GetAnimType(id);
    int index = GetAnimIndex(id);
    switch (type)
    {
        case AnimType::Skeletal:
            return GetSkeletalAnimState(index);
        case AnimType::Material:
            return GetMaterialAnimState(index);
        case AnimType::Shape:
            return GetShapeAnimState(index);
        case AnimType::BoneVisibility:
            return GetBoneVisibilityAnimState(index);
        default:
            break;
    }
    return AnimPlayState::Stop;
}

void ModelAnimObj::SetAnimState(int id, AnimPlayState state) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    AnimType type = GetAnimType(id);
    int index = GetAnimIndex(id);
    switch (type)
    {
        case AnimType::Skeletal:
            SetSkeletalAnimState(index, state);
            break;
        case AnimType::Material:
            SetMaterialAnimState(index, state);
            break;
        case AnimType::Shape:
            SetShapeAnimState(index, state);
            break;
        case AnimType::BoneVisibility:
            SetBoneVisibilityAnimState(index, state);
            break;
        default:
            break;
    }
}

nn::g3d::ModelAnimObj* ModelAnimObj::GetAnimObj(int id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    AnimType type = GetAnimType(id);
    int index = GetAnimIndex(id);
    switch (type)
    {
    case AnimType::Skeletal:
        return GetSkeletalAnimObj(index);
    case AnimType::Material:
        return GetMaterialAnimObj(index);
    case AnimType::Shape:
        return GetShapeAnimObj(index);
    case AnimType::BoneVisibility:
        return GetBoneVisibilityAnimObj(index);
    default:
        break;
    }
    return NULL;
}

const nn::g3d::ModelAnimObj* ModelAnimObj::GetAnimObj(int id) const NN_NOEXCEPT
{
    AnimType type = GetAnimType(id);
    int index = GetAnimIndex(id);
    switch (type)
    {
        case AnimType::Skeletal:
            return GetSkeletalAnimObj(index);
        case AnimType::Material:
            return GetMaterialAnimObj(index);
        case AnimType::Shape:
            return GetShapeAnimObj(index);
        case AnimType::BoneVisibility:
            return GetBoneVisibilityAnimObj(index);
        default:
            break;
    }
    return NULL;
}

int ModelAnimObj::CreateSkeletalAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim, const nn::g3d::SkeletalAnimObj::BindArgument& bindArgument, float weight) NN_NOEXCEPT
{
    int index;
    for (index = 0; index < m_HandleSkeletalAnimCount; ++index)
    {
        if (!m_pSkeletalAnimList[index].IsInitialized())
        {
            break;
        }
    }

    if (index == m_HandleSkeletalAnimCount)
    {
        return -1;
    }

    nn::g3d::SkeletalAnimObj::Builder builder;
    builder.Reserve(pResSkeletalAnim);
    builder.Reserve(m_pModelObj->GetResource());
    if (bindArgument.GetHostResource() != NULL)
    {
        builder.SetRetargetingEnabled();
    }
    nn::g3d::SkeletalAnimObj* pSkeletalAnimObj = CreateAnimObj<nn::g3d::SkeletalAnimObj>(builder);
    pSkeletalAnimObj->SetResource(pResSkeletalAnim);
    pSkeletalAnimObj->Bind(bindArgument);
    m_pSkeletalAnimList[index].Initialize(pSkeletalAnimObj, weight);

    return index;
}

void ModelAnimObj::DestroySkeletalAnim(int index) NN_NOEXCEPT
{
    if (0 <= index && index < m_HandleSkeletalAnimCount)
    {
        SkeletalAnim& skeletalAnim = m_pSkeletalAnimList[index];
        if (skeletalAnim.IsInitialized())
        {
            DestroyAnimObj(skeletalAnim.GetAnimObj());
            skeletalAnim.Finalize();
        }
    }

    return;
}

int ModelAnimObj::FindSkeletalAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim) const NN_NOEXCEPT
{
    int index;
    for (index = 0; index < m_HandleSkeletalAnimCount; index++)
    {
        nns::g3d::ModelAnimObj::SkeletalAnim* pSkeletalAnim = &m_pSkeletalAnimList[index];
        if (pSkeletalAnim->IsInitialized() &&
            pSkeletalAnim->GetAnimObj()->GetResource() == pResSkeletalAnim)
        {
            return index;
        }
    }
    return -1;
}

int ModelAnimObj::CreateMaterialAnim(const nn::g3d::ResMaterialAnim* pResMaterialAnim) NN_NOEXCEPT
{
    int index;
    for (index = 0; index < m_HandleMaterialAnimCount; ++index)
    {
        if (!m_pMaterialAnimList[index].IsInitialized())
        {
            break;
        }
    }

    if (index == m_HandleMaterialAnimCount)
    {
        return -1;
    }

    nn::g3d::MaterialAnimObj::Builder builder;
    builder.Reserve(pResMaterialAnim);
    builder.Reserve(m_pModelObj->GetResource());
    nn::g3d::MaterialAnimObj* pMaterialAnimObj = CreateAnimObj<nn::g3d::MaterialAnimObj>(builder);
    pMaterialAnimObj->SetResource(pResMaterialAnim);
    pMaterialAnimObj->Bind(m_pModelObj);

    m_pMaterialAnimList[index].Initialize(pMaterialAnimObj);

    return index;
}

void ModelAnimObj::DestroyMaterialAnim(int index) NN_NOEXCEPT
{
    if (0 <= index && index < m_HandleMaterialAnimCount)
    {
        MaterialAnim& materialAnim = m_pMaterialAnimList[index];
        if (materialAnim.IsInitialized())
        {
            materialAnim.GetAnimObj()->RevertTo(m_pModelObj);
            DestroyAnimObj(materialAnim.GetAnimObj());
            materialAnim.Finalize();
        }
    }

    return;
}

int ModelAnimObj::FindMaterialAnim(const nn::g3d::ResMaterialAnim* pResMaterialAnim) const NN_NOEXCEPT
{
    int index;
    for (index = 0; index < m_HandleMaterialAnimCount; index++)
    {
        nns::g3d::ModelAnimObj::MaterialAnim* pMaterialAnim = &m_pMaterialAnimList[index];
        if (pMaterialAnim->IsInitialized() &&
            pMaterialAnim->GetAnimObj()->GetResource() == pResMaterialAnim)
        {
            return index;
        }
    }
    return -1;
}

int ModelAnimObj::CreateShapeAnim(const nn::g3d::ResShapeAnim* pResShapeAnim) NN_NOEXCEPT
{
    int index;
    for (index = 0; index < m_HandleShapeAnimCount; ++index)
    {
        if (!m_pShapeAnimList[index].IsInitialized())
        {
            break;
        }
    }

    if (index == m_HandleShapeAnimCount)
    {
        return -1;
    }

    nn::g3d::ShapeAnimObj::Builder builder;
    builder.Reserve(pResShapeAnim);
    builder.Reserve(m_pModelObj->GetResource());
    nn::g3d::ShapeAnimObj* pShapeAnimObj = CreateAnimObj<nn::g3d::ShapeAnimObj>(builder);
    pShapeAnimObj->SetResource(pResShapeAnim);
    pShapeAnimObj->Bind(m_pModelObj);

    m_pShapeAnimList[index].Initialize(pShapeAnimObj);

    return index;
}

void ModelAnimObj::DestroyShapeAnim(int index) NN_NOEXCEPT
{
    if (0 <= index && index < m_HandleShapeAnimCount)
    {
        ShapeAnim& shapeAnim = m_pShapeAnimList[index];
        if (shapeAnim.IsInitialized())
        {
            DestroyAnimObj(shapeAnim.GetAnimObj());
            shapeAnim.Finalize();
        }
    }

    return;
}

int ModelAnimObj::FindShapeAnim(const nn::g3d::ResShapeAnim* pResShapeAnim) const NN_NOEXCEPT
{
    int index;
    for (index = 0; index < m_HandleShapeAnimCount; index++)
    {
        nns::g3d::ModelAnimObj::ShapeAnim* pShapeAnim = &m_pShapeAnimList[index];
        if (pShapeAnim->IsInitialized() &&
            pShapeAnim->GetAnimObj()->GetResource() == pResShapeAnim)
        {
            return index;
        }
    }
    return -1;
}

int ModelAnimObj::CreateBoneVisibilityAnim(const nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim) NN_NOEXCEPT
{
    int index;
    for (index = 0; index < m_HandleBoneVisibilityAnimCount; ++index)
    {
        if (!m_pBoneVisibilityAnimList[index].IsInitialized())
        {
            break;
        }
    }

    if (index == m_HandleBoneVisibilityAnimCount)
    {
        return -1;
    }

    nn::g3d::BoneVisibilityAnimObj::Builder builder;
    builder.Reserve(pResBoneVisibilityAnim);
    builder.Reserve(m_pModelObj->GetResource());
    nn::g3d::BoneVisibilityAnimObj* pBoneVisibilityAnimObj = CreateAnimObj<nn::g3d::BoneVisibilityAnimObj>(builder);
    pBoneVisibilityAnimObj->SetResource(pResBoneVisibilityAnim);
    pBoneVisibilityAnimObj->Bind(m_pModelObj);

    m_pBoneVisibilityAnimList[index].Initialize(pBoneVisibilityAnimObj);

    return index;
}

void ModelAnimObj::DestroyBoneVisibilityAnim(int index) NN_NOEXCEPT
{
    if (0 <= index && index < m_HandleBoneVisibilityAnimCount)
    {
        BoneVisibilityAnim& boneVisibilityAnim = m_pBoneVisibilityAnimList[index];
        if (boneVisibilityAnim.IsInitialized())
        {
            DestroyAnimObj(boneVisibilityAnim.GetAnimObj());
            boneVisibilityAnim.Finalize();
        }
    }

    return;
}

int ModelAnimObj::FindBoneVisibilityAnim(const nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim) const NN_NOEXCEPT
{
    int index;
    for (index = 0; index < m_HandleBoneVisibilityAnimCount; index++)
    {
        nns::g3d::ModelAnimObj::BoneVisibilityAnim* pBoneVisibilityAnim = &m_pBoneVisibilityAnimList[index];
        if (pBoneVisibilityAnim->IsInitialized() &&
            pBoneVisibilityAnim->GetAnimObj()->GetResource() == pResBoneVisibilityAnim)
        {
            return index;
        }
    }
    return -1;
}

void ModelAnimObj::CalculateView(int viewIndex, int bufferIndex, const nn::util::Matrix4x3fType& viewMtx) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    m_pModelObj->CalculateView(viewIndex, viewMtx, bufferIndex);
}

void ModelAnimObj::CalculateAnimations() NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    // マテリアルアニメーション計算
    for (int index = 0; index < m_HandleMaterialAnimCount; ++index)
    {
        MaterialAnim& materialAnim = m_pMaterialAnimList[index];
        if (materialAnim.IsInitialized() && materialAnim.GetState() == AnimPlayState::Play)
        {
            nn::g3d::MaterialAnimObj* pMaterialAnimObj = materialAnim.GetAnimObj();
            pMaterialAnimObj->Calculate();
            pMaterialAnimObj->ApplyTo(m_pModelObj);
            pMaterialAnimObj->GetFrameCtrl().UpdateFrame();
        }
    }

    // シェイプアニメーション計算
    // フラグをクリア
    for (int index = 0; index <  m_pModelObj->GetShapeCount(); ++index)
    {
        nn::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(index);
        if (pShapeObj->HasValidBlendWeight())
        {
            pShapeObj->ClearBlendWeights();
        }
    }
    for (int index = 0; index < m_HandleShapeAnimCount; ++index)
    {
        ShapeAnim& shapeAnim = m_pShapeAnimList[index];
        if (shapeAnim.IsInitialized() && shapeAnim.GetState() == AnimPlayState::Play)
        {
            nn::g3d::ShapeAnimObj* pShapeAnimObj = shapeAnim.GetAnimObj();
            pShapeAnimObj->Calculate();
            pShapeAnimObj->ApplyTo(m_pModelObj);
            pShapeAnimObj->GetFrameCtrl().UpdateFrame();
        }
    }

    // ボーンビジビリティーアニメーション計算
    for (int index = 0; index < m_HandleBoneVisibilityAnimCount; ++index)
    {
        BoneVisibilityAnim& boneVisibilityAnim = m_pBoneVisibilityAnimList[index];
        if (boneVisibilityAnim.IsInitialized() && boneVisibilityAnim.GetState() == AnimPlayState::Play)
        {
            nn::g3d::BoneVisibilityAnimObj* pBoneVisibilityAnimObj = boneVisibilityAnim.GetAnimObj();
            pBoneVisibilityAnimObj->Calculate();
            pBoneVisibilityAnimObj->ApplyTo(m_pModelObj);
            pBoneVisibilityAnimObj->GetFrameCtrl().UpdateFrame();
        }
    }

    int skeletalAnimIndex = -1;
    int skeletalAnimCount = 0;
    float weightSum = 0.0f;
    for (int index = 0; index < m_HandleSkeletalAnimCount; ++index)
    {
        if (m_pSkeletalAnimList[index].IsInitialized() && m_pSkeletalAnimList[index].GetState() == AnimPlayState::Play)
        {
            ++skeletalAnimCount;
            SkeletalAnim& skeletalAnim = m_pSkeletalAnimList[index];
            weightSum += skeletalAnim.GetWeight();
            skeletalAnimIndex = index;
        }
    }
    // スケルタルアニメーション計算
    if (skeletalAnimCount == 1)
    {
        SkeletalAnim& skeletalAnim = m_pSkeletalAnimList[skeletalAnimIndex];
        {
            nn::g3d::SkeletalAnimObj* pSkeletalAnimObj = skeletalAnim.GetAnimObj();
            pSkeletalAnimObj->Calculate();
            pSkeletalAnimObj->ApplyTo(m_pModelObj);
            pSkeletalAnimObj->GetFrameCtrl().UpdateFrame();
        }
    }
    else if (skeletalAnimCount > 1)
    {
        float invWeightSum = nn::util::Rcp(weightSum);
        m_SkeletalAnimBlender.ClearResult();
        for (int index = 0; index < m_HandleSkeletalAnimCount; ++index)
        {
            SkeletalAnim& skeletalAnim = m_pSkeletalAnimList[index];
            if (skeletalAnim.IsInitialized() && skeletalAnim.GetState() == AnimPlayState::Play)
            {
                nn::g3d::SkeletalAnimObj* pSkeletalAnimObj = skeletalAnim.GetAnimObj();
                m_SkeletalAnimBlender.Blend(pSkeletalAnimObj, skeletalAnim.GetWeight() * invWeightSum);
                pSkeletalAnimObj->GetFrameCtrl().UpdateFrame();
            }
        }
        m_SkeletalAnimBlender.ApplyTo(m_pModelObj->GetSkeleton());
    }
}

void ModelAnimObj::CalculateWorld() NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    nn::util::MatrixSetScaleRotateXyz(&m_BaseMtx, m_Scale, m_Rotate);
    nn::util::MatrixSetTranslate(&m_BaseMtx, m_Translate);

    if (m_pBindModel && m_BindBoneIndex >= 0)
    {
        // バインド指定されている場合、指定されたボーンのワールド行列を親の行列として乗算します。
        // バインド対象のモデルが先に計算されている必要があります。
        const nn::g3d::SkeletonObj* pBindSkeletonObj = m_pBindModel->GetSkeleton();
        const nn::util::Matrix4x3fType* pWorldMtxArray = pBindSkeletonObj->GetWorldMtxArray();
        NN_ASSERT_RANGE(m_BindBoneIndex, 0, pBindSkeletonObj->GetBoneCount());
        MatrixMultiply(&m_BaseMtx, m_BaseMtx, pWorldMtxArray[m_BindBoneIndex]);
    }

    m_pModelObj->CalculateWorld(m_BaseMtx);
}

void ModelAnimObj::CalculateBounding() NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    if (m_pModelObj->GetBounding())
    {
        for (int lodLevel = 0; lodLevel < m_pModelObj->GetLodCount(); ++lodLevel)
        {
            m_pModelObj->CalculateBounding(lodLevel);
        }
        for (int shapeIndex = 0; shapeIndex < m_pModelObj->GetShapeCount(); ++shapeIndex)
        {
            nn::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(shapeIndex);
            for (int meshIndex = 0; meshIndex < pShapeObj->GetMeshCount(); ++meshIndex)
            {
                pShapeObj->CalculateSubMeshBounding(m_pModelObj->GetSkeleton(), meshIndex);
            }
        }
    }
}

void ModelAnimObj::Calculate() NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    // アニメーション更新
    CalculateAnimations();

    // ワールド変換行列を更新
    CalculateWorld();

    // バウンディング計算
    CalculateBounding();

} // NOLINT

void ModelAnimObj::CalculateUniformBlock(int bufferIndex) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    m_pModelObj->CalculateMaterial(bufferIndex);
    m_pModelObj->CalculateSkeleton(bufferIndex);
    m_pModelObj->CalculateShape(bufferIndex);
}

}}
