﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/dbgui/dbgui_Interface.h>

#include "dbgui_Allocator.h"
#include "dbgui_Renderer.h"
#include "dbgui_UserInputs.h"
#include "dbgui_ImguiIO.h"

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Tick.h>

namespace nns { namespace dbgui {

namespace {

void* AllocateMemoryForImGui(size_t sz, void* pUserData)
{
    static const size_t DefaultAlignment = 8;
    Allocator* pAllocator = reinterpret_cast<Allocator*>(pUserData);
    return pAllocator->AllocateMemory(sz, DefaultAlignment);
}

void FreeMemoryForImGui(void* ptr, void* pUserData)
{
    Allocator* pAllocator = reinterpret_cast<Allocator*>(pUserData);
    pAllocator->FreeMemory(ptr);
}

struct InterfaceState
{
    bool                isInitialized               = false;
    nn::os::Tick        previousFrameTime           = nn::os::Tick(0);

    Allocator           allocator;
    Renderer            renderer;
    UserInputs          userInputs;

    MouseMoveFunction   pMouseMoveFunction          = nullptr;
    void*               pMouseMoveFunctionUserData  = nullptr;

    static const int    IniFilePathMaxLength = 1024;
    char                iniFilePath[IniFilePathMaxLength];
};

InterfaceState g_State;



}

void Initialize(nn::gfx::Device* pDevice, const InterfaceInfo& info) NN_NOEXCEPT
{
    NN_ASSERT(!g_State.isInitialized);

    g_State.pMouseMoveFunction = info.GetMouseMoveFunction();
    g_State.pMouseMoveFunctionUserData = info.GetMouseMoveFunctionUserData();

    memset(g_State.iniFilePath, 0, sizeof(g_State.iniFilePath));
    if (info.GetIniFilePath() != nullptr)
    {
        strncpy(g_State.iniFilePath, info.GetIniFilePath(), sizeof(g_State.iniFilePath) - 1);
    }

    g_State.allocator.Initialize(
        info.GetMemoryCallbackAllocate(),
        info.GetMemoryCallbackFree(),
        info.GetMemoryCallbackUserData());

    io::ImguiIoInfo imguiIoInfo = {};
    imguiIoInfo.pMemoryCallbackAllocate         = AllocateMemoryForImGui;
    imguiIoInfo.pMemoryCallbackFree             = FreeMemoryForImGui;
    imguiIoInfo.pMemoryCallbackUserData         = &g_State.allocator;
    imguiIoInfo.pCustomFontBuffer               = info.GetCustomTtfBuffer();
    imguiIoInfo.customFontBufferSize            = info.GetCustomTtfBufferSize();
    imguiIoInfo.customFontIsCompressed          = info.IsCustomTtfCompressed();
    imguiIoInfo.customFontCharacterSize         = info.GetCustomTtfCharacterSize();
    imguiIoInfo.displayWidth                    = info.GetImguiDisplayWidth();
    imguiIoInfo.displayHeight                   = info.GetImguiDisplayHeight();
    imguiIoInfo.iniFilePath                     = g_State.iniFilePath[0] == '\0' ? g_State.iniFilePath : nullptr;
    imguiIoInfo.pNpadNavigationKeyBindings      = info.GetNpadNavigationKeyBindings();
    imguiIoInfo.pDebugNpadNavigationKeyBindings = info.GetDebugPadNavigationKeyBindings();
    imguiIoInfo.mouseWheelScrollRatio           = info.GetMouseWheelScrollRatio();

    io::Initialize(&imguiIoInfo);

    // ComputeGpuPoolMemoryRequiredはimguiのフォントの設定に依存しています。
    size_t requiredPoolMemorySize =
        Renderer::ComputeGpuPoolMemoryRequired(pDevice, info.GetBufferCount());
#if defined(NNS_DBGUI_MEMORYPOOL_CPU_CACHED)
    int memoryPoolProprertyFlags = nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_CpuCached;
#else
    int memoryPoolProprertyFlags = nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_CpuUncached;
#endif
    g_State.allocator.InitializeGpuMemoryPool(
        pDevice,
        requiredPoolMemorySize,
        memoryPoolProprertyFlags);

    g_State.renderer.Initialize(info, &g_State.allocator);
    g_State.renderer.InitializeGfxObjects(pDevice);

    int imguiDisplayWidth = g_State.renderer.GetImguiDisplayWidth();
    int imguiDisplayHeight = g_State.renderer.GetImguiDisplayHeight();
    g_State.userInputs.Initialize(
        &g_State.allocator,
        imguiDisplayWidth, imguiDisplayHeight);

    g_State.previousFrameTime = nn::os::Tick(0);

    g_State.isInitialized = true;
}

void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT(g_State.isInitialized);

    g_State.userInputs.Finalize();

    g_State.renderer.FinalizeGfxObjects(pDevice);
    g_State.renderer.Finalize();

    g_State.allocator.FinalizeGpuMemoryPool(pDevice);

    io::Finalize();

    g_State.allocator.Finalize(pDevice);

    g_State.pMouseMoveFunction = nullptr;
    g_State.pMouseMoveFunctionUserData = nullptr;

    g_State.isInitialized = false;
}

bool IsInitialized()  NN_NOEXCEPT
{
    return g_State.isInitialized;
}

void AddInputCharactersFromKeyboardState(const nn::hid::KeyboardState& keyboardState) NN_NOEXCEPT
{
    g_State.userInputs.AddInputCharactersFromKeyboardState(keyboardState);
}

void AddInputCharacter(char c) NN_NOEXCEPT
{
    g_State.userInputs.AddInputCharacter(c);
}

void AddInputCharactersUtf8(const char* c) NN_NOEXCEPT
{
    g_State.userInputs.AddInputCharactersUtf8(c);
}

#if defined(NNS_DBGUI_ENABLE_SOFTWARE_KEYBOARD)
void UpdateSoftwareKeyboard() NN_NOEXCEPT
{
    g_State.userInputs.UpdateSoftwareKeyboard();
}
#endif

void SetKeyboardState(const nn::hid::KeyboardState& keyboardState) NN_NOEXCEPT
{
    NN_ASSERT(g_State.isInitialized);
    g_State.userInputs.SetKeyboardState(keyboardState);
}

void SetDebugPadState(
    const nn::hid::DebugPadButtonSet& debugPadButtonSet,
    const nn::hid::AnalogStickState& analogStickState) NN_NOEXCEPT
{
    NN_ASSERT(g_State.isInitialized);
    g_State.userInputs.SetDebugPadState(debugPadButtonSet, analogStickState);
}

void SetNpadState(
    const nn::hid::NpadButtonSet& npadButtonSet,
    const nn::hid::AnalogStickState& analogStickState) NN_NOEXCEPT
{
    NN_ASSERT(g_State.isInitialized);
    g_State.userInputs.SetNpadState(npadButtonSet, analogStickState);
}

void SetMouseState(const nn::hid::MouseState& mouseState) NN_NOEXCEPT
{
    NN_ASSERT(g_State.isInitialized);
    if (!io::IsMoveMouseRequested())
    {
        g_State.userInputs.SetMouseState(mouseState);
    }
}

void SetTouchState(const nn::hid::TouchState& touchState) NN_NOEXCEPT
{
    NN_ASSERT(g_State.isInitialized);
    if (!io::IsMoveMouseRequested())
    {
        g_State.userInputs.SetTouchState(touchState);
    }
}

void NewFrame() NN_NOEXCEPT
{
    NN_ASSERT(g_State.isInitialized);

    nn::os::Tick currentTime = nn::os::GetSystemTick();
    nn::TimeSpan frameDuration;

    if (g_State.previousFrameTime.GetInt64Value() == 0)
    {
        frameDuration = nn::TimeSpan::FromMicroSeconds(16667);
    }
    else
    {
        frameDuration = (currentTime - g_State.previousFrameTime).ToTimeSpan();
    }
    g_State.previousFrameTime = currentTime;

    if ((g_State.pMouseMoveFunction != nullptr) && io::IsMoveMouseRequested())
    {
        int posX = 0;
        int posY = 0;
        io::GetMoveMouseRequestedPos(&posX, &posY);
        g_State.userInputs.ConvertCoordinatesImguiToInput(&posX, &posY);
        g_State.pMouseMoveFunction(posX, posY, g_State.pMouseMoveFunctionUserData);
    }

    io::NewFrame(frameDuration);
}

void Render(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_ASSERT(g_State.isInitialized);

    io::Render();
    g_State.renderer.RenderDrawLists(pCommandBuffer, io::GetDrawData());

    g_State.userInputs.ResetPadNavigationInputs();
    g_State.userInputs.ResetMouseAndTouchScreenInputs();
}

} } // namespace nns { namespace dbgui {

