﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dbgui_Allocator.h"

#include <nn/nn_Assert.h>

namespace nns { namespace dbgui {

Allocator::Allocator()
: m_GpuMemory()
, m_pMemoryCallbackAllocate(nullptr)
, m_pMemoryCallbackFree(nullptr)
, m_pMemoryCallbackUserData(nullptr)
{
}

Allocator::~Allocator()
{
}

void Allocator::Initialize(
    nn::AlignedAllocateFunctionWithUserData pMemoryCallbackAllocate,
    nn::FreeFunctionWithUserData pMemoryCallbackFree,
    void* pMemoryCallbackUserData)
{
    m_pMemoryCallbackAllocate   = pMemoryCallbackAllocate;
    m_pMemoryCallbackFree       = pMemoryCallbackFree;
    m_pMemoryCallbackUserData   = pMemoryCallbackUserData;

    m_GpuMemory.pBuffer         = nullptr;
    m_GpuMemory.size            = 0;
    m_GpuMemory.offset          = 0;
}

void Allocator::Finalize(nn::gfx::Device* pDevice)
{
    FinalizeGpuMemoryPool(pDevice);

    m_pMemoryCallbackAllocate = nullptr;
    m_pMemoryCallbackFree = nullptr;
    m_pMemoryCallbackUserData = nullptr;
}

void Allocator::InitializeGpuMemoryPool(
    nn::gfx::Device* pDevice, size_t requiredPoolMemorySize, int memoryPoolPropertyFlags)
{
    nn::gfx::MemoryPool::InfoType memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty(memoryPoolPropertyFlags);

    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memoryPoolInfo);
    size_t memoryPoolBufferSize = nn::util::align_up(
        requiredPoolMemorySize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memoryPoolInfo));

    m_GpuMemory.pBuffer = AllocateMemory(memoryPoolBufferSize, alignment);
    m_GpuMemory.size = memoryPoolBufferSize;
    m_GpuMemory.offset = 0;

    memoryPoolInfo.SetPoolMemory(m_GpuMemory.pBuffer, memoryPoolBufferSize);
    m_GpuMemory.m_MemoryPool.Initialize(pDevice, memoryPoolInfo);
}

void Allocator::FinalizeGpuMemoryPool(nn::gfx::Device* pDevice)
{
    if (m_GpuMemory.size > 0)
    {
        m_GpuMemory.m_MemoryPool.Finalize(pDevice);

        FreeMemory(m_GpuMemory.pBuffer);
        m_GpuMemory.pBuffer = nullptr;
        m_GpuMemory.size = 0;
        m_GpuMemory.offset = 0;
    }
}

ptrdiff_t Allocator::AllocateGpuMemoryPool(size_t size, size_t alignement)
{
    ptrdiff_t offset = m_GpuMemory.offset;

    offset = nn::util::align_up(offset, alignement);

    m_GpuMemory.offset = offset + size;

    NN_ASSERT(static_cast<size_t>(m_GpuMemory.offset) <= m_GpuMemory.size);

    return offset;
}

}
} // namespace nns { namespace dbgui {
