﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <cmath>
#include "UIControl.h"
#include "WlanTestTypes.h"


namespace WlanTest {

/*
  UIControl
*/

UIControl::UIControl()
{
    X = 0;
    Y = 0;
    Width  = 0;
    Height = 0;
    Parent = NULL;
}

UIControl::~UIControl()
{}

void UIControl::Show(Display& display)
{
    if(Parent == NULL)
    {
        //display.Clear();
    }

    ShowImpl(display);

    if(Parent == NULL)
    {
        //display.SwapBuffer();
    }
}

void UIControl::InputPad(Pad&)
{}


/*
  LineGraph
*/

LineGraph::LineGraph(uint8_t points)
{
    m_PointSize = 4;
    m_Points = points;
    m_PointsX = new float[points];
    m_PointsY = new float[points];
    for(int i=0; i<points; i++)
    {
        m_PointsX[i] = 0.0;
        m_PointsY[i] = 0.0;
    }
    MaxX = 100;
    MaxY = 100;
    MinX = 0;
    MinY = 0;
    ScaleX = 10;
    ScaleY = 10;
    ScaleOffsetX = 26;
    ScaleOffsetY = 10;
    ShowData = false;
}

LineGraph::~LineGraph()
{
    delete [] m_PointsX;
    delete [] m_PointsY;
}

void LineGraph::ShowImpl(Display& display)
{
    ostringstream oss;
    display.SetColor(m_ScaleColor);

    uint8_t figure = std::log10(MaxY) + 1;
    ScaleOffsetX = figure * 8 + 2;

    float GraphX = GetX() + ScaleOffsetX;
    float GraphY = GetY();
    float GraphWidth = Width - ScaleOffsetX;
    float GraphHeight = Height - ScaleOffsetY;
    float GraphScaleX = MaxX - MinX;
    float GraphScaleY = MaxY - MinY;

    // 目盛りを表示
    display.DrawLine(GetX() + ScaleOffsetX, GetY(), GetX() + ScaleOffsetX, GetY() + GraphHeight);
    display.DrawLine(GetX() + ScaleOffsetX, GetY() + GraphHeight, GetX() + GraphWidth, GetY() + GraphHeight);
    if(ScaleX != 0)
    {
        for(float i=MinX; i<MaxX; i+=ScaleX)
        {
            oss << i;
            display.DrawText( (i - MinX) / GraphScaleX * GraphWidth + GraphX - 4, GraphY + GraphHeight + 4, oss.str());
            oss.str("");
        }
        if((int)GraphScaleX % (int)ScaleX != 0)
        {
            oss << MaxX;
            display.DrawText( GraphWidth + GraphX - 4, GraphY + GraphHeight + 4, oss.str());
            oss.str("");
        }
    }
    if(ScaleY != 0)
    {
        for(float i=MinY; i<=MaxY; i+=ScaleY)
        {
            oss << setw(figure) << right << i << left;
            display.DrawText(GetX(), GetY() + GraphHeight - (i - MinY) / GraphScaleY * GraphHeight - 4, oss.str());
            oss.str("");
        }
        if((int)GraphScaleY % (int)ScaleY != 0)
        {
            oss << setw(figure) << right << MaxY << left;
            display.DrawText(GetX(), GetY() - 4, oss.str());
            oss.str("");
        }
    }

    // グラフを表示
    display.SetColor(m_Color);
    display.SetPointSize(m_PointSize);
    for(int i=0; i<m_Points - 1; i++)
    {
        display.DrawLine(GraphX + (m_PointsX[i] - MinX) / GraphScaleX * GraphWidth,
                         GraphY + GraphHeight - ((m_PointsY[i] - MinY) / GraphScaleY * GraphHeight),
                         GraphX + (m_PointsX[i + 1] - MinX) / GraphScaleX * GraphWidth,
                         GraphY + GraphHeight - ((m_PointsY[i + 1] - MinY) / GraphScaleY * GraphHeight));
        display.DrawPoint(GraphX + (m_PointsX[i] - MinX) / GraphScaleX * GraphWidth,
                          GraphY + GraphHeight - ((m_PointsY[i] - MinY) / GraphScaleY * GraphHeight));
        if(ShowData)
        {
            oss << m_PointsY[i];
            display.DrawText(GraphX + (m_PointsX[i] - MinX) / GraphScaleX * GraphWidth + 2,
                             GraphY + GraphHeight - ((m_PointsY[i] - MinY) / GraphScaleY * GraphHeight) - 8,
                             oss.str());
            oss.str("");
        }
    }
    display.DrawPoint(GraphX + (m_PointsX[m_Points - 1] - MinX) / GraphScaleX * GraphWidth,
                      GraphY + GraphHeight - ((m_PointsY[m_Points - 1] - MinY) / GraphScaleY * GraphHeight));
    if(ShowData)
    {
        oss << m_PointsY[m_Points - 1];
        display.DrawText(GraphX + (m_PointsX[m_Points - 1] - MinX) / GraphScaleX * GraphWidth + 2,
                         GraphY + GraphHeight - ((m_PointsY[m_Points - 1] - MinY) / GraphScaleY * GraphHeight) - 8,
                         oss.str());
        oss.str("");
    }
}



/*
  Rectangle
*/

Rectangle::Rectangle() {}
Rectangle::~Rectangle() {}

void Rectangle::ShowImpl(Display& display)
{
    display.SetColor(m_Color);
    display.DrawRectangle(X, Y,Width, Height);
}


/*
  PercentageBar
*/

PercentageBar::PercentageBar()
{
    m_Percentage = 0;
    m_IsHorizontal = true;
}

PercentageBar::~PercentageBar()
{}

void PercentageBar::ShowImpl(Display& display)
{
    display.SetColor(m_Color);
    if(m_IsHorizontal)
    {
        display.DrawRectangle(X, Y, Width * m_Percentage, Height);
        display.DrawText(X + Width * m_Percentage + 10, Y + Height / 2, m_Text);
    }
    else
    {
        display.DrawRectangle(X, Y + (Height * (1 - m_Percentage)), Width , Height * m_Percentage);
        if(Height * (m_Percentage) > 15)
        {
            display.SetColor(ToNegative(ToBlackOrWhite(m_Color)));
            display.DrawText(X + (Width - (m_Text.size() * 8)) / 2 , Y + (Height * (1 - m_Percentage)) + 5, m_Text);
        }
        else
        {
            display.DrawText(X + (Width - (m_Text.size() * 8)) / 2 , Y + (Height * (1 - m_Percentage)) - 10, m_Text);
        }
    }
}


/*
  PercentageBar2
*/

PercentageBar2::PercentageBar2()
{
    Clear();
}

PercentageBar2::~PercentageBar2()
{}

void PercentageBar2::ShowImpl(Display& display)
{
    PercentageBar::ShowImpl(display);


    display.SetColor(m_PeakColor);
    if(m_IsHorizontal)
    {
        display.DrawLine(X + Width * m_Peak, Y, X + Width * m_Peak, Y + Height);
        display.DrawText(X + Width * m_Peak + 10, Y + Height / 2, m_PeakText);
    }
    else
    {
        display.DrawLine(X, Y + (Height * (1 - m_Peak)), X + Width, Y + (Height * (1 - m_Peak)));
        display.DrawText(X + (Width - (m_PeakText.size() * 8)) / 2 , Y + (Height * (1 - m_Peak)) - 10, m_PeakText);
    }

    display.SetColor(m_AverageColor);
    if(m_IsHorizontal)
    {
        display.DrawLine(X + Width * m_Average.GetValue(), Y, X + Width * m_Peak, Y + Height);
        display.DrawText(X + Width * m_Average.GetValue() + 10, Y + Height / 2, m_AverageText);
    }
    else
    {
        display.DrawLine(X, Y + (Height * (1 - m_Average.GetValue())), X + Width - 5, Y + (Height * (1 - m_Average.GetValue())));
        //display.DrawText(X + (Width - (m_AverageText.size() * 8)) / 2 , Y + (Height * (1 - m_Average.GetValue())) - 10, m_AverageText);
        display.DrawText(X + Width - 7 , Y + (Height * (1 - m_Average.GetValue())) - 5, m_AverageText);
    }
}


/*
  Page
*/

void Page::InvokeInput()
{
    list<UIControl*>::iterator p = controls.begin();
    Pad& pad = Pad::GetInstance();
    InputPad(pad);

    while( p != controls.end() )
    {
        (*p)->InputPad(pad);
        p++;
    }
}
void Page::InvokeShow(Display& display)
{
    list<UIControl*>::iterator p = controls.begin();
    while( p!= controls.end() )
    {
        (*p)->Show(display);
        p++;
    }
}

void Page::ShowImpl(Display& display)
{
    // ##? コンソール出力をするため、InputPadで全てを処理してフラグを変更したのちに表示処理する
    // while(p!= controls.end())
    // {
    //     (*p)->InputPad(pad);
    //     (*p)->Show(display);
    //     p++;
    // }

    InvokeInput();

    // 入力後の要素の値により、表示・非表示の設定や要素の値を更新する
    UpdateElements();

    InvokeShow(display);
}

void Page::Add(UIControl& ui)
{
    ui.Parent = this;
    controls.push_back(&ui);
}

void Page::Remove(UIControl& ui)
{
    ui.Parent = NULL;
    controls.remove(&ui);
}

void Page::Clear()
{
    list<UIControl*>::iterator p = controls.begin();
    while(p !=  controls.end())
    {
        (*p)->Parent = NULL;
        p++;
    }

    controls.clear();
}

void Page::UpdateElements()
{
}

/*
  SwitchablePage
*/

void SwitchablePage::ShowImpl(Display& display)
{
    InputPad(Pad::GetInstance());

    if(m_Pages.empty())
    {
        return;
    }

    if(m_Switching)
    {
        float offset = (0 - m_NextX) / m_Smoothness;
        if(-1 < offset && offset < 1)
        {
            offset = 0 - m_NextX;
            m_Switching = false;
        }
        m_CurrentX += offset;
        m_NextX += offset;

        float orgCurrentX = m_Current->X;
        float orgNextX = m_Next->X;
        m_Next->X    = m_NextX;
        m_Current->X = m_CurrentX;
        m_Next->ShowImpl(display);
        m_Current->ShowImpl(display);
        m_Next->X = orgNextX;
        m_Current->X = orgCurrentX;
    }
    else
    {
        (*m_PagesIterator)->ShowImpl(display);
    }
}

void SwitchablePage::InputPad(Pad& pad)
{
    if(pad.IsTrigger(NextKey))
    {
        pad.Clear();

        m_Switching = true;
        m_CurrentX = m_NextX;
        m_Current = *m_PagesIterator;

        m_NamesIterator++;
        m_PagesIterator++;

        if( m_NamesIterator == m_Names.end() )
        {
            m_NamesIterator = m_Names.begin();
            m_PagesIterator = m_Pages.begin();
        }
        m_Next = *m_PagesIterator;
        m_NextX = m_CurrentX + Width;
    }
    else if(pad.IsTrigger(BackKey))
    {
        pad.Clear();

        m_Switching = true;
        m_CurrentX = m_NextX;
        m_Current = *m_PagesIterator;

        if( m_NamesIterator != m_Names.begin() )
        {
            m_NamesIterator--;
            m_PagesIterator--;
        }
        else
        {
            m_NamesIterator = m_Names.end();
            m_PagesIterator = m_Pages.end();
            m_NamesIterator--;
            m_PagesIterator--;
        }
        m_Next = *m_PagesIterator;
        m_NextX = m_CurrentX - Width;
    }
}

void SwitchablePage::AddPage(string name, Page& page)
{
    m_Names.push_back(name);
    m_Pages.push_back(&page);
    m_NamesIterator = m_Names.begin();
    m_PagesIterator = m_Pages.begin();
}

SwitchablePage::SwitchablePage()
{
    m_Switching = false;
    m_Smoothness = 1.5;
    m_CurrentX = 0;
    m_NextX = 0;
    m_Next = NULL;
    NextKey = Button::R;
    BackKey = Button::L;
}

void SwitchablePage::Clear()
{
    // vector<Page*>::iterator p = m_Pages.begin();
    // while(p != m_Pages.end())
    // {
    //     (*p)->Clear();
    //     p++;
    // }
    m_Pages.clear();
    m_Names.clear();
}

Page& SwitchablePage::GetCurrentPage()
{
    return **m_PagesIterator;
}

void SwitchablePage::SetSmoothness(float val)
{
    m_Smoothness = val;
}
void SwitchablePage::ChangeNextKey(const Button& button)
{
    NextKey = button;
}

void SwitchablePage::ChangeBackKey(const Button& button)
{
    BackKey = button;
}


/*
  Label
*/

void Label::ShowImpl(Display& display)
{
    display.SetFontSize(FontWidth, FontHeight);
    display.SetFixedWidth(FixedWidth);
    display.SetColor(BackgroundColor);
    display.DrawRectangle(GetX(), GetY(), Width, Height);
    display.SetColor(TextColor);

    string t;
    uint32_t displayLength = Width / FixedWidth;

    float x;
    switch(Alignment)
    {
    case TopLeft:
    case MiddleLeft:
    case BottomLeft:
        t = Text.substr(0, displayLength);
        x = GetX();
        break;
    case TopCenter:
    case MiddleCenter:
    case BottomCenter:
        t = Text.substr( Text.size() > displayLength ? (Text.size() / 2 - displayLength / 2) : 0, displayLength);
        x = GetX() + ((Width - (t.size() * FontWidth)) / 2);
        break;
    case TopRight:
    case MiddleRight:
    case BottomRight:
        t = Text.substr( Text.size() > displayLength ? (Text.size() - displayLength) : 0, displayLength);
        x = GetX() + (Width - (t.size() * FontWidth));
        break;
    default:
        break;
    }

    float y;
    switch(Alignment)
    {
    case TopLeft:
    case TopCenter:
    case TopRight:
        y = GetY();
        break;
    case MiddleLeft:
    case MiddleCenter:
    case MiddleRight:
        y = GetY() + ((Height - FontHeight) / 2);
        break;
    case BottomLeft:
    case BottomCenter:
    case BottomRight:
        y = GetY() + (Height - FontHeight);
        break;
    default:
        break;
    }

    display.DrawText(x, y, t);
}

void Label::FitSize()
{
    Width = Text.size() * FontWidth;
    Height = FontHeight;
}

Label::Label()
{
    FontWidth = DEFAULT_FONT_WIDTH;
    FontHeight = DEFAULT_FONT_HEIGHT;
    FixedWidth = DEFAULT_FIXED_WIDTH;
    Width = FontWidth * 100;
    Height = 8;
    BackgroundColor = ToColor(BLACK);
    TextColor = ToColor(WHITE);
    Alignment = TopLeft;
}


/*
  ScrollableTextPage
*/

void ScrollableTextPage::ShowIndicator(Display& display)
{
    // Indicator が動かされたときのオフセット(ページのスイッチ時など)
    float offsetX = GetX();
    float offsetY = GetY();

    if( m_PagesIterator == m_Pages.begin() && !m_IsLoopEnabled )
    {
        m_LeftIndicator.Inactivate();
    }
    else
    {
        m_LeftIndicator.Activate();
    }

    if( m_PagesIterator + 1 == m_Pages.end() && !m_IsLoopEnabled )
    {
        m_RightIndicator.Inactivate();
    }
    else
    {
        m_RightIndicator.Activate();
    }

    float fontSize = Display::GetInstance().GetFixedWidth();
    float displayRange = m_DisplayRangeRight - m_DisplayRangeLeft;
    m_LeftIndicator.X = offsetX - fontSize + 10;
    m_LeftIndicator.Y = offsetY - 15;
    m_LeftIndicator.ShowImpl(display);

    m_RightIndicator.X = offsetX + displayRange - 5;
    m_RightIndicator.Y = offsetY - 15;
    m_RightIndicator.ShowImpl(display);
}

void ScrollableTextPage::ShowTextPage(Display& display)
{
    // ScrollableTextPage が動かされたときのオフセット(ページのスイッチ時など)
    float offsetX = GetX();
    float offsetY = GetY();

    if( m_Switching )
    {
        float offset = (0 - m_NextX) / m_Smoothness;
        if( -5 < offset && offset < 5 )
        {
            offset = 0 - m_NextX;
            m_Switching = false;
        }
        m_CurrentX += offset;
        m_NextX += offset;

        // スクロール前の情報を一時保存
        vector<float> orgCurrentXs;
        vector<float> orgCurrentYs;
        vector<string> orgCurrentTexts;
        for(auto *pLabel : *m_Current)
        {
            orgCurrentXs.push_back(pLabel->X);
            orgCurrentYs.push_back(pLabel->Y);
            pLabel->X += m_CurrentX + offsetX;
            pLabel->Y += offsetY;

            orgCurrentTexts.push_back(pLabel->Text);
            pLabel->Text = Trim(pLabel->Text, pLabel->X, pLabel->FixedWidth, m_DisplayRangeLeft, m_DisplayRangeRight);
        }

        vector<float> orgNextXs;
        vector<float> orgNextYs;
        vector<string> orgNextTexts;
        for(auto *pLabel : *m_Next)
        {
            orgNextXs.push_back(pLabel->X);
            orgNextYs.push_back(pLabel->Y);
            pLabel->X += m_NextX + offsetX;
            pLabel->Y += offsetY;

            orgNextTexts.push_back(pLabel->Text);
            pLabel->Text = Trim(pLabel->Text, pLabel->X, pLabel->FixedWidth, m_DisplayRangeLeft, m_DisplayRangeRight);
        }

        // 描画
        for(auto *pLabel : *m_Next)
        {
            pLabel->ShowImpl(display);
        }
        for(auto *pLabel : *m_Current)
        {
            pLabel->ShowImpl(display);
        }

        // 元の情報に戻す
        int i=0;
        for(auto *pLabel : *m_Current)
        {
            pLabel->X = orgCurrentXs[i];
            pLabel->Y = orgCurrentYs[i];
            pLabel->Text = orgCurrentTexts[i++];
        }

        i = 0;
        for(auto *pLabel : *m_Next)
        {
            pLabel->X = orgNextXs[i];
            pLabel->Y = orgNextYs[i];
            pLabel->Text = orgNextTexts[i++];
        }
    }
    else
    {
        for(auto* pLabel : **m_PagesIterator)
        {
            pLabel->X += offsetX;
            pLabel->Y += offsetY;
            pLabel->ShowImpl(display);
            pLabel->X -= offsetX;
            pLabel->Y -= offsetY;
        }
    }
}

void ScrollableTextPage::ShowImpl(Display& display)
{
    InputPad(Pad::GetInstance());

    if( m_Pages.empty() )
    {
        return;
    }

    ShowIndicator(display);
    ShowTextPage(display);
}

void ScrollableTextPage::InputPad(Pad& pad)
{
    if( pad.IsTrigger(m_NextKey) )
    {
        pad.Clear();

        auto  switching = true;
        auto* currentTextPage = *m_PagesIterator;

        m_PagesIterator++;

        if( m_PagesIterator == m_Pages.end() )
        {
            if( m_IsLoopEnabled )
            {
                m_PagesIterator = m_Pages.begin();
            }
            else
            {
                switching = false;
                --m_PagesIterator;
            }
        }

        m_Switching = switching;
        if( m_Switching )
        {
            m_CurrentX = m_NextX;
            m_Current = currentTextPage;
            m_Next = *m_PagesIterator;
            m_NextX = m_CurrentX + Width;
        }
    }
    else if( pad.IsTrigger(m_BackKey) )
    {
        pad.Clear();

        auto  switching = true;
        auto* currentTextPage = *m_PagesIterator;

        if( m_PagesIterator != m_Pages.begin() )
        {
            m_PagesIterator--;
        }
        else
        {
            if( m_IsLoopEnabled )
            {
                m_PagesIterator = m_Pages.end();
                m_PagesIterator--;
            }
            else
            {
                switching = false;
            }
        }

        m_Switching = switching;
        if( m_Switching )
        {
            m_CurrentX = m_NextX;
            m_Current = currentTextPage;
            m_Next = *m_PagesIterator;
            m_NextX = m_CurrentX - Width;
        }
    }
}

void ScrollableTextPage::AddPage(TextPage& page)
{
    m_Pages.push_back(&page);
    m_PagesIterator = m_Pages.begin();
}

ScrollableTextPage::ScrollableTextPage()
{
    m_IsLoopEnabled = true;
    m_Switching = false;
    m_Smoothness = 1.5;
    m_CurrentX = 0;
    m_NextX = 0;
    m_Next = NULL;
    m_NextKey = Button::RIGHT;
    m_BackKey = Button::LEFT;
    m_DisplayRangeLeft = 0;
    m_DisplayRangeRight = 3000;

    m_LeftIndicator.SetIndicator('<');
    float lx[3] = {5, 0, -5};
    float ly[3] = {0, 0,  0};
    m_LeftIndicator.SetPosition(lx, ly, 3);

    m_RightIndicator.SetIndicator('>');
    float rx[3] = {-5, 0, 5};
    float ry[3] = { 0, 0, 0};
    m_RightIndicator.SetPosition(rx, ry, 3);
}

void ScrollableTextPage::Clear()
{
    m_Pages.clear();
}

ScrollableTextPage::TextPage& ScrollableTextPage::GetCurrentPage()
{
    return **m_PagesIterator;
}

void ScrollableTextPage::SetSmoothness(float val)
{
    m_Smoothness = val;
}
void ScrollableTextPage::ChangeNextKey(const Button& button)
{
    m_NextKey = button;
}

void ScrollableTextPage::ChangeBackKey(const Button& button)
{
    m_BackKey = button;
}

void ScrollableTextPage::SetDisplayRange(const float& left, const float& right)
{
    SetDisplayRangeLeft(left);
    SetDisplayRangeRight(right);
}

void ScrollableTextPage::SetDisplayRangeLeft(const float& left)
{
    m_DisplayRangeLeft = left;
}

void ScrollableTextPage::SetDisplayRangeRight(const float& right)
{
    m_DisplayRangeRight = right;
}

string ScrollableTextPage::Trim(const string& text, const float& textX, const float& fontWidth, const float& left, const float& right)
{
    uint32_t count = 0;
    string rtn;
    for(int i=0; i<text.size(); ++i)
    {
        if( text[i] == '\n' )
        {
            rtn += text[i];
            count = 0;
        }
        else
        {
            float curX = textX + count * fontWidth;
            if( left <= curX && curX + fontWidth <= right )
            {
                rtn += text[i];
            }
            else
            {
                rtn += ' ';
            }
            ++count;
        }
    }
    return rtn;
}

void ScrollableTextPage::SetLoop(const bool& isEnabled)
{
    m_IsLoopEnabled = isEnabled;
}


/*
  BuildInfo
*/

BuildInfo::BuildInfo()
{
    m_Date.Text = __DATE__;
    m_Date.FitSize();
    m_Date.Width = 150;
    m_Date.Height = m_Date.FontHeight + 5;
    m_Date.Alignment = MiddleCenter;
    m_Date.BackgroundColor = GenerateColor((void*)m_Date.Text.c_str(), strlen(m_Date.Text.c_str()));
    m_Date.TextColor = ToNegative(ToBlackOrWhite(m_Date.BackgroundColor));
    m_Page.Add(m_Date);

    m_Time.Text = __TIME__;
    m_Time.FitSize();
    m_Time.Width = 150;
    m_Time.Height = m_Date.FontHeight + 5;
    m_Time.Alignment = MiddleCenter;
    m_Time.Y = m_Date.Y + m_Date.FontHeight + 5;
    m_Time.BackgroundColor = GenerateColor((void*)m_Time.Text.c_str(), strlen(m_Time.Text.c_str()));
    m_Time.TextColor = ToNegative(ToBlackOrWhite(m_Time.BackgroundColor));
    m_Page.Add(m_Time);
}

void BuildInfo::ShowImpl(WlanTest::Display& display)
{
    m_Page.X = X;
    m_Page.Y = Y;
    m_Page.ShowImpl(display);
}


/*
  LabelBar
*/

LabelBar::LabelBar()
{
    m_Space = 0;
    next = 0;
    m_IsHorizontal = true;

    for(int i=0; i<Size; i++)
    {
        labels[i] = NULL;
    }
}

void LabelBar::AddLabel(Label& label)
{
    if(next < Size)
    {
        labels[next] = &label;
        next++;
    }
    else
    {
        NN_LOG("can not add because full!\n");
    }
}

void LabelBar::ShowImpl(Display& display)
{
    if(m_IsHorizontal)
    {
        float totalWidth = 0;
        float originalX;
        float originalY;

        for(int i=0; i<next; i++)
        {
            originalX = labels[i]->X;
            originalY = labels[i]->Y;
            labels[i]->X = totalWidth + GetX();
            labels[i]->Y = GetY();

            labels[i]->ShowImpl(display);

            labels[i]->X = originalX;
            labels[i]->Y = originalY;
            totalWidth += (labels[i]->Width + m_Space);
        }
    }
    else
    {
        float totalHeight = 0;
        float originalX;
        float originalY;

        for(int i=0; i<next; i++)
        {
            originalX = labels[i]->X;
            originalY = labels[i]->Y;
            labels[i]->X = GetX();
            labels[i]->Y = totalHeight + GetY();

            labels[i]->ShowImpl(display);

            labels[i]->X = originalX;
            labels[i]->Y = originalY;
            totalHeight += (labels[i]->Height + m_Space);
        }
    }
}


/*
   Console
*/

Console::Console() : m_Mutex(false, 0)
{
    FontWidth = DEFAULT_FONT_WIDTH;
    FontHeight = DEFAULT_FONT_HEIGHT;
    Width = 1280;
    Height = 720;
    BackgroundColor = ToColor(BLACK);
    TextColor = ToColor(WHITE);
    Charactors = Width / FontWidth;
    Lines = Height / FontHeight;
    m_Buffer = new RingBuffer<string>(Lines - 1);
}

Console::~Console()
{
    delete m_Buffer;
}

// 改行コードの後に文字列を含まない文字列の描画（ただし、はみ出して改行になることはある）
bool Console::WriteOneLine(const string& text)
{
    if(text == "")
    {
        return true;
    }

    if(m_Line.size() + text.size() > Charactors) // 改行が生じるか？
    {
        int position = (Charactors - m_Line.size());
        string splitted, splitted2;
        splitted.assign(text, 0, position);
        splitted2.assign(text, position, (text.size() - (position)));
        m_Line+=splitted;
        m_Buffer->SetValue(m_Line);
        m_Line="";
        WriteOneLine(splitted2);
    }
    else
    {
        m_Line+=text;
        if(text[text.size() - 1] == '\n')
        {
            m_Buffer->SetValue(m_Line);
            m_Line="";
        }
    }

    return true;
}

bool Console::Write(const string& text)
{
    m_Mutex.Lock();
    int position = text.find("\n");

    string splitted, splitted2;
    if(position != string::npos)
    {
        splitted.assign(text, 0, position + 1); // positionは位置、長さに変換するためには+1する
        WriteOneLine(splitted);
        splitted2.assign(text, position + 1, (text.size() - (position + 1))); // positionの次の位置からなので+1する
        m_Mutex.Unlock();
        Write(splitted2);
    }
    else
    {
        WriteOneLine(text);
        m_Mutex.Unlock();
    }

    return true;
}

bool Console::WriteLine(const string& text)
{
    string str = text + "\n";
    return Write(str);
}

void Console::Clear()
{
    m_Mutex.Lock();
    m_Line="";
    delete m_Buffer;
    Charactors = Width / FontWidth;
    Lines = Height / FontHeight;
    m_Buffer = new RingBuffer<string>(Lines - 1);
    m_Mutex.Unlock();
}

void Console::ShowImpl(Display& display)
{
    display.SetFontSize(FontWidth, FontHeight);
    display.SetColor(BackgroundColor);
    display.DrawRectangle(GetX(), GetY(), Width, Height);
    display.SetColor(TextColor);
    m_Mutex.Lock();
    for(int i=0; i<m_Buffer->GetSize(); i++)
    {
        display.DrawText(GetX(), GetY() + i*FontHeight, (*m_Buffer)[i]);
    }
    m_Mutex.Unlock();
    display.DrawText(GetX(), GetY() + m_Buffer->GetSize() * FontHeight, m_Line);
}


/*
  ActiveIndicator
*/

ActiveIndicator::ActiveIndicator()
{
    index = 4;
    Width = 5 * DEFAULT_FONT_WIDTH;
    Height = 2 * DEFAULT_FONT_HEIGHT;
    m_IsActive = false;
    m_Interval = nn::TimeSpan::FromMilliSeconds(70);
    m_LastTick = nn::os::Tick(0);
    ch[0] = '|';
    ch[1] = '/';
    ch[2] = '-';
    ch[3] = '\\';
    ch[4] = '\0';
}

void ActiveIndicator::ShowImpl(Display& display)
{
    label.X = GetX();
    label.Y = GetY();
    label.Width = Width;
    label.Height = Height;
    if( m_IsActive )
    {
        nn::os::Tick tick = nn::os::GetSystemTick();
        if( (tick - m_LastTick).ToTimeSpan() > m_Interval )
        {
            index++;
            index %= 4;
            m_LastTick = tick;
        }
    }
    label.Text = ch[index];
    label.ShowImpl(display);
}


/*
  ArrowIndicator
*/

ArrowIndicator::ArrowIndicator()
{
    animCount = 3;
    Width = 5 * DEFAULT_FONT_WIDTH;
    Height = 2 * DEFAULT_FONT_HEIGHT;
    m_IsActive = true;
    m_Interval = nn::TimeSpan::FromMilliSeconds(150);
    m_LastTick = nn::os::Tick(0);
    ch = '<';

    memset(x, 0, sizeof(x));
    memset(y, 0, sizeof(y));
}

void ArrowIndicator::ShowImpl(Display& display)
{
    label.Text = ch;
    label.X = GetX();
    label.Y = GetY();
    label.Width = Width;
    label.Height = Height;
    if( m_IsActive )
    {
        nn::os::Tick tick = nn::os::GetSystemTick();
        if( (tick - m_LastTick).ToTimeSpan() > m_Interval )
        {
            index++;
            index %= animCount;
            m_LastTick = tick;
        }

        label.X = GetX() + x[index];
        label.Y = GetY() + y[index];
        label.ShowImpl(display);
    }
}

}    // WlanTest
