﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "LcsMaster.h"

namespace WlanTest {

/*---------------------------------------------------------------------------
　　　　　LcsMaster
---------------------------------------------------------------------------*/

LcsMaster::LcsMaster()
{
    m_Param.channel = 1;                           //!> 使用するチャンネルです。1-14の値です。
    m_Param.hiddenSsid = false;                    //!> ステルスSSID設定フラグです。 false:any true:hidden
    m_Param.inactivePeriod = 10;                   //!> 通信がないClientを切断するまでの時間[sec]です。
    m_Param.autoKeepAlive = false; // inactivePeriod = 0で同等のことが出来るので使用しない。（一応値は入れておく）
    m_Param.supportedRates = nn::wlan::RateSetLegacy_11gMask;  //!> IEEE802.11 Supported Rates です。
    m_Param.basicRates = nn::wlan::RateSetLegacy_11gMask;      //!> IEEE802.11 Supported Rates の Basic Rates を示します。
    m_Param.security = nn::wlan::Security();       //!> 接続先のセキュリティプロファイル
    m_Param.ssid = nn::wlan::Ssid("sap_master");   //!> 使用する SSID です。

    m_NumOfClients = 1;
    m_Param.beaconInterval = 100;                  //!> 使用する Beacon Interval[TU] です。
    m_Ctw = 10;
    m_NumOfDeauth = 3;
    m_ConnectionTime = 0;
    m_Updated = 0x00;
    m_ActionFrameSize = 0;
    std::memset(m_respondOui, 0x00, sizeof(m_respondOui));
    std::memset(m_ClientList, 0x00, sizeof(m_ClientList));
    std::memset(m_ClientStatus, 0x00, sizeof(m_ClientStatus));
    std::memset(m_ClientL3Status, 0x00, sizeof(m_ClientL3Status));
    m_ReceiverHandler = nullptr;

    m_Socket.SetSocketMode(SocketModeServer);
    m_Socket.SetSocketType(SOCKET_TYPE_TCP);
    m_Socket.SetSrcIpAddress("169.254.1.1");
    m_Socket.SetDstIpAddress("0.0.0.0");
    m_Socket.SetSubnetMask("255.255.255.0");
    m_Socket.SetDns1("0.0.0.0");
    m_Socket.SetDns2("0.0.0.0");
    m_Socket.SetGwIpAddress("127.0.0.1");
    m_Socket.SetPort(50222);
}

LcsMaster::LcsMaster(nn::wlan::MasterBssParameters param)
{
    m_Param = param;
}

LcsMaster::~LcsMaster()
{
}

nn::Result LcsMaster::Initialize()
{
    nn::Result result = WlanInitialize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanInitialize()\n");
        return result;
    }

    // 前回の BSS 構築処理時間の初期化
    m_ConnectionTime = 0;
    std::memset(&m_Param, 0, sizeof(m_Param));

    ModeChanger::ToLcsMasterMode();

    // 送受信する際に Rx エントリが必須なので初期化時に作成する
    CreateRxEntry();

    m_Socket.Open();

    m_IsInitialized = true;

    return result;
}

nn::Result LcsMaster::Finalize()
{
    m_Socket.Close();

    nn::Result result = DeleteRxEntry();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : DeleteRxEntry()\n");
    }

    result = WlanFinalize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanFinalize()\n");
    }

    ModeChanger::ToReadyMode();
    m_IsInitialized = false;

    return result;
}

nn::Result LcsMaster::Open()
{
    nn::Result result = nn::ResultSuccess();

    // 接続可能なクライアント数を設定
    if( m_NumOfClients != ConnectableClientCount_Any )
    {
        nn::Result setResult = nn::wlan::Local::SetMaxAssociationNumber(m_NumOfClients);
        NN_ASSERT(setResult.IsSuccess());
    }

    char buf[nn::wlan::Ssid::SsidHexStringLengthMax] = {0};
    NN_LOG(" === CreateBss Params ===============================\n");
    NN_LOG("   Channel           : %d\n", m_Param.channel);
    NN_LOG("   Hidden ssid       : %d\n", m_Param.hiddenSsid);
    NN_LOG("   Inactive period   : %u\n", m_Param.inactivePeriod);
    NN_LOG("   Auto keep alive   : %d\n", m_Param.autoKeepAlive);
    NN_LOG("   Security mode     : %d\n", m_Param.security.groupPrivacyMode);
    NN_LOG("   Security key      : %s\n", reinterpret_cast<char*>(m_Param.security.key));
    NN_LOG("   Ssid              : %s\n", m_Param.ssid.GetHexString(buf));
    NN_LOG("   Beacon interval   : %d\n", m_Param.beaconInterval);
    NN_LOG("   Action frame      : %d\n", IsActionFrameEnabled());
    NN_LOG(" ====================================================\n");

    nn::os::Tick startTick = nn::os::GetSystemTick();
    result = nn::wlan::Local::CreateBss(m_Param);
    nn::os::Tick endTick = nn::os::GetSystemTick();

    if( result.IsSuccess() )
    {
        // CreateBss() の時間を計測
        nn::wlan::ConnectionStatus connectionStatus;
        nn::Result conResult = nn::wlan::Local::GetConnectionStatus(&connectionStatus);
        if( conResult.IsSuccess() && connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            m_ConnectionTime = (endTick - startTick).ToTimeSpan().GetMilliSeconds();

            // Auto の場合もあるので改めてチャネルを取得する
            m_Param.channel = connectionStatus.channel;
        }
    }
    else
    {
        NN_LOG(" - failed : CreateBss\n");
        WlanTest::TPrintResult(result);
        NN_ASSERT(false);
        return result;
    }

    return result;
}

nn::Result LcsMaster::Close()
{
    nn::Result result;

    for(int j=0; j<sizeof(m_ClientStatus) / sizeof(m_ClientStatus[0]); j++)
    {
        if(m_ClientStatus[j].state == nn::wlan::ConnectionState_Connected)
        {
            nn::wlan::DisconnectClient client;
            client.clientMacAddress = m_ClientStatus[j].clientMacAddress;
            client.reasonCode = nn::wlan::Dot11ReasonCode_UnspecifiedFailure;

            char str[nn::wlan::MacAddress::MacStringSize];
            NN_LOG("Disconnect client : %s, reason : %d\n", client.clientMacAddress.GetString(str), client.reasonCode);
            for(uint8_t i=0; i<m_NumOfDeauth; i++)
            {
                Disconnect(&client);
            }
        }
    }

    // 更新値を次回のマスター起動時に持ち越さないためここで読み込む
    nn::wlan::Local::GetClientStatus(m_ClientStatus, &m_Updated);
    std::memset(m_ClientList, 0x00, sizeof(m_ClientList));
    std::memset(m_ClientStatus, 0x00, sizeof(m_ClientStatus));
    std::memset(m_ClientL3Status, 0x00, sizeof(m_ClientL3Status));
    m_Updated = 0x00;

    result = nn::wlan::Local::DestroyBss();
    if (result.IsFailure())
    {
        NN_LOG(" - failed : DestroyBss\n");
        WlanTest::TPrintResult(result);
    }
    else
    {
        //NN_LOG(" - success : DestroyBss\n");
    }

    return result;
}

void LcsMaster::MaintainConnection()
{
    m_RequestStop = false;
    m_IsStop = false;

    nn::os::SystemEventType connectionEvent;
    nn::Result result;

    SetActionFrame();
    Open();

    result = nn::wlan::Local::GetConnectionEvent(&connectionEvent);
    if( result.IsFailure() )
    {
        NN_LOG("GetConnectionEvent failed\n");
        WlanTest::TPrintResult(result);
        goto End;
    }

    while(!m_RequestStop)
    {
        nn::wlan::WlanState state = nn::wlan::WlanState_Stop;
        result = nn::wlan::Local::GetState(&state);
        if( result.IsFailure() )
        {
            NN_LOG("GetState failed\n");
            WlanTest::TPrintResult(result);
            goto End;
        }

        m_Cs.Lock();

        result = nn::wlan::Local::GetClientStatus(m_ClientStatus, &m_Updated);
        if( result.IsFailure() )
        {
            NN_LOG("GetClientStatus failed\n");
            WlanTest::TPrintResult(result);
        }

        for(int i = 0; i < sizeof(m_ClientStatus) / sizeof(m_ClientStatus[0]); i++)
        {
            if( (m_Updated >> i) & 1 )
            {
                if(m_ClientStatus[i].state == nn::wlan::ConnectionState_Connected)
                {
                    m_ClientList[i] = m_ClientStatus[i].clientMacAddress;
                    AssignIpAddress(i);
                    m_Statistics.ConnectCount++;
                }
                else
                {
                    RemoveIpAddress(i);
                    m_ClientList[i] = nn::wlan::MacAddress::CreateZeroMacAddress();
                    m_Statistics.Disconnected++;
                }

                char adr[nn::wlan::MacAddress::MacStringSize];
                if(m_ClientStatus[i].state == nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("CONNECTED : %s\n", m_ClientStatus[i].clientMacAddress.GetString(adr));
                }
                else if(m_ClientStatus[i].state == nn::wlan::ConnectionState_Disconnected)
                {
                    NN_LOG("DISCONNECTED : %s\n", m_ClientStatus[i].clientMacAddress.GetString(adr));
                }
                NN_LOG("    State       : %d\n", m_ClientStatus[i].state);
                NN_LOG("    Cause       : %d\n", m_ClientStatus[i].cause);
                NN_LOG("    Reason      : %d\n", m_ClientStatus[i].statusReasonCode);
                NN_LOG("    Capability  : 0x%x\n", m_ClientStatus[i].capabilityInfo);
                NN_LOG("\n");
                NN_UNUSED(adr); // ワーニング対策
            }
        }

        m_Cs.Unlock();

        // L4 の接続の許可
        m_Socket.Accept();

        // L3 の状態を更新
        UpdateL3Status();

        // L4 の接続ができていない Client のため再送する
        //NN_LOG(" L3 status :");
        for(int i=0; i<sizeof(m_ClientL3Status) / sizeof(m_ClientL3Status[0]); ++i)
        {
            //NN_LOG(" %d", m_ClientL3Status[i].state);
            ReassignIpAddress(i);
        }
        //NN_LOG("\n");

        TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(100));
    }
End:
    m_RequestStop = true;
    m_IsStop = true;
    nn::os::DestroySystemEvent(&connectionEvent);
}

void LcsMaster::StopMaintainConnection()
{
    if( !m_IsStop )
    {
        m_RequestStop = true;

        //#?? Detach できないから明示的に破棄する必要がある
        nn::os::DestroyThread(&m_MaintainConnectionThread);
        Close();
    }
}

nn::Result LcsMaster::Disconnect( const nn::wlan::DisconnectClient* pClient )
{
    nn::Result result;
    result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, pClient );
    return result;
}

nn::Result LcsMaster::Disconnect()
{
    nn::wlan::DisconnectClient client;
    client.clientMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    //#?? Dot11ReasonCode を指定しないといけないため、とりあえず下記を指定する
    client.reasonCode = nn::wlan::Dot11ReasonCode_UnspecifiedFailure;
    return Disconnect( &client );
}

nn::Result LcsMaster::Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize)
{
    NN_UNUSED(ieInd);

    nn::Result result = nn::ResultSuccess();
    result = m_Socket.Send(data, size, pSentSize);

    if( result.IsFailure() )
    {
        m_Statistics.SendErrorCount++;
        // NN_LOG("PutFrameRaw failed\n");
        // WlanTest::TPrintResult(result);
        Sleep(nn::TimeSpan::FromMicroSeconds(10));    // 切断時の連続送信失敗を避けるため、スリープする
    }
    else
    {
        if(m_Statistics.SendCount == 0)
        {
            m_Statistics.FirstSendTime = nn::os::GetSystemTick();
        }
        m_Statistics.LastSendTime = nn::os::GetSystemTick();
        m_Statistics.SendCount++;
        m_Statistics.SendSize += *pSentSize;
    }

    return result;
}

nn::Result LcsMaster::Receive(uint8_t pOutput[], size_t* pSize, const size_t maxSize)
{
    nn::Result result = nn::ResultSuccess();
    result = m_Socket.Receive(pOutput, pSize, maxSize);

    return result;
}

nn::Result LcsMaster::WlanGetFrame( uint32_t rxId, uint8_t pOutput[], size_t* pSize, size_t maxSize, int8_t* pRssi)
{
    NN_UNUSED(rxId);
    NN_UNUSED(pRssi);

    return Receive(pOutput, pSize, maxSize);
}

bool LcsMaster::IsConnected()
{
    nn::wlan::ConnectionStatus connectionStatus;
    nn::Result result = nn::wlan::Local::GetConnectionStatus(&connectionStatus);

    if( result.IsFailure() )
    {
        return false;
    }

    return connectionStatus.state == nn::wlan::ConnectionState_Connected;
}

void LcsMaster::UpdateL3Status()
{
    const vector<LcsSocket::SocketInfo>* pDataSocketList = m_Socket.GetSocketInfo();
    for(const auto& dataSocket : *pDataSocketList)
    {
        if( dataSocket.m_IsAccepted )
        {
            for(auto& status : m_ClientL3Status)
            {
                // IP アドレス割り当て中のクライアントが Accept されたらステートを遷移
                if( status.ipAddress == dataSocket.m_IpAddressStr && status.state == State_Assigning )
                {
                    status.state = State_Assigned;
                    break;
                }
            }
        }
    }
}

void LcsMaster::AssignIpAddress(const int& id)
{
    ClientL3Status& status = m_ClientL3Status[id];

    if( status.state == State_NotAssigned )
    {
        uint8_t ip[4] = { 0 };
        string srcIp(m_Socket.GetSrcIpAddress());
        NN_ASSERT(sscanf(srcIp.c_str(), "%hhu.%hhu.%hhu.%hhu", &ip[0], &ip[1], &ip[2], &ip[3]) == 4);

        uint16_t newIp = ip[3] + 1 + id;
        if( newIp >= 255 )
        {
            newIp -= 254;
        }
        ip[3] = newIp;
        status.ipAddress = LcsSocket::GenerateIpAddress(ip);
        status.state = State_Assigning;
    }
    else if( status.state == State_Assigning )
    {
    }
    else
    {
        return;
    }

    uint8_t size = strlen(status.ipAddress.c_str()) + 1;
    uint8_t buf[30] = { '\0' };
    std::memcpy(buf, status.ipAddress.c_str(), size);

    SendCommand(m_ClientList[id], WitCommand_AssignSrcIpAddress, buf, size);
}

void LcsMaster::ReassignIpAddress(const int& id)
{
    if( m_ClientL3Status[id].state == State_Assigning )
    {
        AssignIpAddress(id);
    }
}

void LcsMaster::RemoveIpAddress(const int& id)
{
    ClientL3Status& status = m_ClientL3Status[id];
    // Remove() では L2 接続完了、L4 接続未完了時に false となる
    m_Socket.Remove(status.ipAddress);
    status.ipAddress = "";
    status.state = State_NotAssigned;
}

}
