﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <sstream>
#include <iomanip>
#include <cmath>
#include "Delegate.h"
#include "Timer.h"
#include "Util.h"
#include "UIControl.h"

namespace WlanTest {

/*!--------------------------------------------------------------------------*
  @brief        IEEE802.3 + LLC + SNAPフォーマット

 *---------------------------------------------------------------------------*/
struct DIX_FRAME
{
    uint8_t Destination[6];
    uint8_t Source[6];
    uint16_t Type;
    uint8_t Payload[1];
};

enum FrameType
{
    FrameType_WlanNd = 0x00,
    FrameType_Wit
};

struct WlanNdHeader
{
    uint8_t afType;
    uint8_t oui[3];
    uint8_t subType;
    uint8_t reserved1;
    uint16_t version;
    uint8_t command;
    uint8_t reserved2;
    uint8_t hash[8];
};

struct WitHeader
{
    uint16_t dummy;
    uint8_t version;
    uint8_t command;
    uint16_t length;
    uint16_t reserved;
    uint32_t sequence;
};

enum WitCommand
{
    WitCommand_Data                  = 0x00,

    WitCommand_ActionFrame           = 0x10,
    WitCommand_ActionFrameSearchInfo = 0x11,

    WitCommand_AssignSrcIpAddress    = 0x40,

    WitCommand_Ack                   = 0x80,

    WitCommand_Num

};

struct ActionFrameSearchInfoCommand
{
    uint64_t sequenceNumber;
    int64_t tsfTime;
    int16_t channel;
};

const uint32_t WlanActionFrameHeaderSize = 5; // category(1), oui(3), subtype(1)
const uint32_t WlanNdhpHeaderSize = 18; // Type(7fh), oui(3), subType(7d), reserved(1), ver(2), cmd(1), reserved(1), hash(8)
const uint32_t WlanHeaderSize = 14;
const uint32_t MatchInfoSize = 5;
const uint32_t WitHeaderSize = sizeof(WitHeader);
const uint32_t DataHeaderSize = WlanHeaderSize + MatchInfoSize + WitHeaderSize;
const uint32_t CommandHeaderSize = WlanHeaderSize + WitHeaderSize;

void GenerateWitHeader(WitHeader* pHeader, const WitCommand command, const uint32_t length);

class DixFrame
{
private:

    uint8_t m_Destination[6]; // 802.3 Destination
    uint8_t m_Source[6];      // 802.3 Source
    uint16_t m_Type;          // Ether type
    uint8_t m_Payload[1500];  // Payload (1500 : 最大ペイロード) #??

public:

    DixFrame();

    uint8_t* GetDestinationP() { return m_Destination; }
    nn::wlan::MacAddress GetDestination() { return nn::wlan::MacAddress(m_Destination); }
    void SetDestination(nn::wlan::MacAddress address) { std::memcpy(m_Destination, address.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize); }

    uint8_t* GetSourceP() { return m_Source; }
    nn::wlan::MacAddress GetSource() { return nn::wlan::MacAddress(m_Source); }
    void SetSource(nn::wlan::MacAddress address) { std::memcpy(m_Source, address.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize); }

    // uint16_t GetLength() { return (m_Length << 8 | m_Length >> 8); }
    // void SetLength(uint16_t l) { m_Length = (l << 8 | l >> 8); }

    uint16_t GetType() { return (m_Type << 8 | m_Type >> 8); }
    void SetType(uint16_t type) { m_Type = (type << 8 | type >> 8); /* エンディアン変換している */ }

    uint8_t* GetPayload() { return (uint8_t*)m_Payload; }
    void SetPayload(uint8_t* p, size_t size)
    {
        NN_ASSERT(GetHeaderSize() + size <= sizeof(m_Payload));

        if(GetType() == PID_OUI_EXT)
        {
            uint16_t offset = 0;
            uint16_t size = 0;

            size = sizeof(OUI);
            std::memcpy(m_Payload + offset, OUI, size);
            offset += size;

            size = sizeof(PID_LDN);
            std::memcpy(m_Payload + offset, PID_LDN, size);
            offset += size;
        }
        std::memcpy(m_Payload + GetHeaderSize(), p, size);
    }

    uint32_t GetHeaderSize()
    {
        switch(GetType())
        {
        case PID_OUI_EXT :      return 5;
        case PID_PAYLOAD_ONLY : return 0;
        default :               return 0;
        }
    }

};



/*!--------------------------------------------------------------------------*
  @brief        NintendoIE構造体

 *---------------------------------------------------------------------------*/
struct NintendoIE
{
    nn::Bit8    ElementID;
    nn::Bit8    Length;
    nn::Bit8    OUI[3];
    nn::Bit8    SubType;
    nn::Bit8*   Data;

    NN_IMPLICIT NintendoIE(size_t length);
    virtual ~NintendoIE();

    std::string ToString();

    uint32_t GetSize() { return Length + 2; }
    void GetBinary(nn::Bit8 buf[], size_t length)
    {
        if(length < GetSize())
        {
            NN_LOG("NintendoIE : Buffer size too small.\n");
            return;
        }
        std::memcpy(buf, &(this->ElementID), 6);
        std::memcpy(buf + 6, Data, Length - 4);
    }
};


/*!--------------------------------------------------------------------------*
  @brief        ノード統計値

 *---------------------------------------------------------------------------*/
struct NodeStatistics
{
    uint32_t SendCount;
    uint64_t SendSize;
    uint32_t SendErrorCount;
    uint32_t ReceiveCount;
    uint64_t ReceiveSize;
    uint32_t ReceiveErrorCount;
    uint32_t ConnectCount;
    uint32_t ConnectErrorCount;
    uint32_t Disconnected;
    nn::os::Tick FirstSendTime;
    nn::os::Tick LastSendTime;
    nn::os::Tick FirstReceiveTime;
    nn::os::Tick LastReceiveTime;

    void Clear()
    {
        SendCount = 0;
        SendSize = 0;
        SendErrorCount = 0;
        ReceiveCount = 0;
        ReceiveSize = 0;
        ReceiveErrorCount = 0;
        ConnectCount = 0;
        ConnectErrorCount = 0;
        Disconnected = 0;
        FirstSendTime -= FirstSendTime;
        LastSendTime -= LastSendTime;
        FirstReceiveTime -= FirstReceiveTime;
        LastReceiveTime -= LastReceiveTime;
    }

    NodeStatistics()
    {
        Clear();
    }
};



/*!--------------------------------------------------------------------------*
  @brief        受信器統計値

 *---------------------------------------------------------------------------*/
struct ReceiverStatistics
{
    uint32_t ReceiveCount;
    uint64_t ReceiveSize;
    uint32_t ReceiveErrorCount;
    nn::os::Tick FirstReceiveTime;
    nn::os::Tick LastReceiveTime;

    void Clear()
    {
        ReceiveCount = 0;
        ReceiveSize = 0;
        ReceiveErrorCount = 0;
        FirstReceiveTime -= FirstReceiveTime;
        LastReceiveTime -= LastReceiveTime;
    }

    ReceiverStatistics()
    {
        Clear();
    }
};



/*!--------------------------------------------------------------------------*
  @brief        RSSI 情報

 *---------------------------------------------------------------------------*/
struct Rssi
{
    int32_t value;
    nn::os::Tick recordTime;
    nn::wlan::MacAddress address;

    void Clear()
    {
        value = RSSI_MIN;
        recordTime = nn::os::Tick(0);
        address = nn::wlan::MacAddress();
    }
};



/*!--------------------------------------------------------------------------*
  @brief        受信イベント情報

 *---------------------------------------------------------------------------*/
enum ReceiverInfo
{
    INFORMATION_ELEMENT_RECEIVER = 0,
    PROTOCOL_RECEIVER,
    ACTION_FRAME_RECEIVER,
};

class ReceiveEventArgs : public EventArgs
{
public:
    ReceiverInfo Receiver;
    int8_t Rssi;
    uint8_t* Data;
    size_t Size;
};

class AfReceiveEventArgs : public EventArgs
{
public:
    ReceiverInfo Receiver;
    nn::wlan::MacAddress Source;
    uint8_t* Data;
    size_t Size;
    uint16_t Channel;
    int16_t Rssi;
};

/*!--------------------------------------------------------------------------*
  @brief        イベントハンドラインタフェース

 *---------------------------------------------------------------------------*/
typedef IDelegate<const EventArgs*> IReceiveEventHandler;
#define ReceiveEventHandler(a) Delegate<a,const EventArgs*>
// template typedefが使えないので、こうしておく。



class Node;

/*!--------------------------------------------------------------------------*
  @brief        受信器

 *---------------------------------------------------------------------------*/
class IReceiver
{
private:

protected:

    IReceiveEventHandler* m_Handler;
    Node* m_Target;
    ReceiverStatistics m_Statistics;

public:

    IReceiver();
    virtual ~IReceiver();

    virtual nn::Result Start(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread())) = 0;
    virtual nn::Result Stop() = 0;

    void SetHandler(IReceiveEventHandler& handler) { m_Handler = &handler; }
    void SetTarget(Node& node) { m_Target = &node; }
    ReceiverStatistics& GetStatistics(){ return m_Statistics; }

};



/*!--------------------------------------------------------------------------*
  @brief        プロトコル受信器

 *---------------------------------------------------------------------------*/
class ProtocolReceiver : public IReceiver
{
/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
private:

    static void ReceiveThreadFunction(void* receiver);

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
private:

    nn::os::ThreadType m_ReceiveThread;
    uint16_t           m_ProtocolType;
    bool               m_RequestStop;
    bool               m_IsStop;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void Receive();
    virtual nn::Result Start(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    virtual nn::Result Stop();

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:

    ProtocolReceiver();

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    void SetProtocolType(uint16_t type){ m_ProtocolType = type; };

};



/*!--------------------------------------------------------------------------*
  @brief        InformationElement受信器

 *---------------------------------------------------------------------------*/
class InformationElementReceiver : public IReceiver
{
/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
private:

    static void ReceiveThreadFunction(void* receiver);

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
private:

    nn::os::ThreadType m_ReceiveThread;
    bool               m_RequestStop;
    bool               m_IsStop;
    nn::Bit8           m_Types[4];
    uint8_t            m_TypesNum;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void Receive();
    virtual nn::Result Start(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    virtual nn::Result Stop();

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:

    InformationElementReceiver();

/*---------------------------------------------------------------------------
　　　　　デストラクタ
---------------------------------------------------------------------------*/
public:

    virtual ~InformationElementReceiver();

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    void SetMinSubType(nn::Bit8 t)
    {
        for(int i=0; i<m_TypesNum; i++)
        {
            m_Types[i] = t;
        }
    }

};


/*!--------------------------------------------------------------------------*
  @brief        コマンド受信器

 *---------------------------------------------------------------------------*/
class CommandReceiver : public IReceiver
{
/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
private:

    static void ReceiveThreadFunction(void* receiver);

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
private:

    nn::os::ThreadType m_ReceiveThread;
    bool               m_RequestStop;
    bool               m_IsStop;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void Receive();
    virtual nn::Result Start(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    virtual nn::Result Stop();

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:

    CommandReceiver();

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

};



/*!--------------------------------------------------------------------------*
  @brief        プロトコル受信器

 *---------------------------------------------------------------------------*/
class ActionFrameReceiver : public IReceiver
{
/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
private:

    static void ReceiveThreadFunction(void* receiver);

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
protected:

    nn::wlan::ScanParameters  m_Param;
    uint32_t                  m_BufferSize;
    uint8_t*                  m_pBuffer;
    uint32_t                  m_WaitTime;
    uint32_t                  m_RandomWaitTime;

private:

    nn::os::ThreadType m_ReceiveThread;
    bool               m_RequestStop;
    bool               m_IsStop;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void Receive();
    virtual nn::Result Start(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    virtual nn::Result Stop();

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:

    ActionFrameReceiver();
    virtual ~ActionFrameReceiver();

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    void SetChannel(const int16_t chList[nn::wlan::WirelessChannelsCountMax], const uint8_t count);
    void SetScanTime(const int32_t time);
    void SetWaitTime(const int32_t time);
    void SetRandomWaitTime(const int32_t time);

};


enum Mode
{
    READY,
    LOCAL_MASTER,
    LOCAL_CLIENT,
    LOCAL_SPECTATOR,
    LCS_MASTER,
    LCS_CLIENT,
    INFRA,
    DETECTOR,
    OTHER
};

class ModeChanger
{
private:

    static Mode s_Mode;

public:

    static Mode GetMode();
    static nn::Result ToReadyMode();
    static nn::Result ToLocalMasterMode();
    static nn::Result ToLocalClientMode();
    static nn::Result ToLocalSpectatorMode();
    static nn::Result ToLcsMasterMode();
    static nn::Result ToLcsClientMode();
    static nn::Result ToInfraMode();
    static nn::Result ToDetectorMode(uint16_t channel);

};


/*!--------------------------------------------------------------------------*
  @brief        端末クラス

 *---------------------------------------------------------------------------*/
class Node
{
public:

/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
private:

    static void MaintainConnectionThreadFunction(void* node);

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    IReceiveEventHandler* m_Handler;
    IReceiveEventHandler* m_AfHandler;
    bool m_EnableEcho;
    bool m_RequestStop;
    bool m_IsStop;
    bool m_IsInitialized;
    bool m_IsAfThreadEnabled;
    NodeStatistics m_Statistics;
    uint32_t m_DataRxId;
    uint32_t m_SocketDataRxId;
    uint32_t m_SocketControlRxId;
    uint32_t m_LocalRxId;
    uint32_t m_AfRxId;
    int64_t  m_ConnectionTime; //ms
    nn::wlan::MacAddress m_Address;

    IReceiver*            m_Receiver;
    IReceiveEventHandler* m_ReceiverHandler;
    IReceiver*            m_CommandReceiver;
    IReceiver*            m_AfReceiver;
    IReceiveEventHandler* m_AfReceiverHandler;
    nn::os::ThreadType m_MaintainConnectionThread;

private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual nn::Result Initialize() = 0;
    virtual nn::Result Finalize() = 0;
    virtual nn::Result Open() = 0;
    virtual nn::Result Close() = 0;

    virtual void MaintainConnection();
    virtual nn::Result StartMaintainConnection(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    virtual void StopMaintainConnection();
    virtual nn::Result StartReceiveData(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    virtual nn::Result StopReceiveData();
    virtual nn::Result StartReceiveActionFrame(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    virtual nn::Result StopReceiveActionFrame();
    virtual nn::Result StartReceiveCommand(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    virtual nn::Result StopReceiveCommand();
    virtual nn::Result Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize);
    virtual void OnReceive(const EventArgs* pArgs);
    virtual void OnAfReceive(const EventArgs* pArgs);

    virtual nn::Result CancelGetFrame(uint32_t rxId);
    virtual nn::Result CancelGetActionFrame(uint32_t rxId);

    virtual void PrintInfo(){};

public:

    virtual nn::Result WlanInitialize()=0;
    virtual nn::Result WlanFinalize()=0;
    // virtual nn::Result WlanOpenMode()=0;
    // virtual nn::Result WlanCloseMode()=0;
    virtual nn::Result WlanLocalCreateRxEntry(uint32_t* pRxId, const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity)=0;
    virtual nn::Result WlanSocketCreateRxEntry(uint32_t* pRxid, const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity)=0;
    virtual nn::Result WlanLocalCreateRxEntryForAf(uint32_t* pRxId, const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity)=0;
    virtual nn::Result WlanDetectCreateRxEntryForAf(uint32_t* pRxId, const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity)=0;
    virtual nn::Result WlanLocalDeleteRxEntry(uint32_t* pRxId)=0;
    virtual nn::Result WlanSocketDeleteRxEntry(uint32_t* pRxId)=0;
    virtual nn::Result WlanLocalDeleteRxEntryForAf(uint32_t* pRxId)=0;
    virtual nn::Result WlanDetectDeleteRxEntryForAf(uint32_t* pRxId)=0;
    virtual nn::Result WlanLocalAddMatchingData(const uint32_t& rxId, const nn::wlan::ReceivedDataMatchInfo& pMatchInfo)=0;
    virtual nn::Result WlanLocalRemoveMatchingData(const uint32_t& rxId, const nn::wlan::ReceivedDataMatchInfo& pMatchInfo)=0;

    virtual nn::Result WlanGetFrame(uint32_t rxId, uint8_t pOutput[], size_t* pSize, size_t maxSize, int8_t* pRssi = NULL)=0;
    virtual nn::Result WlanPutFrame(const uint8_t pInput[], size_t size, bool selfCts = false)=0;
    virtual nn::Result WlanGetActionFrame(nn::wlan::MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId, uint16_t* pChannel, int16_t* pRssi)=0;
    virtual nn::Result WlanPutActionFrame(const nn::wlan::MacAddress& dstMac, const uint8_t* pData, size_t size, int16_t channel, uint32_t dwellTime)=0;
    virtual nn::Result WlanGetState(nn::wlan::WlanState *pState)=0;

    virtual nn::Result WlanCancelGetFrame(uint32_t rxId)=0;
    virtual nn::Result WlanCancelGetActionFrame(uint32_t rxId)=0;

    virtual bool WaitReceive(nn::TimeSpan timeout = nn::TimeSpan(0)){ return true; }
    virtual bool IsConnected(){ return false; }
    virtual bool IsAfRxReady(){ return false; }

    nn::Result RunCommand(const WitHeader& witHeader, const uint8_t* data, const size_t& size);
    nn::Result SendCommand(const nn::wlan::MacAddress& dst, const WitCommand& command, const uint8_t* data, const uint32_t size);
    //nn::Result ReceiveCommand(WitHeader* pHeader, uint8_t* pData, uint32_t* pSize, const uint32_t& maxSize);

    void SetIsAfThreadEnabled(const bool isEnabled)
    {
        m_IsAfThreadEnabled = isEnabled;
    }
    bool IsAfThreadEnabled()
    {
        return m_IsAfThreadEnabled;
    }

protected:

    virtual nn::Result RunAssignSrcIpAddressCommand(const string& ipAddress) { return nn::ResultSuccess(); }

private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    Node();
    virtual ~Node();

private:

    Node(const Node& node);
    Node& operator=(const Node& node);

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    virtual void SetReceiveEventHandler(IReceiveEventHandler& handler) { this->m_Handler = &handler; };
    virtual void SetAfReceiveEventHandler(IReceiveEventHandler& handler) { this->m_AfHandler = &handler; };

    virtual int16_t GetChannel()
    {
        return 0;
    }

    virtual bool GetRssi(vector<Rssi>* pRssiList)
    {
        return false;
    }

    virtual nn::wlan::MacAddress GetMacAddress()
    {
        nn::Result result;

        if(m_Address == nn::wlan::MacAddress::CreateZeroMacAddress())
        {
            result = nn::wlan::Local::GetMacAddress(&m_Address);
            if (result.IsFailure())
            {
                NN_LOG("GetMacAddress failed\n");
                WlanTest::TPrintResult(result);
            }
        }

        return m_Address;
    }

    virtual nn::wlan::MacAddress GetApAddress()
    {
        nn::Result result;

        nn::wlan::MacAddress address[1];
        uint32_t num = 1;
        GetMacAddressListInBss(address, num);

        return address[0];
    }

    virtual void GetMacAddressListInBss(nn::wlan::MacAddress list[], uint32_t& num) { num = 0; }

    virtual NodeStatistics& GetTxStatistics()
    {
        return m_Statistics;
    }

    virtual ReceiverStatistics& GetRxStatistics()
    {
        NN_ASSERT(m_Receiver != nullptr);
        return m_Receiver->GetStatistics();
    }

    virtual uint32_t GetRxIdForData(){ return m_SocketDataRxId; }
    virtual uint32_t GetRxIdForControl(){ return m_SocketControlRxId; }
    virtual uint32_t GetRxIdForAf(){ return m_AfRxId; }
    virtual uint32_t GetConnectableCount() { return 0; }
    virtual uint32_t GetConnectedCount() { return 0; }

    virtual void ClearStatistics()
    {
        m_Statistics.Clear();
        m_Receiver->GetStatistics().Clear();
    }

    void SetEnableEcho(bool b) { m_EnableEcho = b; }
    bool IsStop() { return m_IsStop; }
    bool IsInitialized() { return m_IsInitialized; }
    int64_t GetConnectionTime(){ return m_ConnectionTime; }

};

class WlanFwInfo : public UIControl
{
private:

    Console m_Console;

public:

    virtual void ShowImpl(Display& display)
    {
        m_Console.X = GetX();
        m_Console.Y = GetY();
        m_Console.TextColor = ToColor(GRAY);
        m_Console.ShowImpl(display);
    }

    WlanFwInfo()
    {
        char buf[256] = {};
        nn::Result result;

        result =  nn::wlan::Infra::GetFwVersion(buf, sizeof(buf));
        if( result.IsSuccess() )
        {
            ostringstream oss;

            oss << "WIT Info" << endl;
            oss << "  * Date     : " << __DATE__ << ' ' << __TIME__ << endl;
            oss << "  * Version  : " << WIT_VERSION_MAJOR << '.'
                << WIT_VERSION_MINOR << '.'
                << WIT_VERSION_MICRO << ' '
                << WIT_VERSION_SUFFIX << endl;
            oss << endl;

            char date[50] = {};
            char version[50] = {};
            char revision[50] = {};
            char fwid[50] = {};
            sscanf(buf, " wl0%*[: \t]%20c version%*[: \t]%s (r%20[^)]) FWID%*[: \t]%20s ",
                   date, version, revision, fwid);

            oss << "WLAN FW Info" << endl;
            oss << "  * Date     : " << ( strlen(date) == 0 ? "-" : date) << endl;
            oss << "  * Version  : " << ( strlen(version) == 0 ? "-" : version) << endl;
            oss << "  * Revision : " << ( strlen(revision) == 0 ? "-" : revision) << endl;
            oss << "  * FW ID    : " << ( strlen(fwid) == 0 ? "-" : fwid) << endl;
            oss << endl;

            nn::wlan::MacAddress wlanAddr;
            nn::wlan::Local::GetMacAddress(&wlanAddr);
            nn::btm::HostDeviceProperty hdp;
            nn::btm::GetHostDeviceProperty(&hdp);
            oss << "MAC Address" << endl;
            oss << "  * WLAN     : " << ToString(wlanAddr) << endl;
            oss << "  * Bluetooth: " << ToString(hdp.bdAddress) << endl;
            oss << endl;

            m_Console.Write(oss.str());
        }
    }
};

}
