﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//#include <nn/version.h>
#include "Node.h"
#include "Counter.h"
#include "Invoker.h"

namespace WlanTest {

/*---------------------------------------------------------------------------
           DataGeneratorParam
---------------------------------------------------------------------------*/

struct DataGeneratorParam
{
public :

    enum ItemId
    {
        ITEM_ID_UNDEFINED = 0,

        ITEM_ID_TYPE,
        ITEM_ID_PAYLOAD_SIZE,
        ITEM_ID_SEND_INTERVAL,
        ITEM_ID_NUM_OF_PACKETS
    };

    static const char* ITEM_STR_TYPE;
    static const char* ITEM_STR_PAYLOAD_SIZE;
    static const char* ITEM_STR_SEND_INTERVAL;
    static const char* ITEM_STR_NUM_OF_PACKETS;

    uint16_t  type;
    uint32_t  payloadSize;
    uint32_t  sendInterval;
    uint32_t  numOfPacket;

    DataGeneratorParam()
    {
        type         = PID_OUI_EXT;
        payloadSize  = 500;
        sendInterval = 16;
        numOfPacket  = 0;
    }

    void Print()
    {
        NN_LOG("  Type                     : %04x\n", type);
        NN_LOG("  Payload size             : %u byte\n", payloadSize);
        NN_LOG("  Send interval            : %u ms\n", sendInterval);
        NN_LOG("  Number of frames         : ");
        if(numOfPacket == 0)
        {
            NN_LOG("infinity\n");
        }
        else
        {
            NN_LOG("%u packets\n", numOfPacket);
        }
    }

    ItemId GetItemId(const string& item)
    {
        if(strcmp(item.c_str(), ITEM_STR_TYPE) == 0)            return ITEM_ID_TYPE;
        if(strcmp(item.c_str(), ITEM_STR_PAYLOAD_SIZE) == 0)    return ITEM_ID_PAYLOAD_SIZE;
        if(strcmp(item.c_str(), ITEM_STR_SEND_INTERVAL) == 0)   return ITEM_ID_SEND_INTERVAL;
        if(strcmp(item.c_str(), ITEM_STR_NUM_OF_PACKETS) == 0)  return ITEM_ID_NUM_OF_PACKETS;

        NN_LOG("  - failed : Unrecognized item name (%s)\n", item.c_str());

        return ITEM_ID_UNDEFINED;
    }

    bool SetParam(const string& item, const string& value)
    {
        ItemId id = GetItemId(item);
        if(id == ITEM_ID_UNDEFINED)
        {
            return false;
        }

        switch(id)
        {
        case ITEM_ID_TYPE                : return SetType(value);
        case ITEM_ID_PAYLOAD_SIZE        : return SetPayloadSize(value);
        case ITEM_ID_SEND_INTERVAL       : return SetSendInterval(value);
        case ITEM_ID_NUM_OF_PACKETS      : return SetNumOfPackets(value);
        default : return false;
        }

        return false;
    }

    bool SetType(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%4x", &type) == 1)
        {
            if(type == PID_OUI_EXT ||
               type == PID_PAYLOAD_ONLY)
            {
                return true;
            }
            else
            {
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    bool SetPayloadSize(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%u", &payloadSize) == 1)
        {
            if(16 <= payloadSize && payloadSize <= 1500)
            {
                return true;
            }
            else
            {
                NN_LOG("  - failed : Out of range (%u)\n", payloadSize);
                return false;
            }

        }
        else
        {
            return false;
        }
    }

    bool SetSendInterval(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%u", &sendInterval) == 1)
        {
            return true;

        }
        else
        {
            return false;
        }
    }

    bool SetNumOfPackets(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%u", &numOfPacket) == 1)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

};

class DataGenerator;
class FixedSizeDataTransmitter;

class GenerateCommand : public Command
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
private:
    DataGenerator      *m_pDataGenerator;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
protected:
    virtual void CommandContent();

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    NN_IMPLICIT GenerateCommand(DataGenerator* pGenerator);

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    virtual size_t GetInstanceSize() const;

};

class FixedSizeGenerateCommand : public Command
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
private:
    FixedSizeDataTransmitter      *m_pTransmitter;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
protected:
    virtual void CommandContent();
    virtual bool NeedReinvoke();

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    NN_IMPLICIT FixedSizeGenerateCommand(FixedSizeDataTransmitter* transmitter);

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    virtual size_t GetInstanceSize() const;
};


class DataGenerator
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
    Invoker               m_Invoker;
    nn::TimeSpan          m_Interval;
    uint64_t              m_Count;

    //Ieee802_3_Llc_Snap    m_Frame;
    DixFrame              m_Frame;
    size_t                m_Size;
    Node*                 m_Node;

    uint8_t               m_IeIndicator;//IEデータか通常データかの識別子

private:
/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Generate();
    virtual void Generate(uint32_t num);
    virtual void Cancel();

protected:
private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    DataGenerator();
    virtual ~DataGenerator();

protected:
private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    void SetTarget(Node& node)
    {
        m_Node = &node;
    }

    void GetCandidateDestinations(nn::wlan::MacAddress list[], uint32_t& num)
    {
        m_Node->GetMacAddressListInBss(list, num);
    }

    void SetDestination(nn::wlan::MacAddress address)
    {
        m_Frame.SetDestination(address);
    }

    void SetType(uint16_t type)
    {
        m_Frame.SetType(type);
    }

    void SetPayloadSize(size_t size)
    {
        m_Size = size;
    }

    void SetInterval(nn::TimeSpan time)
    {
        m_Interval = time;
    }

    virtual void ClearCount()
    {
        m_Count = 0;
    }

    void ChangePriority(int32_t priority)
    {
        m_Invoker.Stop();
        m_Invoker.Start(priority);
    }

    bool IsIdle()
    {
        return m_Invoker.GetCount() == 0 ? true : false;
    }

    //IEデータの識別子のセット
    void SetIeIndicator(uint8_t ind)
    {
        m_IeIndicator = ind;
    }

    NodeStatistics& GetTxStatistics(){ return m_Node->GetTxStatistics(); }

protected:
private:

};


class FixedSizeDataTransmitter : public DataGenerator
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    uint64_t  m_DataSize;
    uint64_t  m_SentDataSize;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Generate();
    virtual void Generate(uint32_t num);

protected:
private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    FixedSizeDataTransmitter();
    virtual ~FixedSizeDataTransmitter();

protected:
private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    void SetDataSize(uint64_t dataSize)
    {
        m_DataSize = dataSize;
    }
    uint64_t GetDataSize()
    {
        return m_DataSize;
    }

    uint64_t GetSentDataSize()
    {
        return m_SentDataSize;
    }

    bool IsCompleted()
    {
        return m_DataSize <= m_SentDataSize;
    }

    virtual void ClearCount()
    {
        DataGenerator::ClearCount();
        m_DataSize = 0;
        m_SentDataSize = 0;
    }

    virtual void ResetCount()
    {
        DataGenerator::ClearCount();
        m_SentDataSize = 0;
    }

protected:
private:

};


/*!--------------------------------------------------------------------------*
  @brief        データ集計器

 *---------------------------------------------------------------------------*/
class DataSink
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
    //Ieee802_3_Llc_Snap*    m_Frame;
    DixFrame*    m_Frame;

private:
/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Sink(const EventArgs* pArgs) = 0;
    virtual void Clear(){}

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    DataSink();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

};



/*!--------------------------------------------------------------------------*
  @brief        マルチデータ集計器
  　　　　　　　複数のソースからのデータを別々に集計する

 *---------------------------------------------------------------------------*/
class MultiDataSink : public DataSink
{

public:
    struct Statistics
    {
        // 更新間隔
        static nn::TimeSpan        UpdateInterval;
        static nn::os::Tick        LastUpdateTime;

        nn::os::Tick               FirstReceiveTime;
        nn::os::Tick               LastReceiveTime;

        // 受信開始後からの統計情報
        SequenceNumberCounter      Errors;
        LastAverage                Rssi;
        Average                    RxInterval;
        uint64_t                   LastSequenceNo;
        ThroughputMeter            Throughput;
        LatencyCounter             Latency;
        SequenceNumberCounter      ErrorsForLatency;

        // 一定間隔の統計情報
        SequenceNumberCounter      LastErrors;
        ThroughputMeter            LastThroughput;
        Average                    LastRxInterval;
        LatencyCounter             LastLatency;
        SequenceNumberCounter      LastErrorsForLatency;

        // 一定間隔の統計情報の処理変数
        SequenceNumberCounter      WorkErrors;
        ThroughputMeter            WorkThroughput;
        Average                    WorkRxInterval;
        LatencyCounter             WorkLatency;
        SequenceNumberCounter      WorkErrorsForLatency;

        void Clear()
        {
            FirstReceiveTime = nn::os::Tick(0);
            LastReceiveTime = nn::os::Tick(0);

            Errors.Clear();
            Rssi.Clear();
            RxInterval.Clear();
            LastSequenceNo = 0;
            Throughput.Clear();
            Latency.Clear();
            ErrorsForLatency.Clear();

            LastErrors.Clear();
            LastThroughput.Clear();
            LastRxInterval.Clear();
            LastLatency.Clear();
            LastErrorsForLatency.Clear();

            WorkErrors.Clear();
            WorkThroughput.Clear();
            WorkRxInterval.Clear();
            WorkLatency.Clear();
            WorkErrorsForLatency.Clear();
        }

        void ClearLatency()
        {
            Latency.Clear();
            ErrorsForLatency.Clear();

            LastLatency.Clear();
            LastErrorsForLatency.Clear();

            WorkLatency.Clear();
            WorkErrorsForLatency.Clear();
        }

        void ClearRxStats()
        {
            FirstReceiveTime = nn::os::Tick(0);
            LastReceiveTime = nn::os::Tick(0);

            Errors.Clear();
            Rssi.Clear();
            RxInterval.Clear();
            LastSequenceNo = 0;
            Throughput.Clear();

            LastErrors.Clear();
            LastThroughput.Clear();
            LastRxInterval.Clear();

            WorkErrors.Clear();
            WorkThroughput.Clear();
            WorkRxInterval.Clear();
        }
    };

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
    string                         MyAddress;
protected:
private:

    map<uint64_t, string>       m_Sources;
    map<uint64_t, Statistics*>  m_Statistics;
    map<uint64_t, ReceiverInfo> m_ReceiverInfos;
    nn::os::Mutex               m_Cs;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void Sink(const EventArgs* pArgs);
    virtual void Clear();
    virtual void ShowBurstStatistics();
    virtual void UpdateLastStatistics();

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    MultiDataSink();
    virtual ~MultiDataSink();

private:


/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    map<uint64_t, string>& GetSources()
    {
        return m_Sources;
    }

    map<uint64_t, Statistics*>& GetStatistics()
    {
        return m_Statistics;
    }

    map<uint64_t, ReceiverInfo>& GetReceiverInfos()
    {
        return m_ReceiverInfos;
    }
};

/*!--------------------------------------------------------------------------*
  @brief        アクションフレーム用マルチデータ集計器
  　　　　　　　複数のソースからのデータを別々に集計する

 *---------------------------------------------------------------------------*/
class MultiAfSink : public DataSink
{

public:
    struct Statistics
    {
        // 更新間隔
        static nn::TimeSpan        UpdateInterval;
        static nn::os::Tick        LastUpdateTime;

        nn::os::Tick               FirstReceiveTime;
        nn::os::Tick               LastReceiveTime;

        // 受信開始後からの統計情報
        SequenceNumberCounter      Errors;
        LastAverage                Rssi;
        Average                    RxInterval;
        uint64_t                   LastSequenceNo;
        ThroughputMeter            Throughput;
        LatencyCounter             Latency;

        // 一定間隔の統計情報
        SequenceNumberCounter      LastErrors;
        ThroughputMeter            LastThroughput;
        Average                    LastRxInterval;
        LatencyCounter             LastLatency;

        // 一定間隔の統計情報の処理変数
        SequenceNumberCounter      WorkErrors;
        ThroughputMeter            WorkThroughput;
        Average                    WorkRxInterval;
        LatencyCounter             WorkLatency;

        void Clear()
        {
            FirstReceiveTime = nn::os::Tick(0);
            LastReceiveTime = nn::os::Tick(0);

            Errors.Clear();
            Rssi.Clear();
            RxInterval.Clear();
            LastSequenceNo = 0;
            Throughput.Clear();
            Latency.Clear();

            LastErrors.Clear();
            LastThroughput.Clear();
            LastRxInterval.Clear();
            LastLatency.Clear();

            WorkErrors.Clear();
            WorkThroughput.Clear();
            WorkRxInterval.Clear();
            WorkLatency.Clear();
        }
    };

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
    string                         MyAddress;
protected:
private:

    map<uint64_t, string>       m_Sources;
    map<uint64_t, map<int16_t, Statistics*>>  m_Statistics;
    map<uint64_t, ReceiverInfo> m_ReceiverInfos;
    nn::os::Mutex               m_Cs;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void Sink(const EventArgs* pArgs);
    virtual void Clear();
    virtual void UpdateLastStatistics();

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    MultiAfSink();
    virtual ~MultiAfSink();

private:


/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    map<uint64_t, string>& GetSources()
    {
        return m_Sources;
    }

    map<uint64_t, map<int16_t, Statistics*>>& GetStatistics()
    {
        return m_Statistics;
    }

    map<uint64_t, ReceiverInfo>& GetReceiverInfos()
    {
        return m_ReceiverInfos;
    }
};


/*!--------------------------------------------------------------------------*
  @brief        アクションフレーム用マルチデータ集計器
  　　　　　　　複数のソースからのデータを別々に集計する

 *---------------------------------------------------------------------------*/
class MultiNdhpSink : public DataSink
{

public:

    struct Statistics
    {
        bool                       IsWitFormat;

        // 更新間隔
        static nn::TimeSpan        UpdateInterval;
        static nn::os::Tick        LastUpdateTime;

        nn::os::Tick               FirstReceiveTime;
        nn::os::Tick               LastReceiveTime;

        // 受信開始後からの統計情報
        SequenceNumberCounter      Errors;
        Average                    Rssi;
        Average                    RxInterval;
        uint64_t                   LastSequenceNo;
        ThroughputMeter            Throughput;
        LatencyCounter             Latency;

        // 一定間隔の統計情報
        SequenceNumberCounter      LastErrors;
        Average                    LastRssi;
        Average                    LastRxInterval;
        ThroughputMeter            LastThroughput;
        LatencyCounter             LastLatency;

        // 一定間隔の統計情報の処理変数
        SequenceNumberCounter      WorkErrors;
        Average                    WorkRssi;
        Average                    WorkRxInterval;
        ThroughputMeter            WorkThroughput;
        LatencyCounter             WorkLatency;

        void Clear()
        {
            IsWitFormat = false;

            FirstReceiveTime = nn::os::Tick(0);
            LastReceiveTime = nn::os::Tick(0);

            Errors.Clear();
            Rssi.Clear();
            RxInterval.Clear();
            LastSequenceNo = 0;
            Throughput.Clear();
            Latency.Clear();

            LastErrors.Clear();
            LastRssi.Clear();
            LastThroughput.Clear();
            LastRxInterval.Clear();
            LastLatency.Clear();

            WorkErrors.Clear();
            WorkRssi.Clear();
            WorkThroughput.Clear();
            WorkRxInterval.Clear();
            WorkLatency.Clear();
        }
    };

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    map<uint64_t, string>       m_Sources;
    map<uint64_t, map<uint64_t, Statistics*>>  m_Statistics;
    map<uint64_t, ReceiverInfo> m_ReceiverInfos;
    nn::os::Mutex               m_Cs;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void Sink(const EventArgs* pArgs);
    virtual void Clear();
    virtual void UpdateLastStatistics();

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    MultiNdhpSink();
    virtual ~MultiNdhpSink();

private:


/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    map<uint64_t, string>& GetSources()
    {
        return m_Sources;
    }

    map<uint64_t, map<uint64_t, Statistics*>>& GetStatistics()
    {
        return m_Statistics;
    }

    map<uint64_t, ReceiverInfo>& GetReceiverInfos()
    {
        return m_ReceiverInfos;
    }
};


#if 0
/*!--------------------------------------------------------------------------*
  @brief        ソケット集計器
  　　　　　　　複数のソースからのソケットを介したデータを別々に集計する

 *---------------------------------------------------------------------------*/
class SocketSink : public DataSink
{

public:
    struct Statistics
    {
        // 更新間隔
        static nn::TimeSpan        UpdateInterval;
        static nn::os::Tick        LastUpdateTime;

        nn::os::Tick               FirstReceiveTime;
        nn::os::Tick               LastReceiveTime;

        // 受信開始後からの統計情報
        uint64_t                   RxSize;
        ThroughputMeter            Throughput;

        // 一定間隔の統計情報
        uint64_t                   LastRxSize;
        ThroughputMeter            LastThroughput;

        // 一定間隔の統計情報の処理変数
        uint64_t                   WorkRxSize;
        ThroughputMeter            WorkThroughput;

        void Clear()
        {
            FirstReceiveTime = nn::os::Tick(0);
            LastReceiveTime = nn::os::Tick(0);

            RxSize = 0;
            Throughput.Clear();

            LastRxSize = 0;
            LastThroughput.Clear();

            WorkLastRxSize = 0;
            WorkThroughput.Clear();
        }
    };

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
    string                         MyAddress;
protected:
private:

    map<uint64_t, string>       m_Sources;
    map<uint64_t, Statistics*>  m_Statistics;
    map<uint64_t, ReceiverInfo> m_ReceiverInfos;
    nn::os::Mutex               m_Cs;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void Sink(ReceiveEventArgs args);
    virtual void Clear();
    virtual void ShowBurstStatistics();
    virtual void UpdateLastStatistics();

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    MultiDataSink();
    virtual ~MultiDataSink();

private:


/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    map<uint64_t, string>& GetSources()
    {
        return m_Sources;
    }

    map<uint64_t, Statistics*>& GetStatistics()
    {
        return m_Statistics;
    }

    map<uint64_t, ReceiverInfo>& GetReceiverInfos()
    {
        return m_ReceiverInfos;
    }
};

#endif

}
