﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Menu.h"
#include "App.h"
#include <nn/hid/hid_DebugPadMap.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/util/util_BitFlagSet.h>
#include <tuple>


namespace ApConnectivityTest {
namespace UI {

// SubMenuItem を選択したときの処理などをこちらに持たせたい

//----------------------------------------------------------------
// メニュー項目
//----------------------------------------------------------------
MenuItem::MenuItem(const std::string& name, bool enabled, const std::function<void()>& callback) :
    m_Type(MenuItemType_Item),
    m_Name(name),
    m_Enabled(enabled),
    m_Parent(nullptr),
    m_Callback(callback)
{
}


MenuItem::MenuItem(const std::string & name, const std::string & value, bool enabled, const std::function<void()>& callback) :
    m_Type(MenuItemType_Item),
    m_Name(name),
    m_Value(value),
    m_Enabled(enabled),
    m_Parent(nullptr),
    m_Callback(callback)
{
}


MenuItem::~MenuItem()
{
}


void MenuItem::Draw(float x, float y, float nameWidth, float subNameWidth, bool nameOnly)
{
    auto& fontRenderer = App::GetUi().GetGfxContext().GetFontRenderer();
    fontRenderer.EnableFixedFont(false);

    if (!m_Enabled)
    {
        fontRenderer.SetColor(nn::util::Color4u8::Gray());
    }

    fontRenderer.SetPosition(x, y);
    fontRenderer.Print(m_Name.c_str());

    if (nameOnly)
    {
        return;
    }

    fontRenderer.SetPosition(x + nameWidth, y);
    fontRenderer.Print(GetValue().c_str());
}


float MenuItem::GetNameWidth()
{
    auto& fontRenderer = App::GetUi().GetGfxContext().GetFontRenderer();
    fontRenderer.EnableFixedFont(false);

    return fontRenderer.CalculateWidth(m_Name.c_str()) + 20;
}


float MenuItem::GetValueWidth()
{
    auto& fontRenderer = App::GetUi().GetGfxContext().GetFontRenderer();
    fontRenderer.EnableFixedFont(false);

    return fontRenderer.CalculateWidth(m_Value.c_str()) + 20;
}


void MenuItem::Callback()
{
    if (m_Callback)
    {
        m_Callback();
    }
}


// アクセサ
MenuItemType MenuItem::GetType() const
{
    return m_Type;
}

void MenuItem::SetName(const std::string& name)
{
    m_Name = name;
}

const std::string & MenuItem::GetName() const
{
    return m_Name;
}

void MenuItem::SetValue(const std::string& value)
{
    m_Value = value;
}

const std::string & MenuItem::GetValue() const
{
    return m_Value;
}

void MenuItem::SetEnabled(bool enabled)
{
    m_Enabled = enabled;
}

bool MenuItem::GetEnabled() const
{
    return m_Enabled;
}

void MenuItem::SetParent(Menu* parent)
{
    m_Parent = parent;
}

Menu * MenuItem::GetParent() const
{
    return m_Parent;
}

void MenuItem::SetCallback(const std::function<void()>& callback)
{
    m_Callback = callback;
}


// コンストラクタ
MenuItem::MenuItem(MenuItemType type, const std::string& name, bool enabled, const std::function<void()>& callback) :
    m_Type(type),
    m_Name(name),
    m_Enabled(enabled),
    m_Parent(nullptr),
    m_Callback(callback)
{
}


//----------------------------------------------------------------
// メニュー項目：セパレータ
//----------------------------------------------------------------
Separator::Separator() :
    MenuItem(MenuItemType_Separator, "", false)
{
}


Separator::~Separator()
{
}


//----------------------------------------------------------------
// メニュー項目：サブメニュー
//----------------------------------------------------------------
SubMenuItem::SubMenuItem(const std::string& name, bool enabled, const std::shared_ptr<Menu>& subMenu) :
    MenuItem(MenuItemType_SubMenuItem, name, enabled),
    m_SubMenu(subMenu)
{
}


SubMenuItem::~SubMenuItem()
{
}


void SubMenuItem::Draw(float x, float y, float nameWidth, float subNameWidth, bool nameOnly)
{
    auto& fontRenderer = App::GetUi().GetGfxContext().GetFontRenderer();
    fontRenderer.EnableFixedFont(false);

    if (!GetEnabled())
    {
        fontRenderer.SetColor(nn::util::Color4u8::Gray());
    }

    fontRenderer.SetPosition(x, y);
    fontRenderer.Print(GetName().c_str());

    if (nameOnly)
    {
        return;
    }

    fontRenderer.SetPosition(x + nameWidth, y);
    fontRenderer.Print(GetValue().c_str());

    fontRenderer.EnableFixedFont(true);
    fontRenderer.SetPosition(x + nameWidth + subNameWidth - fontRenderer.GetFixedFontWidth(), y);
    fontRenderer.Print(">");
}


std::shared_ptr<Menu> SubMenuItem::GetSubMenu() const
{
    return m_SubMenu;
}


//----------------------------------------------------------------
// メニュー項目：値選択サブメニュー
//----------------------------------------------------------------
ValueSelectSubMenuItem::ValueSelectSubMenuItem(const std::string& name, bool enabled, bool enableInput, int defaultIndex, const std::vector<std::string>& strList) :
    SubMenuItem(name, enabled, CreateMenu(enableInput, defaultIndex + enableInput, strList)),
    m_MinValueLength(0),
    m_MaxValueLength(0),
    m_ValueChangedCallback(std::function<void(const std::string&)>())
{
    SetValue((*GetSubMenu()->activeItem)->GetName());
}


ValueSelectSubMenuItem::ValueSelectSubMenuItem(const std::string& name, bool enabled, bool enableInput, int defaultIndex, const std::function<void(const std::string&)>& valueChangedCallback, const std::vector<std::string>& strList) :
    SubMenuItem(name, enabled, CreateMenu(enableInput, defaultIndex + enableInput, strList)),
    m_MinValueLength(0),
    m_MaxValueLength(0),
    m_ValueChangedCallback(valueChangedCallback)
{
    SetValue((*GetSubMenu()->activeItem)->GetName());
}


ValueSelectSubMenuItem::ValueSelectSubMenuItem(const std::string& name, bool enabled, bool enableInput, size_t minLength, size_t maxLength, int defaultIndex, const std::vector<std::string>& strList) :
    SubMenuItem(name, enabled, CreateMenu(enableInput, defaultIndex + enableInput, strList)),
    m_MinValueLength(minLength),
    m_MaxValueLength(maxLength),
    m_ValueChangedCallback(std::function<void(const std::string&)>())
{
    SetValue((*GetSubMenu()->activeItem)->GetName());

    UpdateEnabledSubMenuItem();
}


ValueSelectSubMenuItem::ValueSelectSubMenuItem(const std::string& name, bool enabled, bool enableInput, size_t minLength, size_t maxLength, int defaultIndex, const std::function<void(const std::string&)>& valueChangedCallback, const std::vector<std::string>& strList) :
    SubMenuItem(name, enabled, CreateMenu(enableInput, defaultIndex + enableInput, strList)),
    m_MinValueLength(minLength),
    m_MaxValueLength(maxLength),
    m_ValueChangedCallback(valueChangedCallback)
{
    SetValue((*GetSubMenu()->activeItem)->GetName());

    UpdateEnabledSubMenuItem();
}


ValueSelectSubMenuItem::~ValueSelectSubMenuItem()
{
}


void ValueSelectSubMenuItem::SetMinValueLength(size_t value)
{
    m_MinValueLength = value;

    UpdateEnabledSubMenuItem();
}


size_t ValueSelectSubMenuItem::GetMinValueLength() const
{
    return m_MinValueLength;
}


void ValueSelectSubMenuItem::SetMaxValueLength(size_t value)
{
    m_MaxValueLength = value;

    UpdateEnabledSubMenuItem();
}


size_t ValueSelectSubMenuItem::GetMaxValueLength() const
{
    return m_MaxValueLength;
}


void ValueSelectSubMenuItem::SelectItemCallback()
{
    SetValue((*GetSubMenu()->activeItem)->GetName());
    if (m_ValueChangedCallback)
    {
        m_ValueChangedCallback(GetValue());
    }
}


void ValueSelectSubMenuItem::ShowKeyboardCallback()
{
    App::GetUi().OpenKeyboard(Keyboard::KeyMap_FullKey, GetValue(), m_MinValueLength, m_MaxValueLength, std::bind(&ValueSelectSubMenuItem::KeyboardResultCallback, this, std::placeholders::_1));
}


void ValueSelectSubMenuItem::KeyboardResultCallback(const std::string& str)
{
    SetValue(str);

    if (m_ValueChangedCallback)
    {
        m_ValueChangedCallback(GetValue());
    }
}


void ValueSelectSubMenuItem::UpdateEnabledSubMenuItem()
{
    for (auto &item : GetSubMenu()->items)
    {
        if (item->GetName() == "Keyboard..." || (item->GetName().size() >= m_MinValueLength && item->GetName().size() <= m_MaxValueLength))
        {
            item->SetEnabled(true);
        }
        else
        {
            item->SetEnabled(false);
        }
    }
}


std::shared_ptr<Menu> ValueSelectSubMenuItem::CreateMenu(bool enableInput, int defaultIndex, const std::vector<std::string>& strList)
{
    std::vector<std::shared_ptr<MenuItem>> items(strList.size() + enableInput);

    if (enableInput)
    {
        items[0] = std::shared_ptr<MenuItem>(new MenuItem("Keyboad...", true, std::bind(&ValueSelectSubMenuItem::ShowKeyboardCallback, this)));
    }

    for (int i = 0; i < strList.size(); ++i)
    {
        items[i + enableInput] = std::shared_ptr<MenuItem>(new MenuItem(strList[i], true, std::bind(&ValueSelectSubMenuItem::SelectItemCallback, this)));
    }

    return std::shared_ptr<Menu>(new Menu(true, defaultIndex, items));
}


//----------------------------------------------------------------
// メニュー
//----------------------------------------------------------------
Menu::Menu(bool vertical, int defaultIndex, const std::vector<std::shared_ptr<MenuItem>>& items) :
    vertical(vertical)
{
    int index = 0;
    for (auto& item : items)
    {
        AddItem(item);
        item->SetParent(this);

        if (index == defaultIndex)
        {
            activeItem = --this->items.end();
        }

        ++index;
    }
}


Menu::~Menu()
{
}


void Menu::AddItem(const std::shared_ptr<MenuItem>& item)
{
    items.push_back(item);
}


void Menu::InsertItem(size_t index, const std::shared_ptr<MenuItem>& item)
{
    auto iter = items.begin();
    while (index--)
    {
        ++iter;
        if (iter == items.end())
        {
            return;
        }
    }

    items.insert(iter, item);
}


void Menu::RemoveItem(size_t index)
{
    auto iter = items.begin();
    while (index--)
    {
        ++iter;
    }

    if (activeItem == iter)
    {
        MoveFocusNext();
        if (activeItem == iter)
        {
            activeItem = items.end();
        }
    }
    items.erase(iter);
}


void Menu::RemoveItem(const std::shared_ptr<MenuItem>& item)
{
    if (*activeItem == item)
    {
        MoveFocusNext();
        if (*activeItem == item)
        {
            activeItem = items.end();
        }
    }
    items.remove(item);
}


void Menu::Draw()
{
    auto& fontRenderer = App::GetUi().GetGfxContext().GetFontRenderer();

    // メニューのサイズを更新する
    float maxNameWidth = 0;
    float maxSubNameWidth = 0;
    float totalWidth = 0;

    for (auto& item : items)
    {
        auto nameWidth = item->GetNameWidth();
        if (nameWidth > maxNameWidth)
        {
            maxNameWidth = nameWidth;
        }

        auto nameSubWidth = item->GetValueWidth();
        if (nameSubWidth > maxSubNameWidth)
        {
            maxSubNameWidth = nameSubWidth;
        }

        totalWidth += nameWidth + nameSubWidth;
    }

    if (vertical)
    {
        rectangle.width = maxNameWidth + maxSubNameWidth + 2;
        rectangle.height = fontRenderer.GetFontHeight() * items.size() + 2;
    }
    else
    {
        rectangle.width = totalWidth + 2;
        rectangle.height = fontRenderer.GetFontHeight() + 2;
    }

    // 枠描画
    {
        nn::util::Float3 points[] = {
            { { { rectangle.x, rectangle.y, 0 } } },
            { { { rectangle.x + rectangle.width, rectangle.y, 0 } } },
            { { { rectangle.x + rectangle.width, rectangle.y + rectangle.height, 0 } } },
            { { { rectangle.x, rectangle.y + rectangle.height, 0 } } },
            { { { rectangle.x, rectangle.y, 0 } } },
        };
        App::GetUi().GetGfxContext().DrawLinkedPoint(points, sizeof(points) / sizeof(*points), { { 255, 127, 0, 255 } });
    }
    App::GetUi().GetGfxContext().DrawQuad(
        nn::util::Vector3f(rectangle.x + 1, rectangle.y + 1, -15),
        NN_UTIL_FLOAT_2_INITIALIZER(rectangle.width - 2, rectangle.height - 2),
        { { 15, 15, 15, 255 } });

    // 子項目の描画
    int itemIndex = 0;
    float left = rectangle.x + 1;
    float top = rectangle.y + 1;
    for (auto& item : items)
    {
        if (item == *activeItem)
        {
            fontRenderer.SetColor(nn::util::Color4u8(255, 127, 0, 255));
        }
        else
        {
            fontRenderer.SetColor(nn::util::Color4u8::White());
        }

        item->Draw(left, top, maxNameWidth, maxSubNameWidth, !vertical);

        ++itemIndex;

        if (vertical)
        {
            top = rectangle.y + 1 + itemIndex * fontRenderer.GetFontHeight();
        }
        else
        {
            left += item->GetNameWidth() + item->GetValueWidth();
        }
    }
}


void Menu::MoveFocusPrevious()
{
    for (int i = items.size(); i--;)
    {
        if (activeItem != items.begin())
        {
            --activeItem;
        }
        else
        {
            activeItem = --items.end();
        }

        if ((*activeItem)->GetType() != MenuItemType_Separator && (*activeItem)->GetEnabled())
        {
            break;
        }
    }
}


void Menu::MoveFocusNext()
{
    for (int i = items.size(); i--;)
    {
        if (++activeItem == items.end())
        {
            activeItem = items.begin();
        }

        if ((*activeItem)->GetType() != MenuItemType_Separator && (*activeItem)->GetEnabled())
        {
            break;
        }
    }
}


void Menu::SelectItem()
{
    int activeIndex = 0;
    for (auto& item : items)
    {
        if (item == *activeItem)
        {
            break;
        }
        ++activeIndex;
    }

    (*activeItem)->Callback();

    if ((*activeItem)->GetType() == MenuItemType_SubMenuItem)
    {
        auto subMenu = reinterpret_cast<SubMenuItem&>(*(*activeItem)).GetSubMenu();

        for (auto iter = subMenu->items.begin(); iter != subMenu->items.end(); ++iter)
        {
            if ((*iter)->GetEnabled())
            {
                subMenu->activeItem = iter;
                break;
            }
        }

        if (vertical)
        {
            subMenu->rectangle.x = rectangle.x + rectangle.width;
            subMenu->rectangle.y = rectangle.y + activeIndex * App::GetUi().GetGfxContext().GetFontRenderer().GetFontHeight();
        }
        else
        {
            float left = rectangle.x;
            for (auto& item : items)
            {
                if (item == *activeItem)
                {
                    break;
                }

                left += item->GetNameWidth() + item->GetValueWidth();
            }

            subMenu->rectangle.x = left;
            subMenu->rectangle.y = rectangle.y + 1 + App::GetUi().GetGfxContext().GetFontRenderer().GetFontHeight();
        }

        App::GetUi().PopupMenu(subMenu);
    }
    else if ((*activeItem)->GetType() == MenuItemType_Item)
    {
        App::GetUi().CloseMenu();
    }
}


std::shared_ptr<MenuItem> Menu::operator[](const std::string& name)
{
    for (auto& item : items)
    {
        if (item->GetName() == name)
        {
            return item;
        }
    }
    return std::shared_ptr<MenuItem>();
}

}
}
