﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Keyboard.h"
#include "App.h"
#include "UI.h"


namespace ApConnectivityTest {
namespace UI {

// コンストラクタ
Keyboard::Keyboard() :
        m_KeyMap(nullptr),
        m_ShiftState(ShiftState_Off)
{
}


// デストラクタ
Keyboard::~Keyboard()
{
}


// 初期化
void Keyboard::Initialize(KeyMap* keyMap, const std::string& initialString, size_t minLength, size_t maxLength, const std::function<void(const std::string&)>& callback)
{
    m_KeyMap = keyMap;
    m_ActiveKey = m_KeyMap->GetKeys();
    m_ShiftState = ShiftState_Off;

    m_MinLength = minLength;
    m_MaxLength = maxLength;

    m_Str = initialString;
    m_CalletPos = initialString.size();

    m_Callback = callback;

    m_KeyboardHeight = 0;
    for (auto key = m_KeyMap->GetKeys(); key->m_Char; ++key)
    {
        auto keyBottomPos = key->m_Position.y + key->m_Size.y / 2;
        if (keyBottomPos > m_KeyboardHeight)
        {
            m_KeyboardHeight = keyBottomPos;
        }
    }
}


// 描画
void Keyboard::Draw()
{
    auto& gfxContext = App::GetUi().GetGfxContext();
    auto& fontRenterer = gfxContext.GetFontRenderer();
    fontRenterer.EnableFixedFont(false);

    // 入力文字列の描画
    {
        // 文字数が指定範囲外なら色で警告する
        // TODO: 色以外でも警告する
        if (m_Str.size() < m_MinLength || (m_MaxLength > 0 && m_Str.size() > m_MaxLength))
        {
            fontRenterer.SetColor(nn::util::Color4u8(255, 63, 15));
        }
        else
        {
            fontRenterer.SetColor(nn::util::Color4u8::White());
        }

        float posX = fontRenterer.GetFixedFontWidth();
        float posY = 720 - m_KeyboardHeight * 1280 / 14 - fontRenterer.GetFontHeight() * 2;
        fontRenterer.SetPosition(posX, posY);

        auto str = m_Str.substr(0, m_CalletPos);
        fontRenterer.Print(str.c_str());

        // キャレットの描画
        if (str.size()) // 長さ 0 の文字列を渡すとおかしい
        {
            posX += fontRenterer.CalculateWidth(str.c_str());
        }
        gfxContext.DrawQuad(nn::util::Vector3f(posX, posY, 0), NN_UTIL_FLOAT_2_INITIALIZER(2, fontRenterer.GetFontHeight()),
                GetUint8x4(255, 255, 255, 255));

        posX += 1;
        fontRenterer.SetPosition(posX, posY);

        fontRenterer.Print(m_Str.substr(m_CalletPos).c_str());
    }

    // キーボードの描画
    std::string printStr;
    for (auto key = m_KeyMap->GetKeys(); key->m_Char; ++key)
    {
        // キーの中心位置
        auto posX = key->m_Position.x * 1280 / 14;
        auto posY = 720 + (-m_KeyboardHeight + key->m_Position.y) * 1280 / 14;

        // キーのサイズ
        auto width = key->m_Size.x * 1280 / 14;
        auto height = key->m_Size.y * 1280 / 14;

        // 描画色設定 & カーソルのあるキーの背景描画
        if (key->m_Char == '\x82' && m_ShiftState == ShiftState_OnLock)
        {
            gfxContext.DrawQuad(
                nn::util::Vector3f(posX - width / 2, posY - height / 2, 0),
                NN_UTIL_FLOAT_2_INITIALIZER(width, height),
                { { 255, 255, 0, 255 } });

            fontRenterer.SetColor(nn::util::Color4u8::Black());
        }
        else if (key == m_ActiveKey || (key->m_Char == '\x82' && m_ShiftState == ShiftState_On))
        {
            gfxContext.DrawQuad(
                    nn::util::Vector3f(posX - width / 2, posY - height / 2, 0),
                    NN_UTIL_FLOAT_2_INITIALIZER(width, height),
                    { { 255, 127, 0, 255 } });

            fontRenterer.SetColor(nn::util::Color4u8::Black());
        }
        else
        {
            fontRenterer.SetColor(nn::util::Color4u8::White());
        }

        // キー文字を設定
        auto keyName = m_ShiftState == ShiftState_Off ? key->m_Name : key->m_ShiftName;

        if (keyName)
        {
            printStr = keyName;
        }
        else
        {
            printStr = m_ShiftState == ShiftState_Off ? key->m_Char : key->m_ShiftChar;
        }

        // キー文字をキー中心に描画
        fontRenterer.SetPosition(
                posX - fontRenterer.CalculateWidth(printStr.c_str()) / 2,
                posY - fontRenterer.GetFontHeight() / 2
                );
        fontRenterer.Print(printStr.c_str());

        // キー枠を描画
        {
            nn::util::Float3 points[] = {
                { { { posX - width / 2, posY - height / 2, 0 } } },
                { { { posX + width / 2, posY - height / 2, 0 } } },
                { { { posX + width / 2, posY + height / 2, 0 } } },
                { { { posX - width / 2, posY + height / 2, 0 } } },
                { { { posX - width / 2, posY - height / 2, 0 } } },
            };
            gfxContext.DrawLinkedPoint(points, sizeof(points) / sizeof(*points), { { 255, 255, 255, 255 } });
        }
    }
}


// キー押下処理
void Keyboard::OnKeyDown(const nn::hid::DebugPadButtonSet& button)
{
    // ShiftLock
    if ((button & (nn::hid::DebugPadButton::ZL::Mask | nn::hid::DebugPadButton::ZR::Mask)).IsAnyOn())
    {
        m_ShiftState = ShiftState_OnLock;
    }
}


// キー開放処理
void Keyboard::OnKeyUp(const nn::hid::DebugPadButtonSet & button)
{
    // ShiftLock
    if ((button & (nn::hid::DebugPadButton::ZL::Mask | nn::hid::DebugPadButton::ZR::Mask)).IsAnyOn())
    {
        m_ShiftState = ShiftState_Off;
    }
}


// キー入力処理
void Keyboard::OnKeyPress(const nn::hid::DebugPadButtonSet& button)
{
    // カーソル移動
    if (button.Test<nn::hid::DebugPadButton::Left>())
    {
        if (m_ActiveKey->m_pLeftKey)
        {
            m_ActiveKey = m_ActiveKey->m_pLeftKey;
        }
    }
    else if (button.Test<nn::hid::DebugPadButton::Right>())
    {
        if (m_ActiveKey->m_pRightKey)
        {
            m_ActiveKey = m_ActiveKey->m_pRightKey;
        }
    }
    else if (button.Test<nn::hid::DebugPadButton::Up>())
    {
        if (m_ActiveKey->m_pUpKey)
        {
            m_ActiveKey = m_ActiveKey->m_pUpKey;
        }
    }
    else if (button.Test<nn::hid::DebugPadButton::Down>())
    {
        if (m_ActiveKey->m_pBottomKey)
        {
            m_ActiveKey = m_ActiveKey->m_pBottomKey;
        }
    }
    // キーボードのキー押下
    else if (button.Test<nn::hid::DebugPadButton::A>())
    {
        auto keyChar = m_ShiftState == ShiftState_Off ? m_ActiveKey->m_Char : m_ActiveKey->m_ShiftChar;

        if (keyChar & 0x80)
        {
            m_KeyMap->SpecialKeyDown(*this, keyChar);
        }
        else
        {
            Input(keyChar);
        }
    }
    // キーボード終了
    else if (button.Test<nn::hid::DebugPadButton::B>())
    {
        App::GetUi().CloseKeyboard();
    }
    // Enter
    else if (button.Test<nn::hid::DebugPadButton::Start>())
    {
        Enter();
    }
    // BackSpace
    else if (button.Test<nn::hid::DebugPadButton::Y>())
    {
        BackSpace();
    }
    // Space
    else if (button.Test<nn::hid::DebugPadButton::X>())
    {
        m_KeyMap->SpecialKeyDown(*this);
    }
    // キャレット左移動
    else if (button.Test<nn::hid::DebugPadButton::L>())
    {
        MoveCalletLeft();
    }
    // キャレット右移動
    else if (button.Test<nn::hid::DebugPadButton::R>())
    {
        MoveCalletRight();
    }
}


// キャレットを左に移動
void Keyboard::MoveCalletLeft()
{
    if (m_CalletPos > 0)
    {
        --m_CalletPos;
    }
}


// キャレットを右に移動
void Keyboard::MoveCalletRight()
{
    if (m_CalletPos < m_Str.size())
    {
        ++m_CalletPos;
    }
}


// 文字入力
void Keyboard::Input(char inputChar)
{
    m_Str.insert(m_CalletPos, 1, inputChar);
    ++m_CalletPos;
    if (m_ShiftState == ShiftState_On)
    {
        m_ShiftState = ShiftState_Off;
    }
}


// 1 文字削除
void Keyboard::BackSpace()
{
    if (m_CalletPos > 0)
    {
        m_Str.erase(m_CalletPos - 1, 1);
        --m_CalletPos;
    }
}


// 決定 (改行 ?)
void Keyboard::Enter()
{
    if (m_Str.size() < m_MinLength || (m_MaxLength > 0 && m_Str.size() > m_MaxLength))
    {
        return;
    }

    m_Callback(m_Str);
    App::GetUi().CloseKeyboard();
}


// シフトキー
void Keyboard::ToggleShift()
{
    switch (m_ShiftState)
    {
    case ShiftState_Off:
        m_ShiftState = ShiftState_On;
        break;

    case ShiftState_On:
        m_ShiftState = ShiftState_OnLock;
        break;

    case ShiftState_OnLock:
        m_ShiftState = ShiftState_Off;
        break;

    default:
        break;
    }
}


// 座標からキーを取得する
Keyboard::Key * Keyboard::GetHitKey(int32_t x, int32_t y)
{
    for (auto key = m_KeyMap->GetKeys(); key->m_Char; ++key)
    {
    }
    return nullptr;
}

}
}
