﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ApConnectivityTestApp.h"

#include <nn/bsdsocket/cfg/cfg.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <cstdlib>
#include <map>


namespace
{

#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
namespace posix
{
#include <netinet/ip_icmp.h>
}
#endif

// ソケット用メモリプール
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

// メニュー項目、 enum 値を引くための辞書
static const std::map<std::string, nn::nifm::Authentication> g_AuthEnumMap = {
    { "Open", nn::nifm::Authentication_Open },
    { "Shared", nn::nifm::Authentication_Shared },
    { "WPA-PSK", nn::nifm::Authentication_WpaPsk },
    { "WPA2-PSK", nn::nifm::Authentication_Wpa2Psk },
    { "WPA/WPA2 mixed", nn::nifm::Authentication_Invalid },
};

static const std::map<std::string, nn::nifm::Encryption> g_EncryptEnumMap = {
    { "None", nn::nifm::Encryption_None },
    { "WEP", nn::nifm::Encryption_Wep },
    { "AES", nn::nifm::Encryption_Aes },
};

static const std::map<nn::nifm::LinkLevel, std::string> g_LevelNameMap = {
    { nn::nifm::LinkLevel_0, "Worst" },
    { nn::nifm::LinkLevel_1, "Bad" },
    { nn::nifm::LinkLevel_2, "Better" },
    { nn::nifm::LinkLevel_3, "Best" },
};

static const std::map<nn::nifm::Authentication, std::string> g_AuthNameMap = {
    { nn::nifm::Authentication_Invalid, "Invalid" },
    { nn::nifm::Authentication_Open, "Open" },
    { nn::nifm::Authentication_Shared, "Shared" },
    { nn::nifm::Authentication_Wpa, "WPA-Enterprise" },
    { nn::nifm::Authentication_WpaPsk, "WPA-Personal" },
    { nn::nifm::Authentication_Wpa2, "WPA2-Enterprise" },
    { nn::nifm::Authentication_Wpa2Psk, "WPA2-Personal" },
};

static const std::map<nn::nifm::Encryption, std::string> g_EncryptNameMap = {
    { nn::nifm::Encryption_Invalid, "Invalid" },
    { nn::nifm::Encryption_None, "None" },
    { nn::nifm::Encryption_Wep, "WEP" },
    { nn::nifm::Encryption_Tkip, "TKIP" },
    { nn::nifm::Encryption_Aes, "AES" },
};

const char g_IperfDefaultCommandline[] = "-s -i 1";

}


namespace ApConnectivityTest
{

// コンストラクタ
ApConnectivityTestApp::ApConnectivityTestApp()
{
    // nifm の初期化
    if (nn::nifm::InitializeAdmin().IsFailure())
    {
        // nifm:a の権限がないので、弱い方を使う
        nn::nifm::Initialize();
        m_NifmAdminMode = false;
    }
    else
    {
        m_NifmAdminMode = true;
    }
    nn::nifm::SetExclusiveClient(nn::nifm::GetClientId());

    // ソケットの初期化
    nn::socket::Initialize(g_SocketConfigWithMemory);

    // ネットワーク機能の初期化
    Network::Initialize();

    // UI の初期化
    InitializeMenu();
    GetUi().SetTopMenu(m_Menu);
}


// デストラクタ
ApConnectivityTestApp::~ApConnectivityTestApp()
{
    Network::Finalize();

    nn::socket::Finalize();
}


// メニューを作成
void ApConnectivityTestApp::InitializeMenu()
{
    m_Menu.reset(new UI::Menu(false, 0, {}));

    m_ApMenuItem.reset(new UI::SubMenuItem("AP", true, std::shared_ptr<UI::Menu>(std::shared_ptr<UI::Menu>(new UI::Menu(true, 0, {})))));
    m_ApSsidMenuItem.reset(new UI::ValueSelectSubMenuItem("SSID", true, true, 0, 32, 0, {
        "test_ap_0",
        "test_ap_1",
        "test_ap_2",
        "test_ap_3",
        "test_ap_4",
        "test_ap_5",
        "test_ap_6",
        "test_ap_7",
        "test_ap_8",
        "test_ap_9",
    }));
    m_ApAuthenticationMenuItem.reset(new UI::ValueSelectSubMenuItem("Authentication", true, false, 0, std::bind(&ApConnectivityTestApp::AuthenticationChangeCallback, this, std::placeholders::_1), {
        "Open",
        "Shared",
        "WPA-PSK",
        "WPA2-PSK",
    }));
    m_ApEncryptionMenuItem.reset(new UI::ValueSelectSubMenuItem("Encryption", true, false, 0, std::bind(&ApConnectivityTestApp::EncryptionChangeCallback, this, std::placeholders::_1), {
        "None",
        "WEP",
        "AES",
    }));
    m_ApSharedKeyMenuItem.reset(new UI::ValueSelectSubMenuItem("SharedKey", true, true, 0, {
        "01234",
        "0123456789abc",
        "0123456789abcdef",
    }));

    m_ApIpSettingMenuItem.reset(new UI::ValueSelectSubMenuItem("IP Setting", true, false, 1, std::bind(&ApConnectivityTestApp::ApIpSettingMenuItem, this, std::placeholders::_1), {
        "Static",
        "DHCP",
    }));
    m_ApIpAddressMenuItem.reset(new UI::MenuItem("IP Address", "", false, std::bind(&ApConnectivityTestApp::ApIpAddressMenuItem, this)));
    m_ApSubnetMaskMenuItem.reset(new UI::MenuItem("Subnet Mask", "", false, std::bind(&ApConnectivityTestApp::ApSubnetMaskMenuItem, this)));
    m_ApDefaultGatewayMenuItem.reset(new UI::MenuItem("Default Gateway", "", false, std::bind(&ApConnectivityTestApp::ApDefaultGatewayMenuItem, this)));
    m_ApDnsSettingMenuItem.reset(new UI::ValueSelectSubMenuItem("DNS Setting", false, false, 1, std::bind(&ApConnectivityTestApp::ApDnsSettingMenuItem, this, std::placeholders::_1), {
        "Manual",
        "Automatic",
    }));
    m_ApPreferredDnsMenuItem.reset(new UI::MenuItem("Preferred DNS", "0.0.0.0", false, std::bind(&ApConnectivityTestApp::ApPreferredDnsMenuItem, this)));
    m_ApAlternateDnsMenuItem.reset(new UI::MenuItem("Alternate DNS", "0.0.0.0", false, std::bind(&ApConnectivityTestApp::ApAlternateDnsMenuItem, this)));
    m_ApMtuMenuItem.reset(new UI::MenuItem("MTU", "1400", true, std::bind(&ApConnectivityTestApp::ApMtuMenuItem, this)));
    m_ApProxySettingMenuItem.reset(new UI::ValueSelectSubMenuItem("Proxy", true, false, 0, std::bind(&ApConnectivityTestApp::ApProxySettingMenuItem, this, std::placeholders::_1), {
        "Disable",
        "Enable",
    }));
    m_ApProxyAddressMenuItem.reset(new UI::MenuItem("Proxy Address", "", false, std::bind(&ApConnectivityTestApp::ApProxyAddressMenuItem, this)));
    m_ApProxyPortMenuItem.reset(new UI::MenuItem("Proxy Port", "8080", false, std::bind(&ApConnectivityTestApp::ApProxyPortMenuItem, this)));

    m_ApRetryCountMenuItem.reset(new UI::ValueSelectSubMenuItem("Retry Count", true, false, 0, {
        "0",
        "1",
        "2",
        "3",
        "4",
        "5",
    }));
    m_ApConnectMenuItem.reset(new UI::MenuItem("Connect", true, std::bind(&ApConnectivityTestApp::ConnectMenuCallback, this)));
    m_ApDisconnectMenuItem.reset(new UI::MenuItem("Disconnect", false, std::bind(&ApConnectivityTestApp::DisconnectMenuCallback, this)));

    m_WpsMenuItem.reset(new UI::SubMenuItem("WPS", true, std::shared_ptr<UI::Menu>(new UI::Menu(true, 0, {}))));
    m_WpsPbcMenuItem.reset(new UI::MenuItem("PBC", true, std::bind(&ApConnectivityTestApp::WpsPbcMenuCallback, this)));
    m_WpsPinMenuItem.reset(new UI::MenuItem("PIN", true, std::bind(&ApConnectivityTestApp::WpsPinMenuCallback, this)));

    m_EthernetMenuItem.reset(new UI::SubMenuItem("Ethernet", true, std::shared_ptr<UI::Menu>(new UI::Menu(true, 0, {}))));
    m_EthernetConnectMenuItem.reset(new UI::MenuItem("Connect", true, std::bind(&ApConnectivityTestApp::EthernetConnectMenuCallback, this)));

    m_PingMenuItem.reset(new UI::SubMenuItem("Ping", false, std::shared_ptr<UI::Menu>(new UI::Menu(true, 0, {}))));
    m_PingDestinationMenuItem.reset(new UI::ValueSelectSubMenuItem("Destination", true, true, 0, {
        "Default gateway",
    }));
    m_PingSizeMenuItem.reset(new UI::MenuItem("Size", "64", true, std::bind(&ApConnectivityTestApp::PingSizeMenuCallback, this)));
    m_PingRepeatCountMenuItem.reset(new UI::ValueSelectSubMenuItem("Repeat Count", true, false, 9, {
        "1",
        "2",
        "3",
        "4",
        "5",
        "6",
        "7",
        "8",
        "9",
        "10",
        "Infinity",
    }));
    m_PingStartMenuItem.reset(new UI::MenuItem("Start", true, std::bind(&ApConnectivityTestApp::StartPingCallback, this)));
    m_PingStopMenuItem.reset(new UI::MenuItem("Stop", false, std::bind(&ApConnectivityTestApp::StopPingCallback, this)));

    m_IperfMenuItem.reset(new UI::SubMenuItem("Iperf", false, std::shared_ptr<UI::Menu>(new UI::Menu(true, 0, {}))));
    m_IperfCommandlineMenuItem.reset(new UI::MenuItem("Commandline", g_IperfDefaultCommandline, true, std::bind(&ApConnectivityTestApp::IperfCommandlineMenuCallback, this, 0)));
    m_IperfStartMenuItem.reset(new UI::MenuItem("Start", true, std::bind(&ApConnectivityTestApp::StartIperfCallback, this)));
    m_IperfStopMenuItem.reset(new UI::MenuItem("Stop", false, std::bind(&ApConnectivityTestApp::StopIperfCallback, this)));

    m_MiscMenuItem.reset(new UI::SubMenuItem("Misc", true, std::shared_ptr<UI::Menu>(new UI::Menu(true, 0, {}))));
    m_MiscQuitApplicationMenuItem.reset(new UI::MenuItem("Quit Application", true, std::bind(&ApConnectivityTestApp::QuitApplicationMenuCallback, this)));

    m_ApMenuItem->GetSubMenu()->AddItem(m_ApSsidMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApAuthenticationMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApEncryptionMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApSharedKeyMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(std::shared_ptr<UI::Separator>(new UI::Separator()));
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApIpSettingMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApIpAddressMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApSubnetMaskMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApDefaultGatewayMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(std::shared_ptr<UI::Separator>(new UI::Separator()));
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApDnsSettingMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApPreferredDnsMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApAlternateDnsMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(std::shared_ptr<UI::Separator>(new UI::Separator()));
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApMtuMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(std::shared_ptr<UI::Separator>(new UI::Separator()));
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApProxySettingMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApProxyAddressMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApProxyPortMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(std::shared_ptr<UI::Separator>(new UI::Separator()));
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApRetryCountMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(std::shared_ptr<UI::Separator>(new UI::Separator()));
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApConnectMenuItem);
    m_ApMenuItem->GetSubMenu()->AddItem(m_ApDisconnectMenuItem);
    m_ApMenuItem->GetSubMenu()->activeItem = m_ApMenuItem->GetSubMenu()->items.begin();
    m_Menu->AddItem(m_ApMenuItem);

    m_WpsMenuItem->GetSubMenu()->AddItem(m_WpsPbcMenuItem);
    m_WpsMenuItem->GetSubMenu()->AddItem(m_WpsPinMenuItem);
    m_WpsMenuItem->GetSubMenu()->activeItem = m_WpsMenuItem->GetSubMenu()->items.begin();
    m_Menu->AddItem(m_WpsMenuItem);

    m_EthernetMenuItem->GetSubMenu()->AddItem(m_EthernetConnectMenuItem);
    m_EthernetMenuItem->GetSubMenu()->activeItem = m_EthernetMenuItem->GetSubMenu()->items.begin();
    m_Menu->AddItem(m_EthernetMenuItem);

    m_PingMenuItem->GetSubMenu()->AddItem(m_PingDestinationMenuItem);
    m_PingMenuItem->GetSubMenu()->AddItem(m_PingSizeMenuItem);
    m_PingMenuItem->GetSubMenu()->AddItem(std::shared_ptr<UI::Separator>(new UI::Separator()));
    m_PingMenuItem->GetSubMenu()->AddItem(m_PingRepeatCountMenuItem);
    m_PingMenuItem->GetSubMenu()->AddItem(std::shared_ptr<UI::Separator>(new UI::Separator()));
    m_PingMenuItem->GetSubMenu()->AddItem(m_PingStartMenuItem);
    m_PingMenuItem->GetSubMenu()->AddItem(m_PingStopMenuItem);
    m_PingMenuItem->GetSubMenu()->activeItem = m_PingMenuItem->GetSubMenu()->items.begin();
    m_Menu->AddItem(m_PingMenuItem);

    m_IperfMenuItem->GetSubMenu()->AddItem(m_IperfCommandlineMenuItem);
    m_IperfMenuItem->GetSubMenu()->AddItem(std::shared_ptr<UI::Separator>(new UI::Separator()));
    m_IperfMenuItem->GetSubMenu()->AddItem(m_IperfStartMenuItem);
    m_IperfMenuItem->GetSubMenu()->AddItem(m_IperfStopMenuItem);
    m_IperfMenuItem->GetSubMenu()->activeItem = m_IperfMenuItem->GetSubMenu()->items.begin();
    m_Menu->AddItem(m_IperfMenuItem);

    m_MiscMenuItem->GetSubMenu()->AddItem(m_MiscQuitApplicationMenuItem);
    m_MiscMenuItem->GetSubMenu()->activeItem = m_MiscMenuItem->GetSubMenu()->items.begin();
    m_Menu->AddItem(m_MiscMenuItem);

    m_Menu->activeItem = m_Menu->items.begin();
} // NOLINT(impl/function_size)


void ApConnectivityTestApp::ApIpSettingMenuItem(const std::string& value)
{
    if (m_ApIpSettingMenuItem->GetValue() == "Static")
    {
        if (m_ApDnsSettingMenuItem->GetValue() == "Automatic")
        {
            m_ApDnsSettingMenuItem->GetSubMenu()->MoveFocusPrevious();
            m_ApDnsSettingMenuItem->SetValue((*m_ApDnsSettingMenuItem->GetSubMenu()->activeItem)->GetName());

            ApDnsSettingMenuItem("Automatic");
        }

        m_ApIpAddressMenuItem->SetEnabled(true);
        m_ApSubnetMaskMenuItem->SetEnabled(true);
        m_ApDefaultGatewayMenuItem->SetEnabled(true);
    }
    else
    {
        m_ApIpAddressMenuItem->SetEnabled(false);
        m_ApSubnetMaskMenuItem->SetEnabled(false);
        m_ApDefaultGatewayMenuItem->SetEnabled(false);
    }
}


void ApConnectivityTestApp::ApIpAddressMenuItem()
{
    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_FullKey, m_ApIpAddressMenuItem->GetValue(), 0, 0, [&](const std::string& value) {
        m_ApIpAddressMenuItem->SetValue(value);
    });
}


void ApConnectivityTestApp::ApSubnetMaskMenuItem()
{
    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_FullKey, m_ApSubnetMaskMenuItem->GetValue(), 0, 0, [&](const std::string& value) {
        m_ApSubnetMaskMenuItem->SetValue(value);
    });
}


void ApConnectivityTestApp::ApDefaultGatewayMenuItem()
{
    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_FullKey, m_ApDefaultGatewayMenuItem->GetValue(), 0, 0, [&](const std::string& value) {
        m_ApDefaultGatewayMenuItem->SetValue(value);
    });
}


void ApConnectivityTestApp::ApDnsSettingMenuItem(const std::string& value)
{
    if (m_ApDnsSettingMenuItem->GetValue() == "Manual")
    {
        if (m_ApDnsSettingMenuItem->GetValue() == "Automatic")
        {
            m_ApDnsSettingMenuItem->GetSubMenu()->MoveFocusPrevious();
            m_ApDnsSettingMenuItem->SetValue((*m_ApDnsSettingMenuItem->GetSubMenu()->activeItem)->GetName());
        }

        m_ApPreferredDnsMenuItem->SetEnabled(true);
        m_ApAlternateDnsMenuItem->SetEnabled(true);
    }
    else
    {
        m_ApPreferredDnsMenuItem->SetEnabled(false);
        m_ApAlternateDnsMenuItem->SetEnabled(false);
    }
}


void ApConnectivityTestApp::ApPreferredDnsMenuItem()
{
    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_FullKey, m_ApPreferredDnsMenuItem->GetValue(), 0, 0, [&](const std::string& value) {
        m_ApPreferredDnsMenuItem->SetValue(value);
    });
}


void ApConnectivityTestApp::ApAlternateDnsMenuItem()
{
    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_FullKey, m_ApAlternateDnsMenuItem->GetValue(), 0, 0, [&](const std::string& value) {
        m_ApAlternateDnsMenuItem->SetValue(value);
    });
}


void ApConnectivityTestApp::ApMtuMenuItem()
{
    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_FullKey, m_ApMtuMenuItem->GetValue(), 0, 0, [&](const std::string& value) {
        m_ApMtuMenuItem->SetValue(value);
    });
}


void ApConnectivityTestApp::ApProxySettingMenuItem(const std::string& value)
{
    if (m_ApProxySettingMenuItem->GetValue() == "Enable")
    {
        m_ApProxyAddressMenuItem->SetEnabled(true);
        m_ApProxyPortMenuItem->SetEnabled(true);
    }
    else
    {
        m_ApProxyAddressMenuItem->SetEnabled(false);
        m_ApProxyPortMenuItem->SetEnabled(false);
    }
}


void ApConnectivityTestApp::ApProxyAddressMenuItem()
{
    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_FullKey, m_ApProxyAddressMenuItem->GetValue(), 0, 0, [&](const std::string& value) {
        m_ApProxyAddressMenuItem->SetValue(value);
    });
}


void ApConnectivityTestApp::ApProxyPortMenuItem()
{
    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_FullKey, m_ApProxyPortMenuItem->GetValue(), 0, 0, [&](const std::string& value) {
        m_ApProxyPortMenuItem->SetValue(value);
    });
}


// Connect メニューコールバック
void ApConnectivityTestApp::ConnectMenuCallback()
{
    SetMenuState(MenuState_Connecting);

    m_ConnectTryCount = 0;
    LoadConnectSettings(&m_ConnectNetworkSetting);

    ConnectNetwork();
}


// Disconnect メニューコールバック
void ApConnectivityTestApp::DisconnectMenuCallback()
{
    SetMenuState(MenuState_Disconnect);

    StopPingCallback();
    StopIperfCallback();

    Network::GetInstance().DisconnectNetwork();

    NN_LOG("Disconnected\n");
}


// WPS PBC メニューコールバック
void ApConnectivityTestApp::WpsPbcMenuCallback()
{
    SetMenuState(MenuState_Connecting);

    ConnectNetworkSetting setting;
    LoadConnectSettings(&setting);

    Network::GetInstance().WpsPbc(setting.profile.ipSetting, std::bind(&ApConnectivityTestApp::WpsConnectCallback, this, std::placeholders::_1, std::placeholders::_2));
}


// WPS PIN メニューコールバック
void ApConnectivityTestApp::WpsPinMenuCallback()
{
    SetMenuState(MenuState_Connecting);

    ConnectNetworkSetting setting;
    LoadConnectSettings(&setting);

    Network::GetInstance().WpsPin(setting.profile.ipSetting, std::bind(&ApConnectivityTestApp::WpsConnectCallback, this, std::placeholders::_1, std::placeholders::_2));
}


// 有線接続メニューコールバック
void ApConnectivityTestApp::EthernetConnectMenuCallback()
{
    SetMenuState(MenuState_Connecting);

    m_ConnectTryCount = 0;
    m_ConnectNetworkSetting.profile = {
        nn::util::InvalidUuid,                    // id
        {},                                       // name[NetworkProfileBasicInfo::NameSize] - 64
        nn::nifm::NetworkProfileType_User,        // networkProfileType
        nn::nifm::NetworkInterfaceType_Ethernet,  // networkInterfaceType
        false,                                     // isAutoConnect
        true,                                     // isLargeCapacity
        {},
        { // ipSetting
            { // ip
                true,  // isAuto
                {},    // ipAddress
                {},    // subnetMask
                {}     // defaultGateway
            },
            { // dns
                true,  // isAuto
                {},    // preferredDns
                {}     // alternateDns
            },
            { // proxy
                false,  // isEnabled
                0,      // port
                "",     // proxy[ProxyNameSize] - 100
                { // authentication
                    false,  // isEnabled
                    "",     // username[UsernameSize] - 32
                    ""      // password[PasswordSize] - 32
                }
            },
            1400  // mtu
        }
    };

    m_ConnectNetworkSetting.tryCount = 1;
    ConnectNetwork();
}


// Ping Size メニューコールバック
void ApConnectivityTestApp::PingSizeMenuCallback()
{
    auto& size = m_PingSizeMenuItem->GetValue();

    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_Number, size, 0, 0, [this](const std::string& value) {
        m_PingSizeMenuItem->SetValue(value);
    });
}


// Start Ping メニューコールバック
void ApConnectivityTestApp::StartPingCallback()
{
    SetMenuState(MenuState_PingStart);

    std::string dest;
    int dataSize;
    int repeatCount;
    {
        auto& destinationStr = m_PingDestinationMenuItem->GetValue();
        if (destinationStr == "Default gateway")
        {
            if (Network::GetInstance().IsConnectionNifm())
            {
                nn::socket::InAddr dummy;
                nn::socket::InAddr destIp;
                nn::nifm::GetCurrentIpConfigInfo(&dummy, &dummy, &destIp, &dummy, &dummy);

                dest = nn::socket::InetNtoa(destIp);
            }
            else
            {
                nn::bsdsocket::cfg::IfState state;
                nn::bsdsocket::cfg::GetIfState("wl0", &state);

                dest = nn::socket::InetNtoa(state.gatewayAddr);
            }
        }
        else
        {
            dest = destinationStr;
        }

        auto& sizeStr = m_PingSizeMenuItem->GetValue();
        dataSize = std::stoi(sizeStr);

        auto& repeatCountStr = m_PingRepeatCountMenuItem->GetValue();
        if (repeatCountStr == "Infinity")
        {
            repeatCount = -1;
        }
        else
        {
            repeatCount = std::stoi(repeatCountStr);
        }
    }

    m_PingSuccessCount = 0;
    m_PingResponseCount = 0;

    m_PingController = Network::GetInstance().Ping(dest, dataSize, repeatCount,
            std::bind(&ApConnectivityTestApp::PingCallback, this));
}


// Stop Ping メニューコールバック
void ApConnectivityTestApp::StopPingCallback()
{
    SetMenuState(MenuState_PingStop);

    m_PingController.Cancel();
}


// iperf 引数メニューコールバック
void ApConnectivityTestApp::IperfCommandlineMenuCallback(size_t index)
{
    GetUi().OpenKeyboard(UI::Keyboard::KeyMap_FullKey, m_IperfCommandlineMenuItem->GetValue(), 0, 0, [&](const std::string& value) {
        m_IperfCommandlineMenuItem->SetValue(value);
    });
}


// iperf 開始メニューコールバック
void ApConnectivityTestApp::StartIperfCallback()
{
    SetMenuState(MenuState_IperfStart);

    Network::GetInstance().StartIperf(m_IperfCommandlineMenuItem->GetValue(), std::bind(&ApConnectivityTestApp::IperfExitCallback, this));
}


// iperf 停止メニューコールバック
void ApConnectivityTestApp::StopIperfCallback()
{
    Network::GetInstance().StopIperf();
}


// アプリ終了メニューコールバック
void ApConnectivityTestApp::QuitApplicationMenuCallback()
{
    Exit();
}


// 認証方式メニューコールバック
void ApConnectivityTestApp::AuthenticationChangeCallback(const std::string& value)
{
}


// 暗号方式メニューコールバック
void ApConnectivityTestApp::EncryptionChangeCallback(const std::string& value)
{
    // 鍵の文字数制限を設定する
    if (value == "None")
    {
        m_ApSharedKeyMenuItem->SetMinValueLength(0);
        m_ApSharedKeyMenuItem->SetMaxValueLength(0);
    }
    else if (value == "WEP")
    {
        m_ApSharedKeyMenuItem->SetMinValueLength(5);
        m_ApSharedKeyMenuItem->SetMaxValueLength(26);
    }
    else if (value == "AES")
    {
        m_ApSharedKeyMenuItem->SetMinValueLength(8);
        m_ApSharedKeyMenuItem->SetMaxValueLength(64);
    }
}


// ネットワーク接続する
void ApConnectivityTestApp::ConnectNetwork()
{
    NN_LOG("\nTry to connecting\n");

    ++m_ConnectTryCount;

    // 接続
    Network::GetInstance().ConnectNetwork(m_ConnectNetworkSetting.profile, m_ConnectNetworkSetting.mixedMode, m_NifmAdminMode,
            std::bind(&ApConnectivityTestApp::ConnectNetworkCallback, this, std::placeholders::_1, std::placeholders::_2));
}


// ネットワーク接続結果のコールバック
void ApConnectivityTestApp::ConnectNetworkCallback(bool result, const nn::TimeSpan& elapsedTime)
{
    // 接続失敗
    if (!result)
    {
        NN_LOG("Connection failed (%d)\n", m_ConnectTryCount);

        if (m_ConnectTryCount < m_ConnectNetworkSetting.tryCount)
        {
            NN_LOG("\n");
            ConnectNetwork();
        }
        else
        {
            NN_LOG("Abort connection (failed %d times)\n\n", m_ConnectTryCount);

            SetMenuState(MenuState_Disconnect);
        }
    }
    // 接続成功
    else
    {
        SetMenuState(MenuState_Connect);

        NN_LOG("Connected [%d ms]\n\n", elapsedTime.GetMilliSeconds());

        if (m_ConnectNetworkSetting.profile.networkInterfaceType == nn::nifm::NetworkInterfaceType_Ieee80211)
        {
            // 接続先 AP の情報を表示
            nn::nifm::AccessPointData ap;
            nn::nifm::GetCurrentAccessPoint(&ap);
            NN_LOG("SSID             : %s\n", std::string(reinterpret_cast<char*>(ap.ssid.hex), ap.ssid.length).c_str());
            NN_LOG("BSSID            : %02X:%02X:%02X:%02X:%02X:%02X\n", ap.bssid.data[0], ap.bssid.data[1], ap.bssid.data[2], ap.bssid.data[3], ap.bssid.data[4], ap.bssid.data[5]);
            NN_LOG("LinkLevel        : %d (%s)\n", ap.linkLevel, g_LevelNameMap.at(ap.linkLevel).c_str());
            NN_LOG("Channel          : %d\n", ap.channel);
            NN_LOG("Auth             : %s\n", g_AuthNameMap.at(ap.authentication).c_str());
            NN_LOG("Encryption       : %s\n", g_EncryptNameMap.at(ap.encryption).c_str());
            NN_LOG("Group Encryption : %s\n", g_EncryptNameMap.at(ap.groupEncryption).c_str());

            NN_LOG("\n");
        }

        if (m_ConnectNetworkSetting.profile.ipSetting.ip.isAuto) {
            // IP 設定を表示
            nn::socket::InAddr ipAddress;
            nn::socket::InAddr subnetMask;
            nn::socket::InAddr defaultGateway;
            nn::socket::InAddr preferredDns;
            nn::socket::InAddr alternateDns;
            nn::nifm::GetCurrentIpConfigInfo(&ipAddress, &subnetMask, &defaultGateway, &preferredDns, &alternateDns);

            NN_LOG("IP              : %d.%d.%d.%d\n",
                   ipAddress.S_addr & 0xff,
                   (ipAddress.S_addr >> 8) & 0xff,
                   (ipAddress.S_addr >> 16) & 0xff,
                   (ipAddress.S_addr >> 24) & 0xff
            );

            NN_LOG("Subnet Mask     : %d.%d.%d.%d\n",
                   subnetMask.S_addr & 0xff,
                   (subnetMask.S_addr >> 8) & 0xff,
                   (subnetMask.S_addr >> 16) & 0xff,
                   (subnetMask.S_addr >> 24) & 0xff
            );

            NN_LOG("Default Gateway : %d.%d.%d.%d\n",
                   defaultGateway.S_addr & 0xff,
                   (defaultGateway.S_addr >> 8) & 0xff,
                   (defaultGateway.S_addr >> 16) & 0xff,
                   (defaultGateway.S_addr >> 24) & 0xff
            );

            NN_LOG("Primary DNS     : %d.%d.%d.%d\n",
                   preferredDns.S_addr & 0xff,
                   (preferredDns.S_addr >> 8) & 0xff,
                   (preferredDns.S_addr >> 16) & 0xff,
                   (preferredDns.S_addr >> 24) & 0xff
            );

            NN_LOG("Secondary DNS   : %d.%d.%d.%d\n",
                   alternateDns.S_addr & 0xff,
                   (alternateDns.S_addr >> 8) & 0xff,
                   (alternateDns.S_addr >> 16) & 0xff,
                   (alternateDns.S_addr >> 24) & 0xff
            );
        } else {
            std::string ipaddr = m_ApIpAddressMenuItem->GetValue();
            std::string subnetmask = m_ApSubnetMaskMenuItem->GetValue();
            std::string gateway = m_ApDefaultGatewayMenuItem->GetValue();
            std::string preferred_dns = m_ApPreferredDnsMenuItem->GetValue();
            std::string alternate_dns = m_ApAlternateDnsMenuItem->GetValue();
            NN_LOG("IP              : %s\n", ipaddr.c_str());
            NN_LOG("Subnet Mask     : %s\n", subnetmask.c_str());
            NN_LOG("Default Gateway : %s\n", gateway.c_str());
            NN_LOG("Primary DNS     : %s\n", preferred_dns.c_str());
            NN_LOG("Secondary DNS   : %s\n", alternate_dns.c_str());
        }

        NN_LOG("\n");
    }
}


// WPS 接続結果コールバック
void ApConnectivityTestApp::WpsConnectCallback(bool result, const nn::TimeSpan& elapsedTime)
{
    NN_LOG("WPS %s [%dms]\n", result ? "succeed" : "failed", elapsedTime.GetMilliSeconds());

    if (result)
    {
        SetMenuState(MenuState_Connect);
    }
    else
    {
        SetMenuState(MenuState_Disconnect);
    }
}


// ping 終了コールバック
void ApConnectivityTestApp::PingCallback()
{
    SetMenuState(MenuState_PingStop);
}


// iperf 終了コールバック
void ApConnectivityTestApp::IperfExitCallback()
{
    SetMenuState(MenuState_IperfStop);
}


// 接続設定をメニューから読み取る
void ApConnectivityTestApp::LoadConnectSettings(ConnectNetworkSetting* pOut)
{
    std::string ssid = m_ApSsidMenuItem->GetValue();
    std::string auth = m_ApAuthenticationMenuItem->GetValue();
    std::string encrypt = m_ApEncryptionMenuItem->GetValue();
    std::string sharedkey = m_ApSharedKeyMenuItem->GetValue();
    std::string ip_option = m_ApIpSettingMenuItem->GetValue();
    std::string ipaddr = m_ApIpAddressMenuItem->GetValue();
    std::string subnetmask = m_ApSubnetMaskMenuItem->GetValue();
    std::string gateway = m_ApDefaultGatewayMenuItem->GetValue();
    std::string dns_option = m_ApDnsSettingMenuItem->GetValue();
    std::string preferred_dns = m_ApPreferredDnsMenuItem->GetValue();
    std::string alternate_dns = m_ApAlternateDnsMenuItem->GetValue();
    std::string mtu = m_ApMtuMenuItem->GetValue();
    std::string proxy_option = m_ApProxySettingMenuItem->GetValue();
    std::string proxy_addr = m_ApProxyAddressMenuItem->GetValue();
    std::string proxy_port = m_ApProxyPortMenuItem->GetValue();
    int retryCount = std::stoi(m_ApRetryCountMenuItem->GetValue());

    // ネットワークプロファイルを作成
    pOut->profile = {
        nn::util::InvalidUuid,                     // id
        {},                                        // name[NetworkProfileBasicInfo::NameSize] - 64
        nn::nifm::NetworkProfileType_User,         // networkProfileType
        nn::nifm::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        false,                                      // isAutoConnect
        true,                                      // isLargeCapacity
        { // union
            //{// wirelessSetting
            //    { // ssidConfig
            //        { // ssid
            //            ,  // length
            //               // hex
            //        },
            //        true  // nonBroadcast
            //    },
            //    { // security
            //    }
            //}
        },
        { // ipSetting
            { // ip
                true,  // isAuto
                {},    // ipAddress
                {},    // subnetMask
                {}     // defaultGateway
            },
            { // dns
                true,  // isAuto
                {},    // preferredDns
                {}     // alternateDns
            },
            { // proxy
                false,  // isEnabled
                0,      // port
                "",     // proxy[ProxyNameSize] - 100
                { // authentication
                    false,  // isEnabled
                    "",     // username[UsernameSize] - 32
                    ""      // password[PasswordSize] - 32
                }
            },
            1400  // mtu
        }
    };

    // SSID 設定
    pOut->profile.wirelessSetting.ssidConfig.ssid.length = static_cast<uint8_t>(ssid.size());
    ssid.copy(reinterpret_cast<char*>(pOut->profile.wirelessSetting.ssidConfig.ssid.hex), nn::nifm::Ssid::HexSize);
    pOut->profile.wirelessSetting.ssidConfig.nonBroadcast = true;

    // セキュリティ設定
    pOut->mixedMode = auth == "WPA/WPA2 mixed";
    pOut->profile.wirelessSetting.security.authEncryption.authentication = g_AuthEnumMap.at(auth);
    pOut->profile.wirelessSetting.security.authEncryption.encryption = g_EncryptEnumMap.at(encrypt);

    // 共通鍵設定
    pOut->profile.wirelessSetting.security.sharedKey.length = static_cast<uint8_t>(sharedkey.size());
    sharedkey.copy(reinterpret_cast<char*>(pOut->profile.wirelessSetting.security.sharedKey.keyMaterial), nn::nifm::SharedKey::KeyMaterialSize);

    // IP 設定
    in_addr inaddr;
    pOut->profile.ipSetting.ip.isAuto = ip_option == "DHCP";
    nn::socket::InetAton(ipaddr.c_str(), &inaddr);
    reinterpret_cast<in_addr&>(pOut->profile.ipSetting.ip.ipAddress) = inaddr;
    nn::socket::InetAton(subnetmask.c_str(), &inaddr);
    reinterpret_cast<in_addr&>(pOut->profile.ipSetting.ip.subnetMask) = inaddr;
    nn::socket::InetAton(gateway.c_str(), &inaddr);
    reinterpret_cast<in_addr&>(pOut->profile.ipSetting.ip.defaultGateway) = inaddr;

    // DNS 設定
    pOut->profile.ipSetting.dns.isAuto = dns_option == "Automatic";
    nn::socket::InetAton(preferred_dns.c_str(), &inaddr);
    reinterpret_cast<in_addr&>(pOut->profile.ipSetting.dns.preferredDns) = inaddr;
    nn::socket::InetAton(alternate_dns.c_str(), &inaddr);
    reinterpret_cast<in_addr&>(pOut->profile.ipSetting.dns.alternateDns) = inaddr;

    // プロキシ設定
    pOut->profile.ipSetting.proxy.isEnabled = proxy_option == "Enable";
    proxy_addr.copy(pOut->profile.ipSetting.proxy.proxy, nn::nifm::ProxySetting::ProxyNameSize - 1);
    pOut->profile.ipSetting.proxy.proxy[std::min(proxy_addr.size(), nn::nifm::ProxySetting::ProxyNameSize - 1)] = '\0';
    pOut->profile.ipSetting.proxy.port = std::stoi(proxy_port);

    // その他の設定
    pOut->tryCount = retryCount + 1;
}


// メニューの状態を更新する
void ApConnectivityTestApp::SetMenuState(MenuState state)
{
    switch (state)
    {
    case MenuState_Disconnect:
        m_ApSsidMenuItem->SetEnabled(true);
        m_ApAuthenticationMenuItem->SetEnabled(true);
        m_ApEncryptionMenuItem->SetEnabled(true);
        m_ApSharedKeyMenuItem->SetEnabled(true);
        m_ApIpSettingMenuItem->SetEnabled(true);
        m_ApDnsSettingMenuItem->SetEnabled(true);
        m_ApMtuMenuItem->SetEnabled(true);
        m_ApProxySettingMenuItem->SetEnabled(true);
        m_ApRetryCountMenuItem->SetEnabled(true);
        m_ApConnectMenuItem->SetEnabled(true);
        m_ApDisconnectMenuItem->SetEnabled(false);

        ApIpSettingMenuItem(m_ApIpSettingMenuItem->GetValue());
        ApDnsSettingMenuItem(m_ApDnsSettingMenuItem->GetValue());
        ApProxySettingMenuItem(m_ApProxySettingMenuItem->GetValue());

        m_WpsMenuItem->SetEnabled(true);

        m_EthernetMenuItem->SetEnabled(true);

        m_PingMenuItem->SetEnabled(false);
        m_IperfMenuItem->SetEnabled(false);
        break;

    case MenuState_Connecting:
        m_ApSsidMenuItem->SetEnabled(false);
        m_ApAuthenticationMenuItem->SetEnabled(false);
        m_ApEncryptionMenuItem->SetEnabled(false);
        m_ApSharedKeyMenuItem->SetEnabled(false);
        m_ApIpSettingMenuItem->SetEnabled(false);
        m_ApIpAddressMenuItem->SetEnabled(false);
        m_ApSubnetMaskMenuItem->SetEnabled(false);
        m_ApDefaultGatewayMenuItem->SetEnabled(false);
        m_ApDnsSettingMenuItem->SetEnabled(false);
        m_ApPreferredDnsMenuItem->SetEnabled(false);
        m_ApAlternateDnsMenuItem->SetEnabled(false);
        m_ApMtuMenuItem->SetEnabled(false);
        m_ApProxySettingMenuItem->SetEnabled(false);
        m_ApProxyAddressMenuItem->SetEnabled(false);
        m_ApProxyPortMenuItem->SetEnabled(false);
        m_ApRetryCountMenuItem->SetEnabled(false);
        m_ApConnectMenuItem->SetEnabled(false);
        m_ApDisconnectMenuItem->SetEnabled(false);

        m_WpsMenuItem->SetEnabled(false);

        m_EthernetMenuItem->SetEnabled(false);

        m_PingMenuItem->SetEnabled(false);
        m_IperfMenuItem->SetEnabled(false);
        break;

    case MenuState_Connect:
        m_ApSsidMenuItem->SetEnabled(false);
        m_ApAuthenticationMenuItem->SetEnabled(false);
        m_ApEncryptionMenuItem->SetEnabled(false);
        m_ApSharedKeyMenuItem->SetEnabled(false);
        m_ApIpSettingMenuItem->SetEnabled(false);
        m_ApIpAddressMenuItem->SetEnabled(false);
        m_ApSubnetMaskMenuItem->SetEnabled(false);
        m_ApDefaultGatewayMenuItem->SetEnabled(false);
        m_ApDnsSettingMenuItem->SetEnabled(false);
        m_ApPreferredDnsMenuItem->SetEnabled(false);
        m_ApAlternateDnsMenuItem->SetEnabled(false);
        m_ApMtuMenuItem->SetEnabled(false);
        m_ApProxySettingMenuItem->SetEnabled(false);
        m_ApProxyAddressMenuItem->SetEnabled(false);
        m_ApProxyPortMenuItem->SetEnabled(false);
        m_ApRetryCountMenuItem->SetEnabled(false);
        m_ApConnectMenuItem->SetEnabled(false);
        m_ApDisconnectMenuItem->SetEnabled(true);

        m_WpsMenuItem->SetEnabled(false);

        m_EthernetMenuItem->SetEnabled(false);

        m_PingMenuItem->SetEnabled(true);
        m_IperfMenuItem->SetEnabled(true);

        m_PingStartMenuItem->SetEnabled(true);
        m_PingStopMenuItem->SetEnabled(false);
        break;

    case MenuState_PingStart:
        m_PingStartMenuItem->SetEnabled(false);
        m_PingStopMenuItem->SetEnabled(true);
        break;

    case MenuState_PingStop:
        m_PingStartMenuItem->SetEnabled(true);
        m_PingStopMenuItem->SetEnabled(false);
        break;

    case MenuState_IperfStart:
        m_IperfStartMenuItem->SetEnabled(false);
        m_IperfStopMenuItem->SetEnabled(true);
        break;

    case MenuState_IperfStop:
        m_IperfStartMenuItem->SetEnabled(true);
        m_IperfStopMenuItem->SetEnabled(false);
        break;

    default:
        NN_ABORT("Unknown MenuState");
        break;
    }
}

}


// エントリポイント
extern "C" void nnMain()
{
    using namespace ApConnectivityTest;

    ApConnectivityTestApp().Run();
}
