﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_ConditionVariable.h>
#include <nn/os/os_Mutex.h>
#include <memory>
#include <vector>


template <typename Type>
class BlockingQueue
{
public:
    NN_IMPLICIT BlockingQueue(size_t max_size);
    virtual ~BlockingQueue();

    void Push(const Type& obj);
    Type Pop();

    void Clear();

private:
    std::vector<uint8_t> m_Buffer;
    size_t m_MaxCount;
    size_t m_ReadIndex;
    size_t m_UsingCount;

    std::unique_ptr<nn::os::Mutex> m_Mutex;
    std::unique_ptr<nn::os::ConditionVariable> m_ConditionReadable;
    std::unique_ptr<nn::os::ConditionVariable> m_ConditionWritable;
};


template <typename Type>
BlockingQueue<Type>::BlockingQueue(size_t maxCount) :
        m_Buffer(sizeof(Type) * maxCount),
        m_MaxCount(maxCount),
        m_ReadIndex(0),
        m_UsingCount(0),
        m_Mutex(new nn::os::Mutex(false, 0)),
        m_ConditionReadable(new nn::os::ConditionVariable()),
        m_ConditionWritable(new nn::os::ConditionVariable())
{
}


template <typename Type>
BlockingQueue<Type>::~BlockingQueue()
{
}


template <typename Type>
void BlockingQueue<Type>::Push(const Type& obj)
{
    m_Mutex->Lock();
    while (m_UsingCount >= m_MaxCount)
    {
        m_ConditionWritable->Wait(*m_Mutex);
    }

    //*reinterpret_cast<Type*>(m_Buffer.data() + (m_ReadIndex + m_UsingCount++) % m_MaxCount * sizeof(Type)) = obj;
    new (m_Buffer.data() + (m_ReadIndex + m_UsingCount++) % m_MaxCount * sizeof(Type))Type(obj);

    m_ConditionReadable->Signal();
    m_Mutex->Unlock();
}


template <typename Type>
Type BlockingQueue<Type>::Pop()
{
    m_Mutex->Lock();
    while (!m_UsingCount)
    {
        m_ConditionReadable->Wait(*m_Mutex);
    }

    auto pObj = reinterpret_cast<Type*>(m_Buffer.data() + m_ReadIndex * sizeof(Type));
    Type obj = *pObj;
    pObj->~Type();

    m_ReadIndex = (m_ReadIndex + 1) % m_MaxCount;
    --m_UsingCount;

    m_ConditionWritable->Signal();
    m_Mutex->Unlock();

    return obj;
}


template <typename Type>
void BlockingQueue<Type>::Clear()
{
    m_Mutex->Lock();

    while (m_UsingCount)
    {
        auto pObj = reinterpret_cast<Type*>(m_Buffer.data() + m_ReadIndex * sizeof(Type));
        pObj->~Type();

        m_ReadIndex = (m_ReadIndex + 1) % m_MaxCount;
        --m_UsingCount;
    }

    m_ConditionWritable->Signal();
    m_Mutex->Unlock();
}
