﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "WebDemo_GraphicsSystem.hpp"

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/fs.h>

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

namespace {
// メモリのサイズ
const size_t MemorySize = 16 * 1024 * 1024;
// メモリプールサイズ
const size_t MemoryPoolMemorySize = 16 * 1024 * 1024;
// デスクリプタスロットの数
const size_t TextureDescriptorPoolSlotCount = 256;
// ディスプレイの幅と高さ
const int DisplayWidth = 1280;
const int DisplayHeight = 720;
const float DisplayWidthHalf = float(DisplayWidth) / 2.0f;
const float DisplayHeightHalf = float(DisplayHeight) / 2.0f;
// スワップチェーンのフォーマット
const nn::gfx::ImageFormat SwapChainFormat
    = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
// コマンドバッファのメモリサイズ
const size_t CommandBufferControlMemorySize = 256;
const size_t CommandBufferMemoryPoolSize = 1024 * 1024;

// 一度に表示できる最大文字数
const int CharCountMax = 2048;

// シェーダファイルへのパス
const char* ShaderFilePath = "Contents:/SampleShader.bnsh";
// テクスチャファイルへのパス
const char* TextureFilePath = "Contents:/GameBg.bntx";

// プリミティブトポロジの種類
const nn::gfx::PrimitiveTopologyType PrimitiveTopologyType =
    nn::gfx::PrimitiveTopologyType_Triangle;

// Position, Uv, Indexのデータ
// Positionの設定
struct Positions
{
    nn::util::Float3 leftTop;
    nn::util::Float3 leftBottom;
    nn::util::Float3 rightBottom;
    nn::util::Float3 rightTop;
};
const float DefaultPositions[] =
{
    -1.0f,  1.0f, 0.0f,
    -1.0f, -1.0f, 0.0f,
     1.0f, -1.0f, 0.0f,
     1.0f,  1.0f, 0.0f,
};
NN_STATIC_ASSERT(sizeof(DefaultPositions) == sizeof(Positions));

const int PositionBufferIndex = 0;
const nn::gfx::AttributeFormat PositionFormat = nn::gfx::AttributeFormat_32_32_32_Float;
const char* PositionSymbolName = "i_Position";
const size_t PositionStride = sizeof(float) * 3;
// フォーマットが変わったら Stride も変更すること
NN_STATIC_ASSERT(PositionFormat == nn::gfx::AttributeFormat_32_32_32_Float);

// Uvの設定(左上原点としています。)
const float Uvs[] =
{
    0.0f, 0.0f,
    0.0f, 1.0f,
    1.0f, 1.0f,
    1.0f, 0.0f,
};
const int UvBufferIndex = 1;
const nn::gfx::AttributeFormat UvFormat = nn::gfx::AttributeFormat_32_32_Float;
const char* UvSymbolName = "i_TexCoord";
const size_t UvStride = sizeof(float) * 2;
// フォーマットが変わったら Stride も変更すること
NN_STATIC_ASSERT(UvFormat == nn::gfx::AttributeFormat_32_32_Float);

// Index
const nn::gfx::PrimitiveTopology PrimitiveTopology = nn::gfx::PrimitiveTopology_TriangleList;
const nn::gfx::IndexFormat IndexFormat = nn::gfx::IndexFormat_Uint16;
const uint16_t Indices[] =
{
    0, 1, 2, 2, 3, 0
};
const int IndexCount = static_cast<int>(sizeof(Indices) / sizeof(uint16_t));
// フォーマットが変わったら Index 数も変更すること
NN_STATIC_ASSERT(IndexFormat == nn::gfx::IndexFormat_Uint16);

// テクスチャリストのインデックスの取得
int GetTextureListIndex(GraphicsSystem::BackgroundKind kind) NN_NOEXCEPT
{
    // BackgroundDrawer::BackgroundKind_None はテクスチャなしなのでここには来ない
    NN_ASSERT_RANGE(
        kind,
        GraphicsSystem::BackgroundKind_TextureMin,
        GraphicsSystem::BackgroundKind_TextureMax + 1
        );
    return kind - 1;
}

struct Constant
{
    nn::util::Float4 color;
};

float ChangeCoordinateX(float value) NN_NOEXCEPT
{
    return (value - DisplayWidthHalf) / DisplayWidthHalf;
}

float ChangeCoordinateY(float value) NN_NOEXCEPT
{
    return -(value - DisplayHeightHalf) / DisplayHeightHalf;
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}
static void NvFreeFunction(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif
}

GraphicsSystem::GraphicsSystem() NN_NOEXCEPT
: m_BackgroundColor()
, m_BackgroundKind(BackgroundKind_None)
, m_pMemoryHeap(nullptr)
, m_pMemory(nullptr)
, m_Device()
, m_pMemoryPoolMemory(nullptr)
, m_pMemoryPoolStart(nullptr)
, m_MemoryPoolOffset(0)
, m_MemoryPool()
, m_pCompressedMemoryPoolMemory(nullptr)
, m_pCompressedMemoryPoolStart(nullptr)
, m_CompressedMemoryPoolOffset(0)
, m_CompressedMemoryPool()
, m_pDisplay(nullptr)
, m_pLayer(nullptr)
, m_SwapChain()
, m_Queue()
, m_CommandBuffer()
, m_pCommandBufferControlMemory(nullptr)
, m_CommandBufferMemoryPoolOffset(0)
, m_ViewportScissor()
, m_pShader()
, m_BlendState()
, m_DepthStencilState()
, m_RasterizerState()
, m_VertexState()
, m_InterfaceSlot()
, m_PositionBuffer()
, m_UvBuffer()
, m_IndexBuffer()
, m_SamplerDescriptorPool()
, m_SamplerDescriptorBaseIndex(0)
, m_SamplerDescriptorSlot()
, m_TextureDescriptorPool()
, m_TextureDescriptorBaseIndex(0)
, m_DisplayFence()
, m_DisplaySemaphore()
, m_GpuDoneFence()
, m_pResShaderFile(nullptr)
, m_pResTextureFile(nullptr)
, m_Sampler()
, m_DebugFontTextWriter()
, m_RectangleColor()
, m_ConstantBuffer()
, m_IsRectangleVisible(true)
, m_NextScanBufferIndex(0)
, m_FrameCount(0)
{
    for( int idx = 0; idx < ScanBufferCount; ++idx )
    {
        m_pScanBufferViews[idx] = nullptr;
    }
    for( int idx = 0; idx < BackgroundKind_TextureCount; ++idx )
    {
        m_TextureDescriptorSlotList[idx].Invalidate();
    }

    // 背景色は白にしておく
    SetBackgroundColor(255, 255, 255);
}

GraphicsSystem::~GraphicsSystem() NN_NOEXCEPT
{
}

void GraphicsSystem::Initialize() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(
            NvAllocateFunction, NvFreeFunction, NvReallocateFunction, nullptr);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

    // メモリの準備
    m_pMemoryHeap.Reset(malloc(MemorySize));
    m_pMemory = m_pMemoryHeap;

    // レイヤの初期化
    nn::vi::Initialize();
    InitializeLayer();

    // gfxライブラリを初期化
    nn::gfx::Initialize();

    InitializeDevice();

    // Nvn ではデスクリプタスロットのシステムによる予約分がある
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(m_Device);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &m_TextureDescriptorBaseIndex);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &m_SamplerDescriptorBaseIndex);

    InitializeMemoryPool();
    InitializeCompressedMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeShader();
    InitializeRasterizerState();
    InitializeBlendState();
    InitializeDepthStencilState();
    InitializeVertexState();

    InitializePositionBuffer();
    InitializeUvBuffer();
    InitializeIndexBuffer();

    InitializeConstantBuffer();

    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    InitializeFence();
    InitializeSemaphore();

    InitializeSampler();
    InitializeTexture();

    InitializeDebugFontTextWriter();

    NN_ASSERT(m_pMemoryHeap.Distance(m_pMemory.Get()) < MemorySize);
    NN_ASSERT(m_MemoryPoolOffset < MemoryPoolMemorySize);
    NN_ASSERT(m_CompressedMemoryPoolOffset < MemoryPoolMemorySize);
}

void GraphicsSystem::Finalize() NN_NOEXCEPT
{
    m_Queue.Sync();

    m_DebugFontTextWriter.Finalize();

    // 各オブジェクトを破棄
    const int count = m_pResTextureFile->GetTextureDic()->GetCount();
    for( int idx = 0; idx < count; ++idx )
    {
        nn::gfx::ResTexture* pResTexture = m_pResTextureFile->GetResTexture(idx);
        NN_ASSERT_NOT_NULL(pResTexture);
        pResTexture->Finalize(&m_Device);
    }
    m_pResTextureFile->Finalize(&m_Device);

    m_Sampler.Finalize(&m_Device);

    m_DisplaySemaphore.Finalize(&m_Device);
    m_DisplayFence.Finalize(&m_Device);
    m_GpuDoneFence.Finalize(&m_Device);

    m_TextureDescriptorPool.Finalize(&m_Device);
    m_SamplerDescriptorPool.Finalize(&m_Device);

    m_ConstantBuffer.Finalize(&m_Device);
    m_IndexBuffer.Finalize(&m_Device);
    m_UvBuffer.Finalize(&m_Device);
    for( int idx = 0; idx < DrawType_End; ++idx )
    {
        m_PositionBuffer[idx].Finalize(&m_Device);
    }

    for( int idx = 0; idx < DrawType_End; ++idx )
    {
        m_VertexState[idx].Finalize(&m_Device);
    }
    m_DepthStencilState.Finalize(&m_Device);
    m_BlendState.Finalize(&m_Device);
    m_RasterizerState.Finalize(&m_Device);

    FinalizeResShaderFile();

    m_ViewportScissor.Finalize(&m_Device);
    m_CommandBuffer.Finalize(&m_Device);
    m_SwapChain.Finalize(&m_Device);
    m_Queue.Finalize(&m_Device);
    m_CompressedMemoryPool.Finalize(&m_Device);
    m_MemoryPool.Finalize(&m_Device);
    m_Device.Finalize();

    // ライブラリを終了
    nn::gfx::Finalize();

    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);
    nn::vi::Finalize();

    free(m_pMemoryHeap.Get());
    free(m_pMemoryPoolMemory);
    free(m_pCompressedMemoryPoolMemory);
}

void GraphicsSystem::InitializeLayer() NN_NOEXCEPT
{
    nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_UNUSED(result);

    result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void GraphicsSystem::InitializeDevice() NN_NOEXCEPT
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
    m_Device.Initialize(info);
}

void GraphicsSystem::InitializeMemoryPool() NN_NOEXCEPT
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached
        | nn::gfx::MemoryPoolProperty_GpuCached);

    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);
    m_pMemoryPoolMemory = malloc(MemoryPoolMemorySize + alignment);

    m_pMemoryPoolStart = nn::util::BytePtr(m_pMemoryPoolMemory).AlignUp(alignment).Get();
    info.SetPoolMemory(m_pMemoryPoolStart, nn::util::align_down(MemoryPoolMemorySize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info)));
    m_MemoryPool.Initialize(&m_Device, info);

    m_MemoryPoolOffset = 0;
}

void GraphicsSystem::InitializeCompressedMemoryPool() NN_NOEXCEPT
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuCached |
        nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);

    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);
    m_pCompressedMemoryPoolMemory = malloc(MemoryPoolMemorySize + alignment);

    m_pCompressedMemoryPoolStart =
        nn::util::BytePtr(m_pCompressedMemoryPoolMemory).AlignUp(alignment).Get();
    info.SetPoolMemory(m_pCompressedMemoryPoolStart, nn::util::align_down(MemoryPoolMemorySize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info)));
    m_CompressedMemoryPool.Initialize(&m_Device, info);

    m_CompressedMemoryPoolOffset = 0;
}

void GraphicsSystem::InitializeSwapChain() NN_NOEXCEPT
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer(m_pLayer);
    info.SetWidth(DisplayWidth);
    info.SetHeight(DisplayHeight);
    info.SetFormat(SwapChainFormat);
    info.SetBufferCount(2);
    if( NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired) )
    {
        size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, info);
        m_CompressedMemoryPoolOffset = nn::util::align_up(m_CompressedMemoryPoolOffset,
            nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, info));
        m_SwapChain.Initialize(
            &m_Device, info, &m_CompressedMemoryPool, m_CompressedMemoryPoolOffset, size);
        m_CompressedMemoryPoolOffset += size;
    }
    else
    {
        m_SwapChain.Initialize(&m_Device, info, nullptr, 0, 0);
    }
    NN_ASSERT(nn::gfx::IsInitialized(m_SwapChain));

    m_SwapChain.GetScanBufferViews(m_pScanBufferViews, ScanBufferCount);
}

void GraphicsSystem::InitializeQueue() NN_NOEXCEPT
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);
    m_Queue.Initialize(&m_Device, info);
    NN_ASSERT(nn::gfx::IsInitialized(m_Queue));
}

void GraphicsSystem::InitializeCommandBuffer() NN_NOEXCEPT
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
    m_CommandBuffer.Initialize(&m_Device, info);

    // コントロールメモリを確
    const size_t alignment = 256;
    m_pMemory.AlignUp(alignment);
    m_pCommandBufferControlMemory = m_pMemory.Get();
    m_pMemory.Advance(CommandBufferControlMemorySize);

    // メモリプール領域確保
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));
    m_CommandBufferMemoryPoolOffset = m_MemoryPoolOffset;
    m_MemoryPoolOffset += CommandBufferMemoryPoolSize;
}

void GraphicsSystem::InitializeViewport() NN_NOEXCEPT
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth(static_cast< float >(DisplayWidth));
        viewportInfo.SetHeight(static_cast< float >(DisplayHeight));
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth(DisplayWidth);
        scissorInfo.SetHeight(DisplayHeight);
    }
    info.SetViewportStateInfoArray(&viewportInfo, 1);
    info.SetScissorStateInfoArray(&scissorInfo, 1);
    m_ViewportScissor.Initialize(&m_Device, info);
    NN_ASSERT(nn::gfx::IsInitialized(m_ViewportScissor));
}

void GraphicsSystem::InitializeShader() NN_NOEXCEPT
{
    void* pResource = ReadResource(ShaderFilePath);
    m_pResShaderFile = nn::gfx::ResShaderFile::ResCast(pResource);
    NN_ASSERT_NOT_NULL(m_pResShaderFile);

    nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL(pContainer);
    pContainer->Initialize(&m_Device);

    const int variationCount = pContainer->GetShaderVariationCount();
    // バリエーションの種類と描画の種類が一致しているかをチェックしています。
    NN_ASSERT(variationCount == DrawType_End);
    for( int idx = 0; idx < variationCount; ++idx )
    {
        const nn::gfx::ShaderInitializeResult result =
            pContainer->GetResShaderProgram(idx)->Initialize(&m_Device);
        NN_ASSERT(result == nn::gfx::ShaderInitializeResult_Success);

        // インターフェーススロットの取得
        nn::gfx::Shader* pShader = pContainer->GetResShaderProgram(idx)->GetShader();
        NN_ASSERT_NOT_NULL(pShader);
        static const char* s_TexName[] = { "s_Tex", "u_Constant" };
        if( idx == 0 )
        {
            m_InterfaceSlot[idx] = pShader->GetInterfaceSlot(
                nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, s_TexName[idx]);
        }
        else
        {
            m_InterfaceSlot[idx] = pShader->GetInterfaceSlot(
                nn::gfx::ShaderStage_Pixel,
                nn::gfx::ShaderInterfaceType_ConstantBuffer, s_TexName[idx]);
        }
        m_pShader[idx] = pShader;
    }
}

void GraphicsSystem::FinalizeResShaderFile() NN_NOEXCEPT
{
    auto pContainer = m_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL(pContainer);

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram(idxVariation);
        pProgram->Finalize(&m_Device);
    }

    pContainer->Finalize(&m_Device);
}

void GraphicsSystem::InitializeRasterizerState() NN_NOEXCEPT
{
    nn::gfx::RasterizerStateInfo info;
    info.SetDefault();
    info.SetPrimitiveTopologyType(PrimitiveTopologyType);
    info.SetScissorEnabled(true);
    info.SetFillMode(nn::gfx::FillMode_Solid);
    info.SetCullMode(nn::gfx::CullMode_None);
    info.SetFrontFace(nn::gfx::FrontFace_Ccw);
    m_RasterizerState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeBlendState() NN_NOEXCEPT
{
    nn::gfx::BlendTargetStateInfo targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetBlendEnabled(true);
    targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
    targetInfo.SetDestinationColorBlendFactor(
        nn::gfx::BlendFactor_OneMinusSourceAlpha);
    targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
    nn::gfx::BlendStateInfo info;
    info.SetDefault();
    info.SetAlphaToCoverageEnabled(true);
    info.SetBlendTargetStateInfoArray(&targetInfo, 1);

    const size_t size = nn::gfx::BlendState::GetRequiredMemorySize(info);
    m_pMemory.AlignUp(nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
    m_BlendState.SetMemory(m_pMemory.Get(), size);
    m_pMemory.Advance(size);
    m_BlendState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeDepthStencilState() NN_NOEXCEPT
{
    nn::gfx::DepthStencilStateInfo info;
    info.SetDefault();
    info.SetDepthTestEnabled(false);
    info.SetDepthWriteEnabled(false);
    m_DepthStencilState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeVertexState() NN_NOEXCEPT
{
    nn::gfx::VertexStateInfo info;
    nn::gfx::VertexAttributeStateInfo attribute[2];
    nn::gfx::VertexBufferStateInfo    buffer[2];

    info.SetDefault();
    attribute[0].SetDefault();
    attribute[0].SetNamePtr(PositionSymbolName);
    attribute[0].SetBufferIndex(PositionBufferIndex);
    attribute[0].SetFormat(PositionFormat);
    attribute[0].SetOffset(0);
    attribute[1].SetDefault();
    attribute[1].SetNamePtr(UvSymbolName);
    attribute[1].SetBufferIndex(UvBufferIndex);
    attribute[1].SetFormat(UvFormat);
    attribute[1].SetOffset(0);
    buffer[0].SetDefault();
    buffer[0].SetStride(PositionStride);
    buffer[1].SetDefault();
    buffer[1].SetStride(UvStride);

    info.SetVertexAttributeStateInfoArray(attribute, 2);
    info.SetVertexBufferStateInfoArray(buffer, 2);

    for( int idx = 0; idx < DrawType_End; ++idx )
    {
        const size_t size = nn::gfx::VertexState::GetRequiredMemorySize(info);
        m_pMemory.AlignUp(nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
        m_VertexState[idx].SetMemory(m_pMemory.Get(), size);
        m_pMemory.Advance(size);
        m_VertexState[idx].Initialize(&m_Device, info, m_pShader[idx]);
    }
}

void GraphicsSystem::InitializePositionBuffer() NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(sizeof(Positions));
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);

    for( int idx = 0; idx < DrawType_End; ++idx )
    {
        if( NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired) )
        {
            m_MemoryPoolOffset = nn::util::align_up(
                m_MemoryPoolOffset, nn::gfx::Buffer::GetBufferAlignment(&m_Device, info));
            m_PositionBuffer[idx].Initialize(
                &m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, info.GetSize());
            m_MemoryPoolOffset += info.GetSize();
        }
        else
        {
            m_PositionBuffer[idx].Initialize(&m_Device, info, NULL, 0, 0);
        }
        NN_ASSERT(nn::gfx::IsInitialized(m_PositionBuffer[idx]));

        void* pMapped = m_PositionBuffer[idx].Map();
        memcpy(pMapped, DefaultPositions, info.GetSize());
        m_PositionBuffer[idx].Unmap();
    }
}

void GraphicsSystem::InitializeUvBuffer() NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(sizeof(Uvs));
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);

    if( NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired) )
    {
        m_MemoryPoolOffset = nn::util::align_up(
            m_MemoryPoolOffset, nn::gfx::Buffer::GetBufferAlignment(&m_Device, info));
        m_UvBuffer.Initialize(
            &m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, info.GetSize());
        m_MemoryPoolOffset += info.GetSize();
    }
    else
    {
        m_UvBuffer.Initialize(&m_Device, info, nullptr, 0, 0);
    }
    NN_ASSERT(nn::gfx::IsInitialized(m_UvBuffer));

    void* pMapped = m_UvBuffer.Map();
    memcpy(pMapped, Uvs, info.GetSize());
    m_UvBuffer.Unmap();
}

void GraphicsSystem::InitializeIndexBuffer() NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(sizeof(Indices));
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);

    if( NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired) )
    {
        m_MemoryPoolOffset = nn::util::align_up(
            m_MemoryPoolOffset, nn::gfx::Buffer::GetBufferAlignment(&m_Device, info));
        m_IndexBuffer.Initialize(
            &m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, info.GetSize());
        m_MemoryPoolOffset += info.GetSize();
    }
    else
    {
        m_IndexBuffer.Initialize(&m_Device, info, NULL, 0, 0);
    }
    NN_ASSERT(nn::gfx::IsInitialized(m_IndexBuffer));

    void* pMapped = m_IndexBuffer.Map();
    memcpy(pMapped, Indices, info.GetSize());
    m_IndexBuffer.Unmap();
}

void GraphicsSystem::InitializeConstantBuffer() NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(sizeof(Constant));
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);

    if( NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired) )
    {
        m_MemoryPoolOffset = nn::util::align_up(
            m_MemoryPoolOffset, nn::gfx::Buffer::GetBufferAlignment(&m_Device, info));
        m_ConstantBuffer.Initialize(
            &m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, info.GetSize());
        m_MemoryPoolOffset += info.GetSize();
    }
    else
    {
        m_ConstantBuffer.Initialize(&m_Device, info, NULL, 0, 0);
    }
    NN_ASSERT(nn::gfx::IsInitialized(m_ConstantBuffer));
}

void GraphicsSystem::InitializeSamplerDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
    info.SetSlotCount(m_SamplerDescriptorBaseIndex + 1);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, info);
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, info));
    m_SamplerDescriptorPool.Initialize(&m_Device, info,
        &m_MemoryPool, m_MemoryPoolOffset, size);
    m_MemoryPoolOffset += size;
}

void GraphicsSystem::InitializeTextureDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
    info.SetSlotCount(m_TextureDescriptorBaseIndex + TextureDescriptorPoolSlotCount);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, info);
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, info));
    m_TextureDescriptorPool.Initialize(&m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, size);
    m_MemoryPoolOffset += size;
}

void GraphicsSystem::InitializeFence() NN_NOEXCEPT
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();
    m_GpuDoneFence.Initialize(&m_Device, info);
    m_DisplayFence.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeSemaphore() NN_NOEXCEPT
{
    nn::gfx::Semaphore::InfoType info;
    info.SetDefault();
    m_DisplaySemaphore.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeTexture() NN_NOEXCEPT
{
    // テクスチャリソースの初期化
    void* pTextureResorce = ReadResource(TextureFilePath);
    NN_ASSERT_NOT_NULL(pTextureResorce);
    nn::gfx::ResTextureFile* pResTextureFile = nn::gfx::ResTextureFile::ResCast(pTextureResorce);
    NN_ASSERT_NOT_NULL(pResTextureFile);
    pResTextureFile->Initialize(&m_Device);
    const int count = pResTextureFile->GetTextureDic()->GetCount();
    for( int idx = 0; idx < count; ++idx )
    {
        // 各テクスチャの初期化
        nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(idx);
        NN_ASSERT_NOT_NULL(pResTexture);
        pResTexture->Initialize(&m_Device);
        nn::gfx::TextureView* pTextureView = pResTexture->GetTextureView();
        NN_ASSERT_NOT_NULL(pTextureView);
        m_pResTextureFile = pResTextureFile;

        // デスクリプタプールへの登録
        m_TextureDescriptorPool.BeginUpdate();
        m_TextureDescriptorPool.SetTextureView(m_TextureDescriptorBaseIndex, pTextureView);
        m_TextureDescriptorPool.EndUpdate();
        m_TextureDescriptorPool.GetDescriptorSlot(
            &m_TextureDescriptorSlotList[idx], m_TextureDescriptorBaseIndex);
        ++m_TextureDescriptorBaseIndex;
    }
}

void GraphicsSystem::InitializeSampler() NN_NOEXCEPT
{
    // サンプラの初期化
    nn::gfx::SamplerInfo info;
    info.SetDefault();
    m_Sampler.Initialize(&m_Device, info);

    // デスクリプタプールへの登録
    m_SamplerDescriptorPool.BeginUpdate();
    m_SamplerDescriptorPool.SetSampler(m_SamplerDescriptorBaseIndex, &m_Sampler);
    m_SamplerDescriptorPool.EndUpdate();
    m_SamplerDescriptorPool.GetDescriptorSlot(
        &m_SamplerDescriptorSlot, m_SamplerDescriptorBaseIndex);
}

void GraphicsSystem::InitializeDebugFontTextWriter() NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(CharCountMax);
    info.SetUserMemoryPoolEnabled(true);

    const size_t memorySize =
        nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(&m_Device, info);
    const size_t memoryPoolSize =
        nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(&m_Device, info);

    m_DebugFontTextWriter.Initialize(&m_Device,
        info, m_pMemory.Get(), memorySize, &m_MemoryPool, m_MemoryPoolOffset,
        memoryPoolSize);
    m_DebugFontTextWriter.SetDisplayWidth(DisplayWidth);
    m_DebugFontTextWriter.SetDisplayHeight(DisplayHeight);

    m_pMemory.Advance(memorySize);
    m_MemoryPoolOffset += memoryPoolSize;

    m_DebugFontTextWriter.SetTextureDescriptor(
        &m_TextureDescriptorPool,
        m_TextureDescriptorBaseIndex);
    ++m_TextureDescriptorBaseIndex;
    m_DebugFontTextWriter.SetSamplerDescriptor(
        &m_SamplerDescriptorPool,
        m_SamplerDescriptorBaseIndex);
    ++m_SamplerDescriptorBaseIndex;
}

void* GraphicsSystem::ReadResource(const char* filename) NN_NOEXCEPT
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile(&hFile, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::fs::GetFileSize(&fileSize, hFile);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile(
        &readSize, hFile, 0, &fileHeader, sizeof(nn::util::BinaryFileHeader));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ASSERT(readSize == sizeof(nn::util::BinaryFileHeader));
    size_t alignment = fileHeader.GetAlignment();

    m_pMemory.AlignUp(alignment);
    void* pBuffer = m_pMemory.Get();
    result = nn::fs::ReadFile(&readSize, hFile, 0, pBuffer, static_cast< size_t >(fileSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ASSERT(readSize == static_cast< size_t >(fileSize));
    m_pMemory.Advance(static_cast< ptrdiff_t >(fileSize));

    nn::fs::CloseFile(hFile);

    return pBuffer;
}

void GraphicsSystem::ResetCommandBuffer() NN_NOEXCEPT
{
    m_CommandBuffer.Reset();
    m_CommandBuffer.AddCommandMemory(
        &m_MemoryPool, m_CommandBufferMemoryPoolOffset, CommandBufferMemoryPoolSize);
    m_CommandBuffer.AddControlMemory(
        m_pCommandBufferControlMemory, CommandBufferControlMemorySize);
}

void GraphicsSystem::SetBackgroundColor(int r, int g, int b) NN_NOEXCEPT
{
    // RGBを設定する（アルファは1で固定）
    static const float s_MaxValue = 255.0f;
    m_BackgroundColor.valueFloat[0] = float(r) / s_MaxValue;
    m_BackgroundColor.valueFloat[1] = float(g) / s_MaxValue;
    m_BackgroundColor.valueFloat[2] = float(b) / s_MaxValue;
    m_BackgroundColor.valueFloat[3] = 1.0f;
}

void GraphicsSystem::SetRectangleColor(int r, int g, int b, int a) NN_NOEXCEPT
{
    static const float s_MaxValue = 255.0f;
    Constant* pMapped = m_ConstantBuffer.Map<Constant>();
    pMapped->color.x = float(r) / s_MaxValue;
    pMapped->color.y = float(g) / s_MaxValue;
    pMapped->color.z = float(b) / s_MaxValue;
    pMapped->color.w = float(a) / s_MaxValue;
    m_ConstantBuffer.Unmap();
}

void GraphicsSystem::SetRectangleVisible(bool value) NN_NOEXCEPT
{
    m_IsRectangleVisible = value;
}

bool GraphicsSystem::GetRectangleVisible() const NN_NOEXCEPT
{
    return m_IsRectangleVisible;
}

void GraphicsSystem::SetBackgroundKind(BackgroundKind kind) NN_NOEXCEPT
{
    m_BackgroundKind = kind;
}

nn::gfx::util::DebugFontTextWriter* GraphicsSystem::GetTextWriter() NN_NOEXCEPT
{
    return &m_DebugFontTextWriter;
}

void GraphicsSystem::SetRectanglePositions(const RectanglePositions& positions) NN_NOEXCEPT
{
    // 初期化
    NN_ASSERT(nn::gfx::IsInitialized(m_PositionBuffer[1]));
    // 左上原点、右下 1280, 720 となる座標からの変換
    Positions* pBuffer = m_PositionBuffer[1].Map<Positions>();
    pBuffer->leftTop = NN_UTIL_FLOAT_3_INITIALIZER(
        ChangeCoordinateX(positions.leftTop.x), ChangeCoordinateY(positions.leftTop.y), 0.0f);
    pBuffer->leftBottom = NN_UTIL_FLOAT_3_INITIALIZER(
        ChangeCoordinateX(positions.leftBottom.x),
        ChangeCoordinateY(positions.leftBottom.y), 0.0f);
    pBuffer->rightTop = NN_UTIL_FLOAT_3_INITIALIZER(
        ChangeCoordinateX(positions.rightTop.x), ChangeCoordinateY(positions.rightTop.y), 0.0f);
    pBuffer->rightBottom = NN_UTIL_FLOAT_3_INITIALIZER(
        ChangeCoordinateX(positions.rightBottom.x),
        ChangeCoordinateY(positions.rightBottom.y), 0.0f);
    m_PositionBuffer[1].Unmap();

}

void GraphicsSystem::MakeCommandDrawTexture() NN_NOEXCEPT
{
    m_CommandBuffer.SetVertexState(&m_VertexState[DrawType_Texture]);
    m_CommandBuffer.SetShader(
        m_pShader[DrawType_Texture],
        nn::gfx::ShaderStageBit_Vertex | nn::gfx::ShaderStageBit_Pixel);
    // 背景として表示すべきテクスチャがあるなら表示するための処理を行う。
    const int textureIndex = GetTextureListIndex(m_BackgroundKind);
    m_CommandBuffer.SetTextureAndSampler(
        m_InterfaceSlot[DrawType_Texture], nn::gfx::ShaderStage_Pixel,
        m_TextureDescriptorSlotList[textureIndex], m_SamplerDescriptorSlot);
    nn::gfx::GpuAddress gpuAddress;
    m_PositionBuffer[DrawType_Texture].GetGpuAddress(&gpuAddress);
    m_CommandBuffer.SetVertexBuffer(
        PositionBufferIndex, gpuAddress, PositionStride, sizeof(Positions));
    m_UvBuffer.GetGpuAddress(&gpuAddress);
    m_CommandBuffer.SetVertexBuffer(
        UvBufferIndex, gpuAddress, UvStride, sizeof(Uvs));
    m_IndexBuffer.GetGpuAddress(&gpuAddress);
    m_CommandBuffer.DrawIndexed(
        PrimitiveTopology, IndexFormat, gpuAddress, IndexCount, 0);
}

void GraphicsSystem::MakeCommandDrawRectangle() NN_NOEXCEPT
{
    m_CommandBuffer.SetVertexState(&m_VertexState[DrawType_Constant]);
    m_CommandBuffer.SetShader(
        m_pShader[DrawType_Constant],
        nn::gfx::ShaderStageBit_Vertex | nn::gfx::ShaderStageBit_Pixel);
    nn::gfx::GpuAddress gpuAddress;
    m_ConstantBuffer.GetGpuAddress(&gpuAddress);
    m_CommandBuffer.SetConstantBuffer(
        m_InterfaceSlot[DrawType_Constant],
        nn::gfx::ShaderStage_Pixel, gpuAddress, sizeof(Constant));

    m_PositionBuffer[DrawType_Constant].GetGpuAddress(&gpuAddress);
    m_CommandBuffer.SetVertexBuffer(
        PositionBufferIndex, gpuAddress, PositionStride, sizeof(Positions));

    m_UvBuffer.GetGpuAddress(&gpuAddress);
    m_CommandBuffer.SetVertexBuffer(
        UvBufferIndex, gpuAddress, UvStride, sizeof(Uvs));

    m_IndexBuffer.GetGpuAddress(&gpuAddress);
    m_CommandBuffer.DrawIndexed(
        PrimitiveTopology, IndexFormat, gpuAddress, IndexCount, 0);
}

void GraphicsSystem::Draw() NN_NOEXCEPT
{
    // Immediate フレームワークで描画しています。
    if(m_FrameCount == 0)
    {
        // 0 フレーム目はフレームの頭で次のフレームのスキャンバッファの取得しておく
        const nn::gfx::AcquireScanBufferResult acquireResult =
            m_SwapChain.AcquireNextScanBufferIndex(
                &m_NextScanBufferIndex, &m_DisplaySemaphore, &m_DisplayFence);
        NN_ASSERT(acquireResult == nn::gfx::AcquireScanBufferResult_Success);
        NN_UNUSED(acquireResult);
        // スキャンバッファの取得を同期（GPU）は不要
        // m_Queue.SyncSemaphore(&m_DisplaySemaphore);
        nn::gfx::SyncResult syncResult = m_DisplayFence.Sync(nn::TimeSpan::FromSeconds(1));
        NN_ASSERT(syncResult == nn::gfx::SyncResult_Success);
        NN_UNUSED(syncResult);
    }
    nn::gfx::ColorTargetView* pTarget = m_pScanBufferViews[m_NextScanBufferIndex];

    ResetCommandBuffer();
    m_CommandBuffer.Begin();
    m_CommandBuffer.ClearColorTarget(pTarget, m_BackgroundColor, nullptr);

    m_CommandBuffer.SetRenderTargets(1, &pTarget, nullptr);
    m_CommandBuffer.SetViewportScissorState(&m_ViewportScissor);

    m_CommandBuffer.SetRasterizerState(&m_RasterizerState);
    m_CommandBuffer.SetBlendState(&m_BlendState);
    m_CommandBuffer.SetDepthStencilState(&m_DepthStencilState);

    m_CommandBuffer.SetDescriptorPool(&m_TextureDescriptorPool);
    m_CommandBuffer.SetDescriptorPool(&m_SamplerDescriptorPool);

    m_CommandBuffer.InvalidateMemory(
        nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer |
        nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_ConstantBuffer |
        nn::gfx::GpuAccess_ColorBuffer);

    // 背景なしでなければテクスチャの表示
    if( m_BackgroundKind != BackgroundKind_None )
    {
        MakeCommandDrawTexture();
    }

    // 表示の場合は矩形の表示
    if( m_IsRectangleVisible )
    {
        MakeCommandDrawRectangle();
    }

    // デバッグフォントの表示
    m_DebugFontTextWriter.Draw(&m_CommandBuffer);
    m_CommandBuffer.End();

    m_Queue.ExecuteCommand(&m_CommandBuffer, &m_GpuDoneFence);
    m_Queue.Present(&m_SwapChain, 1);

    // 次のフレームのレンダリング対象となるスキャンバッファを取得する
    const nn::gfx::AcquireScanBufferResult acquireResult =
        m_SwapChain.AcquireNextScanBufferIndex(
            &m_NextScanBufferIndex, &m_DisplaySemaphore, &m_DisplayFence);
    NN_ASSERT(acquireResult == nn::gfx::AcquireScanBufferResult_Success);
    NN_UNUSED(acquireResult);
    // スキャンバッファの取得を同期（GPU）は不要
    // m_Queue.SyncSemaphore(&m_DisplaySemaphore);
    // 次のフレームでレンダリング対象となるスキャンバッファが使用可能になるまで待つ
    nn::gfx::SyncResult syncResult = m_DisplayFence.Sync(nn::TimeSpan::FromSeconds(1));
    NN_ASSERT(syncResult == nn::gfx::SyncResult_Success);
    // 現在行っているレンダリングの待ち
    syncResult = m_GpuDoneFence.Sync(nn::TimeSpan::FromSeconds(1));
    NN_ASSERT(syncResult == nn::gfx::SyncResult_Success);

    // VSync
    ++m_FrameCount;
}
