﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SimpleModel.h"
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/g3d.h>
#include <nn/g3d/g3d_Configuration.h>
#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_SkeletalAnimObj.h>
#include <nn/g3d/g3d_BoneVisibilityAnimObj.h>
#include <nn/g3d/g3d_MaterialAnimObj.h>
#include <nn/g3d/g3d_ShapeAnimObj.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/util/util_BitArray.h>

#include <nn/util/util_MatrixApi.h>


//------------------------------------------------------------------------------
// ※ グラフィックスに関する処理は、g3dのデモを参考にしてください。
//------------------------------------------------------------------------------
enum
{
    Max_Descriptor = 512
};

// 使用する頂点属性の名前を列挙します。
// ここでは配列中のインデックスを explicit attribute location と一致させています。
const char* g_AttribName[] = { "_p0", "_n0", "_u0", "_i0", "_w0" };
const char* g_AttribId[]   = { "i_Position", "i_Normal", "i_UV", "i_Index", "i_Weight" };

// 使用するサンプラーの名前と ID を列挙します。
// 事前にシェーダを関連付けたマテリアルは ResSamplerAssign に名前と ID の関連付けを持っています。
const char* g_SamplerName[] = { "_a0" };
const char* g_SamplerId[]   = { "albedoTex" };

const int g_NumAttrib  = sizeof(g_AttribName) / sizeof(char*);
const int g_NumSampler = sizeof(g_SamplerName) / sizeof(char*);
const int g_NumBuffer  = 16;

static const size_t g_CpuUncachedMemoryPoolSize = 4 * 1024 * 1024;

//--------------------------------------------------------------------------------------------------
// File
enum
{
    DEFAULT_ALIGNMENT = 4
};

nn::gfx::Shader* g_pSimpleShader = NULL;
void*   g_pSimpleShaderWork = NULL;


//---------------------------------------------------------------------------
// ファイルからシェーダーを読み込みます。
//---------------------------------------------------------------------------
nn::gfx::Shader* LoadShaderFromMemory(nn::gfx::Device* pDevice, void* pShaderResource)
{
    nn::gfx::ResShaderFile* pResShaderFile = NULL;
    nn::gfx::Shader* pShader = NULL;
    if(pShaderResource)
    {
        pResShaderFile = nn::gfx::ResShaderFile::ResCast(pShaderResource);
        nn::gfx::ResShaderContainer* pShaderContainer = pResShaderFile->GetShaderContainer();
        pShaderContainer->Initialize(pDevice);
        nn::gfx::ResShaderVariation* pShaderVariation = pShaderContainer->GetResShaderVariation(0);
        NN_SDK_ASSERT_NOT_NULL(pShaderVariation);
        nn::gfx::ResShaderProgram* pShaderProgram = pShaderVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Binary);

        nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_InvalidType;
        if (pShaderProgram != NULL)
        {
            shaderResult = pShaderProgram->Initialize(pDevice);
        }

        // シェーダーバイナリが入っていない、もしくはバイナリでの初期化に失敗したときのみ
        // シェーダーソースを使用して初期化する。
        if( pShaderProgram == NULL ||
            shaderResult != nn::gfx::ShaderInitializeResult_Success )
        {
            pShaderProgram = pShaderVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Source );
            NN_SDK_ASSERT_NOT_NULL(pShaderProgram);
            shaderResult = pShaderProgram->Initialize( pDevice );
        }
        NN_SDK_ASSERT_EQUAL( shaderResult, nn::gfx::ShaderInitializeResult_Success );
        pShader = pShaderProgram->GetShader();
    }

    return pShader;
}

//------------------------------------------------------------------------------
//     シェーダを取得します。
//------------------------------------------------------------------------------
nn::gfx::Shader* getSimpleShader()
{
    return g_pSimpleShader;
}


//------------------------------------------------------------------------------
//     テクスチャを初期化します。
//------------------------------------------------------------------------------
nn::g3d::TextureRef TextureBindCallback(const char* name, void* pUserData)
{
    nn::g3d::TextureRef textureRef;
    const nn::gfx::ResTextureFile* pTextureFile = reinterpret_cast<const nn::gfx::ResTextureFile*>(pUserData);
    const nn::util::ResDic* pDic = pTextureFile->GetTextureDic();
    int index = pDic->FindIndex(name);
    if (index == nn::util::ResDic::Npos)
    {
        return textureRef;
    }
    const nn::gfx::ResTexture* pResTexture = pTextureFile->GetResTexture(index);
    nn::gfx::DescriptorSlot descriptorSlot;
    pResTexture->GetUserDescriptorSlot(&descriptorSlot);
    textureRef.SetTextureView(pResTexture->GetTextureView());
    textureRef.SetDescriptorSlot(descriptorSlot);
    return textureRef;
}

//------------------------------------------------------------------------------
//     テクスチャを初期化します。
//------------------------------------------------------------------------------
nn::gfx::ResTextureFile* SetupTexture(nn::gfx::Device* pDevice, nn::g3d::ResFile* res3dFile, void* textureResource, nn::g3d::TextureBindCallback pCallback)
{
    nn::gfx::ResTextureFile* pResTextureFile = nn::gfx::ResTextureFile::ResCast( textureResource );
    NN_SDK_ASSERT_NOT_NULL(pResTextureFile);
    pResTextureFile->Initialize(pDevice);

    int textureCount = pResTextureFile->GetTextureDic()->GetCount();

    for (int idxTex = 0; idxTex < textureCount; ++idxTex)
    {
        nn::gfx::ResTexture* resTexture = pResTextureFile->GetResTexture(idxTex);
        resTexture->Initialize(pDevice);
    }

    // テクスチャをバインド
    res3dFile->BindTexture(pCallback, pResTextureFile);

    return pResTextureFile;
}

//------------------------------------------------------------------------------
//     テクスチャを破棄します。
//------------------------------------------------------------------------------
void CleanupTexture( nn::gfx::Device* pDevice, nn::gfx::ResTextureFile* pResTextureFile )
{
    // 付属ファイルが存在するか確認
    int textureCount = pResTextureFile->GetTextureDic()->GetCount();

    for (int idxTex = 0; idxTex < textureCount; ++idxTex)
    {
        nn::gfx::ResTexture* resTexture = pResTextureFile->GetResTexture(idxTex);
        resTexture->Finalize(pDevice);
    }

    pResTextureFile->Finalize(pDevice);
}

// コンパイル済みシェーダバイナリをヘッダ化したものを取り込む
#if defined( NN_BUILD_APISET_NX )
#include "SimpleModel_BuiltinShader_Nx.h"
#elif defined( NN_BUILD_APISET_GENERIC )
#include "SimpleModel_BuiltinShader_Win32.h"
#else
static uint8_t SimpleModelBuildinShader[1] = { 0 };
#endif

//------------------------------------------------------------------------------
//  シェーダリソースを初期化処理を行います。
//------------------------------------------------------------------------------
void InitializeG3dShaderResource( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap )
{
    // 決め打ちのシェーダーをセットアップ
    // アライメント調整のためデータをコピーする
    size_t alignment = reinterpret_cast<nn::util::BinaryFileHeader*>(SimpleModelBuildinShader)->GetAlignment();
    g_pSimpleShaderWork = pHeap->Alloc(sizeof(SimpleModelBuildinShader), alignment);
    memcpy( g_pSimpleShaderWork, SimpleModelBuildinShader, sizeof(SimpleModelBuildinShader) );
    g_pSimpleShader = LoadShaderFromMemory(pDevice, g_pSimpleShaderWork);

    NN_SDK_ASSERT_NOT_NULL(g_pSimpleShader);
}

//------------------------------------------------------------------------------
//　　　シェーダリソースの終了処理を行います。
//------------------------------------------------------------------------------
void FinalizeG3dShaderResource( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap )
{
    g_pSimpleShader->Finalize(pDevice);
    pHeap->Free(g_pSimpleShaderWork);
}

//---------------------------------------------------------------------------
//! @brief     コンストラクタ
//---------------------------------------------------------------------------
SimpleModel::SimpleModel()
    : m_ResFile(NULL)
    , m_ModelObj(NULL)
    , m_pVertexStateArray(NULL)
    , m_pSamplerTableArray(NULL)
    , m_pResSkeletalAnim(NULL)
    , m_pVertexStateBuffer(NULL)
    , m_pMemoryPoolPtr(NULL)
    , m_MemoryPoolOffset(0)
    , m_CurrentFace(0)
{
    for( uint32_t i = 0; i < MODEL_ANIM_MAX; ++i )
    {
        m_ModelAnimObjArray[i] = NULL;
    }
    nn::util::MatrixIdentity(&m_DrawMatrix);
    m_AnimFrame          = 0.0f;
    m_AnimIndex          = 0;
}


//---------------------------------------------------------------------------
//! @brief     デストラクタ
//---------------------------------------------------------------------------
SimpleModel::~SimpleModel()
{
//    Finalize();
}

//---------------------------------------------------------------------------
//! @brief     初期化処理を行います。
//---------------------------------------------------------------------------
bool SimpleModel::Initialize( void* modelResource, size_t modelResourceSize, void* textureResource, size_t textureResourceSize, nn::vfx::Heap* pHeap, nn::gfx::Device* pDevice )
{
    NN_UNUSED(modelResourceSize);
    NN_UNUSED(textureResourceSize);

    NN_SDK_ASSERT(nn::g3d::ResFile::IsValid( modelResource ));
    nn::g3d::ResFile* pResFile = nn::g3d::ResFile::ResCast( modelResource );
    NN_SDK_ASSERT(pResFile->GetModelCount() == 1);

    // メモリプールの初期化
    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );

        size_t cpuUncachedMemoryPoolSize = nn::util::align_up(g_CpuUncachedMemoryPoolSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( pDevice, info ));
        m_pMemoryPoolPtr = pHeap->Alloc( cpuUncachedMemoryPoolSize, nn::gfx::MemoryPool::GetPoolMemoryAlignment( pDevice, info ));
        NN_ASSERT_NOT_NULL( m_pMemoryPoolPtr );
        info.SetPoolMemory( m_pMemoryPoolPtr, cpuUncachedMemoryPoolSize);

        m_MemoryPool.Initialize( pDevice, info );
    }

    pResFile->Setup(pDevice);

    // テクスチャを初期化
    {
        m_ResTextureFile = SetupTexture( pDevice, pResFile, textureResource, TextureBindCallback );
    }

    // 決め打ちのシェーダー
    nn::gfx::Shader* pShader = getSimpleShader();
    NN_ASSERT_NOT_NULL(pShader);

    nn::g3d::ResModel* pResModel = pResFile->GetModel(0);

    // 頂点ステートを初期化
    {
        // VertexStateは ResVertex と1対1で対応します。
        // 複数の ResShape が ResVertex を共有している場合があります。
        int numVertex = pResModel->GetVertexCount();
        size_t vertexStateArraySize = sizeof(nn::gfx::VertexState) * numVertex;
        m_pVertexStateArray = static_cast<nn::gfx::VertexState*>(pHeap->Alloc(vertexStateArraySize, 4));
        NN_ASSERT_NOT_NULL(m_pVertexStateArray);

        for (int idxVertex = 0; idxVertex < numVertex; ++idxVertex)
        {
            nn::gfx::VertexState* pVertexState = new(&m_pVertexStateArray[idxVertex]) nn::gfx::VertexState;
            nn::gfx::VertexState::InfoType info;
            info.SetDefault();
            nn::g3d::ResVertex* pResVertex = pResModel->GetVertex(idxVertex);

            // 頂点属性を設定
            int idxElement = 0;
            nn::gfx::VertexAttributeStateInfo vertexAttribs[g_NumAttrib];
            for (int idxAttrib = 0; idxAttrib < g_NumAttrib; ++idxAttrib)
            {
                const char* name = g_AttribName[idxAttrib];
                vertexAttribs[idxAttrib].SetDefault();
                nn::g3d::ResVertexAttr* pResVertexAttr = pResVertex->FindVertexAttr(name);
                if (pResVertexAttr)
                {
                    vertexAttribs[idxElement].SetBufferIndex(pResVertexAttr->GetBufferIndex());
                    vertexAttribs[idxElement].SetFormat(pResVertexAttr->GetFormat());
                    vertexAttribs[idxElement].SetOffset(pResVertexAttr->GetOffset());
                    int slotAttr = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, g_AttribId[idxAttrib]);
                    vertexAttribs[idxElement].SetShaderSlot(slotAttr);
                    idxElement++;
                }
            }

            // 頂点バッファ情報を設定
            int numVertexBuffer = pResVertex->GetVertexBufferCount();
            nn::gfx::VertexBufferStateInfo bufferStateInfo[g_NumBuffer];
            NN_ASSERT(numVertexBuffer < g_NumBuffer);
            for (int idxBuffer = 0; idxBuffer < numVertexBuffer; ++idxBuffer)
            {
                bufferStateInfo[idxBuffer].SetDefault();
                bufferStateInfo[idxBuffer].SetStride(pResVertex->GetVertexBufferStride(idxBuffer));
            }

            // 頂点属性と頂点バッファ情報を頂点ステートに設定
            info.SetVertexAttributeStateInfoArray(vertexAttribs, idxElement);
            info.SetVertexBufferStateInfoArray(bufferStateInfo, numVertexBuffer);

            size_t memorySize = nn::gfx::VertexState::GetRequiredMemorySize(info);
            m_pVertexStateBuffer = pHeap->Alloc(memorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
            NN_ASSERT_NOT_NULL(m_pVertexStateBuffer);
            pVertexState->SetMemory(m_pVertexStateBuffer, memorySize);
            pVertexState->Initialize(pDevice, info, NULL);
            pResVertex->SetUserPtr(pVertexState);
        }
    }

    // マテリアル毎にサンプラーのインデックスをキャッシュします。
    {
        int materialCount = pResModel->GetMaterialCount();
        size_t samplerTableArraySize = sizeof(SamplerTable) * materialCount;
        m_pSamplerTableArray = static_cast<SamplerTable*>(pHeap->Alloc(samplerTableArraySize, 4));
        NN_ASSERT_NOT_NULL(m_pSamplerTableArray);

        for (int idxMat = 0; idxMat < materialCount; ++idxMat)
        {
            nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(idxMat);

            // SamplerTable の初期化します。
            SamplerTable* pSamplerTable = &m_pSamplerTableArray[idxMat];
            memset(pSamplerTable->index, -1, sizeof(pSamplerTable->index));

            // サンプラーをディスクリプタプールに登録
            for (int idxTex = 0; idxTex < g_NumSampler; ++idxTex)
            {
                const char* name = g_SamplerName[idxTex];
                pSamplerTable->index[idxTex] = pResMaterial->FindSamplerIndex(name);
            }

            // Uniform Block のサイズを指定します。
            pResMaterial->SetRawParamSize(16);
        }
    }

    // モデルを初期化
    {
        m_ModelObj = new(pHeap->Alloc( sizeof(nn::g3d::ModelObj), nn::g3d::ModelObj::Alignment_Buffer ) ) nn::g3d::ModelObj();
        NN_SDK_ASSERT_NOT_NULL(m_ModelObj);
        nn::g3d::ModelObj::Builder builder(pResFile->GetModel(0));

        builder.MaterialBufferingCount( 2 );
        builder.SkeletonBufferingCount( 2 );
        builder.ShapeBufferingCount( 2 );
        builder.ViewCount( 2 );

        builder.CalculateMemorySize();
        size_t bufferSize = builder.GetWorkMemorySize();
        void* pBuffer = pHeap->Alloc(bufferSize);
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        bool success = builder.Build(m_ModelObj, pBuffer, bufferSize);
        NN_SDK_ASSERT(success);

        // Uniform Block のバッファは所定のアライメントが必要です。
        size_t blockBufferSize = m_ModelObj->CalculateBlockBufferSize(pDevice);
        ptrdiff_t blockBufferOffset = -1;
        blockBufferOffset = AllocateFromMemoryPool(blockBufferSize, m_ModelObj->GetBlockBufferAlignment(pDevice));
        success = m_ModelObj->SetupBlockBuffer(pDevice,
                                              &m_MemoryPool,
                                              blockBufferOffset,
                                              blockBufferSize);
        NN_ASSERT(success);
    }

    // ビューの Uniform Block
    // モデルに依存しないので通常はビューごとに1つ使用します。
    // 定数バッファ用のバッファを初期化
    nn::gfx::Buffer::InfoType viewBlockInfo;
    ptrdiff_t viewBlockMemoryPoolOffset = -1;
    viewBlockInfo.SetDefault();
    viewBlockInfo.SetSize(sizeof(ViewBlock));
    viewBlockInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
    viewBlockMemoryPoolOffset = AllocateFromMemoryPool(
                                    viewBlockInfo.GetSize(),
                                    nn::gfx::Buffer::GetBufferAlignment( pDevice, viewBlockInfo ));
    m_ViewBlock.Initialize( pDevice, viewBlockInfo,
                            &m_MemoryPool,
                            viewBlockMemoryPoolOffset,
                            viewBlockInfo.GetSize());

    // ブレンドステートを初期化
    {
        nn::gfx::BlendState::InfoType info;
        info.SetDefault();
        nn::gfx::BlendTargetStateInfo targetInfo;
        targetInfo.SetDefault();
        info.SetBlendTargetStateInfoArray( &targetInfo, 1 );
        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize( info );
        void* pMemoryBuffer = pHeap->Alloc( memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        NN_ASSERT_NOT_NULL(pMemoryBuffer);
        m_BlendState.SetMemory(pMemoryBuffer, memorySize);
        m_BlendState.Initialize(pDevice, info);
    }

    // 深度ステンシルステートを初期化
    {
        nn::gfx::DepthStencilState::InfoType info;
        info.SetDefault();
        info.SetDepthTestEnabled(true);
        info.SetDepthWriteEnabled(true);
        m_DepthStencilState.Initialize(pDevice, info);
    }

    // スケルタルアニメの構築。
    if(pResFile->GetSkeletalAnimCount() > 0)
    {
        m_pResSkeletalAnim = pResFile->GetSkeletalAnim(0);
        if (m_pResSkeletalAnim)
        {
            nn::g3d::SkeletalAnimObj::Builder builder;
            builder.Reserve(pResModel);
            builder.Reserve(m_pResSkeletalAnim);
            builder.CalculateMemorySize();
            size_t bufferSize = builder.GetWorkMemorySize();
            void* pBuffer = pHeap->Alloc(bufferSize, nn::g3d::SkeletalAnimObj::Alignment_Buffer);
            NN_ASSERT_NOT_NULL(pBuffer);
            bool success = builder.Build(&m_SkeletalAnimObj, pBuffer, bufferSize);
            if(success)
            {
                // リソースの設定を行います。再設定可能です。
                m_SkeletalAnimObj.SetResource(m_pResSkeletalAnim);
                // モデルへの関連付けを行います。
                m_SkeletalAnimObj.Bind(m_ModelObj);
            }
        }
    }

    // リソースを記録しておく
    m_ResFile = pResFile;

    return true;
} // NOLINT(impl/function_size)


//---------------------------------------------------------------------------
//! @brief     終了処理を行います。
//---------------------------------------------------------------------------
void SimpleModel::Finalize( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap )
{
    void* pBlendStateMemory = m_BlendState.GetMemory();

    m_DepthStencilState.Finalize(pDevice);
    m_BlendState.Finalize(pDevice);

    if (pBlendStateMemory!= NULL)
    {
        pHeap->Free(pBlendStateMemory);
    }

    if (m_pSamplerTableArray != NULL)
    {
        pHeap->Free(m_pSamplerTableArray);
        m_pSamplerTableArray = NULL;
    }
    if (m_pVertexStateArray != NULL)
    {
        pHeap->Free(m_pVertexStateArray);
        m_pVertexStateArray = NULL;
    }
    if (m_pVertexStateBuffer != NULL)
    {
        pHeap->Free(m_pVertexStateBuffer);
        m_pVertexStateBuffer = NULL;
    }
    m_ViewBlock.Finalize(pDevice);

    // テクスチャの破棄
    CleanupTexture(pDevice, m_ResTextureFile);

    // モデルの破棄
    m_ModelObj->CleanupBlockBuffer( pDevice );

    m_ResFile->Cleanup(pDevice);

    pHeap->Free( m_ModelObj->GetBufferPtr() );
    pHeap->Free( m_ModelObj );

    // スケルタルアニメの破棄。
    if (void* pBuffer = m_SkeletalAnimObj.GetBufferPtr())
    {
        pHeap->Free(pBuffer);
    }

    // メモリプールの終了処理
    m_MemoryPool.Finalize(pDevice);
    pHeap->Free(m_pMemoryPoolPtr);
    m_pMemoryPoolPtr = NULL;
}


//---------------------------------------------------------------------------
//! @brief     計算処理を行います。
//---------------------------------------------------------------------------
void SimpleModel::Calc()
{
    // スケルタルアニメーション。
    if (m_SkeletalAnimObj.GetResource())
    {
        m_SkeletalAnimObj.GetFrameCtrl().SetFrame( m_AnimFrame );
        m_SkeletalAnimObj.Calculate();
        m_SkeletalAnimObj.ApplyTo( m_ModelObj );
        m_SkeletalAnimObj.GetFrameCtrl().UpdateFrame();
    }

    // ベース行列。
    m_ModelObj->CalculateWorld( m_DrawMatrix );
}


//---------------------------------------------------------------------------
//! @brief     描画マトリクスを設定します。
//---------------------------------------------------------------------------
void SimpleModel::SetDrawMatrix( const nn::util::Matrix4x3fType& drawMatrix )
{
    m_DrawMatrix = drawMatrix;
}


//---------------------------------------------------------------------------
//! @brief     アニメーションフレームを設定します。
//---------------------------------------------------------------------------
void SimpleModel::SetAnimationFrame( float frame )
{
    m_AnimFrame = frame;
}


//---------------------------------------------------------------------------
//! @brief     アニメーションタイプを設定します。
//---------------------------------------------------------------------------
void SimpleModel::SetAnimationIndex( uint32_t index )
{
    m_AnimIndex = index;
}


//---------------------------------------------------------------------------
//! @brief     ブロックの更新処理を行います。
//---------------------------------------------------------------------------
void SimpleModel::CalcBlock( const nn::util::Matrix4x4fType* projMtx, const nn::util::Matrix4x3fType* viewMtx, const nn::util::Vector3fType* lightPos, const float animaSpeed )
{
    NN_UNUSED(lightPos);
    NN_UNUSED(animaSpeed);

    // バッファの更新
    m_CurrentFace = 1 - m_CurrentFace;

    // 計算
    {
        // GPU 待ちが必要な計算
        m_ModelObj->CalculateSkeleton(m_CurrentFace);
        m_ModelObj->CalculateMaterial(m_CurrentFace);
        m_ModelObj->CalculateShape(m_CurrentFace);
        m_ModelObj->CalculateView(m_CurrentFace, *viewMtx, m_CurrentFace);


        // モデルのマテリアルに混載した領域にプログラムから値を差し込む場合、
        // 所定のエンディアンで書き込みます。Unmap処理でCPUキャッシュはメモリに吐き出されます。
        for (int idxMat = 0, materialCount = m_ModelObj->GetMaterialCount(); idxMat < materialCount; ++idxMat)
        {
            nn::g3d::MaterialObj* pMaterialObj = m_ModelObj->GetMaterial(idxMat);
            void* pMaterialBlock = pMaterialObj->GetMaterialBlock(m_CurrentFace)->Map<void>();

            float colorOffset[] = { 0.0f, 0.0f, 0.0f };
            memcpy(pMaterialBlock, colorOffset, sizeof(colorOffset));
            pMaterialObj->GetMaterialBlock(m_CurrentFace)->FlushMappedRange(
                0, pMaterialObj->GetMaterialBlockSize());
            pMaterialObj->GetMaterialBlock(m_CurrentFace)->Unmap();
        }

        // ビュー用の Uniform Block の更新。
        ViewBlock* pViewBlockBuffer = m_ViewBlock.Map< ViewBlock >();
        MatrixStore(&pViewBlockBuffer->projMtx, *projMtx);
        MatrixStore(&pViewBlockBuffer->cameraMtx, *viewMtx);
        m_ViewBlock.FlushMappedRange(0, sizeof(ViewBlock));
        m_ViewBlock.Unmap();
    }
}


//---------------------------------------------------------------------------
//! @brief     描画処理を行います。
//---------------------------------------------------------------------------
void SimpleModel::Draw( nn::gfx::CommandBuffer* pCommandBuffer, nn::vfx::Heap* pHeap )
{
    NN_UNUSED(pHeap);
    // シェーダーを設定
    pCommandBuffer->SetShader(g_pSimpleShader, nn::gfx::ShaderStageBit_Vertex | nn::gfx::ShaderStageBit_Pixel);
    pCommandBuffer->SetBlendState(&m_BlendState);
    pCommandBuffer->SetDepthStencilState(&m_DepthStencilState);
    int slotSkeleton = g_pSimpleShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Skeleton");
    int slotShape    = g_pSimpleShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Shape");
    int slotView     = g_pSimpleShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "View");
    int slotMaterial = g_pSimpleShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel , nn::gfx::ShaderInterfaceType_ConstantBuffer, "Material");

    // ユニフォームブロック設定
    const nn::g3d::SkeletonObj* pSkeletonObj = m_ModelObj->GetSkeleton();
    nn::gfx::GpuAddress gpuAddress;
    if ( pSkeletonObj->GetMtxBlockSize() > 0 )
    {
        pSkeletonObj->GetMtxBlock(m_CurrentFace)->GetGpuAddress(&gpuAddress);
        pCommandBuffer->SetConstantBuffer(slotSkeleton, nn::gfx::ShaderStage_Vertex, gpuAddress, pSkeletonObj->GetMtxBlockSize());
    }
    m_ViewBlock.GetGpuAddress(&gpuAddress);
    pCommandBuffer->SetConstantBuffer(slotView, nn::gfx::ShaderStage_Vertex, gpuAddress, sizeof(ViewBlock));

    // ここではソートやカリング、同一の描画ステートの省略などは行っていません。
    // アプリケーションの仕様にとって最適な方法で描画処理を行うことを推奨します。
    int shapeCount = m_ModelObj->GetShapeCount();
    for (int idxShape = 0; idxShape < shapeCount; ++idxShape)
    {
        const nn::g3d::ShapeObj* pShapeObj = m_ModelObj->GetShape(idxShape);

        if (!m_ModelObj->IsShapeVisible(idxShape)) // ビジビリティの制御を行います。
        {
            continue;
        }

        int idxMaterial = pShapeObj->GetMaterialIndex();
        const nn::g3d::MaterialObj* pMaterialObj = m_ModelObj->GetMaterial(idxMaterial);

        // ユニフォームブロックを設定
        pShapeObj->GetShapeBlock(0, 0)->GetGpuAddress(&gpuAddress);
        pCommandBuffer->SetConstantBuffer(slotShape, nn::gfx::ShaderStage_Vertex, gpuAddress, sizeof(nn::g3d::ShapeBlock));
        pMaterialObj->GetMaterialBlock(m_CurrentFace)->GetGpuAddress(&gpuAddress);

        pCommandBuffer->SetConstantBuffer(slotMaterial, nn::gfx::ShaderStage_Pixel, gpuAddress, pMaterialObj->GetMaterialBlockSize());

        // 頂点属性を設定
        int idxVertex = pShapeObj->GetVertexIndex();
        nn::g3d::ResModel* pResModel = const_cast<nn::g3d::ResModel*>(m_ModelObj->GetResource());
        nn::g3d::ResVertex* pResVertex = pResModel->GetVertex(idxVertex);
        nn::gfx::VertexState* pVertexState = static_cast<nn::gfx::VertexState*>(pResVertex->GetUserPtr());
        pCommandBuffer->SetVertexState(pVertexState);

        // バッファを設定
        int numVertexBuffer = pResVertex->GetVertexBufferCount();
        for (int idxBuffer = 0; idxBuffer < numVertexBuffer; ++idxBuffer)
        {
            pResVertex->GetVertexBuffer(idxBuffer)->GetGpuAddress(&gpuAddress);
            pCommandBuffer->SetVertexBuffer(idxBuffer,
                                            gpuAddress,
                                            pResVertex->GetVertexBufferStride(idxBuffer),
                                            pResVertex->GetVertexBufferInfo(idxBuffer)->GetSize());
        }

        // サンプラーを設定
        SamplerTable* pSamplerTable = &m_pSamplerTableArray[idxMaterial];
        for (int idxTex = 0; idxTex < g_NumSampler; ++idxTex)
        {
            const char* name = g_SamplerId[idxTex];
            int slotSampler = g_pSimpleShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, name);
            int idxSampler = pSamplerTable->index[idxTex];
            if (slotSampler != -1 && idxSampler != -1)
            {
                nn::g3d::SamplerRef samplerRef = pMaterialObj->GetSampler(idxSampler);
                nn::g3d::TextureRef textureRef = pMaterialObj->GetTexture(idxSampler);
                pCommandBuffer->SetTextureAndSampler(slotSampler,
                                                     nn::gfx::ShaderStage_Pixel,
                                                     textureRef.GetDescriptorSlot(),
                                                     samplerRef.GetDescriptorSlot());
            }
        }

        // 描画コール
        pShapeObj->GetResMesh()->Draw(pCommandBuffer, 1);
    }
}

void SimpleModel::UpdateDescriptorSlot( RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pTextureUserData,
                                        RegisterSamplerSlot     pRegisterSamplerSlotCallback, void* pSamplerUserData )
{
    nn::g3d::ResModel* pResModel = m_ResFile->GetModel(0);

    int materialCount = pResModel->GetMaterialCount();
    for (int idxMat = 0; idxMat < materialCount; ++idxMat)
    {
        nn::g3d::ResMaterial* pRes = pResModel->GetMaterial(idxMat);
        nn::g3d::MaterialObj* pMaterialObj = m_ModelObj->GetMaterial(idxMat);
        int textureCount    = pRes->GetTextureCount();
        for (int idxTexture = 0; idxTexture < textureCount; ++idxTexture)
        {
            nn::gfx::DescriptorSlot descriptorSlot;
            const nn::gfx::TextureView* pTextureView = pRes->GetTexture(idxTexture);
            if ( pTextureView )
            {
                pRegisterTextureSlotCallback(&descriptorSlot, *pTextureView, pTextureUserData);
                pRes->SetTextureDescriptorSlot(idxTexture, descriptorSlot);

                nn::g3d::TextureRef texRef;
                texRef.SetDescriptorSlot(descriptorSlot);
                texRef.SetTextureView(pTextureView);
                pMaterialObj->SetTexture(idxTexture, texRef);
            }
        }

        SamplerTable* pSamplerTable = &m_pSamplerTableArray[idxMat];
        for (int idxTex = 0; idxTex < g_NumSampler; ++idxTex)
        {
            const char* name = g_SamplerId[idxTex];
            int slotSampler = g_pSimpleShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, name);
            int idxSampler = pSamplerTable->index[idxTex];
            if (slotSampler != -1 && idxSampler != -1)
            {
                nn::gfx::Sampler* pSampler = pRes->GetSampler( idxSampler );
                nn::gfx::DescriptorSlot descriptorSlot;
                pRegisterSamplerSlotCallback( &descriptorSlot, *pSampler, pSamplerUserData );
                pRes->SetSamplerDescriptorSlot(idxTex, descriptorSlot);
            }
        }
    }
}

//---------------------------------------------------------------------------
//      メモリプールから領域を確保します
//---------------------------------------------------------------------------
ptrdiff_t SimpleModel::AllocateFromMemoryPool(size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/)
{
    ptrdiff_t offset = nn::util::align_up(m_MemoryPoolOffset, alignment);
    NN_ASSERT(g_CpuUncachedMemoryPoolSize - offset >= size, "AllocateMemoryPool failed.\n");
    m_MemoryPoolOffset= offset + size;
    return offset;
}

