﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{VfxSimple/main.cpp,PageSampleVfxSimple}
 *
 * @brief
 *  シンプルなエフェクト表示のサンプルプログラム
 */

/**
 * @page PageSampleVfxSimple シンプルなエフェクト表示
 * @tableofcontents
 *
 * @image html Applications\VfxSimple\VfxSimple.png
 *
 * @brief
 *  サンプルプログラム VfxSimple の解説です。
 *
 * @section PageSampleVfxSimple_SectionBrief 概要
 *  nn::vfx を使用してエフェクトバイナリを再生する最もシンプルなサンプルです。
 *
 *  エフェクトランタイムの機能の使い方については、
 *  @ref nn::vfx "Vfx の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleVfxSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/VfxSimple
 *  Samples/Sources/Applications/VfxSimple @endlink 以下にあります。
 *
 * @section PageSampleVfxSimple_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleVfxSimple_SectionHowToOperate 操作方法
 *  このサンプルを実行するとエフェクトが自動的に再生され、一定回数描画した後終了します。
 *
 * @section PageSampleVfxSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleVfxSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleVfxSimple_SectionDetail 解説
 *
 * @subsection PageSampleVfxSimple_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  VfxSimple/main.cpp
 *  @includelineno VfxSimple/main.cpp
 *
 * @subsection PageSampleVfxSimple_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *  - Gfxの初期化
 *  - カメラ・プロジェクション行列の初期化
 *  - VFXランタイムの初期化
 *  - VFXバイナリロード
 *  - エフェクト生成
 *  - ループ開始
 *  - VFXランタイムの計算処理
 *  - VFXランタイムの描画処理
 *  - ループ開始に戻る
 *  - VFXランタイムの終了処理
 *  - Gfxの終了処理
 *
 */

#include <nn/TargetConfigs/build_Base.h>

#include <nn/nn_Result.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs.h>
#include <nn/gfx/util/gfx_DescriptorPoolAllocator.h>
#include <nn/vfx.h>
#include <nn/vi.h>
#include <nn/vfx/viewer/vfx_Viewer.h>
#include <nn/util/util_StringUtil.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_ColorBuffer.h>
#include <nns/gfx/gfx_DepthStencilBuffer.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/hws/hws_Message.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

void*                           g_HeapPtr;              // エフェクトヒープバッファポインタ
nn::vfx::StandardHeap           g_VfxHeap;              // エフェクトヒープ

//------------------------------------------------------------------------------
//  テクスチャデスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateDescriptorSlotForTexture(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(dstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  テクスチャデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForTexture(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);

    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*dstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateDescriptorSlotForSampler(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(dstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForSampler(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);

    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        int slotIndex = pPool->GetDescriptorSlotIndex(*dstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ取得用関数
//------------------------------------------------------------------------------
void* Allocate(size_t size)
{
    return malloc(size);
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ解放用関数
//------------------------------------------------------------------------------
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    free(p);
}

//------------------------------------------------------------------------------
//  nn::fs を利用したファイルロード関数
//------------------------------------------------------------------------------
bool FileLoad(
    void**           pOutBuffer,
    size_t*          pOutBufferSize,
    const char*      filePath,
    nn::vfx::Heap*   pAllocator,
    int              alignment)
{
    char filePathName[nn::fs::EntryNameLengthMax];
    memcpy(reinterpret_cast< void* >(&filePathName), filePath, strlen(filePath) + 1);

    nn::fs::FileHandle fsHandle;
    nn::Result ret = nn::fs::OpenFile(&fsHandle, static_cast< char* >(filePathName), static_cast< nn::fs::OpenMode >(nn::fs::OpenMode_Read));
    if (ret.IsFailure())
    {
        return false;
    }

    int64_t fsSize;
    nn::fs::GetFileSize(&fsSize, fsHandle);
    *pOutBufferSize = static_cast< size_t >(fsSize);

    *pOutBuffer = pAllocator->Alloc(*pOutBufferSize + 4, alignment);
    if (!(*pOutBuffer))
    {
        nn::fs::CloseFile(fsHandle);
        return false;
    }

    memset(*pOutBuffer, 0, *pOutBufferSize + 4);
    nn::fs::ReadFile(fsHandle, 0, *pOutBuffer, *pOutBufferSize);
    nn::fs::CloseFile(fsHandle);

    return true;
}


//------------------------------------------------------------------------------
//  GraphicsFramework
//------------------------------------------------------------------------------
nns::gfx::GraphicsFramework::FrameworkInfo g_GraphicsFrameworkInfo;
nns::gfx::GraphicsFramework                g_GraphicsFramework;


//------------------------------------------------------------------------------
//  nn::gfx オブジェクト
//------------------------------------------------------------------------------
nn::gfx::DescriptorSlot         g_ColorCopyBufferDescSlot;              // カラーコピーコピーバッファデスクリプタスロット
nn::gfx::DescriptorSlot         g_DepthStencilCopyDescSlot;             // デプス・ステンシルコピーデスクリプタスロット
nns::gfx::ColorBuffer           g_ColorBufferCopy;                      // コピー用カラーバッファ
nns::gfx::DepthStencilBuffer    g_DepthStencilBufferCopy;               // コピー用デプス・ステンシル


//------------------------------------------------------------------------------
//  VfxSimple デモ関連
//------------------------------------------------------------------------------
nn::util::Vector3fType          g_CamPosVec         = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 30, 50 );  // カメラ位置
nn::util::Vector3fType          g_CamLookAtVec      = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );    // カメラ注視点
nn::util::Vector3fType          g_CamUpVec          = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 1, 0 );    // カメラ上方向
nn::util::Matrix4x3fType        g_ViewMtx;                              // ビュー行列
nn::util::Matrix4x4fType        g_ProjectionMatrix;                     // 射影変換行列
float                           g_NearZ             = 0.1f;             // Z Near
float                           g_FarZ              = 10000.0f;         // Z Far
nn::vfx::System*                g_pVfxSystem        = nullptr;          // nn::vfx システム


//------------------------------------------------------------------------------
//  Gfx初期処理
//------------------------------------------------------------------------------
void InitializeGfx()
{
    nn::gfx::MemoryPool* memoryPool = g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);

    int width  = g_GraphicsFramework.GetDisplayWidth();
    int height = g_GraphicsFramework.GetDisplayHeight();

    nn::gfx::Device* device = g_GraphicsFramework.GetDevice();

    // カラーコピーバッファ用のテクスチャを初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth( width );
        info.SetHeight( height );
        info.SetGpuAccessFlags( nn::gfx::GpuAccess_ColorBuffer );
        info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
        info.SetImageFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
        info.SetMipCount( 1 );

        size_t size = nn::gfx::Texture::CalculateMipDataSize( device, info );
        size_t align = nn::gfx::Texture::CalculateMipDataAlignment( device, info );
        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, align );

        g_ColorBufferCopy.Initialize( device, info, memoryPool, offset, size );

        // デスクリプタスロット設定
        AllocateDescriptorSlotForTexture( &g_ColorCopyBufferDescSlot, *g_ColorBufferCopy.GetTextureView(), &g_GraphicsFramework );
    }


    // 深度ステンシルコピーバッファ用のテクスチャを初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth( width );
        info.SetHeight( height );
        info.SetGpuAccessFlags( nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_Texture );
        info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
        info.SetImageFormat( nn::gfx::ImageFormat_D32_Float );
        info.SetMipCount( 1 );

        size_t size = nn::gfx::Texture::CalculateMipDataSize( device, info );
        size_t align = nn::gfx::Texture::CalculateMipDataAlignment( device, info );
        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, align );

        g_DepthStencilBufferCopy.Initialize( device, info, memoryPool, offset, size );

        // デスクリプタスロット設定
        AllocateDescriptorSlotForTexture( &g_DepthStencilCopyDescSlot, *g_DepthStencilBufferCopy.GetTextureView(), &g_GraphicsFramework );
    }



} // NOLINT(readability/fn_size)


//------------------------------------------------------------------------------
//  Gfx終了処理
//------------------------------------------------------------------------------
void FinalizeGfx()
{
    nn::gfx::Device* device = g_GraphicsFramework.GetDevice();

    g_ColorBufferCopy.Finalize(device);
    g_DepthStencilBufferCopy.Finalize(device);

    FreeDescriptorSlotForTexture(&g_ColorCopyBufferDescSlot, *g_ColorBufferCopy.GetTextureView(), &g_GraphicsFramework);
    FreeDescriptorSlotForTexture(&g_DepthStencilCopyDescSlot, *g_DepthStencilBufferCopy.GetTextureView(), &g_GraphicsFramework);
}

//!--------------------------------------------------------------------------------------
//! @brief GpuBuffer 用メモリ確保関数
//!--------------------------------------------------------------------------------------
void* AllocateGpuBuffer(size_t size, size_t alignment, void* pUserData)
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);

    // ユーザーデータポインタでアロケータを渡してもらう.
    nn::vfx::StandardHeap*    pAllocator = static_cast<nn::vfx::StandardHeap*>(pUserData);

    return pAllocator->Alloc(size, alignment);
}

//!--------------------------------------------------------------------------------------
//! @brief GpuBuffer 用メモリ開放関数
//!--------------------------------------------------------------------------------------
void FreeGpuBuffer(void* ptr, void* pUserData)
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);

    // ユーザーデータポインタでアロケータを渡してもらう.
    nn::vfx::StandardHeap*    pAllocator = static_cast<nn::vfx::StandardHeap*>(pUserData);

    pAllocator->Free(ptr);
}

//------------------------------------------------------------------------------
//  PrimitiveRenderer 初期化処理
//------------------------------------------------------------------------------
nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
void InitializePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator( AllocateGpuBuffer, &g_VfxHeap );

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // g_pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )する
    info.SetMultiBufferQuantity( 2 );

    // PrimitiveRendererのインスタンス
    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer( g_GraphicsFramework.GetDevice(), info);

    g_pPrimitiveRenderer->SetScreenWidth( g_GraphicsFramework.GetDisplayWidth() );
    g_pPrimitiveRenderer->SetScreenHeight(g_GraphicsFramework.GetDisplayHeight());
}

//------------------------------------------------------------------------------
//  PrimitiveRenderer 破棄処理
//------------------------------------------------------------------------------
void FinalizePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::DestroyRenderer( g_pPrimitiveRenderer, g_GraphicsFramework.GetDevice(), FreeGpuBuffer, &g_VfxHeap );
}


//------------------------------------------------------------------------------
//  グリッドの描画処理
//------------------------------------------------------------------------------
void GridDraw( nn::gfx::CommandBuffer* commandBuffer, float scaleValue )
{
    g_pPrimitiveRenderer->Update(0);

    // Blend
    commandBuffer->SetBlendState( g_pPrimitiveRenderer->GetBlendState( nns::gfx::PrimitiveRenderer::BlendType::BlendType_Opacity ) );

    // Depth Enable
    g_pPrimitiveRenderer->SetDepthStencilState( commandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest );

    g_pPrimitiveRenderer->SetProjectionMatrix( &g_ProjectionMatrix );

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Vector3fType cameraPos;
    nn::util::VectorMultiply(&cameraPos, g_CamPosVec, scaleValue);

    nn::util::MatrixLookAtRightHanded( &viewMatrix, cameraPos, g_CamLookAtVec, 0.0f );

    g_pPrimitiveRenderer->SetViewMatrix( &viewMatrix );

    nn::util::Matrix4x3fType modelMatrix;
    nn::util::MatrixIdentity(&modelMatrix);
    const nn::util::Vector3f offset( 0, -15, -15 );
    nn::util::MatrixSetTranslate( &modelMatrix, offset );
    g_pPrimitiveRenderer->SetModelMatrix( &modelMatrix );

    int cnt = 0;
    for ( int i = -5; i < 5; i++ )
    {
        for ( int j = -5; j < 5; j++ )
        {
            nn::util::Vector3fType center = { i * 10.f + 5.f, 0.f, j * 10.f + 5.f };
            nn::util::Vector3fType size   = { 10.f, 0.1f, 10.f };

            nn::util::Uint8x4 color = { { 32, 32, 32, 255 } };
            if ( ( cnt % 2 ) == 0 )
            {
                color.v[0] = 64;
                color.v[1] = 64;
                color.v[2] = 64;
            }
            g_pPrimitiveRenderer->SetColor(color);
            g_pPrimitiveRenderer->DrawCube( commandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
            cnt++;
        }
        cnt++;
    }
}

//  コマンドバッファイベントコールバック
void OutOfMemoryEventCallback( nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg& arg )
{
    NN_UNUSED( pCommandBuffer );
    NN_UNUSED( arg );
}

// textureをコピー
void CopyTextureBuffer(nn::gfx::CommandBuffer* commandBuffer, nn::gfx::Texture* dstTexture, nn::gfx::Texture* srcTexture )
{
    nn::gfx::TextureSubresource dstSubResource;
    dstSubResource.SetDefault();
    dstSubResource.SetMipLevel( 0 );
    dstSubResource.SetArrayIndex( 0 );

    nn::gfx::TextureCopyRegion srcCopyRegion;
    srcCopyRegion.SetDefault();
    srcCopyRegion.SetWidth( g_GraphicsFramework.GetDisplayWidth() );
    srcCopyRegion.SetHeight( g_GraphicsFramework.GetDisplayHeight() );
    srcCopyRegion.EditSubresource().SetDefault();
    srcCopyRegion.EditSubresource().SetMipLevel( 0 );
    srcCopyRegion.EditSubresource().SetArrayIndex( 0 );

    commandBuffer->CopyImage( dstTexture, dstSubResource, 0, 0, 0, srcTexture, srcCopyRegion );
}

struct EmitterParamter
{
    const char* emitterName;
    float rotate;
    float scale;
    bool  moveCircle;
} emitterParam[] = { { "Afterburner",          90.0f, 1.0f, false },
                     { "Carnival",              0.0f, 2.0f, false },
                     { "Fire_01",               0.0f, 0.5f, false },
                     { "Fire_02",               0.0f, 0.5f, false },
                     { "Fire_03",               0.0f, 0.5f, false },
                     { "Fire_04",               0.0f, 0.5f, false },
                     { "Fire_05",               0.0f, 0.5f, false },
                     { "Fire_06",               0.0f, 0.5f, false },
                     { "Fireworks",             0.0f, 2.2f, false },
                     { "Fireworks_01",          0.0f, 2.2f, false },
                     { "Fireworks_02",          0.0f, 2.0f, false },
                     { "Fireworks_03",          0.0f, 2.0f, false },
                     { "Fireworks_04",          0.0f, 2.0f, false },
                     { "Fireworks_Template",    0.0f, 2.0f, false },
                     { "LightSword",            0.0f, 1.0f, true  },
                     { "Muzzleflash",          90.0f, 1.0f, false },
                     { "snow",                  0.0f, 2.0f, false },
                     { "Tornado",               0.0f, 2.0f, false },
                     { "UnderSea",              0.0f, 2.0f, false }};

// エフェクトの見え方を設定
void GetViewParamaters( nn::vfx::Handle* emitterSetHandle, nn::util::Matrix4x3fType* viewMatrix, nn::util::Vector3fType* cameraPositon, float* scaleValue, bool* circleMove )
{
    int maxEmitters = sizeof(emitterParam) / sizeof(EmitterParamter);
    const char* emitterName = emitterSetHandle->GetEmitterSet()->GetName();

    float twist = 0.0f;
    float scale = 1.0f;

    *circleMove = false;
    for ( int i = 0; i <  maxEmitters; i++ )
    {
        if ( strcmp( emitterParam[i].emitterName, emitterName ) == 0 )
        {
            twist       = emitterParam[i].rotate;
            scale       = emitterParam[i].scale;
            *circleMove = emitterParam[i].moveCircle;
            break;
        }
    }

    *scaleValue    = scale;
    *cameraPositon = g_CamPosVec;

    // 回転
    {
        float sinV;
        float cosV;
        nn::util::SinCosEst( &sinV, &cosV, nn::util::DegreeToRadian(twist) );

        nn::util::VectorSetX( cameraPositon, nn::util::RadianToDegree(sinV) );
        nn::util::VectorSetZ( cameraPositon, nn::util::RadianToDegree(cosV) );
    }

    // スケール
    nn::util::VectorMultiply( cameraPositon, *cameraPositon, scale );
    nn::util::MatrixLookAtRightHanded( viewMatrix, *cameraPositon, g_CamLookAtVec, 0.0f );
}

//------------------------------------------------------------------------------
//  計算処理コールバック
//------------------------------------------------------------------------------
struct CalculateCallbackUserData
{
    bool isLoopExit;
    int* pFrame;
    int resourceId;
    nn::vfx::Handle* pEmitterSetHandle;
    nn::util::Matrix4x3fType* pViewMatrix;
    nn::util::Vector3fType* pCameraPos;
    float* pScaleValue;
    int groupID;
    int* pEmitterSetId;
    int emitterSetMaxCount;
    bool* pIsCircleMove;
    int* pMoveTime;
    float rotSpeed;
};

void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    CalculateCallbackUserData* pCallbackUserData = reinterpret_cast<CalculateCallbackUserData*>(pUserData);

    // 300フレーム毎に次のエミッタを再生する
    if ((*pCallbackUserData->pFrame != 0) && (*pCallbackUserData->pFrame % 300) == 0)
    {
        if (pCallbackUserData->emitterSetMaxCount > *pCallbackUserData->pEmitterSetId)
        {
            g_pVfxSystem->KillEmitterSetGroup(pCallbackUserData->groupID);
            (*pCallbackUserData->pEmitterSetId)++;
            g_pVfxSystem->CreateEmitterSetId(pCallbackUserData->pEmitterSetHandle, *pCallbackUserData->pEmitterSetId, pCallbackUserData->resourceId, pCallbackUserData->groupID);
            if (pCallbackUserData->pEmitterSetHandle->IsValid())
            {
                GetViewParamaters(pCallbackUserData->pEmitterSetHandle, pCallbackUserData->pViewMatrix, pCallbackUserData->pCameraPos, pCallbackUserData->pScaleValue, pCallbackUserData->pIsCircleMove);
                NN_SDK_LOG("%s\n", pCallbackUserData->pEmitterSetHandle->GetEmitterSet()->GetName());
            }
            (*pCallbackUserData->pMoveTime) = 0;
        }
    }
    (*pCallbackUserData->pFrame)++;

    // 最後のエミッタの再生時間
    if (pCallbackUserData->emitterSetMaxCount <= *pCallbackUserData->pEmitterSetId)
    {
        if ((*pCallbackUserData->pFrame % 300) == 0)
        {
            pCallbackUserData->isLoopExit = true;
            return;
        }
    }

    // 円形移動がある場合
    if (*pCallbackUserData->pIsCircleMove && pCallbackUserData->pEmitterSetHandle->IsValid())
    {
        nn::util::Matrix4x3fType matrix = pCallbackUserData->pEmitterSetHandle->GetEmitterSet()->GetMatrixSrt();

        float moveRot = pCallbackUserData->rotSpeed * (*pCallbackUserData->pMoveTime)++;
        float sinV;
        float cosV;
        nn::util::Vector3fType pos = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 0.0f, 0.0f);
        nn::util::SinCosEst(&sinV, &cosV, nn::util::DegreeToRadian(moveRot));
        nn::util::VectorSetX(&pos, sinV * 15.0f);
        nn::util::VectorSetZ(&pos, cosV * 15.0f);
        nn::util::MatrixSetTranslate(&matrix, pos);

        pCallbackUserData->pEmitterSetHandle->GetEmitterSet()->SetMatrix(matrix);
    }
}

//------------------------------------------------------------------------------
//  コマンド生成コールバック
//------------------------------------------------------------------------------
struct MakeCommandCallbackUserData
{
    nn::util::Matrix4x3fType* pViewMatrix;
    nn::util::Vector3fType* pCameraPos;
    float* pScaleValue;
    int groupID;
};

void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    MakeCommandCallbackUserData* pCallbackUserData = reinterpret_cast<MakeCommandCallbackUserData*>(pUserData);

    //------------------------------------------
    // VFXランタイムの計算処理
    //------------------------------------------
    g_pVfxSystem->BeginFrame();
    g_pVfxSystem->Calculate(pCallbackUserData->groupID, 1.0f, true);

    g_GraphicsFramework.BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer*    commandBuffer = g_GraphicsFramework.GetRootCommandBuffer(bufferIndex);
        nn::gfx::ColorTargetView*  pColorTargetView = g_GraphicsFramework.GetColorTargetView();
        nn::gfx::DepthStencilView* pDepthStecilView = g_GraphicsFramework.GetDepthStencilView();

        commandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);
        commandBuffer->ClearColor(pColorTargetView, 0.2f, 0.2f, 0.2f, 0.0f, nullptr);
        commandBuffer->ClearDepthStencil(pDepthStecilView, 1.0f, 0,
            nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        commandBuffer->SetRenderTargets(1, &pColorTargetView, pDepthStecilView);
        commandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());

        //------------------------------------------
        // グリッドの描画処理
        //------------------------------------------
        GridDraw(commandBuffer, *pCallbackUserData->pScaleValue);

        //------------------------------------------
        // カラー・深度バッファのコピー
        //------------------------------------------
        // カラーバッファをコピー
        commandBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
        CopyTextureBuffer(commandBuffer, g_ColorBufferCopy.GetTexture(), g_GraphicsFramework.GetColorBuffer());

        // 深度ステンシルバッファをコピー
        commandBuffer->FlushMemory(nn::gfx::GpuAccess_DepthStencil);
        CopyTextureBuffer(commandBuffer, g_DepthStencilBufferCopy.GetTexture(), g_GraphicsFramework.GetDepthStencilBuffer());

        int processingIndex = 0;

        // フレームバッファ/デプスバッファをvfxに設定
        g_pVfxSystem->SetFrameBufferTexture(processingIndex, g_ColorCopyBufferDescSlot);
        g_pVfxSystem->SetDepthBufferTexture(processingIndex, g_DepthStencilCopyDescSlot);

        //------------------------------------------
        // VFXランタイムの描画処理
        //------------------------------------------
        nn::vfx::ViewParam view;

        view.Set(*pCallbackUserData->pViewMatrix, g_ProjectionMatrix, *pCallbackUserData->pCameraPos, g_NearZ, g_FarZ);
        g_pVfxSystem->SwapBuffer();
        g_pVfxSystem->SetViewParam(processingIndex, &view);
        g_pVfxSystem->AddSortBuffer(processingIndex, pCallbackUserData->groupID);
        g_pVfxSystem->DrawSortBuffer(processingIndex, commandBuffer, true, nullptr);
    }
    g_GraphicsFramework.EndFrame(bufferIndex);
}


//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // VFXヒープ初期化
    int memSize = 1024 * 1024 * 256 ;
    g_HeapPtr = malloc( memSize );
    NN_ASSERT_NOT_NULL( g_HeapPtr );
    g_VfxHeap.Initialize( g_HeapPtr, memSize );

#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    // ファイルシステムのアロケータ設定
    nn::fs::SetAllocator(Allocate, Deallocate);
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    static const size_t   GraphicsSystemMemorySize = 8 * 1024 * 1024;
    g_GraphicsFramework.InitializeGraphicsSystem(GraphicsSystemMemorySize);
#endif

    //------------------------------------------
    // GraphicsFrameWorkの初期化
    //------------------------------------------
    g_GraphicsFrameworkInfo.SetDefault();
    g_GraphicsFrameworkInfo.SetDisplayWidth( 1280 );
    g_GraphicsFrameworkInfo.SetDisplayHeight( 720 );
    g_GraphicsFrameworkInfo.SetBufferCount(2);
    g_GraphicsFrameworkInfo.SetSwapChainBufferCount(2);
    g_GraphicsFrameworkInfo.SetColorBufferFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    g_GraphicsFrameworkInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 32 * 1024 * 1024);
    g_GraphicsFramework.Initialize(g_GraphicsFrameworkInfo);

    // Gfxの初期化
    InitializeGfx();

    // カメラ・プロジェクション行列初期化
    nn::util::MatrixLookAtRightHanded( &g_ViewMtx, g_CamPosVec, g_CamLookAtVec, 0.0f );
    const float fovy = nn::util::FloatPi / 3.0f;
    const float aspect = static_cast< float >(g_GraphicsFramework.GetDisplayWidth()) / static_cast< float >(g_GraphicsFramework.GetDisplayHeight());
    nn::util::MatrixPerspectiveFieldOfViewRightHanded( &g_ProjectionMatrix, fovy, aspect, g_NearZ, g_FarZ );

    // プリミティブレンダラの初期化
    InitializePrimitiveRenderer();

    //------------------------------------------
    // VFXランタイムの初期化
    //------------------------------------------
    // Vfx System 生成
    nn::vfx::Config  config;
    config.SetEffectHeap( &g_VfxHeap );
    config.SetEffectDynamicHeap( &g_VfxHeap );
    config.SetGfxDevice(g_GraphicsFramework.GetDevice());
    config.SetEmitterSetCount( 256 );
    config.SetEmitterCount( 1024 );
    config.SetDoubleBufferSize( 1024 * 32 );
    config.SetStripeCount( 128 );
    config.SetSuperStripeCount( 128 );
    config.SetGpuBufferSize( 1024 * 1024 * 64 );
    g_pVfxSystem = new nn::vfx::System( config );
    g_pVfxSystem->RegisterTextureViewToDescriptorPool( AllocateDescriptorSlotForTexture, &g_GraphicsFramework );
    g_pVfxSystem->RegisterSamplerToDescriptorPool(     AllocateDescriptorSlotForSampler, &g_GraphicsFramework );


    //------------------------------------------
    // VFXバイナリロード
    //------------------------------------------
    char* mountRomCacheBuffer = nullptr;
    {
        size_t cacheSize = 0;
        nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
        NN_ASSERT(result.IsSuccess());

        mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
        NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

        result = nn::fs::MountRom("Contents", mountRomCacheBuffer, cacheSize );
        NN_ABORT_UNLESS_RESULT_SUCCESS( result );
    }

    const char* DemoParticleFilePath = "Contents:/demo.ptcl";
    void* binary = nullptr;
    size_t binarySize = 0;
    bool resultFileLoad = FileLoad( &binary, &binarySize, DemoParticleFilePath, &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment );
    if( resultFileLoad == false )
    {
        return;
    }

    // VFXシステムにバイナリを設定する
    int resourceId = 0;
    g_pVfxSystem->EntryResource( &g_VfxHeap, binary, resourceId, true );

    // リソース内テクスチャをディスクリプタプールに登録。textureCount 枚数分登録が実行されます。
    //int textureCount = g_pVfxSystem->GetResource(resourceId)->->GetTextureCount();
    g_pVfxSystem->GetResource( resourceId )->RegisterTextureViewToDescriptorPool( AllocateDescriptorSlotForTexture, &g_GraphicsFramework );

    //------------------------------------------
    // エフェクト生成
    //------------------------------------------
    nn::vfx::Handle          emitterSetHandle;
    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Vector3fType   cameraPos;
    float                    scaleValue         = 0;
    int                      groupID            = 0;
    int                      emitterSetId       = 0;
    int                      emitterSetMaxCount = 0;
    bool                     isCircleMove       = false;
    int                      moveTime           = 0;
    float                    rotSpeed           = nn::util::FloatPi * 2.0f / 5;

    g_pVfxSystem->CreateEmitterSetId( &emitterSetHandle, emitterSetId, resourceId, groupID );
    if ( emitterSetHandle.IsValid() )
    {
        emitterSetMaxCount = g_pVfxSystem->GetResource( resourceId )->GetEmitterSetCount() - 1;
        NN_SDK_LOG( "%s\n", emitterSetHandle.GetEmitterSet()->GetName() );

        GetViewParamaters( &emitterSetHandle, &viewMatrix, &cameraPos, &scaleValue, &isCircleMove );
    }

    int frame = 0;

    // フレームワーク設定
    CalculateCallbackUserData calculateUserData;
    calculateUserData.isLoopExit = false;
    calculateUserData.pFrame = &frame;
    calculateUserData.resourceId = resourceId;
    calculateUserData.pEmitterSetHandle = &emitterSetHandle;
    calculateUserData.pViewMatrix = &viewMatrix;
    calculateUserData.pCameraPos = &cameraPos;
    calculateUserData.pScaleValue = &scaleValue;
    calculateUserData.groupID = groupID;
    calculateUserData.pEmitterSetId = &emitterSetId;
    calculateUserData.emitterSetMaxCount = emitterSetMaxCount;
    calculateUserData.pIsCircleMove = &isCircleMove;
    calculateUserData.pMoveTime = &moveTime;
    calculateUserData.rotSpeed = rotSpeed;

    MakeCommandCallbackUserData makeCommandUserData;
    makeCommandUserData.pViewMatrix = &viewMatrix;
    makeCommandUserData.pCameraPos = &cameraPos;
    makeCommandUserData.pScaleValue = &scaleValue;
    makeCommandUserData.groupID = groupID;

    g_GraphicsFramework.SetCalculateCallback(CalculateCallback, &calculateUserData);
    g_GraphicsFramework.SetMakeCommandCallback(MakeCommandCallback, &makeCommandUserData);

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    MSG  msg;
#endif
    // メインループ
    for( ; ; )
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
        //------------------------------------------
        // Windows メッセージ処理
        //------------------------------------------
        if( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE ) )
        {
            TranslateMessage( &msg );
            if( msg.message == WM_QUIT )
            {
                break;
            }
            DispatchMessage( &msg );
        }
#endif

        g_GraphicsFramework.ProcessFrame();
        if (calculateUserData.isLoopExit)
        {
            break;
        }
    }
    g_GraphicsFramework.QueueFinish();

    //------------------------------------------
    // VFXランタイムの終了処理
    //------------------------------------------
    g_pVfxSystem->GetResource( resourceId )->UnregisterTextureViewFromDescriptorPool( FreeDescriptorSlotForTexture, &g_GraphicsFramework );
    g_pVfxSystem->UnregisterTextureViewFromDescriptorPool( FreeDescriptorSlotForTexture, &g_GraphicsFramework );
    g_pVfxSystem->UnregisterSamplerFromDescriptorPool( FreeDescriptorSlotForSampler, &g_GraphicsFramework );
    g_pVfxSystem->ClearResource( &g_VfxHeap, resourceId );
    delete g_pVfxSystem;

    nn::fs::Unmount("Contents");
    delete[] mountRomCacheBuffer;

    // プリミティブレンダラの終了処理
    FinalizePrimitiveRenderer();

    // Gfxの終了処理
    FinalizeGfx();

    g_VfxHeap.Finalize();
    free( g_HeapPtr );

    return;
} // NOLINT(readability/fn_size)
