﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{VfxManualEmitterSet/main.cpp,PageSampleVfxManualEmitterSet}
 *
 * @brief
 *  マニュアル放出を使用したエフェクト表示のサンプルプログラム
 */

/**
 * @page PageSampleVfxManualEmitterSet マニュアル放出を使用したエフェクト表示
 * @tableofcontents
 *
 * @image html Applications\VfxManualEmitterSet\VfxManualEmitterSet.png
 *
 * @brief
 *  サンプルプログラム VfxManualEmitterSet の解説です。
 *
 * @section PageSampleVfxManualEmitterSet_SectionBrief 概要
 *  nn::vfx のマニュアル放出機能を使用してエフェクトバイナリを再生するサンプルです。
 *
 *  エフェクトランタイムの機能の使い方については、
 *  @ref nn::vfx "Vfx の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleVfxManualEmitterSet_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/VfxManualEmitterSet
 *  Samples/Sources/Applications/VfxManualEmitterSet @endlink 以下にあります。
 *
 * @section PageSampleVfxManualEmitterSet_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleVfxManualEmitterSet_SectionHowToOperate 操作方法
 *  このサンプルを実行するとエフェクトが自動的に再生され、一定回数描画した後終了します。
 *
 * @section PageSampleVfxManualEmitterSet_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleVfxManualEmitterSet_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleVfxManualEmitterSet_SectionDetail 解説
 *
 * @subsection PageSampleVfxManualEmitterSet_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  VfxManualEmitterSet/main.cpp
 *  @includelineno VfxManualEmitterSet/main.cpp
 *
 * @subsection PageSampleVfxManualEmitterSet_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *  - Gfxの初期化
 *  - カメラ・プロジェクション行列の初期化
 *  - VFXランタイムの初期化
 *  - VFXバイナリロード
 *  - エフェクト生成
 *  - ループ開始
 *  - VFXランタイムの計算処理
 *  - ここで一定期間ごとにパーティクルを手動放出
 *  - VFXランタイムの描画処理
 *  - ループ開始に戻る
 *  - VFXランタイムの終了処理
 *  - Gfxの終了処理
 *
 */

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Result.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs.h>
#include <nn/vfx.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/hws/hws_Message.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
#include <nv/nv_MemoryManagement.h>
#endif


//------------------------------------------------------------------------------
//  nn::fs用メモリ取得用関数
//------------------------------------------------------------------------------
void* Allocate(size_t size)
{
    return malloc(size);
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ解放用関数
//------------------------------------------------------------------------------
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    free(p);
}

//------------------------------------------------------------------------------
//  nn::fs を利用したファイルロード関数
//------------------------------------------------------------------------------
bool FileLoad(
    void**           pOutBuffer,
    size_t*          pOutBufferSize,
    const char*      filePath,
    nn::vfx::Heap*   pAllocator,
    int              alignment)
{
    char filePathName[nn::fs::EntryNameLengthMax];
    memcpy(reinterpret_cast< void* >(&filePathName), filePath, strlen(filePath) + 1);

    nn::fs::FileHandle fsHandle;
    nn::Result ret = nn::fs::OpenFile(&fsHandle, static_cast< char* >(filePathName), static_cast< nn::fs::OpenMode >(nn::fs::OpenMode_Read));
    if (ret.IsFailure())
    {
        return false;
    }

    int64_t fsSize;
    nn::fs::GetFileSize(&fsSize, fsHandle);
    *pOutBufferSize = static_cast< size_t >(fsSize);

    *pOutBuffer = pAllocator->Alloc(*pOutBufferSize + 4, alignment);
    if (!(*pOutBuffer))
    {
        nn::fs::CloseFile(fsHandle);
        return false;
    }

    memset(*pOutBuffer, 0, *pOutBufferSize + 4);
    nn::fs::ReadFile(fsHandle, 0, *pOutBuffer, *pOutBufferSize);
    nn::fs::CloseFile(fsHandle);

    return true;
}



//------------------------------------------------------------------------------
//  GraphicsFramework
//------------------------------------------------------------------------------
nns::gfx::GraphicsFramework::FrameworkInfo g_GraphicsFrameworkInfo;
nns::gfx::GraphicsFramework                g_GraphicsFramework;


//------------------------------------------------------------------------------
//  VfxManualEmitter デモ関連
//------------------------------------------------------------------------------
nn::util::Vector3fType      g_CamPosVec = NN_UTIL_VECTOR_3F_INITIALIZER(0, 30, 50);     // カメラ位置
nn::util::Vector3fType      g_CamLookAtVec = NN_UTIL_VECTOR_3F_INITIALIZER(0, 0, 0);    // カメラ注視点
nn::util::Vector3fType      g_CamUpVec = NN_UTIL_VECTOR_3F_INITIALIZER(0, 1, 0);        // カメラ上方向
nn::util::Matrix4x3fType    g_ViewMtx;                                                  // ビュー行列
nn::util::Matrix4x4fType    g_ProjectionMatrix;                                         // 射影変換行列
float                       g_NearZ = 0.1f;                                             // Z Near
float                       g_FarZ = 10000.0f;                                          // Z Far
nn::vfx::System*            g_pVfxSystem = nullptr;                                     // nn::vfx システム
nn::util::Vector3fType      g_ParticleScale = NN_UTIL_VECTOR_3F_INITIALIZER(1, 1, 1);   // マニュアル放出時に与えるパーティクルスケール値
void*                       g_HeapPtr;                                                  // エフェクトヒープバッファポインタ
nn::vfx::StandardHeap       g_VfxHeap;                                                  // エフェクトヒープ

//------------------------------------------------------------------------------
//  テクスチャデスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateDescriptorSlotForTexture(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(dstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  テクスチャデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForTexture(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);

    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*dstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateDescriptorSlotForSampler(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(dstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForSampler(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);

    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        int slotIndex = pPool->GetDescriptorSlotIndex(*dstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }
}

//------------------------------------------------------------------------------
//  パーティクル生成時に呼ばれるコールバック
//------------------------------------------------------------------------------
bool ParticleEmitCallback(nn::vfx::ParticleEmitArg& arg)
{
    // ユーザーデータを取り出す
    nn::util::Vector3fType* scale = reinterpret_cast< nn::util::Vector3fType* >(arg.pUserData);

    // スケール値を設定
    arg.SetInitialScale(*scale);

    // GPU パーティクルでワンエミッタを行う場合、生成時コールバック以外のコールバックは呼ばれないため、
    // ここでユーザーデータを null にしておく必要があります。
    arg.pUserData = nullptr;

    return true;
}

//------------------------------------------------------------------------------
//  パーティクル削除時に呼ばれるコールバック
//------------------------------------------------------------------------------
bool ParticleRemoveCallback(nn::vfx::ParticleRemoveArg& arg)
{
    //------------------------------------------------------------------------------
    // ※このコールバックは CPU エミッタの時しか呼ばれません。
    //------------------------------------------------------------------------------

    // ユーザーデータを使用する場合、ここで中身を適切に削除し NULL を入れておく必要がある
    // このサンプルでは外部の変数のポインタを持っているだけなので、NULL 代入だけしておく
    arg.pUserData = nullptr;

    return true;
}

//------------------------------------------------------------------------------
//  計算処理コールバック
//------------------------------------------------------------------------------
struct CalculateCallbackUserData
{
    int frame;
    nn::vfx::Handle* pEmitterSetHandle;
};

void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    CalculateCallbackUserData* pCallbackUserData = reinterpret_cast<CalculateCallbackUserData*>(pUserData);

    const float r = 16.0f;
    const int interval = 10;
    const int roundFrame = 300;
    if (pCallbackUserData->frame % interval == 0)
    {
        const float t = static_cast< float >(pCallbackUserData->frame % roundFrame) / static_cast< float >(roundFrame);
        const float radian = nn::util::DegreeToRadian(360.0f * t);
        nn::util::Vector3fType pos =
        { {
                nn::util::SinEst(radian) * r,
                nn::util::CosEst(radian) * r,
                0
            } };

        // 初期スケール値をユーザーデータとして渡す
        const float scale = 5.0f * t + 2.0f;
        nn::util::VectorSet(&g_ParticleScale, scale, scale, 1.0f);    // このケースではパーティクルは板ポリゴンなので Z は無視されます。

        //------------------------------------------------------------------------------
        // EmitParticle() の第二引数として、グローバル変数 g_ParticleScale のアドレスを渡しています。
        // 1フレーム中に 2つ以上のパーティクルをマニュアル放出する場合で、
        // かつパーティクルごとに異なるユーザーデータを持たせたい場合は、
        // 「1フレーム中に放出する最大数（ 上述の maxParticleEmitCount ）」分のユーザーデータを用意して、
        // それぞれのアドレスを EmitParticle() に渡す必要があります。
        //------------------------------------------------------------------------------
        pCallbackUserData->pEmitterSetHandle->GetEmitterSet()->EmitParticle(pos, &g_ParticleScale);
    }
}

//------------------------------------------------------------------------------
//  コマンド生成コールバック
//------------------------------------------------------------------------------
void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    int* pGroupID = reinterpret_cast<int*>(pUserData);

    //------------------------------------------
    // VFXランタイムの計算処理
    //------------------------------------------
    g_pVfxSystem->BeginFrame();
    g_pVfxSystem->Calculate(*pGroupID, 1.0f, true);

    nn::gfx::CommandBuffer*    commandBuffer = g_GraphicsFramework.GetRootCommandBuffer(bufferIndex);
    nn::gfx::ColorTargetView*  pColorTargetView = g_GraphicsFramework.GetColorTargetView();
    nn::gfx::DepthStencilView* pDepthStecilView = g_GraphicsFramework.GetDepthStencilView();

    g_GraphicsFramework.BeginFrame(bufferIndex);
    {
        commandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);
        commandBuffer->ClearColor(pColorTargetView, 0.2f, 0.2f, 0.2f, 0.0f, nullptr);
        commandBuffer->ClearDepthStencil(pDepthStecilView, 1.0f, 0,
            nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        commandBuffer->SetRenderTargets(1, &pColorTargetView, pDepthStecilView);
        commandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());

        //------------------------------------------
        // VFXランタイムの描画処理
        //------------------------------------------
        int processingIndex = 0;
        nn::vfx::ViewParam view;
        view.Set(g_ViewMtx, g_ProjectionMatrix, g_CamPosVec, g_NearZ, g_FarZ);
        g_pVfxSystem->SwapBuffer();
        g_pVfxSystem->SetViewParam(processingIndex, &view);
        g_pVfxSystem->AddSortBuffer(processingIndex, *pGroupID);
        g_pVfxSystem->DrawSortBuffer(processingIndex, commandBuffer, true, nullptr);
    }
    g_GraphicsFramework.EndFrame(bufferIndex);
}

//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // ヒープ初期化
    int memSize = 1024 * 1024 * 32 ;
    g_HeapPtr = malloc( memSize );
    NN_ASSERT_NOT_NULL( g_HeapPtr );
    g_VfxHeap.Initialize( g_HeapPtr, memSize );

#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    // ファイルシステムのアロケータ設定
    nn::fs::SetAllocator(Allocate, Deallocate);
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    static const size_t   GraphicsSystemMemorySize = 8 * 1024 * 1024;
    g_GraphicsFramework.InitializeGraphicsSystem( GraphicsSystemMemorySize );
#endif

    //------------------------------------------
    // GraphicsFrameWorkの初期化
    //------------------------------------------
    g_GraphicsFrameworkInfo.SetDefault();
    g_GraphicsFrameworkInfo.SetDisplayWidth( 1280 );
    g_GraphicsFrameworkInfo.SetDisplayHeight( 720 );
    g_GraphicsFrameworkInfo.SetBufferCount(2);
    g_GraphicsFrameworkInfo.SetSwapChainBufferCount(2);
    g_GraphicsFramework.Initialize( g_GraphicsFrameworkInfo );

    // カメラ・プロジェクション行列初期化
    nn::util::MatrixLookAtRightHanded(&g_ViewMtx, g_CamPosVec, g_CamLookAtVec, 0.0f);
    const float fovy = nn::util::FloatPi / 3.0f;
    const float aspect = static_cast< float >(g_GraphicsFramework.GetDisplayWidth()) / static_cast< float >(g_GraphicsFramework.GetDisplayHeight());
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&g_ProjectionMatrix, fovy, aspect, g_NearZ, g_FarZ);

    //------------------------------------------
    // VFXランタイムの初期化
    //------------------------------------------
    nn::vfx::Config  config;
    config.SetEffectHeap(&g_VfxHeap);
    config.SetEffectDynamicHeap(&g_VfxHeap);
    config.SetGfxDevice(g_GraphicsFramework.GetDevice());
    config.SetEmitterCount(4);
    config.SetEmitterSetCount(4);
    config.SetResourceCount(2);
    config.SetStripeCount(10);
    config.SetSuperStripeCount(10);
    config.SetParticleSortBufferCount(100);
    config.SetProcessingCount(1);
    g_pVfxSystem = new nn::vfx::System(config);
    g_pVfxSystem->RegisterTextureViewToDescriptorPool( AllocateDescriptorSlotForTexture, &g_GraphicsFramework );
    g_pVfxSystem->RegisterSamplerToDescriptorPool( AllocateDescriptorSlotForSampler, &g_GraphicsFramework );

    //------------------------------------------
    // VFXバイナリロード
    //------------------------------------------
    char* mountRomCacheBuffer = nullptr;
    {
        size_t cacheSize = 0;
        nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
        NN_ASSERT(result.IsSuccess());

        mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
        NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

        result = nn::fs::MountRom( "Contents", mountRomCacheBuffer, cacheSize );
        NN_ABORT_UNLESS_RESULT_SUCCESS( result );
    }

    const char* DemoParticleFilePath = "Contents:/demo.ptcl";
    void* binary = nullptr;
    size_t binarySize = 0;
    bool result = FileLoad(&binary, &binarySize, DemoParticleFilePath, &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment);
    if (result == false)
    {
        return;
    }

    // VFXシステムにバイナリを設定する
    int resourceId = 0;
    g_pVfxSystem->EntryResource(&g_VfxHeap, binary, resourceId, true);
    g_pVfxSystem->GetResource(resourceId)->RegisterTextureViewToDescriptorPool(AllocateDescriptorSlotForTexture, &g_GraphicsFramework);

    //------------------------------------------
    // ManualEmitterSet 生成
    //------------------------------------------
    nn::vfx::Handle emitterSetHandle;
    const int groupID = 0;
    const int emitterSetId = g_pVfxSystem->SearchEmitterSetId( "Fire_01", resourceId );
    const int maxParticleCount = 64;
    const int maxParticleEmitCount = maxParticleCount;
    nn::vfx::CallbackSet callbackSet;
    callbackSet.particleEmit = ParticleEmitCallback;
    callbackSet.particleRemove = ParticleRemoveCallback;
    nn::vfx::EmitReservationInfo reservationList[maxParticleCount];
    g_pVfxSystem->CreateManualEmitterSetId(&emitterSetHandle, emitterSetId, resourceId, groupID, maxParticleCount, maxParticleEmitCount, reservationList, nullptr, &callbackSet);

    // フレームワーク設定
    CalculateCallbackUserData calculateUserData;
    calculateUserData.pEmitterSetHandle = &emitterSetHandle;
    int groupid = groupID;
    g_GraphicsFramework.SetCalculateCallback(CalculateCallback, &calculateUserData);
    g_GraphicsFramework.SetMakeCommandCallback(MakeCommandCallback, &groupid);

    // メインループ
    for (int frame = 0; frame < 1800; ++frame)
    {
        calculateUserData.frame = frame;
        g_GraphicsFramework.ProcessFrame();
    }
    g_GraphicsFramework.QueueFinish();

    //------------------------------------------
    // VFXランタイムの終了処理
    //------------------------------------------
    g_pVfxSystem->GetResource(resourceId)->UnregisterTextureViewFromDescriptorPool(FreeDescriptorSlotForTexture, &g_GraphicsFramework);
    g_pVfxSystem->UnregisterTextureViewFromDescriptorPool(FreeDescriptorSlotForTexture, &g_GraphicsFramework);
    g_pVfxSystem->UnregisterSamplerFromDescriptorPool(FreeDescriptorSlotForSampler, &g_GraphicsFramework);
    g_pVfxSystem->ClearResource(&g_VfxHeap, resourceId);
    delete g_pVfxSystem;

    nn::fs::Unmount("Contents");
    delete[] mountRomCacheBuffer;

    // GraphicsFrameWork の破棄
    g_GraphicsFramework.Finalize();

    g_VfxHeap.Finalize();
    free( g_HeapPtr );

    return;
}//NOLINT(impl/function_size)

