﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{VfxCustomShader/main.cpp,PageSampleVfxCustomShader}
 *
 * @brief
 *  カスタムシェーダを利用したエフェクト表示のサンプルプログラム
 */

/**
 * @page PageSampleVfxCustomShader カスタムシェーダを利用したエフェクト表示
 * @tableofcontents
 *
  * @image html Applications\VfxCustomShader\VfxCustomShader.png
 *
 * @brief
 *  サンプルプログラム VfxCustomShader の解説です。
 *
 * @section PageSampleVfxCustomShader_SectionBrief 概要
 *  nn::vfx のカスタムシェーダ機能を利用したエフェクトバイナリを再生するサンプルです。
 *
 *  エフェクトランタイムの機能の使い方については、
 *  @ref nn::vfx "Vfx の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleVfxCustomShader_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/VfxCustomShader
 *  Samples/Sources/Applications/VfxCustomShader @endlink 以下にあります。
 *
 * @section PageSampleVfxCustomShader_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleVfxCustomShader_SectionHowToOperate 操作方法
 *  このサンプルを実行するとエフェクトが自動的に再生され、一定回数描画した後終了します。
 *
 * @section PageSampleVfxCustomShader_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleVfxCustomShader_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleVfxCustomShader_SectionDetail 解説
 *
 * @subsection PageSampleVfxCustomShader_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  VfxCustomShader/main.cpp
 *  @includelineno VfxCustomShader/main.cpp
 *
 * @subsection PageSampleVfxCustomShader_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *  - Gfxの初期化
 *  - カメラ・プロジェクション行列の初期化
 *  - VFXランタイムの初期化
 *  - VFXバイナリロード
 *  - カスタムシェーダのコールバック登録
 *  - エフェクト生成
 *  - ループ開始
 *  - VFXランタイムの計算処理
 *  - VFXランタイムの描画処理
 *  - ループ開始に戻る
 *  - VFXランタイムの終了処理
 *  - Gfxの終了処理
 *
 */

#include <nn/TargetConfigs/build_Base.h>

#include "Resources/CustomShader/CustomShader1.h"
#include "Resources/CustomShader/CustomShader2.h"

#include <nn/nn_Result.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs.h>
#include <nn/gfx/util/gfx_DescriptorPoolAllocator.h>
#include <nn/vfx.h>
#include <nn/vi.h>
#include <nn/vfx/viewer/vfx_Viewer.h>
#include <nn/util/util_StringUtil.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/hws/hws_Message.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
#include <cafe/hio.h>
#else
#include <nn/htcs.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

//------------------------------------------------------------------------------
//  nn::fs用メモリ取得用関数
//------------------------------------------------------------------------------
void* Allocate(size_t size)
{
    return malloc(size);
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ解放用関数
//------------------------------------------------------------------------------
void Deallocate(void* ptr, size_t size)
{
    NN_UNUSED(size);
    free(ptr);
}

//------------------------------------------------------------------------------
//  GraphicsFramework
//------------------------------------------------------------------------------
nns::gfx::GraphicsFramework::FrameworkInfo g_GraphicsFrameworkInfo;
nns::gfx::GraphicsFramework                g_GraphicsFramework;

//------------------------------------------------------------------------------
//  nn::fs を利用したファイルロード関数
//------------------------------------------------------------------------------
bool FileLoad(
    void**           pOutBuffer,
    size_t*          pOutBufferSize,
    const char*      filePath,
    nn::vfx::Heap*   pAllocator,
    size_t           alignment,
    size_t           granularity )
{
    char filePathName[nn::fs::EntryNameLengthMax];
    memcpy( reinterpret_cast< void* >( &filePathName ), filePath, strlen( filePath ) + 1 );

    nn::fs::FileHandle fsHandle;
    nn::Result ret = nn::fs::OpenFile( &fsHandle, static_cast< char* >( filePathName ), static_cast< nn::fs::OpenMode >( nn::fs::OpenMode_Read ) );
    if ( ret.IsFailure() )
    {
        return false;
    }

    int64_t fsSize;
    nn::fs::GetFileSize( &fsSize, fsHandle );
    *pOutBufferSize = static_cast< size_t >( fsSize );

    *pOutBufferSize = nn::util::align_up( *pOutBufferSize, granularity );

    *pOutBuffer = pAllocator->Alloc( *pOutBufferSize, alignment );
    if ( !( *pOutBuffer ) )
    {
        nn::fs::CloseFile( fsHandle );
        return false;
    }

    memset( *pOutBuffer, 0, *pOutBufferSize );
    nn::fs::ReadFile( fsHandle, 0, *pOutBuffer, static_cast< size_t >( fsSize ) );
    nn::fs::CloseFile( fsHandle );
    return true;
}

//------------------------------------------------------------------------------
//  VfxCustomShader デモ関連
//------------------------------------------------------------------------------
nn::util::Vector3fType          g_CamPosVec         = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 30, 50 );  // カメラ位置
nn::util::Vector3fType          g_CamLookAtVec      = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );    // カメラ注視点
nn::util::Vector3fType          g_CamUpVec          = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 1, 0 );    // カメラ上方向
nn::util::Matrix4x3fType        g_ViewMtx;                              // ビュー行列
nn::util::Matrix4x4fType        g_ProjectionMatrix;                     // 射影変換行列
float                           g_NearZ             = 0.1f;             // Z Near
float                           g_FarZ              = 10000.0f;         // Z Far
nn::vfx::System*                g_pVfxSystem        = nullptr;          // nn::vfx システム

void*                           g_HeapPtr;                              // ヒープバッファポインタ
nn::vfx::StandardHeap           g_VfxHeap;                              // エフェクトヒープ

//------------------------------------------------------------------------------
//  テクスチャデスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateDescriptorSlotForTexture(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(dstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  テクスチャデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForTexture(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);

    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*dstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateDescriptorSlotForSampler(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(dstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForSampler(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);

    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        int slotIndex = pPool->GetDescriptorSlotIndex(*dstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }
}

//---------------------------------------------------------------------------
//  カスタムシェーダコールバック
//---------------------------------------------------------------------------
bool CustomShaderSampleCallback( nn::vfx::RenderStateSetArg& arg )
{
    if ( !arg.pEmitter->GetEmitterResource()->m_pCustomShaderParam ) return false;

    nn::vfx::detail::Shader*    shader      = arg.GetShader();
    nn::vfx::TemporaryBuffer*   pTempBuffer = arg.pDrawParameterArg->m_pDrawTempBuffer;

    struct CustomShaderSampleUniformBlock
    {
        nn::util::Float4      param0;
        nn::util::Float4      param1;
    };

    // コンスタントバッファ用のバッファを確保
    nn::gfx::GpuAddress address;
    CustomShaderSampleUniformBlock* param = static_cast<CustomShaderSampleUniformBlock *>( pTempBuffer->Map( &address, sizeof(CustomShaderSampleUniformBlock) ) );
    if( param )
    {
        param->param0.x = 0.0f;
        param->param0.y = 0.0f;
        param->param0.z = 0.0f;
        param->param0.w = 0.0f;
        param->param1.x = 0.0f;
        param->param1.y = 0.0f;
        param->param1.z = 0.0f;
        param->param1.w = 0.0f;

        // コンスタントバッファを設定
        // シェーダ上では、
        shader->BindCustomShaderUniformBlock( arg.pCommandBuffer, nn::vfx::CustomShaderConstantBufferIndex_0, &address, sizeof(CustomShaderSampleUniformBlock) );
    }

    // カスタムUIで指定されたパラメータ をセット
    // 下記メソッドで設定されたコンスタントバッファは、シェーダ上では、sysCustomShaderReservedUniformBlockParamXX で参照できます。
    nn::vfx::BindReservedCustomShaderConstantBuffer( arg );

    return true;
}

//------------------------------------------------------------------------------
//  コマンド生成コールバック
//------------------------------------------------------------------------------
void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    int* pGroupID = reinterpret_cast<int*>(pUserData);

    //------------------------------------------
    // VFXランタイムの計算処理
    //------------------------------------------
    g_pVfxSystem->BeginFrame();
    g_pVfxSystem->Calculate(*pGroupID, 1.0f, true);

    nn::gfx::CommandBuffer*    commandBuffer = g_GraphicsFramework.GetRootCommandBuffer(bufferIndex);
    nn::gfx::ColorTargetView*  pColorTargetView = g_GraphicsFramework.GetColorTargetView();
    nn::gfx::DepthStencilView* pDepthStecilView = g_GraphicsFramework.GetDepthStencilView();

    g_GraphicsFramework.BeginFrame(bufferIndex);
    {
        commandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);
        commandBuffer->ClearColor(pColorTargetView, 0.2f, 0.2f, 0.2f, 0.0f, nullptr);
        commandBuffer->ClearDepthStencil(pDepthStecilView, 1.0f, 0,
            nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        commandBuffer->SetRenderTargets(1, &pColorTargetView, pDepthStecilView);
        commandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());

        //------------------------------------------
        // VFXランタイムの描画処理
        //------------------------------------------
        int processingIndex = 0;
        nn::vfx::ViewParam view;
        view.Set(g_ViewMtx, g_ProjectionMatrix, g_CamPosVec, g_NearZ, g_FarZ);
        g_pVfxSystem->SwapBuffer();
        g_pVfxSystem->SetViewParam(processingIndex, &view);
        g_pVfxSystem->AddSortBuffer(processingIndex, *pGroupID);
        g_pVfxSystem->DrawSortBuffer(processingIndex, commandBuffer, true, nullptr);
    }
    g_GraphicsFramework.EndFrame(bufferIndex);
}

//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // VFXヒープ初期化
    int memSize = 1024 * 1024 * 32 ;
    g_HeapPtr = malloc( memSize );
    NN_ASSERT_NOT_NULL( g_HeapPtr );
    g_VfxHeap.Initialize( g_HeapPtr, memSize );

#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    // ファイルシステムのアロケータ設定
    nn::fs::SetAllocator( Allocate, Deallocate );
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    static const size_t   GraphicsSystemMemorySize = 8 * 1024 * 1024;
    g_GraphicsFramework.InitializeGraphicsSystem(GraphicsSystemMemorySize);
#endif

    //------------------------------------------
    // GraphicsFrameWorkの初期化
    //------------------------------------------
    g_GraphicsFrameworkInfo.SetDefault();
    g_GraphicsFrameworkInfo.SetDisplayWidth( 1280 );
    g_GraphicsFrameworkInfo.SetDisplayHeight( 720 );
    g_GraphicsFrameworkInfo.SetBufferCount(2);
    g_GraphicsFrameworkInfo.SetSwapChainBufferCount(2);
    g_GraphicsFramework.Initialize(g_GraphicsFrameworkInfo);

    // カメラ・プロジェクション行列初期化
    nn::util::MatrixLookAtRightHanded(&g_ViewMtx, g_CamPosVec, g_CamLookAtVec, 0.0f);
    const float fovy = nn::util::FloatPi / 3.0f;
    const float aspect = static_cast< float >(g_GraphicsFramework.GetDisplayWidth()) / static_cast< float >(g_GraphicsFramework.GetDisplayHeight());
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&g_ProjectionMatrix, fovy, aspect, g_NearZ, g_FarZ);


    //------------------------------------------
    // VFXランタイムの初期化
    //------------------------------------------
    // Vfx System 生成
    nn::vfx::Config  config;
    config.SetEffectHeap( &g_VfxHeap );
    config.SetEffectDynamicHeap( &g_VfxHeap );
    config.SetGfxDevice( g_GraphicsFramework.GetDevice() );
    config.SetEmitterCount( 36 );
    config.SetEmitterSetCount( 8 );
    config.SetResourceCount( 2 );
    config.SetStripeCount( 10 );
    config.SetSuperStripeCount( 10 );
    config.SetParticleSortBufferCount( 100 );
    config.SetProcessingCount( 1 );
    g_pVfxSystem = new nn::vfx::System( config );
    g_pVfxSystem->RegisterTextureViewToDescriptorPool( AllocateDescriptorSlotForTexture, &g_GraphicsFramework);
    g_pVfxSystem->RegisterSamplerToDescriptorPool(     AllocateDescriptorSlotForSampler, &g_GraphicsFramework);


    //------------------------------------------
    // VFXバイナリロード
    //------------------------------------------
    char* mountRomCacheBuffer = nullptr;
    {
        size_t cacheSize = 0;
        nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
        NN_ASSERT(result.IsSuccess());

        mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
        NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

        result = nn::fs::MountRom("Contents", mountRomCacheBuffer, cacheSize );
        NN_ABORT_UNLESS_RESULT_SUCCESS( result );
    }

    void* binary = nullptr;
    void* textureBinary = nullptr;
    nn::gfx::MemoryPool effectBinaryMemoryPool;
    nn::vfx::CallbackSet callback;
    nn::vfx::Handle emitterSetHandle;
    size_t effectBinarySize = 0;
    size_t textureBinarySize = 0;
    nn::gfx::ResTextureFile* pResTextureFile = nullptr;
    int resourceId = 0;
    int groupID = 0;
    int emitterSetId = 0;

    // エフェクトバイナリをロードした空間をメモリプールに割り当てる
    nn::gfx::MemoryPoolInfo info;
    info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_ShaderCode );
    size_t granularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( g_GraphicsFramework.GetDevice(), info );

    // エフェクトバイナリをロード
    {
        const char* DemoParticleFilePath = "Contents:/demo.ptcl";
        bool result = FileLoad( &binary, &effectBinarySize, DemoParticleFilePath, &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment, granularity );
        if ( result == false )
        {
            NN_ASSERT( result, "Vfx BinaryFile Load Failed." );
        }
    }

    // エフェクトテクスチャバイナリをロード
    {
        const char* DemoTextureFilePath = "Contents:/texture.bntx";
        bool result = FileLoad( &textureBinary, &textureBinarySize, DemoTextureFilePath, &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment, granularity );
        if (result == false)
        {
            NN_ASSERT(result, "Vfx TextureBinaryFile Load Failed.");
        }
    }

    //------------------------------------------
    // 個別にロードしたバイナリをバインドする
    //------------------------------------------

    // ロードしたエフェクトバイナり空間をメモリプールとして初期化
    info.SetPoolMemory( binary, effectBinarySize );
    effectBinaryMemoryPool.Initialize( g_GraphicsFramework.GetDevice(), info );

    // VFXシステムにバイナリを設定する
    g_pVfxSystem->EntryResource( &g_VfxHeap, binary, &effectBinaryMemoryPool, effectBinarySize, 0, resourceId, true );
    g_pVfxSystem->GetResource(resourceId)->RegisterTextureViewToDescriptorPool(AllocateDescriptorSlotForTexture, &g_GraphicsFramework);

    // エフェクトテクスチャバイナリを初期化
    pResTextureFile = nn::gfx::ResTextureFile::ResCast( textureBinary );
    pResTextureFile->Initialize( g_GraphicsFramework.GetDevice() );

    for (int idxTexture = 0, textureCount = pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture)
    {
        pResTextureFile->GetResTexture(idxTexture)->Initialize(g_GraphicsFramework.GetDevice());
    }

    // ディスクリプタスロットを割り当てる
    for (int i = 0, textureCount = pResTextureFile->GetTextureDic()->GetCount();
        i < textureCount; i++)
    {
        nn::gfx::ResTexture* resTexture = pResTextureFile->GetResTexture(i);
        nn::gfx::TextureView& textureView = *(static_cast<nn::gfx::TextureView*>(resTexture->GetTextureView()));

        nn::gfx::DescriptorSlot slot;
        AllocateDescriptorSlotForTexture(&slot, textureView, &g_GraphicsFramework);

        resTexture->SetUserDescriptorSlot(slot);
    }

    // リソースとテクスチャをバインドする
    g_pVfxSystem->GetResource(resourceId)->BindExternalResTextureFile(pResTextureFile);

    //------------------------------------------
    // カスタムシェーダ
    //------------------------------------------
    callback.renderStateSet = CustomShaderSampleCallback;
    g_pVfxSystem->SetCallback( nn::vfx::CallbackId_CustomShader1, callback );

    //------------------------------------------
    // エフェクト生成
    //------------------------------------------
    emitterSetId = g_pVfxSystem->SearchEmitterSetId( "CustomShader", resourceId );
    g_pVfxSystem->CreateEmitterSetId( &emitterSetHandle, emitterSetId, resourceId, groupID );

    // フレームワーク設定
    g_GraphicsFramework.SetMakeCommandCallback(MakeCommandCallback, &groupID);

    // メインループ
    for( int frame = 0; frame < 60 * 1000; ++frame )
    {
        g_GraphicsFramework.ProcessFrame();
    }
    g_GraphicsFramework.QueueFinish();

    //------------------------------------------
    // VFXランタイムの終了処理
    //------------------------------------------
    if ( g_pVfxSystem->GetResource( resourceId ) )
    {
        g_pVfxSystem->GetResource( resourceId )->UnregisterTextureViewFromDescriptorPool( FreeDescriptorSlotForTexture, &g_GraphicsFramework );
    }
    g_pVfxSystem->UnregisterTextureViewFromDescriptorPool(FreeDescriptorSlotForTexture, &g_GraphicsFramework);
    g_pVfxSystem->UnregisterSamplerFromDescriptorPool(FreeDescriptorSlotForSampler, &g_GraphicsFramework);
    g_pVfxSystem->ClearResource( &g_VfxHeap, resourceId );
    delete g_pVfxSystem;

    // エフェクトバイナリを配置したメモリプールの破棄
    if ( binary )
    {
        effectBinaryMemoryPool.Finalize( g_GraphicsFramework.GetDevice() );
        g_VfxHeap.Free( binary );
    }

    // エフェクトテクスチャリソースの解放
    if ( textureBinary )
    {
        pResTextureFile->Finalize( g_GraphicsFramework.GetDevice() );
        g_VfxHeap.Free( textureBinary );
    }

    // GraphicsFrameWork の破棄
    g_GraphicsFramework.Finalize();
    nn::fs::Unmount( "Contents" );
    delete[] mountRomCacheBuffer;
    g_VfxHeap.Finalize();
    free( g_HeapPtr );

    return;
}  // NOLINT(readability/fn_size)

