﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{SjisEncoding.cpp,PageSampleUtilConvertSjis}
 *
 * @brief
 *  Shift_JIS と UTF-16 の相互変換関数の実装
 */

#include <nn/nn_Assert.h>

#include "SjisEncoding.h"
#include "SjisToUtf16Table.h"
#include "Utf16ToSjisTable.h"

namespace {

//
// 変換テーブルから変換先の文字コードを探し、 pDst に格納します。
//
void ReferUtf16ToSjisTable(unsigned char *pDst, uint16_t src)
{
    unsigned char utf16Upper = static_cast<unsigned char>(src >> 8);    // 探索する変換元の UTF-16 上位 8 bit
    unsigned char tableIndex = static_cast<unsigned char>(src & 0xFFu);

    // g_pUtf16ToSjisTable の UTF-16 の下位 8bit が一致する範囲を二分探索
    int begin = g_pUtf16ToSjisIndex[tableIndex];
    int end = g_pUtf16ToSjisIndex[tableIndex + 1] - 1;
    const unsigned char* pMiddle;

    while (begin <= end)
    {
        int middleIndex = (begin + end) / 2;
        int middleVal = g_pUtf16ToSjisTable[middleIndex * 3];
        pMiddle = &g_pUtf16ToSjisTable[middleIndex * 3];

        if (*pMiddle == utf16Upper)
        {
            pDst[0] = *(++pMiddle);
            pDst[1] = *(++pMiddle);
            return;
        }

        if (middleVal > utf16Upper)
        {
            end = middleIndex - 1;
        }
        else if (middleVal < utf16Upper)
        {
            begin = middleIndex + 1;
        }
    }

    // 変換できなかった
    pDst[0] = pDst[1] = 0;
}

//
// UTF-16 1 文字を Shift_JIS に変換します。
//
void ConvertCharacterUnicodeToSjis(unsigned char *pDst, uint16_t src)
{
    if (src < 0x0080u)
    {
        // 制御文字と ASCII
        pDst[0] = static_cast<unsigned char>(src);
        pDst[1] = 0x00u;
        return;
    }
    if (src >= 0xFF61u && src <= 0xFF9Fu)
    {
        // 半角カナ
        pDst[0] = static_cast<unsigned char>(src - 0xFEC0u);
        pDst[1] = 0;
        return;
    }
    if (src < 0xD800u || src >= 0xF900u)
    {
        // ひらがな、カタカナ、 CJK 統合漢字は変換テーブルから引く
        ReferUtf16ToSjisTable(pDst, src);
        return;
    }
    if (src >= 0xE000u && src <= 0xE757u)
    {
        // Private Use Area
        // Shift_JIS の外字コードに変換
        uint32_t modNum = (src - 0xE000u) % (94 * 2);
        uint32_t divNum = (src - 0xE000u - modNum) / (94 * 2);
        pDst[0] = static_cast<unsigned char>(divNum + 0xF0u);
        pDst[1] = static_cast<unsigned char>(modNum < 0x3Fu ? modNum + 0x40u : modNum + 0x80u - 0x3Fu);
        return;
    }
    if (src >= 0xF8F0 && src <= 0xF8F3)
    {
        // Private Use Area の一部を Windows の WideCharToMultiByte の挙動に合わせる
        if (src == 0xF8F0)
        {
            pDst[0] = 0xA0;
        }
        else
        {
            pDst[0] = static_cast<unsigned char>(src - 0xF8F1 + 0xFD);
        }
        pDst[1] = 0;
        return;
    }
    // 変換できなかった
    pDst[0] = pDst[1] = 0x00u;
}

}   // namespace

//
// 文字コードを Shift_JIS から UTF-16 に変換します。終端文字まで変換されます。
//
nn::util::CharacterEncodingResult ConvertStringSjisToUtf16(uint16_t *pDst, int dstLength, const char *pSrc)
{
    NN_ASSERT_NOT_NULL(pDst);
    NN_ASSERT_NOT_NULL(pSrc);
    NN_ASSERT_GREATER_EQUAL(dstLength, 0);

    int index = 0;

    while (*pSrc != '\0')
    {
        unsigned char sjisUpper = static_cast<unsigned char>(*pSrc);
        unsigned char sjisLower;
        uint16_t utf16Char;

        if (index >= dstLength)
        {
            return nn::util::CharacterEncodingResult_InsufficientLength;
        }

        if ((sjisUpper >= 0x81u && sjisUpper <= 0x9Fu) || (sjisUpper >= 0xE0u && sjisUpper <= 0xFCu))
        {
            // JIS 漢字の範囲はテーブルを利用して変換
            ++pSrc;
            sjisLower = static_cast<unsigned char>(*pSrc);
            if (sjisLower == '\0')
            {
                return nn::util::CharacterEncodingResult_InvalidFormat;
            }
            if ((sjisLower >= 0x40u && sjisLower <= 0x7Eu) || (sjisLower >= 0x80u && sjisLower <= 0xFCu))
            {
                if (sjisUpper < 0xF0u)
                {
                    // JIS 漢字
                    int tableIndex = 0;
                    tableIndex += 94 * 2 * (sjisUpper < 0xA0u ? sjisUpper - 0x81u : sjisUpper - 0xE0u + 0x1Fu);
                    tableIndex += (sjisLower < 0x7Fu ? sjisLower - 0x40u : sjisLower - 0x80u + 0x3Fu);
                    utf16Char = g_pSjisToUtf16Table[tableIndex];
                }
                else if (sjisUpper >= 0xFAu)
                {
                    // IBM 拡張文字
                    int tableIndex = 0;
                    tableIndex += 94 * 2 * (sjisUpper - 0xE0u + 0x1Fu - 0x0Au);
                    tableIndex += (sjisLower < 0x7Fu ? sjisLower - 0x40u : sjisLower - 0x80u + 0x3Fu);
                    utf16Char = g_pSjisToUtf16Table[tableIndex];
                }
                else
                {
                    // 外字コード
                    // Unicode の Private Use Area に変換
                    int utf16Index = 0;
                    utf16Index += 94 * 2 * (sjisUpper - 0xF0u);
                    utf16Index += (sjisLower < 0x7Fu ? sjisLower - 0x40u : sjisLower - 0x80u + 0x3Fu);
                    utf16Char = static_cast<uint16_t>(0xE000u + utf16Index);
                }
            }
            else
            {
                return nn::util::CharacterEncodingResult_InvalidFormat;
            }
        }
        else
        {
            if (sjisUpper <= 0x80u)
            {
                // Windows の MultiByteToWideChar に合わせ 0x80 も変換可能とする
                utf16Char = static_cast<uint16_t>(sjisUpper);
            }
            else if (sjisUpper == 0xA0u)
            {
                // Windows の MultiByteToWideChar に合わせ 0xA0 も変換可能とする
                utf16Char = 0xF8F0u;
            }
            else if (sjisUpper < 0xE0u)
            {
                // 半角カナ
                utf16Char = static_cast<uint16_t>(sjisUpper + 0xFEC0u);
            }
            else if (sjisUpper >= 0xFDu)
            {
                // Windows の MultiByteToWideChar に合わせ 0xFDXX-0xFFXX も変換可能とする
                utf16Char = static_cast<uint16_t>(0xF8F1u + (sjisUpper - 0xFDu));
            }
            else
            {
                return nn::util::CharacterEncodingResult_InvalidFormat;
            }
        }

        if (utf16Char == 0x0000u)
        {
            return nn::util::CharacterEncodingResult_InvalidFormat;
        }
        *pDst = utf16Char;
        ++pDst;
        ++index;
        ++pSrc;
    }
    return nn::util::CharacterEncodingResult_Success;
}

//
// 文字コードを UTF-16 から Shift_JIS に変換します。終端文字まで変換されます。
//
nn::util::CharacterEncodingResult ConvertStringUtf16ToSjis(char *pDst, int dstLength, const uint16_t *pSrc)
{
    NN_ASSERT_NOT_NULL(pDst);
    NN_ASSERT_NOT_NULL(pSrc);
    NN_ASSERT_GREATER_EQUAL(dstLength, 0);

    int index = 0;

    while (*pSrc != L'\0')
    {
        uint16_t utf16Char = *pSrc;
        unsigned char pSjisChar[2];
        unsigned char sjisUpper, sjisLower;

        if (index >= dstLength)
        {
            return nn::util::CharacterEncodingResult_InsufficientLength;
        }

        // 1 文字の変換
        ConvertCharacterUnicodeToSjis(pSjisChar, utf16Char);

        sjisUpper = pSjisChar[0];
        sjisLower = pSjisChar[1];

        if (sjisUpper == 0x00u)
        {
            return nn::util::CharacterEncodingResult_InvalidFormat;
        }
        if (sjisLower == 0x00u)
        {
            // 1 バイトの場合
            *pDst = static_cast<char>(sjisUpper);
            ++pDst;
        }
        else
        {
            // 2 バイトの場合
            if (dstLength - index < 2)
            {
                return nn::util::CharacterEncodingResult_InsufficientLength;
            }
            *pDst = sjisUpper;
            ++pDst;
            *pDst = sjisLower;
            ++pDst;
            ++index;
        }
        ++index;
        ++pSrc;
    }

    return nn::util::CharacterEncodingResult_Success;
}
