﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/migration/migration_AsyncContext.h>
#include <nn/migration/migration_Result.h>
#include <nn/util/util_FormatString.h>

#include "UserMigration_ScreenIo.h"

#include <nn/nn_Log.h>

class SequenceBase
{
public:
    class DialogBase
    {
    private:
        Window::Components m_Components;
        bool m_CloseRequired;

    protected:
        template <typename DerivedType>
        static void CloseHandler(void* data, size_t dataSize) NN_NOEXCEPT
        {
            NN_UNUSED(dataSize);
            auto& obj = *reinterpret_cast<typename std::remove_reference<DerivedType>::type*>(data);
            obj.RequestClose();
        }

        DialogBase() NN_NOEXCEPT
            : m_CloseRequired(false)
        {
        }
        void RequestClose() NN_NOEXCEPT
        {
            m_CloseRequired = true;
        }
        Window::Components* GetComponentsPtr() NN_NOEXCEPT
        {
            return &m_Components;
        }

    public:
        virtual ~DialogBase() NN_NOEXCEPT {}
        virtual void Update() NN_NOEXCEPT {}

        bool IsClosed() const NN_NOEXCEPT
        {
            return m_CloseRequired;
        }
        const Window::Components* GetComponentsPtr() const NN_NOEXCEPT
        {
            return &m_Components;
        }
    };

    struct AsyncTask
    {
        nn::migration::AsyncContext context;
        bool (*resultHandler)(void*, size_t, nn::Result);
        void *data;
        size_t dataSize;
    };

    struct SyncTask
    {
        std::function<nn::Result(void)> context;
        bool (*resultHandler)(void*, size_t, nn::Result);
        void *data;
        size_t dataSize;
    };

    struct DialogCaller
    {
        SequenceBase* _obj;

        void PushError(const char* message) NN_NOEXCEPT
        {
            _obj->PushError(message);
        }
        void PushAsyncProgress(const char* label, AsyncTask&& asyncTask) NN_NOEXCEPT
        {
            _obj->PushAsyncProgress(label, std::move(asyncTask));
        }
        void PushSyncProgress(const char* label, SyncTask&& syncTask) NN_NOEXCEPT
        {
            _obj->PushSyncProgress(label, std::move(syncTask));
        }
    };

private:
    Window& m_Window;
    DialogBase* m_DialogStack[8];
    int m_DialogStackDepth;
    bool m_ErrorOccurred;
    bool m_CloseRequired;
    DialogCaller m_DialogCaller;

    void PushError(const char* message) NN_NOEXCEPT;
    void PushAsyncProgress(const char* label, AsyncTask&& asyncTask) NN_NOEXCEPT;
    void PushSyncProgress(const char* label, SyncTask&& syncTask) NN_NOEXCEPT;
    void Pop() NN_NOEXCEPT;

protected:
    // 派生クラスで利用
    explicit SequenceBase(Window& window) NN_NOEXCEPT;
    void NotifyCurrentSceneUpdate() NN_NOEXCEPT;
    bool IsErrorOccurred() const NN_NOEXCEPT;
    DialogCaller& GetDialogCallerRef() NN_NOEXCEPT;

    // 派生クラスでの実装が必要
    virtual const Window::Components* GetCurrentSceneComponentsPtr() const NN_NOEXCEPT = 0;
    virtual void UpdateImpl() NN_NOEXCEPT = 0;

public:
    virtual ~SequenceBase() NN_NOEXCEPT;
    void Update() NN_NOEXCEPT;
    void RequestClose() NN_NOEXCEPT;
    bool IsClosed() const NN_NOEXCEPT;
};

template <typename SceneKindType>
class SceneBase
{
public:
    typedef SceneKindType SceneKind;

private:
    SequenceBase::DialogCaller* m_pDialogCaller;
    Window::Components m_Components;
    bool m_SceneEnded;
    SceneKind m_NextScene;

protected:
    template <typename DerivedType, SceneKind NextScene>
    static void EndHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<typename std::remove_reference<DerivedType>::type*>(data);
        obj.EndScene(NextScene);
    }

    SceneBase() NN_NOEXCEPT
        : m_pDialogCaller(nullptr)
        , m_SceneEnded(false)
    {
    }
    explicit SceneBase(SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : m_pDialogCaller(&dialogCaller)
        , m_SceneEnded(false)
    {
    }
    void PushError(const char* message) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pDialogCaller);
        m_pDialogCaller->PushError(message);
    }
    void PushAsyncProgress(const char* label, SequenceBase::AsyncTask&& asyncTask) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pDialogCaller);
        m_pDialogCaller->PushAsyncProgress(label, std::move(asyncTask));
    }
    void PushSyncProgress(const char* label, SequenceBase::SyncTask&& sync) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pDialogCaller);
        m_pDialogCaller->PushSyncProgress(label, std::move(sync));
    }
    void EndScene(SceneKind nextScene) NN_NOEXCEPT
    {
        m_NextScene = nextScene;
        m_SceneEnded = true;
    }

    bool HandleResult(nn::Result r) NN_NOEXCEPT
    {
        if (!r.IsSuccess())
        {
            if (nn::migration::ResultCanceled::Includes(r))
            {
                PushError("Operation is canceled.");
            }
            else
            {
                char message[64];
                nn::util::SNPrintf(message, sizeof(message), "Result: %03d-%04d (%08x)", r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
                PushError(message);
            }
            return true;
        }
        return false;
    }

    Window::Components* GetComponentsPtr() NN_NOEXCEPT
    {
        return &m_Components;
    }

public:
    virtual void Update() NN_NOEXCEPT {}

    bool IsEnded() const NN_NOEXCEPT
    {
        return m_SceneEnded;
    }
    SceneKind GetNextScene() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_SceneEnded);
        return m_NextScene;
    }
    const Window::Components* GetComponentsPtr() const NN_NOEXCEPT
    {
        return &m_Components;
    }
};
