﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/vi.h>

#include <nn/gfx.h>

#include <nn/nn_Assert.h>

#include <nn/ui2d.h>
#include <nn/font.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>
#include <nn/mem.h>
#include <nn/os.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <filesystem>
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/init.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

// Calculate をマルチスレッドで処理するフラグです。
static bool g_MultiThreading = true;
// マルチスレッド動作時に GpuBuffer を共有するフラグです。
static bool g_ShareGpuBuffer = true;
// 並列処理を行うスレッドの数です。
static uint32_t g_NumberOfThread = 1;

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------
static int g_AllocationCount = 0;

// アプリケーションヒープとして確保されるメモリ領域の大きさ(byte)
static const size_t ApplicationHeapSize = 128 * 1024 * 1024;
// メモリプールとして確保されるメモリ領域の大きさ(byte)
static const size_t MemoryPoolSize          = 64 * 1024 * 1024;   // 64k の倍数である必要がある
static const size_t MemoryPoolSizeInvisible = 32 * 1024 * 1024;   // 64k の倍数である必要がある
// コマンドバッファで使用するメモリ領域の大きさ(byte)
static const size_t CommandBufferMemorySize = 16 * 1024 * 1024;
static const size_t CommandBufferControlMemorySize = 256 * 1024;
static const size_t NestedCommandBufferMemorySize = 4 * 1024 * 1024;

// 各種ディスクリプタプールの最大数
static const int DescriptorPoolSlotCount = 2048;

// 画面サイズ
static const uint32_t ScreenWidth = 1280;
static const uint32_t ScreenHeight = 720;

// アプリケーションヒープ
static nn::util::BytePtr g_pMemoryHeap(NULL);
static nn::mem::StandardAllocator g_ApplicationHeapHandle;

// コマンドバッファコントロール用メモリ
static void* g_pCommandBufferControlMemory = NULL;

// メモリプール
static void* g_MemoryPoolMemory;
static nn::gfx::MemoryPool  g_MemoryPool;
static ptrdiff_t g_MemoryPoolOffset;

static void* g_MemoryPoolInvisibleMemory;
static nn::gfx::MemoryPool  g_MemoryPoolInvisible;
static ptrdiff_t g_MemoryPoolInvisibleOffset;

// vi ハンドル
static nn::vi::Display* g_pDisplay;
// vi レイヤハンドル
static nn::vi::Layer* g_pLayer;

// gfx デバイス
static nn::gfx::Device g_Device;
// gfx スワップチェイン
static nn::gfx::SwapChain g_SwapChain;
// gfx キュー
static nn::gfx::Queue g_Queue;
// gfx コマンドバッファ
static nn::gfx::CommandBuffer g_CommandBuffer;
static ptrdiff_t g_CommandBufferStartOffset;
// gfx ビューポートシザーステート
static nn::gfx::ViewportScissorState g_ViewportScissor;
// gfx ラスタライザステート
static nn::gfx::RasterizerState g_RasterizerState;
// gfx デプスステンシルステート
static nn::gfx::DepthStencilState g_DepthStencilState;
// gfx ブレンドステート
static nn::gfx::BlendState  g_BlendState;
// gfx カラーバッファ用テクスチャ
static nn::gfx::Texture g_ColorBuffer;
// gfx カラーターゲットビュー
static nn::gfx::ColorTargetView g_ColorBufferView;
// gfx デプスステンシル用テクスチャ
static nn::gfx::Texture g_DepthStencilBuffer;
// gfx デプスステンシルビュー
static nn::gfx::DepthStencilView g_DepthStencilView;

// ui2d レイアウト
static nn::ui2d::Layout* g_pLayout = NULL;
// ui2d アーカイブリソースアクセッサ
static nn::ui2d::ArcResourceAccessor* g_pArcResourceAccessor = NULL;
// レイアウトアーカイブバイナリデータ
static void* g_pLayoutArchiveBinary;
// レイアウトで使用するフォントリソース
static nn::font::ResFont* g_pFont = NULL;
// ui2d が描画に使用するグラフィックスリソース
static nn::ui2d::GraphicsResource g_GraphicsResource;
// ui2d のコンスタントバッファのダブルバッファリング用インデックス
static uint32_t     g_ConstantBufferIndex = 0;

// サンプルで使用するレイアウトデータリスト
const char* g_pLayoutDataFileName[] =
{
    "Projection_Sample_00.bflyt",
    "Projection_Sample_01.bflyt",
    "Write_Sample.bflyt",
    "Font_Sample.bflyt",
};

//------------------------------------------------------------------------------
// レイアウトで使用するコンスタントバッファの情報
class LayoutBufferContext
{
public:
    // コンストラクタ
    LayoutBufferContext() :
        m_pMemoryPoolPtr(NULL)
    {
    }

    // デストラクタ
    ~LayoutBufferContext()
    {
    }

    // 初期化処理
    void Initialize(nn::ui2d::DrawInfo& drawInfo, nn::gfx::Device* pDevice, const nn::ui2d::BuildResultInformation& buildResultInformation);

    // 終了処理
    void Finalize(nn::gfx::Device* pDevice);

    // ui2d 用コンスタントバッファを取得します。
    nn::font::GpuBuffer& GetUi2dConstantBuffer()
    {
        return m_Ui2dConstantBuffer;
    }

    // font 用コンスタントバッファを取得します。
    nn::font::GpuBuffer& GetFontConstantBuffer()
    {
        return m_FontConstantBuffer;
    }
private:
    // ui2d 用コンスタントバッファ
    nn::font::GpuBuffer m_Ui2dConstantBuffer;
    // font 用コンスタントバッファ
    nn::font::GpuBuffer m_FontConstantBuffer;
    // GPU バッファのためのメモリプールポインタ
    nn::util::BytePtr   m_pMemoryPoolPtr;
    // GPU バッファのためのメモリプール
    nn::gfx::MemoryPool m_MemoryPool;
};

// レイアウトで使用するコンスタントバッファ
static LayoutBufferContext* g_pLayoutBufferContext = NULL;

//------------------------------------------------------------------------------
// スレッドで分散処理するためのコマンドバッファ
struct NestedCommandBuffer
{
    // 描画コマンドを発行するコマンドバッファ
    nn::gfx::CommandBuffer* m_pCommandBuffer;
    // コマンドバッファのメモリプール先頭からのオフセット
    ptrdiff_t   m_StartOffset;
    // コマンドバッファコントロールメモリ
    void* m_pControlMemory;

    // デフォルト値を設定します
    void SetDefault()
    {
        m_pCommandBuffer = NULL;
        m_StartOffset = 0;
        m_pControlMemory = NULL;
    }
};

// スレッドごとのコマンドバッファをまとめる情報
static NestedCommandBuffer* g_pNestedCommandBuffers = NULL;

//------------------------------------------------------------------------------
// レイアウトデータをスレッドで分散処理ための情報
struct LayoutThreadContext
{
    // スレッドで更新する nn::ui2d::Layout
    nn::ui2d::Layout*   m_pLayout;
    // 更新処理時に使用する nn::ui2d::DrawInfo
    nn::ui2d::DrawInfo* m_pDrawInfo;
    // 更新するコンスタントバッファのインデックス
    uint32_t            m_ConstantBufferIndex;
    // 描画コマンドを発行するコマンドバッファ
    NestedCommandBuffer* m_pCommandBuffer;

    // デフォルト値を設定します
    void SetDefault()
    {
        m_pLayout = NULL;
        m_pDrawInfo = NULL;
        m_ConstantBufferIndex = 0;
        m_pCommandBuffer = NULL;
    }
};

// レイアウトデータをスレッドで分散処理ための情報
LayoutThreadContext*        g_pLayoutThreadContext = NULL;

//------------------------------------------------------------------------------
// スレッド関連
const size_t              threadStackSize = 1024 * 1024;

struct ThreadContext
{
    // スレッドのスタック領域
    void* m_pThreadStack;
    // スレッドハンドル
    nn::os::ThreadType  m_Thread;

    // デフォルト値を設定します
    void SetDefault()
    {
        m_pThreadStack = NULL;
    }
};

// スレッド関連の情報
static ThreadContext* g_pThreadContext = NULL;

// ui2d 描画情報
nn::ui2d::DrawInfo*         g_pDrawInfo = NULL;

//------------------------------------------------------------------------------
// DescriptorPool 関連
const  int                             DescriptorPoolStartSlot = 256;
static nn::gfx::DescriptorPool         g_TextureDescriptorPool;
static nn::gfx::DescriptorPool         g_SamplerDescriptorPool;
static int                             g_SamplerSlotCount = DescriptorPoolStartSlot;
static int                             g_TextureSlotCount = DescriptorPoolStartSlot;

//------------------------------------------------------------------------------
//  アプリケーションヒープからメモリを確保する
//------------------------------------------------------------------------------
static void* AllocateFromApplicationHeap(size_t size, size_t alignment)
{
    void* ptr = g_ApplicationHeapHandle.Allocate(size, alignment);
    if (ptr != NULL)
    {
    g_AllocationCount++;
    }
    return ptr;
}

static void* AllocateFromApplicationHeap(size_t size)
{
    return AllocateFromApplicationHeap(size, 4);
}

//------------------------------------------------------------------------------
//  アプリケーションヒープから確保したメモリを開放する
//------------------------------------------------------------------------------
static void DeallocateApplicationHeap(void* ptr)
{
    if (ptr == NULL)
    {
        return ;
    }

    g_AllocationCount--;
    g_ApplicationHeapHandle.Free(ptr);
}

template <typename TObject>
TObject* AllocAndConstruct()
{
    return new(AllocateFromApplicationHeap(sizeof(TObject))) TObject();
}

template <typename TObject>
void DestructAndFree(TObject* ptr)
{
    ptr->TObject::~TObject();
    DeallocateApplicationHeap(ptr);
}

template <typename TObject>
TObject* NewArray(int  count)
{
    void* pMem = AllocateFromApplicationHeap(sizeof(TObject) * count);

    TObject *const objAry = static_cast<TObject*>(pMem);

    for (int  i = 0; i < count; ++i)
    {
        new (&objAry[i]) TObject();
    }

    return objAry;
}

template <typename TObject>
void DeleteArray(TObject objAry[], int  count)
{
    if (objAry)
    {
        for (int  i = 0; i < count; ++i)
        {
            objAry[i].~TObject();
        }
        DeallocateApplicationHeap(objAry);
    }
}

//------------------------------------------------------------------------------
// ui2d 用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* Ui2dAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    return AllocateFromApplicationHeap(size, alignment);
}

static void Ui2dDeallocateFunction(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// nn::fs用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* FsAllocateFunction( size_t size )
{
    return AllocateFromApplicationHeap(size);
}

static void FsDeallocateFunction( void* ptr, size_t size )
{
    NN_UNUSED( size );
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

//------------------------------------------------------------------------------
// オフセットの切り上げ
//------------------------------------------------------------------------------
static ptrdiff_t AlignupOffset(ptrdiff_t offset, size_t alignment)
{
    return ((offset + alignment - 1) / alignment) * alignment;
}

//------------------------------------------------------------------------------
void LayoutBufferContext::Initialize(nn::ui2d::DrawInfo& drawInfo, nn::gfx::Device* pDevice, const nn::ui2d::BuildResultInformation& buildResultInformation)
{
    // バッファの再セットアップ
    // ui2d 描画に使用する各種 gfx バッファのためのメモリプールを作成
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );

    // BuildWithName でレイアウトデータをセットアップした際に BuildResultInfomation に収集した
    // 各種バッファのサイズを満たすメモリプールを作成する。

    const size_t constantBufferTotalSize = buildResultInformation.requiredUi2dConstantBufferSize
                                            + buildResultInformation.requiredFontConstantBufferSize;

    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(
                                                                pDevice,
                                                                memPoolInfo);

    const size_t memoryPoolSize = nn::util::align_up(
                                   constantBufferTotalSize,
                                    memoryPoolGranularity);

    const size_t MemoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(
                                                                pDevice,
                                                                memPoolInfo);

    // ui2d 用コンスタントバッファ、font 用コンスタントバッファの順番で
    // メモリプール上に領域を確保する。

    if (memoryPoolSize > 0)
    {
        size_t  totalMemoryPoolSize = memoryPoolSize * nn::ui2d::ConstantBufferCount;
        void*   pMemoryPoolMemory = Ui2dAllocateFunction(totalMemoryPoolSize, MemoryPoolAlignment, NULL);

        m_pMemoryPoolPtr.Reset(pMemoryPoolMemory);
        memPoolInfo.SetPoolMemory(m_pMemoryPoolPtr.Get(), totalMemoryPoolSize);

        m_MemoryPool.Initialize(pDevice, memPoolInfo);
    }

    size_t  bufferOffset = 0;

    // ui2d 用のコンスタントバッファを作成する。
    if (buildResultInformation.requiredUi2dConstantBufferSize > 0)
    {
        nn::font::GpuBuffer::InitializeArg  arg;

        arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
        arg.SetBufferSize(buildResultInformation.requiredUi2dConstantBufferSize);
        arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
        arg.SetMemoryPool(&m_MemoryPool);
        arg.SetMemoryPoolOffset(bufferOffset);
        arg.SetAllocator(Ui2dAllocateFunction, NULL);
        arg.SetAllocateSyncFlag(g_ShareGpuBuffer);

        // フォント描画のためのコンスタントバッファを作成
        m_Ui2dConstantBuffer.Initialize(pDevice, arg);
        drawInfo.SetUi2dConstantBuffer(&m_Ui2dConstantBuffer);
        bufferOffset += buildResultInformation.requiredUi2dConstantBufferSize * nn::ui2d::ConstantBufferCount;

    }
    else
    {
        drawInfo.SetUi2dConstantBuffer(NULL);
    }

    // ui2d 内で使用される font 用のコンスタントバッファを作成する。
    if (buildResultInformation.requiredFontConstantBufferSize > 0)
    {
        nn::font::GpuBuffer::InitializeArg  arg;

        arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
        arg.SetBufferSize(buildResultInformation.requiredFontConstantBufferSize);
        arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
        arg.SetMemoryPool(&m_MemoryPool);
        arg.SetMemoryPoolOffset(bufferOffset);
        arg.SetAllocator(Ui2dAllocateFunction, NULL);
        arg.SetAllocateSyncFlag(g_ShareGpuBuffer);

        m_FontConstantBuffer.Initialize(pDevice, arg);
        drawInfo.SetFontConstantBuffer(&m_FontConstantBuffer);
        bufferOffset += buildResultInformation.requiredFontConstantBufferSize * nn::ui2d::ConstantBufferCount;
    }
    else
    {
        drawInfo.SetFontConstantBuffer(NULL);
    }
}

//------------------------------------------------------------------------------
void LayoutBufferContext::Finalize(nn::gfx::Device* pDevice)
{
    if (m_Ui2dConstantBuffer.IsInitialized())
    {
        m_Ui2dConstantBuffer.Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    }
    if (m_FontConstantBuffer.IsInitialized())
    {
        m_FontConstantBuffer.Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    }

    m_MemoryPool.Finalize(pDevice);
    DeallocateApplicationHeap(m_pMemoryPoolPtr.Get());
}



//------------------------------------------------------------------------------
// gfx で使用するメモリプール初期化関数
//------------------------------------------------------------------------------
void InitializeMemoryPool()
{
    // 単純さのため全体で 2種類のメモリプールを使います
    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );
        info.SetPoolMemory( g_MemoryPoolMemory, MemoryPoolSize );

        g_MemoryPool.Initialize( &g_Device, info );
        g_MemoryPoolOffset = 0;
    }

    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);
        info.SetPoolMemory( g_MemoryPoolInvisibleMemory, MemoryPoolSizeInvisible );

        g_MemoryPoolInvisible.Initialize( &g_Device, info );
        g_MemoryPoolInvisibleOffset = 0;
    }
}

//-----------------------------------------------------------------------------
//  gfx のディスクリプタプール初期化関数
//-----------------------------------------------------------------------------
static void InitialzieGfxDescriptorPool_()
{
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
        info.SetSlotCount( DescriptorPoolSlotCount );

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );

        g_TextureDescriptorPool.Initialize( &g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, size );
        g_MemoryPoolOffset += size;
    }

    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler);
        info.SetSlotCount( DescriptorPoolSlotCount );

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );

        g_SamplerDescriptorPool.Initialize( &g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, size );
        g_MemoryPoolOffset += size;
    }
}

//------------------------------------------------------------------------------

bool RegisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(pUserData);

    const int slot = g_TextureSlotCount;
    NN_ASSERT(g_TextureSlotCount < DescriptorPoolSlotCount);

    g_TextureDescriptorPool.BeginUpdate();
    {
        g_TextureDescriptorPool.SetTextureView( slot, &textureView);
        g_TextureSlotCount++;
    }
    g_TextureDescriptorPool.EndUpdate();

    g_TextureDescriptorPool.GetDescriptorSlot(pDstSlot, slot);

    return true;
}

//------------------------------------------------------------------------------

bool RegisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(pUserData);

    const int slot = g_SamplerSlotCount;
    NN_ASSERT(g_SamplerSlotCount < DescriptorPoolSlotCount);
    g_SamplerDescriptorPool.BeginUpdate();
    {
        g_SamplerDescriptorPool.SetSampler( slot, &sampler);
        g_SamplerSlotCount++;
    }
    g_SamplerDescriptorPool.EndUpdate();

    g_SamplerDescriptorPool.GetDescriptorSlot(pDstSlot, slot);

    return true;
}

//------------------------------------------------------------------------------

void UnregisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    // このサンプルではなにも行いません。
    // ユーザーがアプリケーションに適した方法で、ディスクリタプールの解放処理を行ってください。
    NN_UNUSED(pDstSlot);
    NN_UNUSED(textureView);
    NN_UNUSED(pUserData);
}

//------------------------------------------------------------------------------

void UnregisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    // このサンプルではなにも行いません。
    // ユーザーがアプリケーションに適した方法で、ディスクリタプールの解放処理を行ってください。
    NN_UNUSED(pDstSlot);
    NN_UNUSED(sampler);
    NN_UNUSED(pUserData);
}

//------------------------------------------------------------------------------
// メモリを確保してファイルを読み込みます
//------------------------------------------------------------------------------
void* ReadFileWithAllocate(const char* pFileName, size_t alignment)
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;
    int64_t fileSize;

    result = nn::fs::OpenFile(&fileHandle, pFileName, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    void* binary = AllocateFromApplicationHeap(static_cast<size_t>(fileSize), alignment);
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, binary, static_cast<size_t>(fileSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::fs::CloseFile(fileHandle);

    return binary;
}

//------------------------------------------------------------------------------
// gfx の一部のクラスで共通するメモリ割り当て処理
//------------------------------------------------------------------------------
template <typename TGfxClass, typename TGfxInfo>
static void
SetGfxMemoryTmpl_(TGfxClass * pGfxObject, TGfxInfo& info )
{
    size_t memorySize = TGfxClass::GetRequiredMemorySize( info );
    void* pWork = AllocateFromApplicationHeap(memorySize);

    pGfxObject->SetMemory( pWork, memorySize );
}

//-----------------------------------------------------------------------------
// コマンドバッファーのコールバック処理
//-----------------------------------------------------------------------------
void OutOfMemoryEventCallback( nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::OutOfMemoryEventArg& arg )
{
    NN_UNUSED( pCommandBuffer );
    NN_UNUSED( arg );

    // このサンプルでは予めコマンドバッファ用に十分な大きさのメモリを確保しているためメモリが不足することはありません。
    // ユーザーがアプリケーションに適した方法で追加のメモリを確保し、AddCommandMemory(...) や AddControlMemory(...) を使って
    // コマンドバッファにメモリを追加してください。
    NN_ABORT("CommandMemory shortage!\n");
}

#if defined( NN_BUILD_CONFIG_OS_WIN )
//------------------------------------------------------------------------------
static void InitializeGfxWin_(HWND hwnd, int width, int height )
{
    // 2015/4/1現在、gfxではウインドウサイズが設定出来ないようなので
    // Win32APIを呼び出して強制的にウインドウサイズを設定
    // 将来的には削除予定
    {
        RECT rw, rc;
        ::GetWindowRect(hwnd, &rw);
        ::GetClientRect(hwnd, &rc);

        int new_width = (rw.right - rw.left) - (rc.right - rc.left) + width;
        int new_height = (rw.bottom - rw.top) - (rc.bottom - rc.top) + height;

        SetWindowPos(hwnd, NULL, 0, 0, new_width, new_height, SWP_NOMOVE | SWP_NOZORDER);
        SetWindowText(hwnd, L"Ui2dMultiThread");
    }
}
#endif

//-----------------------------------------------------------------------------
// デバイスを初期化
//-----------------------------------------------------------------------------
void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetDebugMode(nn::gfx::DebugMode_Enable);
    g_Device.Initialize(info);
}


//-----------------------------------------------------------------------------
// スワップチェーンを初期化
//-----------------------------------------------------------------------------
void InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;
    info.SetDefault();
    info.SetLayer(g_pLayer);
    info.SetWidth(ScreenWidth);
    info.SetHeight(ScreenHeight);

    size_t size = g_SwapChain.CalculateScanBufferSize(&g_Device, info);
    g_MemoryPoolInvisibleOffset = AlignupOffset(g_MemoryPoolInvisibleOffset, nn::gfx::SwapChain::GetScanBufferAlignment(&g_Device, info));
    g_SwapChain.Initialize(&g_Device, info, &g_MemoryPoolInvisible, g_MemoryPoolInvisibleOffset, size);
    g_MemoryPoolInvisibleOffset += size;
}

//-----------------------------------------------------------------------------
// キューを初期化
//-----------------------------------------------------------------------------
void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);
    g_Queue.Initialize(&g_Device, info);
}

//-----------------------------------------------------------------------------
// カラーターゲット関連を初期化
//-----------------------------------------------------------------------------
void InitializeColorBuffer()
{
    // カラーバッファ用のテクスチャを初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(ScreenWidth);
        info.SetHeight(ScreenHeight);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetMipCount(1);
        info.SetDepth(1);

        g_MemoryPoolInvisibleOffset = AlignupOffset(g_MemoryPoolInvisibleOffset, nn::gfx::Texture::CalculateMipDataAlignment(&g_Device, info));
        size_t size = g_ColorBuffer.CalculateMipDataSize(&g_Device, info);
        g_ColorBuffer.Initialize(&g_Device, info, &g_MemoryPoolInvisible, g_MemoryPoolInvisibleOffset, size);
        g_MemoryPoolInvisibleOffset += size;
    }

    // カラーバッファビューを初期化
    {
        nn::gfx::ColorTargetView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetTexturePtr(&g_ColorBuffer);
        g_ColorBufferView.Initialize(&g_Device, info);
    }
}

//-----------------------------------------------------------------------------
// 深度ステンシルバッファ関連を初期化
//-----------------------------------------------------------------------------
void InitializeDepthStencilBuffer()
{
    // 深度ステンシルバッファ用のテクスチャを初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(ScreenWidth);
        info.SetHeight(ScreenHeight);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_D16_Unorm);
        info.SetMipCount(1);
        info.SetDepth(1);

        g_MemoryPoolInvisibleOffset = AlignupOffset(g_MemoryPoolInvisibleOffset, nn::gfx::Texture::CalculateMipDataAlignment(&g_Device, info));
        size_t size = g_DepthStencilBuffer.CalculateMipDataSize(&g_Device, info);
        g_DepthStencilBuffer.Initialize(&g_Device, info, &g_MemoryPoolInvisible, g_MemoryPoolInvisibleOffset, size);
        g_MemoryPoolInvisibleOffset += size;
    }

    // 深度ステンシルビューの初期化
    {
        nn::gfx::DepthStencilView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
        info.SetTexturePtr(&g_DepthStencilBuffer);
        g_DepthStencilView.Initialize(&g_Device, info);
    }
}

//-----------------------------------------------------------------------------
// コマンドバッファを初期化
//-----------------------------------------------------------------------------
void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    g_CommandBuffer.Initialize(&g_Device, info);

    // バッファ（メモリプールより確保）
    g_MemoryPoolOffset = AlignupOffset(g_MemoryPoolOffset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
    g_CommandBufferStartOffset = g_MemoryPoolOffset;
    g_MemoryPoolOffset += CommandBufferMemorySize;

    // コントロールメモリ（通常メモリより確保）
    g_pCommandBufferControlMemory = AllocateFromApplicationHeap(CommandBufferControlMemorySize, nn::gfx::CommandBuffer::GetControlMemoryAlignment(&g_Device) );
}

//-----------------------------------------------------------------------------
// ビューポートシザーを初期化
//-----------------------------------------------------------------------------
void InitializeViewportScissorState()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);

    nn::gfx::ViewportStateInfo viewportInfo;
    viewportInfo.SetDefault();
    viewportInfo.SetWidth(static_cast<float>(ScreenWidth));
    viewportInfo.SetHeight(static_cast<float>(ScreenHeight));
    info.SetViewportStateInfoArray(&viewportInfo, 1);

    nn::gfx::ScissorStateInfo scissorInfo;
    scissorInfo.SetDefault();
    scissorInfo.SetWidth(ScreenWidth);
    scissorInfo.SetHeight(ScreenHeight);
    info.SetScissorStateInfoArray(&scissorInfo, 1);

    SetGfxMemoryTmpl_(&g_ViewportScissor, info);
    g_ViewportScissor.Initialize(&g_Device, info);
}

//-----------------------------------------------------------------------------
// ブレンドステートを初期化
//-----------------------------------------------------------------------------
void InitializeBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
        targetInfo.SetChannelMask( nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue );
        targetInfo.SetBlendEnabled(true);
        targetInfo.SetColorBlendFunction( nn::gfx::BlendFunction_Add );
        targetInfo.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
        targetInfo.SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    };
    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );

    SetGfxMemoryTmpl_(&g_BlendState, info);

    g_BlendState.Initialize(&g_Device, info);
}

//-----------------------------------------------------------------------------
// ラスタライザステートを初期化
//-----------------------------------------------------------------------------
void InitializeRasterizerState()
{
    // ui2dライブラリ内ではカリングの設定は行われないので、
    // 明示的にCullMode_BackもしくはCullMode_Noneに設定する必要があります。
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode(nn::gfx::CullMode_Back);
    info.SetScissorEnabled(true);
    info.SetDepthClipEnabled(false);
    g_RasterizerState.Initialize(&g_Device, info);
}

//-----------------------------------------------------------------------------
// 深度ステンシルステートを初期化
//-----------------------------------------------------------------------------
void InitializeDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled(false);
    info.SetDepthWriteEnabled(false);
    g_DepthStencilState.Initialize(&g_Device, info);
}

//-----------------------------------------------------------------------------
// gfx 初期化処理
//-----------------------------------------------------------------------------
void InitializeGfx()
{
    // ライブラリを初期化
    nn::gfx::Initialize();

    // デバイスを初期化
    InitializeDevice();

    // gfx で使用するメモリプールを初期化
    {
        nn::gfx::MemoryPoolInfo info;
        info.SetDefault();

        const size_t MemoryPoolAligment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
        g_MemoryPoolMemory = AllocateFromApplicationHeap(MemoryPoolSize, MemoryPoolAligment);
        g_MemoryPoolOffset = 0;

        g_MemoryPoolInvisibleMemory = AllocateFromApplicationHeap(MemoryPoolSizeInvisible, MemoryPoolAligment);
        g_MemoryPoolInvisibleOffset = 0;

        InitializeMemoryPool();
    }

#if defined( NN_BUILD_CONFIG_OS_WIN )
    HWND hwnd;
    nn::vi::GetNativeWindow((nn::vi::NativeWindowHandle*)&hwnd, g_pLayer);
    InitializeGfxWin_(hwnd, ScreenWidth, ScreenHeight);
#endif

    // スワップチェーンを初期化
    InitializeSwapChain();

    // キューを初期化
    InitializeQueue();

    // カラーバッファ関連を初期化
    InitializeColorBuffer();

    // 深度ステンシルバッファ関連を初期化
    InitializeDepthStencilBuffer();

    // ディスクリプションプールを初期化
    InitialzieGfxDescriptorPool_();

    // コマンドバッファを初期化
    InitializeCommandBuffer();

    // ビューポートシザーを初期化
    InitializeViewportScissorState();

    // ブレンドステートを初期化
    InitializeBlendState();

    // ラスタライザステートを初期化
    InitializeRasterizerState();

    // 深度ステンシルステートを初期化
    InitializeDepthStencilState();
}

//-----------------------------------------------------------------------------
// gfx 終了処理
//-----------------------------------------------------------------------------
void FinalizeGfx()
{
    g_TextureDescriptorPool.Finalize( &g_Device );
    g_SamplerDescriptorPool.Finalize( &g_Device );

    void* pViewportScissorMemory = g_ViewportScissor.GetMemory();
    void* pBlendMemory = g_BlendState.GetMemory();

    g_DepthStencilView.Finalize(&g_Device);
    g_DepthStencilBuffer.Finalize(&g_Device);
    g_ColorBufferView.Finalize(&g_Device);
    g_ColorBuffer.Finalize(&g_Device);
    g_DepthStencilState.Finalize(&g_Device);
    g_RasterizerState.Finalize(&g_Device);
    g_BlendState.Finalize(&g_Device);
    g_ViewportScissor.Finalize(&g_Device);
    g_CommandBuffer.Finalize(&g_Device);
    g_Queue.Finalize(&g_Device);
    g_SwapChain.Finalize(&g_Device);

    DeallocateApplicationHeap(pViewportScissorMemory);
    DeallocateApplicationHeap(pBlendMemory);
    DeallocateApplicationHeap(g_pCommandBufferControlMemory);

    g_MemoryPool.Finalize(&g_Device);
    g_MemoryPoolInvisible.Finalize(&g_Device);
    DeallocateApplicationHeap(g_MemoryPoolMemory);
    DeallocateApplicationHeap(g_MemoryPoolInvisibleMemory);

    g_Device.Finalize();
}

//-----------------------------------------------------------------------------
// レイヤ 初期化処理
//-----------------------------------------------------------------------------
void InitializeLayer()
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::InitializeGraphics( malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
    }
#endif

    // vi ライブラリを初期化
    nn::vi::Initialize();

    nn::Result result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
    NN_UNUSED( result );

    result = nn::vi::CreateLayer( &g_pLayer, g_pDisplay );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    result = nn::vi::SetLayerScalingMode( g_pLayer, nn::vi::ScalingMode_FitToLayer );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
}

//-----------------------------------------------------------------------------
// レイヤ 終了処理
//-----------------------------------------------------------------------------
void FinalizeLayer()
{
    nn::vi::DestroyLayer( g_pLayer );
    nn::vi::CloseDisplay( g_pDisplay );
    nn::vi::Finalize();
}
//-----------------------------------------------------------------------------
// ファイルシステムの初期化
//-----------------------------------------------------------------------------
void InitializeFileSystem()
{
    nn::fs::SetAllocator( FsAllocateFunction, FsDeallocateFunction );

    // 注意：正式なファイルアクセス方法が整備されるまでのワークアラウンドです。
    // 第一引数から、実行ファイルの host 上での位置を取得します。
    char contentPath[260];
    char* pExeFilePath = nn::os::GetHostArgv()[ 0 ];
    char* pEnd = strrchr( pExeFilePath, '\\' );
    NN_ASSERT_NOT_NULL( pEnd );

    // 実行ファイルからの相対パスで、リソースフォルダを定義します。
    size_t length = pEnd - pExeFilePath + 1;
    strncpy( contentPath, pExeFilePath, length );
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    strcpy( contentPath + length, "..\\Contents\\NX" );
#elif defined(NN_BUILD_CONFIG_SPEC_GENERIC)
    strcpy( contentPath + length, "..\\Contents\\Generic" );
#else
    NN_ABORT("Unexpected spec.");
#endif

    nn::Result result = nn::fs::MountHost("Contents", contentPath);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_UNUSED(result);
}

//------------------------------------------------------------------------------
// ui2d 関連の初期化処理
//------------------------------------------------------------------------------
void InitializeUi2dRuntime()
{
    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    {
        g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/Ui2dMultiThread.arc", nn::ui2d::ArchiveResourceAlignment);
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの初期化
    g_pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    {
        void* pFont = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool    result = g_pFont->SetResource(&g_Device, pFont);
        NN_ASSERT(result);
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, NULL);

        g_pArcResourceAccessor->RegisterFont("sample.bffnt", g_pFont);
    }

    // グラフィックスリソースの設定
    g_GraphicsResource.Setup(&g_Device, 512);
    g_GraphicsResource.RegisterCommonSamplerSlot(RegisterSlotForSampler_, NULL);

    // レイアウトの初期化
    // マルチスレッドフラグを見て作成するスレッド数を決定する
    if (g_MultiThreading)
    {
        g_NumberOfThread = 4;
    }
    else
    {
        g_NumberOfThread = 1;
    }

    g_pLayout = NewArray<nn::ui2d::Layout>(g_NumberOfThread);

    // スレッドで並列処理するために必要な情報をスレッド数分作成する
    g_pDrawInfo = NewArray<nn::ui2d::DrawInfo>(g_NumberOfThread);
    g_pLayoutBufferContext = NewArray<LayoutBufferContext>(g_NumberOfThread);
    g_pNestedCommandBuffers = static_cast<NestedCommandBuffer*>(AllocateFromApplicationHeap(sizeof(NestedCommandBuffer) * g_NumberOfThread));
    memset(g_pNestedCommandBuffers, 0x00, sizeof(NestedCommandBuffer) * g_NumberOfThread);
    g_pLayoutThreadContext = static_cast<LayoutThreadContext*>(AllocateFromApplicationHeap(sizeof(LayoutThreadContext) * g_NumberOfThread));
    memset(g_pLayoutThreadContext, 0x00, sizeof(LayoutThreadContext) * g_NumberOfThread);
    g_pThreadContext = static_cast<ThreadContext*>(AllocateFromApplicationHeap(sizeof(ThreadContext) * g_NumberOfThread));
    memset(g_pThreadContext, 0x00, sizeof(ThreadContext) * g_NumberOfThread);

    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );

    nn::ui2d::BuildResultInformation totalMemoryRequirement;
    totalMemoryRequirement.SetDefault();

    // 必要バッファサイズを計算するために一通りビルドする。
    for (uint32_t i = 0; i < g_NumberOfThread;++i)
    {
        // レイアウトデータをセットアップします
        nn::ui2d::Layout::BuildOption   opt;
        opt.SetDefault();
        nn::ui2d::BuildResultInformation   buildResultInformation;
        buildResultInformation.SetDefault();

        bool result = g_pLayout[i].BuildWithName(&buildResultInformation, &g_Device, g_pArcResourceAccessor, NULL, NULL, opt, g_pLayoutDataFileName[i % 4]);
        NN_SDK_ASSERT(result);
        NN_UNUSED(result);

        // 必要な各種バッファのサイズを集計します。
        totalMemoryRequirement.requiredUi2dConstantBufferSize += buildResultInformation.requiredUi2dConstantBufferSize;
        totalMemoryRequirement.requiredFontConstantBufferSize += buildResultInformation.requiredFontConstantBufferSize;
    }

    for (uint32_t i = 0; i < g_NumberOfThread;++i)
    {
        // 必要サイズに基づいて各種バッファを初期化する。
        // 下記のスレッド間で GpuBuffer を共有するテストもあるので
        // すべてのデータを表示できるだけの容量を確保したバッファをすべてのスレッドで作成する。
        g_pLayoutBufferContext[i].Initialize(g_pDrawInfo[i], &g_Device, totalMemoryRequirement);

        // GpuBuffer をスレッド間で共有するテストのためのコード。
        // 各 DrawInfo が参照するコンスタントバッファを 0 版の物に強制設定する。
        if (g_ShareGpuBuffer)
        {
            g_pDrawInfo[i].SetUi2dConstantBuffer(g_pDrawInfo[0].GetUi2dConstantBuffer());
        }

        // 描画に使用する情報の設定
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(
            &projection,
            rect.left * 4.0f,
            rect.right * 4.0f,
            rect.bottom * 4.0f,
            rect.top * 4.0f,
            0.0f,
            300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, i * -200.0f + 300.0f, i * 150.0f - 200.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, i * -200.0f + 300.0f, i * 150.0f - 200.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(&view, pos, target, up);

        g_pDrawInfo[i].SetGraphicsResource(&g_GraphicsResource);
        g_pDrawInfo[i].SetProjectionMtx(projection);
        g_pDrawInfo[i].SetViewMtx(view);

        // 各スレッドでコマンドを発行するためのコマンドバッファを作成します
        nn::gfx::CommandBuffer::InfoType info;
        info.SetDefault();
        info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
        info.SetCommandBufferType( nn::gfx::CommandBufferType_Nested);

        // 各スレッドで描画コマンドをためるための CommandBuffer を初期化する。
        g_pNestedCommandBuffers[i].SetDefault();
        g_pNestedCommandBuffers[i].m_pCommandBuffer = AllocAndConstruct<nn::gfx::CommandBuffer>();
        g_pNestedCommandBuffers[i].m_pCommandBuffer->Initialize(&g_Device, info);

        g_MemoryPoolOffset = AlignupOffset(g_MemoryPoolOffset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
        g_pNestedCommandBuffers[i].m_StartOffset = g_MemoryPoolOffset;
        g_MemoryPoolOffset += NestedCommandBufferMemorySize;

        g_pNestedCommandBuffers[i].m_pControlMemory = AllocateFromApplicationHeap(CommandBufferControlMemorySize, nn::gfx::CommandBuffer::GetControlMemoryAlignment(&g_Device) );

        // スレッドに必要な情報を初期化します
        g_pThreadContext[i].SetDefault();
        // スレッドのスタック作成
        g_pThreadContext[i].m_pThreadStack = AllocateFromApplicationHeap(threadStackSize, nn::os::ThreadStackAlignment);
    }

    // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, NULL);
}

//------------------------------------------------------------------------------
// ui2d 関連の終了処理
//------------------------------------------------------------------------------
void FinalizeUi2dRuntime()
{
    for (uint32_t i = 0; i < g_NumberOfThread;++i)
    {
        // サブのコマンドバッファの終了処理を行う
        g_pNestedCommandBuffers[i].m_pCommandBuffer->Finalize(&g_Device);
        DestructAndFree(g_pNestedCommandBuffers[i].m_pCommandBuffer);
        DeallocateApplicationHeap(g_pNestedCommandBuffers[i].m_pControlMemory);

        g_pLayoutBufferContext[i].Finalize(&g_Device);
        g_pLayout[i].Finalize(&g_Device);

        DeallocateApplicationHeap(g_pThreadContext[i].m_pThreadStack);
    }

    DeallocateApplicationHeap(g_pThreadContext);
    DeallocateApplicationHeap(g_pLayoutThreadContext);
    DeallocateApplicationHeap(g_pNestedCommandBuffers);
    DeleteArray(g_pLayoutBufferContext, g_NumberOfThread);

    DeleteArray(g_pDrawInfo, g_NumberOfThread);
    DeleteArray(g_pLayout, g_NumberOfThread);

    g_GraphicsResource.UnregisterCommonSamplerSlot(UnregisterSlotForSampler_, NULL);
    g_GraphicsResource.Finalize(&g_Device);

    void* pFontResource = g_pFont->RemoveResource(&g_Device);
    DeallocateApplicationHeap(pFontResource);
    g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, NULL);
    g_pFont->Finalize(&g_Device);

    DestructAndFree<nn::font::ResFont>(g_pFont);

    g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, NULL);
    g_pArcResourceAccessor->Detach();
    g_pArcResourceAccessor->Finalize(&g_Device);
    DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);

    // ArcResourceAccessor の Finalize でアクセスされるため最後に開放する。
    DeallocateApplicationHeap(g_pLayoutArchiveBinary);
}

//------------------------------------------------------------------------------
// レイアウトデータの更新を行うスレッドの実装
//------------------------------------------------------------------------------
void UpdateLayoutBufferThread(void* arg)
{
    LayoutThreadContext*    pContext = static_cast<LayoutThreadContext*>(arg);

    NN_SDK_ASSERT_NOT_NULL(pContext->m_pLayout);
    NN_SDK_ASSERT_NOT_NULL(pContext->m_pDrawInfo);

    // レイアウトの更新処理
    pContext->m_pLayout->AnimateAndUpdateAnimFrame();

    // コンスタントバッファの更新
    pContext->m_pLayout->Calculate(*(pContext->m_pDrawInfo));
}

//------------------------------------------------------------------------------
// レイアウトデータの更新と描画
//------------------------------------------------------------------------------
void UpdateAndDrawLayoutData()
{
    // Unmap 後に Draw する必要があるため現時点では Draw までしかマルチスレッド化出来ない。
    // 現時点では Calculate までマルチスレッドで動かして、残りの処理はメインスレッドで処理するようにしておく。

    // スレッドごとに処理するデータなどの設定を LayoutThreadContext に設定してスレッドを作成する。
    // 本スレッドは LayoutThreadContext で与えられたレイアウトデータを更新し、各種バッファを更新するまでを並列処理しています。
    for (uint32_t i = 0; i < g_NumberOfThread;++i)
    {
        // ui2d のコンスタントバッファを更新するために Map する。
        g_pDrawInfo[i].Map(g_ConstantBufferIndex);
        g_pLayoutThreadContext[i].m_ConstantBufferIndex = g_ConstantBufferIndex;
        g_pLayoutThreadContext[i].m_pDrawInfo = &g_pDrawInfo[i];
        g_pLayoutThreadContext[i].m_pLayout = &g_pLayout[i];
        g_pLayoutThreadContext[i].m_pCommandBuffer = &g_pNestedCommandBuffers[i];

        nn::Result result = nn::os::CreateThread(
            &g_pThreadContext[i].m_Thread,
            UpdateLayoutBufferThread,
            &g_pLayoutThreadContext[i],
            g_pThreadContext[i].m_pThreadStack,
            threadStackSize,
            nn::os::DefaultThreadPriority );
        if ( !result.IsSuccess() )
        {
            // スレッド生成に失敗
            break;
        }
        nn::os::StartThread(&g_pThreadContext[i].m_Thread);
    }
    // スレッドの終了待ち処理です。
    // すべてのスレッドが終了するまで待ちます。
    for (uint32_t i = 0; i < g_NumberOfThread;++i)
    {
        nn::os::WaitThread(&g_pThreadContext[i].m_Thread);
        nn::os::DestroyThread(&g_pThreadContext[i].m_Thread);
    }

    // スレッド間で GpuBuffer を共有している場合、各スレッドの処理が完全に終わってから Unmap しないと
    // 別スレッドで使用している最中の GpuBuffer を Unmap してしまう可能性があるため
    // すべてのスレッドの終了を待って Unmap します。
    for (uint32_t i = 0; i < g_NumberOfThread;++i)
    {
        g_pDrawInfo[i].Unmap();
        g_pDrawInfo[i].SetGpuAccessBufferIndex(g_ConstantBufferIndex);
    }

    // レイアウトデータの更新処理と各種バッファデータの作成まで完了しているため
    // コマンドバッファへ描画コマンドを作成します。
    // 将来的に描画コマンドの作成まで並列化することを見越して、スレッドごとに別々のコマンドバッファを作成し
    // メインのコマンドバッファから呼び出すようにしています。
    for (uint32_t i = 0; i < g_NumberOfThread;++i)
    {
        g_pNestedCommandBuffers[i].m_pCommandBuffer->Begin();
        g_pLayout[i].Draw(g_pDrawInfo[i], *g_pNestedCommandBuffers[i].m_pCommandBuffer);
        g_pNestedCommandBuffers[i].m_pCommandBuffer->End();

        g_CommandBuffer.CallCommandBuffer(g_pLayoutThreadContext[i].m_pCommandBuffer->m_pCommandBuffer);
    }
}

//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // アプリケーションで使用するヒープ領域を初期化
    g_pMemoryHeap.Reset(new uint8_t[ApplicationHeapSize]);
    g_ApplicationHeapHandle.Initialize(g_pMemoryHeap.Get(), ApplicationHeapSize);

    // ファイルシステムの初期化(gfx の初期化より前に実行します)
    InitializeFileSystem();

    // レイヤ を初期化
    InitializeLayer();

    // gfx を初期化
    InitializeGfx();

    // ui2d ランタイムの初期化処理
    InitializeUi2dRuntime();

    // メインループ
    NN_LOG("Start demo.\n");

    for(;;)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        // WindowsMessege の処理
        {
            MSG  msg;
            if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE) )
            {
                TranslateMessage(&msg);

                if ( msg.message == WM_QUIT)
                {
                    break;
                }

                DispatchMessage(&msg);
            }
        }
#elif !defined(NN_BUILD_CONFIG_OS_HORIZON)
        if(g_Pad.IsHold(g_Pad.MASK_L|g_Pad.MASK_R))
        {
            break;
        }
#endif
        // 描画処理
        g_CommandBuffer.Reset();

        // コマンドバッファ用のメモリを設定します
        {
            g_CommandBuffer.AddCommandMemory( &g_MemoryPool, g_CommandBufferStartOffset, CommandBufferMemorySize);
            g_CommandBuffer.AddControlMemory(g_pCommandBufferControlMemory, CommandBufferControlMemorySize);
            g_CommandBuffer.SetOutOfCommandMemoryEventCallback( OutOfMemoryEventCallback );
            g_CommandBuffer.SetOutOfControlMemoryEventCallback( OutOfMemoryEventCallback );

            g_CommandBuffer.AddControlMemory(g_pCommandBufferControlMemory, CommandBufferControlMemorySize);

            for (uint32_t i = 0; i < g_NumberOfThread;++i)
            {
                g_pNestedCommandBuffers[i].m_pCommandBuffer->AddCommandMemory(&g_MemoryPool, g_pNestedCommandBuffers[i].m_StartOffset, NestedCommandBufferMemorySize);
                g_pNestedCommandBuffers[i].m_pCommandBuffer->AddControlMemory(g_pNestedCommandBuffers[i].m_pControlMemory, CommandBufferControlMemorySize);
                g_pNestedCommandBuffers[i].m_pCommandBuffer->SetOutOfCommandMemoryEventCallback( OutOfMemoryEventCallback );
                g_pNestedCommandBuffers[i].m_pCommandBuffer->SetOutOfControlMemoryEventCallback( OutOfMemoryEventCallback );
            }
        }

        g_CommandBuffer.Begin();
        {
            // ディスクリプタプール や シェーダーを更新した場合は、GPU のキャッシュを無効化します。
            g_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);
            // ディスクリプタプールをセットする。
            g_CommandBuffer.SetDescriptorPool( &g_TextureDescriptorPool );
            g_CommandBuffer.SetDescriptorPool( &g_SamplerDescriptorPool );
        }
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::gfx::ColorTargetView* pTarget = g_SwapChain.AcquireNextScanBufferView();
#else
        nn::gfx::ColorTargetView* pTarget = &g_ColorBufferView;
#endif

        g_CommandBuffer.ClearColor(pTarget, 0.0f, 0.0f, 0.5f, 1.0f, NULL);
        g_CommandBuffer.ClearDepthStencil(&g_DepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);
        g_CommandBuffer.SetRenderTargets(1, &pTarget, &g_DepthStencilView);

        g_CommandBuffer.SetViewportScissorState(&g_ViewportScissor);
        g_CommandBuffer.SetRasterizerState(&g_RasterizerState);
        g_CommandBuffer.SetDepthStencilState(&g_DepthStencilState);
        g_CommandBuffer.SetBlendState(&g_BlendState);

        // レイアウトデータの更新と描画
        UpdateAndDrawLayoutData();

        g_CommandBuffer.End();

        ++g_ConstantBufferIndex;
        g_ConstantBufferIndex %= nn::ui2d::ConstantBufferCount;

        // コマンドの実行
        g_Queue.ExecuteCommand(&g_CommandBuffer, NULL);

#if !defined(NN_BUILD_CONFIG_OS_HORIZON)
        g_Queue.CopyToScanBuffer(&g_SwapChain, &g_ColorBufferView);
#endif
        // Queue をフラッシュします。
        g_Queue.Flush();

        g_Queue.Present(&g_SwapChain, 1);
        g_Queue.Sync();
        g_Queue.Flush();
    }

    // 終了処理
    FinalizeUi2dRuntime();

    nn::fs::Unmount("Contents");

    // gfx 終了処理
    FinalizeGfx();

    // レイヤ 終了処理
    FinalizeLayer();

    g_ApplicationHeapHandle.Finalize();

    delete[] reinterpret_cast<uint8_t*>(g_pMemoryHeap.Get());

    NN_ASSERT(g_AllocationCount == 0);
}



