﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d.h>
#include <nn/font/font_ScalableFont.h>
#include <nn/fs.h>

#include "Ui2dFontChange.h"

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------

static nn::font::ScalableFont* g_pScFont;
static nn::font::TextureCache* g_pTextureCache;
static void* g_pScalableFontBinary;
static void* g_pScalableFontRawResource;

//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeFontChange()
{
    g_pScFont = AllocAndConstruct<nn::font::ScalableFont>();
    g_pTextureCache = AllocAndConstruct<nn::font::TextureCache>();

    size_t fontBinarySize;
    {
        nn::Result result;
        nn::fs::FileHandle fileHandle;
        int64_t fileSize;

        result = nn::fs::OpenFile(&fileHandle, "Contents:/nintendo_NTLG-DB_002.bfttf", nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = nn::fs::GetFileSize(&fileSize, fileHandle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        void* pFontFileImage = Ui2dAllocateFunction(static_cast<size_t>(fileSize), nn::font::ResourceAlignment, NULL);
        g_pScalableFontRawResource = pFontFileImage;

        size_t readSize;
        result = nn::fs::ReadFile(&readSize, fileHandle, 0, pFontFileImage, static_cast<size_t>(fileSize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        fontBinarySize = static_cast<size_t>(fileSize);
        g_pScalableFontBinary = pFontFileImage;

        nn::fs::CloseFile(fileHandle);
    }
    NN_ASSERT_NOT_NULL(g_pScalableFontBinary);

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    //-----------------------------------------
    // テクスチャキャッシュの初期化
    {
        nn::font::TextureCache::InitializeArg* pArg = static_cast<nn::font::TextureCache::InitializeArg*>(nn::ui2d::Layout::AllocateMemory(sizeof(nn::font::TextureCache::InitializeArg)));
        pArg->SetDefault();
        pArg->allocateFunction = nn::ui2d::Layout::GetAllocateFunction();
        pArg->textureCacheWidth = 1024;
        pArg->textureCacheHeight = 1024;
        pArg->pFontDatas[0][0] = g_pScalableFontBinary;
        pArg->pFontDataSizes[0][0] = fontBinarySize;
        pArg->pFontDataTypes[0][0] = nn::font::TextureCache::FontDataType_Bfttf;
        pArg->fontCount = 1;
        pArg->fontListCount[0] = 1;
        g_pTextureCache->Initialize(pDevice, *pArg);
        g_pTextureCache->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
        nn::ui2d::Layout::FreeMemory(pArg);
    }

    //-----------------------------------------
    // フォントの初期化
    {
        nn::font::ScalableFont::InitializeArg arg;
        arg.SetDefault();
        arg.pTextureCache = g_pTextureCache;
        arg.fontSize = 10;
        arg.fontFace = 0;
        arg.isAlternateCharSpaceWithOriginalWidthForNotReadyChar = true;
        g_pScFont->Initialize(arg);

        // 使用する文字をフォントエンジンに登録します。
        // ここでは ASCII 文字をすべて登録しています。
        for (uint16_t j = 0; j < 255; j++)
        {
            g_pScFont->RegisterGlyph(j);
        }
    }

    // CPU 側のメモリへグリフのレンダリングを行います。
    g_pTextureCache->UpdateTextureCache();

    // CPU 側のメモリへレンダリングしたグリフイメージをテクスチャとして使用できるようにします。
    g_pTextureCache->CompleteTextureCache();

    // テキストボックスのフォントを差し替えます
    nn::ui2d::TextBox* pTextBox = static_cast<nn::ui2d::TextBox*>(g_pLayout->GetRootPane()->FindPaneByName("textBox_4"));
    pTextBox->SetFont(g_pScFont);

    // テキストボックスを読み込んだフォントのサイズに更新します。
    // この処理を行わない場合は、テキストボックスに設定されていたフォントのサイズが使われます。
    nn::ui2d::Size fontSize;
    fontSize.Set(static_cast<float>(g_pScFont->GetWidth()), static_cast<float>(g_pScFont->GetHeight()));
    pTextBox->SetFontSize(fontSize);
}

//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeFontChange()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pScFont->Finalize(pDevice);
    g_pTextureCache->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
    g_pTextureCache->Finalize(pDevice, nn::ui2d::Layout::GetFreeFunction(), NULL);


    DestructAndFree<nn::font::ScalableFont>(g_pScFont);
    DestructAndFree<nn::font::TextureCache>(g_pTextureCache);
    g_pScFont = NULL;
    g_pTextureCache = NULL;

    Ui2dDeallocateFunction(g_pScalableFontRawResource, NULL);
    g_pScalableFontRawResource = NULL;
}
