﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d.h>
#include <nn/fs.h>
#include <nn/font/font_ScalableFontUtil.h>

#include "Ui2dComplexFont.h"

//------------------------------------------------------------------------------
//  定数の宣言
//------------------------------------------------------------------------------

static const int g_FileCount = 4;
static const char* g_FileNames[g_FileCount] =
{
    "sample.bffnt",
    "nintendo_NTLG-DB_002.bfttf",
    "color.bffnt",
    "IconFont-Regular.ttf",
};

static const int g_FcpxCount = 2;
//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------

static nn::font::Font* g_pComplexFontTree[g_FcpxCount] = { NULL };
static size_t g_FontBinarySize[g_FileCount] = { 0 };
static void* g_pFontBinary[g_FileCount] = { NULL };
static void* g_pFontRawResource[g_FileCount] = { NULL };
static nn::font::TextureCache* g_pTextureCache;

//------------------------------------------------------------------------------
// フォント名からフォントデータを探すコールバック関数
//------------------------------------------------------------------------------
static void* AcquireFontFunctionForComplexFont(size_t* pOutFontDataSize, const char* pFontName, nn::ui2d::ResType resType, void* pUserData)
{
    for (int i = 0; i < g_FileCount; i++)
    {
        // フォントを直接ロードする場合
        if (strcmp(pFontName, g_FileNames[i]) == 0)
        {
            if (g_pFontBinary[i] != NULL)
            {
                *pOutFontDataSize = g_FontBinarySize[i];
                return g_pFontBinary[i];
            }
            else
            {
                nn::Result result;
                nn::fs::FileHandle fileHandle;
                int64_t fileSize;

                char path[256];
                strcpy(path, "Contents:/");
                strcat(path, g_FileNames[i]);
                result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                result = nn::fs::GetFileSize(&fileSize, fileHandle);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                void* pFontFileImage = Ui2dAllocateFunction(static_cast<size_t>(fileSize), nn::font::ResourceAlignment, NULL);
                g_pFontRawResource[i] = pFontFileImage;

                size_t readSize;
                result = nn::fs::ReadFile(&readSize, fileHandle, 0, pFontFileImage, static_cast<size_t>(fileSize));
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                g_FontBinarySize[i] = static_cast<size_t>(fileSize);
                g_pFontBinary[i] = pFontFileImage;

                nn::fs::CloseFile(fileHandle);

                NN_ASSERT_NOT_NULL(g_pFontBinary[i]);
                *pOutFontDataSize = g_FontBinarySize[i];
                return g_pFontBinary[i];
            }
        }
    }

    // リソース内からデータを探す場合
    {
        nn::ui2d::ResourceAccessor* pResAccsessor = reinterpret_cast<nn::ui2d::ResourceAccessor*>(pUserData);
        void* pFont = pResAccsessor->FindResourceByName(pOutFontDataSize, resType, pFontName);
        NN_ASSERT(pFont != NULL, "The font resource [%s] was not found.\n", pFontName);
        return pFont;
    }
}

//------------------------------------------------------------------------------
// テクスチャキャッシュのラインの種類を 8 の倍数で決定する関数
//------------------------------------------------------------------------------
static uint8_t CalculateLineKind(uint16_t height)
{
    if (height == 0)
    {
        return 1;
    }

    // height を 8 の倍数に切り上げて 8 で割った数をラインの種類とする
    uint32_t kind = nn::util::align_up(static_cast<uint32_t>(height), 8) / 8;
    if (kind > 255)
    {
        NN_SDK_ASSERT(false, "The font height(%d) is too large.", height);
        return 0;
    }
    return static_cast<uint8_t>(kind);
}
static uint32_t CalculateLineHeight(uint8_t kind)
{
    // ラインの種類をラインの高さに変換する
    return static_cast<uint32_t>(kind) * 8;
}

//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeComplexFont()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/ComplexFontSample.arc", nn::ui2d::ArchiveResourceAlignment);
    {
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの読み込み
    {
        const void* pFcpxData[g_FcpxCount];
        int fontFaceHead[g_FcpxCount];

        pFcpxData[0] = g_pArcResourceAccessor->FindResourceByName(nn::ui2d::ResourceTypeComplexFont, "ComplexFont.bfcpx");
        pFcpxData[1] = g_pArcResourceAccessor->FindResourceByName(nn::ui2d::ResourceTypeComplexFont, "ComplexFont2.bfcpx");

        nn::font::TextureCache::InitializeArg* pArg = static_cast<nn::font::TextureCache::InitializeArg*>(nn::ui2d::Layout::AllocateMemory(sizeof(nn::font::TextureCache::InitializeArg)));
        pArg->SetDefault();
        pArg->workMemorySize = 1024 * 1024;
        pArg->noPlotWorkMemorySize = 1024 * 1024;
        pArg->allocateFunction = nn::ui2d::Layout::GetAllocateFunction();
        pArg->textureCacheWidth = 1024;
        pArg->textureCacheHeight = 1024;

        // ラインの種類を決定する関数を自前で用意して登録することができる
        pArg->pCalculateLineKindFunction = CalculateLineKind;
        pArg->pCalculateLineHeightFunction = CalculateLineHeight;

        // ユーティリティ関数を使って bfcpx から TextureCacheArg を構築する
        for (int i = 0; i < g_FcpxCount; i++)
        {
            fontFaceHead[i] = nn::ui2d::ComplexFontHelper::SetupTextureCacheArg(
                pArg,
                AcquireFontFunctionForComplexFont,
                g_pArcResourceAccessor,
                pFcpxData[i]);
        }

        // TextureCache を初期化する
        g_pTextureCache = AllocAndConstruct<nn::font::TextureCache>();
        g_pTextureCache->Initialize(pDevice, *pArg);
        g_pTextureCache->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

        nn::ui2d::Layout::FreeMemory(pArg);

        // ユーティリティ関数を使って bfcpx からフォントを構築する
        for (int i = 0; i < g_FcpxCount; i++)
        {
            g_pComplexFontTree[i] = nn::ui2d::ComplexFontHelper::InitializeComplexFontTree(
                pDevice,
                RegisterSlotForTexture,
                &g_GfxFramework,
                g_pTextureCache,
                fontFaceHead[i],
                AcquireFontFunctionForComplexFont,
                g_pArcResourceAccessor,
                pFcpxData[i]);
        }

        // アーカイバに .fcpx として登録します。
        g_pArcResourceAccessor->RegisterFont("ComplexFont.fcpx", g_pComplexFontTree[0]);
        g_pArcResourceAccessor->RegisterFont("ComplexFont2.fcpx", g_pComplexFontTree[1]);
    }

    // レイアウトの初期化
    nn::ui2d::BuildResultInformation buildResult;
    {
        g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

        nn::ui2d::Layout::BuildOption    opt;
        opt.SetDefault();
        buildResult.SetDefault();

        g_pLayout->BuildWithName(&buildResult, pDevice, g_pArcResourceAccessor, NULL, NULL, opt, "ComplexFontSample.bflyt");
    }

    // グラフィックスリソースの設定
    InitializeGraphicsResource();

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    g_pDrawInfo = AllocAndConstruct<nn::ui2d::DrawInfo>();
    NN_ASSERT_NOT_NULL(g_pDrawInfo);
    g_pUi2dConstantBuffer = AllocAndConstruct<nn::font::GpuBuffer>();
    NN_ASSERT_NOT_NULL(g_pUi2dConstantBuffer);
    InitializeUi2dBuffers(*g_pDrawInfo, buildResult, g_pUi2dConstantBuffer);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(&projection, rect.left, rect.right, rect.bottom, rect.top, 0.0f, 300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(&view, pos, target, up);

        g_pDrawInfo->SetGraphicsResource(g_pGraphicsResource);
        g_pDrawInfo->SetProjectionMtx(projection);
        g_pDrawInfo->SetViewMtx(view);
    }

    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

    // 複合フォント用のテクスチャキャッシュの状態をを更新する
    nn::font::RegisterGlyphFromTextBoxRecursive(g_pLayout->GetRootPane());
    g_pTextureCache->UpdateTextureCache();
    g_pTextureCache->CompleteTextureCache();
}

//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeComplexFont()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pUi2dConstantBuffer->Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    DestructAndFree<nn::font::GpuBuffer>(g_pUi2dConstantBuffer);
    g_pUi2dConstantBuffer = NULL;
    DestructAndFree<nn::ui2d::DrawInfo>(g_pDrawInfo);
    g_pDrawInfo = NULL;

    FinalizeGraphicsResource();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);
    g_pLayout = NULL;

    g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
    g_pArcResourceAccessor->Detach();
    g_pArcResourceAccessor->Finalize(pDevice);
    DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);
    g_pArcResourceAccessor = NULL;

    for (int i = 0; i < g_FcpxCount; i++)
    {
        nn::ui2d::ComplexFontHelper::FinalizeComplexFontTree(
            pDevice,
            g_pComplexFontTree[i],
            UnregisterSlotForTexture,
            &g_GfxFramework);

        g_pComplexFontTree[i] = NULL;
    }

    g_pTextureCache->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
    g_pTextureCache->Finalize(pDevice, nn::ui2d::Layout::GetFreeFunction(), NULL);
    DestructAndFree<nn::font::TextureCache>(g_pTextureCache);
    g_pTextureCache = NULL;

    for (int i = 0; i < g_FileCount; i++)
    {
        Ui2dDeallocateFunction(g_pFontRawResource[i], NULL);

        g_FontBinarySize[i] = 0;
        g_pFontBinary[i] = NULL;
        g_pFontRawResource[i] = NULL;
    }

    // ArcResourceAccessor の Finalize でアクセスされるため最後に開放する。
    Ui2dDeallocateFunction(g_pLayoutArchiveBinary, NULL);
    g_pLayoutArchiveBinary = NULL;
}
