﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d.h>

#include "Ui2dAlignmentPane.h"

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------

const uint16_t* g_ReplaceText = reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("abcdefghijklm"));

//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeAlignmentPane()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/Ui2dAlignmentPane.arc", nn::ui2d::ArchiveResourceAlignment);
    {
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの初期化
    g_pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    {
        void* pFont = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool    result = g_pFont->SetResource(pDevice, pFont);
        NN_ASSERT(result);
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

        g_pArcResourceAccessor->RegisterFont("sample.bffnt", g_pFont);
    }

    // レイアウトの初期化
    nn::ui2d::BuildResultInformation buildResult;
    {
        g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

        nn::ui2d::Layout::BuildOption    opt;
        opt.SetDefault();
        buildResult.SetDefault();

        g_pLayout->BuildWithName(&buildResult, pDevice, g_pArcResourceAccessor, NULL, NULL, opt, "AlignmentPane.bflyt");
    }

    // グラフィックスリソースの設定
    InitializeGraphicsResource();

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    g_pDrawInfo = AllocAndConstruct<nn::ui2d::DrawInfo>();
    NN_ASSERT_NOT_NULL(g_pDrawInfo);
    g_pUi2dConstantBuffer = AllocAndConstruct<nn::font::GpuBuffer>();
    NN_ASSERT_NOT_NULL(g_pUi2dConstantBuffer);
    InitializeUi2dBuffers(*g_pDrawInfo, buildResult, g_pUi2dConstantBuffer);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(&projection, rect.left, rect.right, rect.bottom, rect.top, 0.0f, 300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(&view, pos, target, up);

        g_pDrawInfo->SetGraphicsResource(g_pGraphicsResource);
        g_pDrawInfo->SetProjectionMtx(projection);
        g_pDrawInfo->SetViewMtx(view);
    }

    // アニメーションの初期化
    {
        nn::ui2d::Animator* pAnimator = g_pLayout->CreateGroupAnimator(pDevice, "loop");
        pAnimator->PlayAuto(1.0f);
    }

    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
}

//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeAlignmentPane()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pUi2dConstantBuffer->Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    DestructAndFree<nn::font::GpuBuffer>(g_pUi2dConstantBuffer);
    g_pUi2dConstantBuffer = NULL;
    DestructAndFree<nn::ui2d::DrawInfo>(g_pDrawInfo);
    g_pDrawInfo = NULL;

    FinalizeGraphicsResource();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);
    g_pLayout = NULL;

    {
        g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        void* pFontResource = g_pFont->RemoveResource(pDevice);
        Ui2dDeallocateFunction(pFontResource, NULL);
        g_pFont->Finalize(pDevice);
        DestructAndFree<nn::font::ResFont>(g_pFont);
        g_pFont = NULL;
    }

    {
        g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        g_pArcResourceAccessor->Detach();
        g_pArcResourceAccessor->Finalize(pDevice);

        Ui2dDeallocateFunction(g_pLayoutArchiveBinary, NULL);
        g_pLayoutArchiveBinary = NULL;

        DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);
        g_pArcResourceAccessor = NULL;
    }
}

//------------------------------------------------------------------------------
// 計算処理
//------------------------------------------------------------------------------
void CalculateAlignmentPane(int frame)
{
    int length = nn::font::CalculateWideCharString16Length(g_ReplaceText);
    int count = ((frame / 8) % length ) + 1;
    {
        nn::ui2d::TextBox *pTextbox = nn::ui2d::DynamicCast<nn::ui2d::TextBox*>(g_pLayout->GetRootPane()->FindPaneByName("T_text_00"));
        NN_ASSERT_NOT_NULL(pTextbox);
        NN_ASSERT(pTextbox->GetUtf8() == false);
        pTextbox->SetString(g_ReplaceText, 0, static_cast<uint16_t>(count));
    }

    // 水平方向の整列ペイン配下の文字列の長さを変更して整列処理を毎回行います。
    {
        nn::ui2d::TextBox *pTextbox = nn::ui2d::DynamicCast<nn::ui2d::TextBox*>(g_pLayout->GetRootPane()->FindPaneByName("T_text_01"));
        NN_ASSERT_NOT_NULL(pTextbox);
        NN_ASSERT(pTextbox->GetUtf8() == false);
        pTextbox->SetString(g_ReplaceText, 0, static_cast<uint16_t>(count));

        // 配下のペインを整列します。
        nn::ui2d::Alignment *pAlignment = nn::ui2d::DynamicCast<nn::ui2d::Alignment*>(g_pLayout->GetRootPane()->FindPaneByName("A_HorizontalAlignment"));
        NN_ASSERT_NOT_NULL(pAlignment);
        pAlignment->RequestAlignment();
    }

    // 垂直方向の整列ペイン配下の整列処理を毎回行います。
    {
        nn::ui2d::Alignment *pAlignment = nn::ui2d::DynamicCast<nn::ui2d::Alignment*>(g_pLayout->GetRootPane()->FindPaneByName("A_VerticalAlignment"));
        NN_ASSERT_NOT_NULL(pAlignment);
        pAlignment->RequestAlignment();
    }
}


