﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{TimeCalculateUserSystemClockDifference.cpp,PageSampleTimeCalculateUserSystemClockDifference}
 *
 * @brief
 *  ユーザー時計( @ref nn::time::StandardUserSystemClock )がユーザーによって意図的に操作された量を計算するサンプルプログラム
 */

/**
 * @page PageSampleTimeCalculateUserSystemClockDifference CalculateStandardUserSystemClockDifferenceByUser
 * @tableofcontents
 *
 * @brief
 *  ユーザー時計( @ref nn::time::StandardUserSystemClock )がユーザーによって意図的に操作された量を計算するサンプルプログラムの解説です。
 *
 * @section TimeCalculateUserSystemClockDifference_SectionBrief 概要
 *  ここでは、2つの nn::time::ClockSnapshot と @ref nn::time::CalculateStandardUserSystemClockDifferenceByUser 関数を利用して、
 *  ユーザー時計( @ref nn::time::StandardUserSystemClock )がユーザーによって意図的に操作された量を計算するサンプルプログラムの解説をします。
 *
 *  @ref nn::time::ClockSnapshot "ClockSnapshot の関数リファレンス" ,
 *  @ref nn::time::CalculateStandardUserSystemClockDifferenceByUser "CalculateStandardUserSystemClockDifferenceByUser の関数リファレンス"
 *  も併せて参照して下さい。
 *
 * @section TimeCalculateUserSystemClockDifference_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/TimeCalculateUserSystemClockDifference Samples/Sources/Applications/TimeCalculateUserSystemClockDifference @endlink 以下にあります。
 *
 * @section TimeCalculateUserSystemClockDifference_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section TimeCalculateUserSystemClockDifference_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section TimeCalculateUserSystemClockDifference_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section TimeCalculateUserSystemClockDifference_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section TimeCalculateUserSystemClockDifference_SectionDetail 解説
 *
 * @subsection TimeCalculateUserSystemClockDifference_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  TimeCalculateUserSystemClockDifference.cpp
 *  @includelineno TimeCalculateUserSystemClockDifference.cpp
 *
 * @subsection TimeCalculateUserSystemClockDifference_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 * - TIME ライブラリの初期化
 * - ユーザーによる時刻の操作検知を開始する ClockSnapshot の生成
 * - 3秒間スリープ
 * - ユーザーによる時刻の操作検知を終了する ClockSnapshot の生成
 * - 時刻がユーザによって意図的に操作された量を計算するサンプル
 * - TIME ライブラリの終了
 *
 * サンプルの実行途中、"Waiting 3 seconds." が表示されたタイミングで HOME ボタンを押して DevMenu もしくは HOME メニューを表示し、
 * そこで時刻やタイムゾーンの設定を変更することで、処理を時計操作検知、およびタイムゾーンの変更検知側へ分岐させることができます。
 *
 * nn::time::ClockSnapshot の値をセーブデータ等で永続化することで
 * アプリケーションのシャットダウンや本体の電源断をまたいだ操作量の計算を行うことができます。
 *
 * nn::Result を返す API において、事前条件を満たしていれば必ず成功するものは
 * @ref NN_ABORT_UNLESS_RESULT_SUCCESS を利用してハンドリングしています。
 *
 * このサンプルの実行結果を以下に示します。
 *
 * @verbinclude TimeCalculateUserSystemClockDifference_ExampleOutput.txt
 *
 */

#include <nn/time.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os/os_ThreadApi.h>

extern "C" void nnMain()
{
    // TIME ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );
    NN_LOG("nn::time::Initialize\n");

    NN_LOG("---- Sample of calculating nn::time::StandardUserSystemClock difference by user.\n");

    // 比較する元となるコンテキストの生成
    // AdjustableUserSystemClock, AdjustableNetworkSystemClock を利用している場合には ClockSnapshot::CreateWithAdjustableSystemClock で初期化します
    nn::time::ClockSnapshot oldContext;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&oldContext);


    // 上記の ClockSnapshot をセーブデータ等に保存して永続化し、次回起動時に読み出すことで、
    // ここでアプリケーションのシャットダウンや本体の電源断が発生したとしても、時計の操作量を計算することができます。
    // なお、ClockSnapshot は C++ 標準の TriviallyCopyable の要件を満たしており、 std::memcpy() が可能なクラスです。


    NN_LOG("Waiting 3 seconds.\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    // このスリープの間に、HOMEボタンを押して DevMenu もしくは HOME メニューを表示し、
    // そこで時刻やタイムゾーンの設定を変更することで、以降の処理を時計操作検知、およびタイムゾーンの変更検知側へ分岐させることができます。


    // 現在のコンテキストの生成
    nn::time::ClockSnapshot latestContext;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&latestContext);

    // CalculateStandardUserSystemClockDifferenceByUser() によって、
    // oldContext と latestContext の間に、
    // ユーザー時計( nn::time::StandardUserSystemClock )がユーザーによって意図的に操作された量を計算します。
    nn::TimeSpan differenceByUser = nn::time::CalculateStandardUserSystemClockDifferenceByUser(oldContext, latestContext);

    if(differenceByUser == nn::TimeSpan(0))
    {
        // 時計がユーザーによって意図的に操作されていない場合、 nn::TimeSpan(0) が返ります。
        NN_LOG("StandardUserSystemClock was not operated by user.\n");

        // CalculateStandardUserSystemClockDifferenceByUser() から nn::TimeSpan(0) が返った場合でも、
        // ユーザーによって時計が意図的に操作されていないことを示すだけであり、
        // StandardUserSystemClock::GetCurrentTime() で得られる時刻が大きく変化していないことは保証されません。
    }
    else
    {
        // 時計がユーザーによって意図的に操作されている場合ここに来ます。
        NN_LOG("StandardUserSystemClock was operated by user : %lld [sec]\n", differenceByUser.GetSeconds());
    }

    // CalculateStandardUserSystemClockDifferenceByUser() は、本体設定のタイムゾーン変更はチェックしません。
    // 必要であれば、ClockSnapshot::GetLocationName() 同士を比較してください。
    if(oldContext.GetLocationName() == latestContext.GetLocationName())
    {
        // 本体設定のタイムゾーンが変更されていない場合ここにきます。
        NN_LOG("LocationName was not changed.\n");
    }
    else
    {
        // 本体設定のタイムゾーンが変更された場合ここにきます。
        NN_LOG("LocationName was changed.\n");

        // 現地時刻へ適応される時差の変化量が必要であれば、以下のように計算可能です。
        // ただし、本体設定のタイムゾーンが変更されていなくても、夏時間による影響などで時差は変わることがある点に注意してください。
        auto oldUtcOffsetSeconds = oldContext.GetStandardUserSystemClockCalendarAdditionalInfo().timeZone.utcOffsetSeconds;
        auto latestUtcOffsetSeconds = latestContext.GetStandardUserSystemClockCalendarAdditionalInfo().timeZone.utcOffsetSeconds;
        auto diffOfUtcOffsetSeconds = latestUtcOffsetSeconds - oldUtcOffsetSeconds;
        NN_LOG("UTC offset seconds at local time : %d [sec] -> %d [sec], Difference : %d [sec]\n",
            oldUtcOffsetSeconds, latestUtcOffsetSeconds, diffOfUtcOffsetSeconds);
    }

    // TIME ライブラリの終了
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );
    NN_LOG("nn::time::Finalize\n");
}

